//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERA_TRANSFORMABLE2D_H)
    #define CANDERA_TRANSFORMABLE2D_H

#include <Candera/EngineBase/Common/CanderaObject.h>
#include <Candera/System/Mathematics/Vector2.h>
#include <Candera/System/Mathematics/Matrix3x2.h>
#include <Candera/System/Mathematics/MathDataTypes.h>

namespace Candera {

/** @addtogroup Core2D
 *  @{
 */

    /**
     * @brief Defines common methods for manipulating node and texture transformations in local coordinate space (object space).
     *        The transformation consists of following components: position, rotation, scale and a generic transform matrix.
     */
    class Transformable2D: public CanderaObject
    {
        FEATSTD_TYPEDEF_BASE(CanderaObject);

    public:
        /**
         * Constructor constructs a Transformable object with default values.
         */
        Transformable2D();

        /**
         * Copy-Constructor constructs a new Transformable object by copying the contents of the Transformable object given.
         * @param transformable defines the values to copy in this Transformable object.
         */
        Transformable2D(const Transformable2D& transformable);

        /**
         * Destructor
         */
        virtual ~Transformable2D() override;

        /**
         * Sets the local position of this object.
         * @param x The x coordinate
         * @param y The y coordinate
         */
        void SetPosition(Float x, Float y);
        /**
         * Sets the local position of this object.
         * @param position The x and y coordinate as vector
         */
        void SetPosition(const Vector2& position);
        /**
         * Gets the local position of this object.
         * @return The position as vector
         */
        const Vector2& GetPosition() const { return m_position; }

        /**
         * Sets the local rotation of this object.
         * @param rotation The local rotation angle in degrees
         */
        void SetRotation(Float rotation);
        /**
         * Gets the local rotation of this object.
         * @return The local rotation value
         */
        Float GetRotation() const;

        /**
         * Sets the local scale factor of this object.
         * @param xScale    The x scale
         * @param yScale    The y scale
         */
        void SetScale(Float xScale, Float yScale)
        {
            SetScale(Vector2(xScale, yScale));
        }

        /**
         * Sets the local scale factor of this object.
         * @param scale The x and y scale as vector
         */
        void SetScale(const Vector2& scale);

        /**
         * Gets the local scale factor of this object.
         * @return The scale as vector
         */
        const Vector2& GetScale() const;


        /**
         * Sets the pivot point for this object in local coordinate space. The pivot point 
         * is the point where rotation and scale is centered around. 
         * @param xPivot     The local x coordinate of the pivot point.
         * @param yPivot     The local y coordinate of the pivot point.
         */
        void SetPivotPoint(Float xPivot, Float yPivot)
        {
            SetPivotPoint(Vector2(xPivot, yPivot));
        }

        /**
         * Sets the pivot point for this object in local coordinate space. The pivot point
         * is the point where rotation and scale is centered around.
         * @param pivot     The pivot point in local coordinate space.
         */
        void SetPivotPoint(const Vector2& pivot);

        /**
         * Gets the pivot point for this object in local coordinate space. The pivot point
         * is the point where rotation and scale is centered around.
         * @return  The pivot point in local coordinate space.
         */
        const Vector2& GetPivotPoint() const;


        /**
         * Sets the pivot offset of this object. The pivot offset is the local
         * space translation applied to the object before applying the other transformations:
         * scale, rotation and translation.
         * @param xPivot    The pivot offset x-coordinate relative to the x-position.
         * @param yPivot    The pivot offset y-coordinate relative to the y-position.
         */
        CANDERA_DEPRECATED_3_6_0("The pivot offset has been replaced with the pivot point, which represents an actual point of reference for transformations instead of an offset. " \
            "Therefore, the respective pivot point methods should be used.",
            void SetPivotOffset(Float xPivot, Float yPivot));

        /**
         * Sets the pivot offset of this object.
         * The pivot offset is the local
         * space translation applied to the object before applying the other transformations:
         * scale, rotation and translation.
         * @param pivot The pivot offset as 2D vector.
         */
        CANDERA_DEPRECATED_3_6_0("The pivot offset has been replaced with the pivot point, which represents an actual point of reference for transformations instead of an offset. " \
            "Therefore, the respective pivot point methods should be used.",
            void SetPivotOffset(const Vector2& pivot));

        /**
         * Gets the pivot offset of this object.
         * @return The pivot offset
         */
        CANDERA_DEPRECATED_3_6_0("The pivot offset has been replaced with the pivot point, which represents an actual point of reference for transformations instead of an offset. " \
            "Therefore, the respective pivot point methods should be used.",
        const Vector2& GetPivotOffset() const);

        /**
         * Concatenates the given movement (translation) with current position component.
         * @param translation The translation that shall be applied.
         */
        void Translate(const Vector2& translation);

        /**
         * Multiplies the given scale factors with current scale component.
         * @param scale The scale that shall be applied.
         */
        void Scale(const Vector2& scale);

        /**
         * Concatenates the given rotation angles with current rotation component.
         * @param   rotation    Defines the rotation angle in degrees.
         */
        void Rotate(const Float rotation);

        /**
         * Concatenates the given movement (translation) with current position component.
         * In addition to Translate, this also adjusts the pivot offset such that the object
         * preserves its appearance (global transfomation matrix remains the same.)
         * @param translation The translation that shall be applied.
         */
        CANDERA_DEPRECATED_3_6_0("The pivot offset has been replaced with the pivot point, which represents an actual point of reference for transformations instead of an offset. " \
            "Therefore, the respective pivot point methods should be used.",
            void TranslatePivot(const Vector2& translation));

        /**
        * Translates the pivot point from its current position by the input Vector2 value.
        * The translation is applied locally and relatively to the object's rotation.
        * @param translation The translation that shall be applied.
        */
        void TranslatePivotPoint(const Vector2& translation);

        /**
         * Retrieves the local composite transformation matrix. The composite transformation matrix is a concatenation of the
         * position, rotation, scaling, and transformation matrix components.
         * Formal logic: CompositeTransform = Rotation * Scale * Position * Transform.
         * @return The composite transformation matrix
         */
        const Matrix3x2& GetCompositeTransform() const;

        /**
         *  Provides the parent for dynamic property scene graph inheritance.
         *  @param node The node, whose parent is sought.
         *  @return The parent of node if available, null otherwise.
         */
        CANDERA_SUPPRESS_LINT_FOR_SYMBOL(1511, Candera::Transformable2D::ParentProvider, CANDERA_LINT_REASON_EXPLICITHIDING)
        static const Candera::DynamicProperties::DynamicPropertyHost* ParentProvider(const Candera::DynamicProperties::DynamicPropertyHost* /*node*/) {
            return 0;
        }

        FEATSTD_RTTI_DECLARATION();

    protected:
        /**
          * Determine if the cache for local transform cache is valid.
          * @return true if cache is valid.
          */
        bool IsCompositeTransformCacheValid() const { return m_isCompositeTransformCacheValid; }

        /**
         * Called when one of the transforms (translation, rotation, scale, general transform) has changed.
         * Useful for updating dependent properties. Implementing classes will probably want to propagate the call up the hierarchy.
         */
        virtual void OnCompositeTransformChanged();

    private:
        friend class Transformable2DDynamicProperties;

        mutable bool m_isCompositeTransformCacheValid;
        Vector2 m_position;
        mutable Matrix3x2 m_compositeTransformCache;

        /* avoid assignment */
        Transformable2D& operator=(const Transformable2D& transformable);

        /**
         * Invalidate composite transform cache and notify descendant objects in order to update their world transform cache.
         */
        void InvalidateCompositeCacheAndNotifyDerivedClasses();

        CdaDynamicPropertiesDeclaration();
    };

 /** @} */ // end of Core2D

}   // namespace Candera

#endif  // CANDERA_TRANSFORMABLE2D_H
