//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERA_TEXTNODERENDERER_H)
#define CANDERA_TEXTNODERENDERER_H

#include <Candera/Environment.h>
#include <Candera/System/Rtti/Rtti.h>

namespace Candera {

    class TextNode2D;
    class RenderTarget2D;
    class Matrix3x2;

    namespace TextRendering {
        class TextRenderContext;
    }

    /** @addtogroup Core2D
    *  @{
    */

    /**
     *  @brief Base class for text to Image2D generation strategies.
     */
    class TextNodeRenderer
    {
        friend class TextNode2D;        
    public:                   
        /**
         * Prepare text for rendering.
         * @param textNode Node that contains data to be rendered.
         * @return True if text is prerendered and not needed durin Render anymore. False otherwise.
         */
        bool PreRender(const TextNode2D& textNode);

        /**
         *  Render text to the given render target.
         *  @param textNode Node that contains data to be rendered.
         *  @param renderTarget Target for the text rendering.
         *  @param localTransform Transformation matrix of the text.
         */
        void Render(TextNode2D& textNode, RenderTarget2D* renderTarget, const Matrix3x2& localTransform);
        
        /**
         * @return context that provides GlyphCacheAccess information for re-using it during successive measurements.
         */
        virtual const TextRendering::TextRenderContext* GetMeasureReferenceContext() const = 0;
       
        /**
        *  Clones this TextNodeRenderer by creating a new instance or returning 
        *  the static instance of this TextNodeRenderer with the same parameters 
        *  as this instance.        
        *  @return    The cloned TextNodeRenderer if successful, null otherwise.
        */
        virtual TextNodeRenderer* Clone() const = 0;
       
        /**
        *  Disposes this TextNodeRenderer
        *  Pure virtual function that must be overridden in each derived object.
        */
        virtual void Dispose() = 0;

        /**
         *  Uploads GPU resources that are used by a TextNodeRenderer derived class.
         *  @return  True if successful, false otherwise.
         */
        virtual bool UploadSelf() { return true; }

        /**
         *  Unloads GPU resources that are used by a TextNodeRenderer derived class.
         *  @return  True if successful, false otherwise.
         */
        virtual bool UnloadSelf() { return true; }

        /**
         *  Indicates if the text is valid after unloading the TextNodeRenderer derived class.
         *  @return  True if the text is valid after unloading the TextNodeRenderer derived class, false otherwise.
         */
        virtual bool IsTextValidAfterUnload() { return true; }

#ifdef CANDERA_DEPRECATED_3_3_0
        /**
        * Destructor
        */
        virtual ~TextNodeRenderer();
#else
#error  Make destructor protected.
#endif

        FEATSTD_RTTI_DECLARATION();
    
    private:
        /**
         * Do optional prerendering now (e.g. render to an offscreen surface).
         * PreRenderInternal is always called before RenderInternal if there is any change of text properties in TextNode.
         * Return true if text properties are not required for actual RenderInternal processing, as TextNode cand dispose
         * data. In this case PreRenderInternal will only be called again when there is a change in the text properties.
         * If returns false, PreRenderInternal will be called before calling next RenderInternal once again.
         */
        virtual bool PreRenderInternal(const TextNode2D& textNode) = 0;

        /**
         * Do rendering of text or prerendered data now.
         */
        virtual void RenderInternal(TextNode2D& textNode, RenderTarget2D* renderTarget, const Matrix3x2& localTransform) = 0;

    };

    inline bool TextNodeRenderer::PreRender(const TextNode2D& textNode)
    { return PreRenderInternal(textNode); }

    inline void TextNodeRenderer::Render(TextNode2D& textNode, RenderTarget2D* renderTarget, const Matrix3x2& localTransform)
    { RenderInternal(textNode, renderTarget, localTransform); }

    /** @} */ // end of Core2D

}   // namespace Candera

#endif  // CANDERA_TEXTNODERENDERER_H
