//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERA_TEXTNODE2DLAYOUTER_H)
#define CANDERA_TEXTNODE2DLAYOUTER_H

#include <Candera/EngineBase/Layout/Layouter.h>
#include <Candera/TextEngine/Async/TextRenderArgs.h>

namespace Candera {

    class TextNode2D;

    namespace TextRendering {
        class LayoutingOptions;
        class SharedStyle;
        class TextProperties;
        class TextRect;
        class TextSize;
    }
    /** @addtogroup Layout2D
     *  @{
     */

    /**
     * @brief Base Layouter for TextNode2D nodes.
     *
     * The TextNode2DLayouter arranges and truncates the text associated of a TextNode2D.
     */
    class TextNode2DLayouter : public Layouter
    {
    public:
#if defined(CANDERA_LAYOUT_ENABLED)

        /**
         *  Trimming defines the behavior to employ when content overflows the content area.
         */
        enum Trimming
        {
            None,       ///< Text is cut off.
            Ellipsis,    ///< Text is trimmed and an ellipsis (...) is drawn in place of remaining text.
            CustomText ///< Text is truncated by using a custom text.
        };

        /**
        * Text alignment.
        */
        enum TextAlignment
        {
            Auto,      ///< TextAlignment inhierits layout alignment.
            Left,      ///< Text is aligned along the left margin.
            Center,    ///< Text is not aligned, but an equal amount of padding is added relative to the right and left margin.
            Right,     ///< Text is aligned along the right margin.
            Justified  ///< Text is left aligned, with word spacing increased in a way that the text fills the entire available width for non end of paragraph lines.
        };

        virtual ~TextNode2DLayouter() {}

        // overrides Layouter::OnMeasure
        virtual Vector2 OnMeasure(const AbstractNodePointer& node, const Vector2& clientArea);

        // overrides Layouter::OnArrange
        virtual void OnArrange(const AbstractNodePointer& node, const Rectangle& clientArea);

        // overrides Layouter::GetComputedLayoutRectangle
        CANDERA_DEPRECATED_3_3_0("Use Node2D::GetComputedLayoutRectangle instead.",
            virtual Rectangle GetComputedLayoutRectangle(const Node2D& node) const);

        /**
         * @return The default TextNode2DLayouter instance.
         */
        static TextNode2DLayouter& GetDefault();

        /**
         * @param node TextNode2D for which the word wrapping configuration is requested.
         * @return The word wrapping configuration for the given TextNode2D.
         */
        static bool IsWordWrapEnabled(const Node2D& node) { return node.GetValue(CdaDynamicPropertyInstance(WordWrapEnabled)); }

        /**
         * Sets the word wrapping configuration.
         * If word wrapping is enabled then the lines that exceed the client area will be split
         * during arrangement into multiple lines. The lines will be split at word boundaries.
         * @param node       TextNode2D for which the word wrapping is configured.
         * @param enabled    The word wrapping enabled flag (disabled by default).
         */
        static void SetWordWrapEnabled(TextNode2D& node, bool enabled);
        
        /**
         * @param node TextNode2D for which the multi line configuration is requested.
         * @return The multi line configuration for the given TextNode2D.
         */
        static bool IsMultiLineEnabled(const Node2D& node) { return node.GetValue(CdaDynamicPropertyInstance(MultiLineEnabled)); }

        /**
         * Sets the multi line configuration.
         * If multi line is disabled then the new line control characters will be ignored and text
         * will always be rendered in one line.
         * @param node       TextNode2D for which the multi line is configured.
         * @param enabled    The multi line enabled flag (enabled by default).
         */
        static void SetMultiLineEnabled(TextNode2D& node, bool enabled);

        /**
         * @param node TextNode2D for which the TextAlignment is requested.
         * @return The TextAlignment configured for the given TextNode2D.
         */
        static TextAlignment GetTextAlignment(const TextNode2D& node);

        /**
         * Sets the TextAlignment type.
         * The TextAlignment type, combined with the culture text direction, is used to
         * define how text will be horizontally aligned within its layout area.
         * @param node             TextNode2D for which the TextAlignment is set.
         * @param textAlignment    The TextAlignment value.
         * @see TextAlignment
         */
        static void SetTextAlignment(TextNode2D& node, TextAlignment textAlignment);

         /**
          * @param node TextNode2D for which the Trimming is requested.
          * @return The Trimming configured for the given TextNode2D.
          */
        static Trimming GetTrimming(const TextNode2D& node);

        /**
         * Sets the trimming type.
         * @param node     TextNode2D for which the Trimming is set.
         * @param trimming The Trimming value.
         * @see Trimming
         */
        static void SetTrimming(TextNode2D& node, Trimming trimming);

        /**
          * @param node TextNode2D for which the line spacing is requested.
          * @return The line spacing configured for the given TextNode2D.
         */
        static TextRendering::PixelSize GetLineSpacing(const TextNode2D& node);
        
        /**
         * Sets the line spacing value.
         * The line spacing is a metric usually defined by the Font configured for text rendering. To use
         * the default value of the Font, set 0 as the line spacing property. Any other positive value 
         * will override the default value with an object space fixed height (independent on the font height).
         * @param node         TextNode2D for which the line spacing is set.
         * @param lineSpacing  Line spacing value.
         */
        static void SetLineSpacing(TextNode2D& node, TextRendering::PixelSize lineSpacing);

        static void SetTrimmingText(TextNode2D& node, FeatStd::String const& text);

        static FeatStd::String GetTrimmingText(const TextNode2D& textNode);

        /*
         * @param textNode TextNode2D for which the LayoutingOptions is requested.
         * @return The configured TextRendering::LayoutingOptions, based on set properties.
         */
        static TextRendering::LayoutingOptions GetLayoutingOptions(const TextNode2D& textNode);

    protected:
        static Candera::HorizontalAlignment GetLanguageSensitiveTextAlignment(const TextNode2D& textNode);

        static bool PreprocessTextInternal(TextRendering::TextRenderArgs::SharedPointer renderArgs);

        bool IsLayoutValid(TextNode2D& node) const;

        void ValidateLayout(TextNode2D& node) const;

    private:
        static bool BoolFalse() { return false; }
        static bool BoolTrue() { return true; }
        static const TextAlignment& TextAlignmentDefault() {
            static TextAlignment defaultValue = Auto;
            return defaultValue;
        }
        static const Trimming& TrimmingDefault() {
            static Trimming defaultValue = None;
            return defaultValue;
        }
        static const FeatStd::String& TrimTextDefault()
        {
            static FeatStd::String defaultValue = "...";
            return defaultValue;
        }

        static void OnPropertyChanged(DynamicPropertyHost* obj, const DynamicProperties::ValueChangedArgs<bool>& /*args*/) { InvalidateLayout(obj); }
        static void OnPropertyChanged(DynamicPropertyHost* obj, const DynamicProperties::ValueChangedArgs<TextAlignment>& /*args*/) { InvalidateLayout(obj); }
        static void OnPropertyChanged(DynamicPropertyHost* obj, const DynamicProperties::ValueChangedArgs<Trimming>& /*args*/) { InvalidateLayout(obj); }
        static void OnPropertyChanged(DynamicPropertyHost* obj, const DynamicProperties::ValueChangedArgs<TextRendering::PixelSize>& /*args*/) { InvalidateLayout(obj); }
        static void OnPropertyChanged(DynamicPropertyHost* obj, const DynamicProperties::ValueChangedArgs<FeatStd::String>& /*args*/) { InvalidateLayout(obj); }

        /// @cond Doxygen ignore - start
        CdaDynamicProperties(Candera::TextNode2DLayouter, Candera::Layouter);
            CdaDynamicProperty(WordWrapEnabled, bool);
                CdaDynamicPropertyValueChangedCb(&TextNode2DLayouter::OnPropertyChanged);
                CdaDynamicPropertyDefaultValue(BoolFalse());
            CdaDynamicPropertyEnd();
            CdaDynamicProperty(MultiLineEnabled, bool);
                CdaDynamicPropertyValueChangedCb(&TextNode2DLayouter::OnPropertyChanged);
                CdaDynamicPropertyDefaultValue(BoolTrue());
                CdaDynamicPropertyEnd();
            CdaDynamicPropertyUnregistered(TextAlignment, TextAlignment);
                CdaDynamicPropertyValueChangedCb(&TextNode2DLayouter::OnPropertyChanged);
                CdaDynamicPropertyDefaultValue(TextAlignmentDefault());
                CdaDynamicPropertyDescription("The horizontal text alignment, using the culture text direction, "
                    "is used to define how text will be horizontally aligned within its layout area.")
                CdaDynamicPropertyCategory("Item")
            CdaDynamicPropertyEnd();
            CdaDynamicPropertyUnregistered(Trimming, Trimming);
                CdaDynamicPropertyValueChangedCb(&TextNode2DLayouter::OnPropertyChanged);
                CdaDynamicPropertyDefaultValue(TrimmingDefault());
                CdaDynamicPropertyDescription("Trimming defines the behavior to employ when content overflows "
                    "the content area.")
                CdaDynamicPropertyCategory("Item")
            CdaDynamicPropertyEnd();
            CdaDynamicPropertyUnregistered(TrimmingText, FeatStd::String);
                CdaDynamicPropertyValueChangedCb(&Candera::TextNode2DLayouter::OnPropertyChanged);
                CdaDynamicPropertyDefaultValue(TrimTextDefault());
                CdaDynamicPropertyDescription("Trimming text defines the text which replaces content overflowing "
                    "the content area.")
                CdaDynamicPropertyCategory("Item")
            CdaDynamicPropertyEnd();
            CdaDynamicProperty(LineSpacing, TextRendering::PixelSize);
                CdaDynamicPropertyValueChangedCb(&TextNode2DLayouter::OnPropertyChanged);
                CdaDynamicPropertyDefaultValue(0);
                CdaDynamicPropertyDescription("Defines the distance between the baselines of successive "
                    "lines of type.")
                CdaDynamicPropertyCategory("Item")
            CdaDynamicPropertyEnd();
        CdaDynamicPropertiesEnd();
        /// @endcond Doxygen ignore - end

        virtual TextRendering::TextSize OnMeasureText(TextNode2D& node, const TextRendering::TextSize& clientArea) = 0;

        virtual TextRendering::TextRect OnArrangeText(TextNode2D& node, const TextRendering::TextRect& clientArea) = 0;
#endif
FEATSTD_SUPPRESS_DEPRECATION_WARNING_BEGIN()
    };
FEATSTD_SUPPRESS_DEPRECATION_WARNING_END()
    /** @} */ // end of Layout2D
}   // namespace Candera

#endif  // CANDERA_TEXTNODE2DLAYOUTER_H
