//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "SurfaceTextNodeRenderer.h"

#include <CanderaPlatform/Device/Common/Base/DevicePackageInterface.h>
#include <CanderaPlatform/Device/Common/Base/GraphicDeviceUnit.h>
#include <CanderaPlatform/Device/Common/Base/GraphicDeviceUnitAttributes.h>
#include <CanderaPlatform/Device/Common/Base/ImageSource2D.h>
#include <CanderaPlatform/Device/Common/Base/RenderTarget2D.h>
#include <Candera/Engine2D/Core/TextNodeRenderer/TextNodeRendererTools.h>
#include <Candera/Engine2D/Core/TextNodeRenderer/SurfaceTextRenderContext.h>
#include <Candera/Engine2D/Core/TextNode2D.h>
#include <Candera/Engine2D/Core/Camera2D.h>

namespace Candera {
    using namespace Candera::TextRendering;
    using namespace Candera::Internal;

    /******************************************************************************
     *  Constructor
     ******************************************************************************/
    SurfaceTextNodeRenderer::SurfaceTextNodeRenderer() :
        m_mustBeDisposed(false),
        m_gdu()
    {
        m_proxyImage = ProxyImage2D::Create();
        if (m_proxyImage != 0) {
            m_proxyImage->SetSyncEnabled(false);
        }
    }

    /******************************************************************************
     *  Copy Constructor
     ******************************************************************************/
    SurfaceTextNodeRenderer::SurfaceTextNodeRenderer
        (const SurfaceTextNodeRenderer& /*surfaceTextNodeRenderer*/) :
        m_mustBeDisposed(true),
        m_gdu(0)
    {        
        m_proxyImage = ProxyImage2D::Create();
        if (m_proxyImage != 0) {
            m_proxyImage->SetSyncEnabled(false);
        }
         
    }


    /******************************************************************************
     *  User defined constructor
     ******************************************************************************/
    SurfaceTextNodeRenderer::SurfaceTextNodeRenderer(const Int& /*dummyInt*/) :
        m_mustBeDisposed(true),
        m_gdu(0)
    {        
        m_proxyImage = ProxyImage2D::Create();
        if (m_proxyImage != 0) {
            m_proxyImage->SetSyncEnabled(false);
        }

    }

    /******************************************************************************
     *  Destructor
     ******************************************************************************/
    SurfaceTextNodeRenderer::~SurfaceTextNodeRenderer()
    {
        Finalize();
    }

    /******************************************************************************
    *  Update
    ******************************************************************************/
    bool SurfaceTextNodeRenderer::PreRenderInternal(const TextNode2D& textNode)
    {
        if (textNode.GetText().IsEmpty() || textNode.GetStyle().PointsToNull()) {
            Finalize();
            return false;
        }

        const TextRect& boundRect = textNode.GetBoundingTextRectangle();
        if (boundRect.IsEmpty()) {
            Finalize();
            return true;
        }

        const UInt32 textWidth = boundRect.GetWidth();
        const UInt32 textHeight = boundRect.GetHeight();

        bool updateImageSize = (m_proxyImage == 0) ||
            (m_proxyImage->GetWidth() < textWidth) ||
            (m_proxyImage->GetHeight() < textHeight) ||
            (m_proxyImage->GetWidth() * m_proxyImage->GetWidth() > (2U * textWidth * textHeight));

        if (updateImageSize) {
            Finalize();

            //TO DO: 2D objects should be aware of which display they are uploaded to.
            const GraphicDeviceUnitTypeHandle *type;
            Int count;
            type = DevicePackageDescriptor::GetUnitTypes(
                DevicePackageDescriptor::OffscreenTarget2D,
                0,
                count);

            if ((type == 0) || (count < 1)) {
                return false;
            }

            const Int gduConfiguration[] = {
                static_cast<Int>(GduWidthAttr), static_cast<Int>(textWidth),
                static_cast<Int>(GduHeightAttr), static_cast<Int>(textHeight),
                static_cast<Int>(GduRedSizeAttr), 0,
                static_cast<Int>(GduGreenSizeAttr), 0,
                static_cast<Int>(GduBlueSizeAttr), 0,
#ifdef CANDERA_4BIT_GLYPH_CACHE_ENABLED
                static_cast<Int>(GduAlphaSizeAttr), 4,
#else
                static_cast<Int>(GduAlphaSizeAttr), 8,
#endif
                static_cast<Int>(GduAttributeListEnd)};

            m_gdu = DevicePackageInterface::CreateGraphicDeviceUnit(type[0], &gduConfiguration[0]);
            if (m_gdu == 0) {
                return false;
            }

            Camera2D* camera = Renderer2D::GetActiveCamera();
            bool isOwnerAvailable = (camera != 0)
                && (camera->GetRenderTarget() != 0)
                && (camera->GetRenderTarget()->GetGraphicDeviceUnit() != 0)
                && (camera->GetRenderTarget()->GetGraphicDeviceUnit()->ToRenderTarget3D() != 0);

            if (isOwnerAvailable) {
                GraphicDeviceUnitOwnerAccess::SetGraphicDeviceUnitOwner(m_gdu, camera->GetRenderTarget()->GetGraphicDeviceUnit()->ToRenderTarget3D());
            }

            bool uploadAndValidate =
                m_gdu->Upload() &&
                (m_gdu->ToImageSource2D() != 0) &&
                (m_gdu->ToRenderTarget2D() != 0);

            if (!uploadAndValidate) {
                Finalize();
                return false;
            }

            m_proxyImage = ProxyImage2D::Create();
            if (m_proxyImage == 0) {
                Finalize();
                return true;
            }
            //m_proxyImage->SetSyncEnabled(false);
            m_proxyImage->SetImageSource(m_gdu->ToImageSource2D());
        }

        SurfaceTextNodeTextRenderContext<> textRenderContext;
        textRenderContext.Set2DContext(m_gdu->ToRenderTarget2D()->Get2DContextHandle());
        textRenderContext.SetClipRect(TextRect(TextCoordinate(), boundRect.GetSize()));
        m_gdu->ToRenderTarget2D()->BeginDraw();
        ContextHandle2D output = m_gdu->ToRenderTarget2D()->Get2DContextHandle();

        if (!Renderer2D::Clear(output, 0.0F, 0.0F, 0.0F, 0.0F)) {
            return false;
        }

        if (!RenderDevice2D::SetBlendOperation(output,
            RenderDevice2D::SourceSurface,
            RenderDevice2D::Add,
            RenderDevice2D::One,
            RenderDevice2D::One)) {
            return false;
        }

        if (!TextNodeRendererTools::RenderText(textRenderContext, textNode, TextCoordinate(-boundRect.GetLeft(), -boundRect.GetTop()))) {
            return false;
        }

        m_gdu->ToRenderTarget2D()->EndDraw();
        m_gdu->ToRenderTarget2D()->SwapBuffers();
        return true;
    }
    /******************************************************************************
    *  Render
    ******************************************************************************/
    void SurfaceTextNodeRenderer::RenderInternal(TextNode2D& textNode, RenderTarget2D* renderTarget, const Matrix3x2& localTransform)
    {
        if (m_proxyImage != 0) {
            const TextRect boundRect = textNode.GetBoundingTextRectangle();
            const Vector2 position(static_cast<Float>(boundRect.GetLeft()), static_cast<Float>(boundRect.GetTop()));
            TextNodeRendererTools::RenderImage(*m_proxyImage,
                textNode,
                renderTarget,
                localTransform,
                position);
        }
    }


    void SurfaceTextNodeRenderer::Finalize()
    {
        if (m_gdu != 0) {
            DevicePackageInterface::DestroyGraphicDeviceUnit(m_gdu);
            m_gdu = 0;
        }
        m_proxyImage.Release();
    }

    /******************************************************************************
     *  Create
     ******************************************************************************/
    SurfaceTextNodeRenderer* SurfaceTextNodeRenderer::Create()
    {
        return FEATSTD_NEW(SurfaceTextNodeRenderer)(0);
    }

    /******************************************************************************
     *  Clone
     ******************************************************************************/
    SurfaceTextNodeRenderer* SurfaceTextNodeRenderer::Clone() const
    {
        return FEATSTD_NEW(SurfaceTextNodeRenderer)(*this);
    }

    /******************************************************************************
     *  Dispose
     ******************************************************************************/
    void SurfaceTextNodeRenderer::Dispose()
    {
        if (m_mustBeDisposed) {            
            FEATSTD_DELETE(this);
        }
    }

    bool SurfaceTextNodeRenderer::UnloadSelf()
    {
        Finalize();
        return true;
    }

    FEATSTD_RTTI_DEFINITION(SurfaceTextNodeRenderer, TextNodeRenderer)
}   // namespace Candera
