//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERA_DEFAULTTEXTNODE2DLAYOUTER_H)
#define CANDERA_DEFAULTTEXTNODE2DLAYOUTER_H

#include <Candera/Engine2D/Core/TextNodeRenderer/TextNode2DLayouter.h>
#include <Candera/TextEngine/GlyphBitmap.h>
#include <Candera/TextEngine/TextSize.h>
#include <Candera/TextEngine/TextRenderContexts/TruncationToGlyphIteratorContext.h>

namespace Candera {

    class TextNode2D;
    class RenderTarget2D;
    class Image2D;

#if defined(CANDERA_LAYOUT_ENABLED)
    /** @addtogroup Layout2D
     *  @{
     */

    /**
    * @brief Default layouter for TextNode.
    *
    * The DefaultTextNode2DLayouter arranges and truncates the text associated to a TextNode2D.
    */
    class DefaultTextNode2DLayouter : public TextNode2DLayouter
    {
    public:
        virtual ~DefaultTextNode2DLayouter() {}
        /**
         *  Gets the instance of this class.
         *  @return A pointer to the instance of this class.
         */
        static DefaultTextNode2DLayouter& GetInstance();

        // overrides Layouter::Clone
        virtual Layouter* Clone() const override;

        // overrides Layouter::Dispose
        virtual void Dispose() override;

    protected:
        /**
        * Triggers the layout and glyph generation process of a TextNode2D.
        * @param textNode is the text which will be measured.
        * @param clientArea is the maximum possible area.
        * @param[out] layoutSize is the actual used area of the text.
        * @return is true in case of the text could be successfully rendered. Otherwise it is false.
        */
        bool MeasureText(TextNode2D& textNode, const TextRendering::TextSize& clientArea, TextRendering::TextSize& layoutSize) const;

    private:
        static const TextRendering::TextSize& GetDefaultTextSize();
       
        static const TextRendering::TruncationToGlyphIteratorContext::SharedPointer GetTruncateContext(Node2D const& textNode) { return textNode.GetValue(CdaDynamicPropertyInstance(TruncateContext)); }
        static bool SetTruncateContext(Node2D& textNode, TextRendering::TruncationToGlyphIteratorContext::SharedPointer value) { return textNode.SetValue(CdaDynamicPropertyInstance(TruncateContext), value); }


        static UInt16 GetStyleVersion(const DynamicPropertyHost& host) { return host.GetValue(CdaDynamicPropertyInstance(StyleVersion)); }
        static bool SetStyleVersion(DynamicPropertyHost& host, UInt16 value) { return host.SetValue(CdaDynamicPropertyInstance(StyleVersion), value); }

        static TextRendering::PixelPosition GetTruncateTextWith(const DynamicPropertyHost& host) { return host.GetValue(CdaDynamicPropertyInstance(TruncateTextWith)); }
        static bool SetTruncateTextWith(DynamicPropertyHost& host, TextRendering::PixelPosition value) { return host.SetValue(CdaDynamicPropertyInstance(TruncateTextWith), value); }

        static TextRendering::TextSize GetClientArea(const DynamicPropertyHost& host) { return host.GetValue(CdaDynamicPropertyInstance(ClientArea)); }
        static bool SetClientArea(DynamicPropertyHost& host, TextRendering::TextSize value) { return host.SetValue(CdaDynamicPropertyInstance(ClientArea), value); }

        static TextRendering::TextSize GetPreviousInputClientArea(const DynamicPropertyHost& host) { return host.GetValue(CdaDynamicPropertyInstance(PreviousInputClientArea)); }
        static bool SetPreviousInputClientArea(DynamicPropertyHost& host, TextRendering::TextSize value) { return host.SetValue(CdaDynamicPropertyInstance(PreviousInputClientArea), value); }

        static LayoutAlignment::LayoutDirection::Enum GetPreviousLayoutDirection(const DynamicPropertyHost& host) { return host.GetValue(CdaDynamicPropertyInstance(PreviousLayoutDirection)); }
        static bool SetPreviousLayoutDirection(DynamicPropertyHost& host, LayoutAlignment::LayoutDirection::Enum value) { return host.SetValue(CdaDynamicPropertyInstance(PreviousLayoutDirection), value); }

        static TextRendering::TextSize GetLayoutingSize(const DynamicPropertyHost& host) { return host.GetValue(CdaDynamicPropertyInstance(LayoutingSize)); }
        static bool SetLayoutingSize(DynamicPropertyHost& host, TextRendering::TextSize value) { return host.SetValue(CdaDynamicPropertyInstance(LayoutingSize), value); }

        /// @cond Doxygen ignore - start
        CdaDynamicProperties(DefaultTextNode2DLayouter, TextNode2DLayouter);
        
            CdaDynamicPropertyUnregistered(TruncateContext, TextRendering::TruncationToGlyphIteratorContext::SharedPointer);
                CdaDynamicPropertyDefaultValue(TextRendering::TruncationToGlyphIteratorContext::GetDefault());
            CdaDynamicPropertyEnd();
            
            CdaDynamicPropertyUnregistered(StyleVersion, UInt16);
            CdaDynamicPropertyEnd();

            CdaDynamicPropertyUnregistered(TruncateTextWith, TextRendering::PixelPosition);
            CdaDynamicPropertyEnd();

            CdaDynamicPropertyUnregistered(ClientArea, TextRendering::TextSize);
                CdaDynamicPropertyDefaultValue(GetDefaultTextSize());
            CdaDynamicPropertyEnd();

            CdaDynamicPropertyUnregistered(PreviousInputClientArea, TextRendering::TextSize);
            CdaDynamicPropertyDefaultValue(GetDefaultTextSize());
            CdaDynamicPropertyEnd();

            CdaDynamicPropertyUnregistered(PreviousLayoutDirection, LayoutAlignment::LayoutDirection::Enum);
            CdaDynamicPropertyDefaultValue(LayoutAlignment::LayoutDirection::LeftToRightDirection);
            CdaDynamicPropertyEnd();


            CdaDynamicPropertyUnregistered(LayoutingSize, TextRendering::TextSize);
                CdaDynamicPropertyDefaultValue(GetDefaultTextSize());
            CdaDynamicPropertyEnd();

        CdaDynamicPropertiesEnd();
        /// @endcond Doxygen ignore - end

        FEATSTD_MAKE_CLASS_STATIC(DefaultTextNode2DLayouter);

        // overrides Layouter::OnMeasure
        virtual TextRendering::TextSize OnMeasureText(TextNode2D& node, const TextRendering::TextSize& clientArea);

        // overrides Layouter::OnArrange
        virtual TextRendering::TextRect OnArrangeText(TextNode2D& node, const TextRendering::TextRect& clientArea);
    };

    inline DefaultTextNode2DLayouter::DefaultTextNode2DLayouter() : TextNode2DLayouter() {}
    inline Layouter* DefaultTextNode2DLayouter::Clone() const { return &GetInstance(); }
    inline void DefaultTextNode2DLayouter::Dispose() {}

#endif

    /** @} */ // end of Layout2D

}   // namespace Candera

#endif  // CANDERA_DEFAULTTEXTNODE2DLAYOUTER_H
