//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "BitmapTextNodeRenderer.h"

#include <Candera/System/Diagnostics/Log.h>
#include <Candera/EngineBase/Common/Bitmap.h>
#include <Candera/EngineBase/Common/BitmapTextRenderContext.h>
#include <Candera/Engine2D/Core/TextNodeRenderer/TextNodeRendererTools.h>
#include <Candera/Engine2D/Core/TextNode2D.h>
#include <Candera/EngineBase/Common/BitmapGlyphCacheAccess.h>
#include <FeatStd/Util/StaticObject.h>

namespace Candera {
    FEATSTD_LOG_SET_REALM(Diagnostics::LogRealm::CanderaPlatformDevice);
    using namespace TextRendering;

    /******************************************************************************
     * Constructor
     ******************************************************************************/
    BitmapTextNodeRenderer::BitmapTextNodeRenderer():
        m_mustBeDisposed(false), 
        m_bitmapImage(0)
    {
    }    

    BitmapTextNodeRenderer::BitmapTextNodeRenderer(
        const BitmapTextNodeRenderer& /*BitmapTextNodeRenderer*/):        
        m_mustBeDisposed(true),
        m_bitmapImage(0)
    {            
    }

    BitmapTextNodeRenderer::BitmapTextNodeRenderer(const Int& /*dummyInt*/) :
        m_mustBeDisposed(true),
        m_bitmapImage(0)
    {
    }

    /******************************************************************************
     *  Create
     ******************************************************************************/
    BitmapTextNodeRenderer* BitmapTextNodeRenderer::Create()
    {        
        return FEATSTD_NEW(BitmapTextNodeRenderer)(0);
    }

    BitmapTextNodeRenderer::~BitmapTextNodeRenderer()
    {
    }

    bool BitmapTextNodeRenderer::PreRenderInternal(const TextNode2D& textNode)
    {
        const TextRect& boundRect = textNode.GetBoundingTextRectangle();
        if (boundRect.IsEmpty()) {
            m_bitmapImage.Release();
            return true;
        }

        const UInt16 textWidth = boundRect.GetWidth();
        const UInt16 textHeight = boundRect.GetHeight();
        const UInt32 textSize = static_cast<UInt32>(textWidth) * textHeight;

        bool updateImageSize = (m_bitmapImage == 0) ||
            (m_bitmapImage->GetWidth() < textWidth) ||
            (m_bitmapImage->GetHeight() < textHeight) ||
            (m_bitmapImage->GetWidth() * m_bitmapImage->GetWidth() > (2U * textSize));

        if (updateImageSize) {
            UInt8* pixels = FEATSTD_NEW_ARRAY(UInt8, textSize);
            if (pixels == 0) {
                return false;
            }

            Bitmap::SharedPointer bitmap = Bitmap::Create(
                textWidth, textHeight,
                Bitmap::AlphaUnsignedBytePixelFormat, Bitmap::PackAlignment1,
                pixels, Bitmap::Disposer::Dispose, textSize);

            if (bitmap == 0) {
                FEATSTD_DELETE_ARRAY(pixels);
                return false;
            }

            m_bitmapImage = BitmapImage2D::Create();
            if (m_bitmapImage.PointsToNull() || (!m_bitmapImage->SetBitmap(bitmap))) {
                return false;
            }
        }

        //unload the old bitmap
        if (m_bitmapImage->IsUploaded()) {
            bool unloadResult = m_bitmapImage->Unload(DeviceObject2D::NoHint);
            if ((!unloadResult) || m_bitmapImage->IsUploaded()) {
                return false;
            }
        }
        //update bitmap
        const Bitmap::SharedPointer& bitmap = m_bitmapImage->GetMutableBitmap();
        Bitmap::PixelsResource pixelsResource(bitmap->GetPixelsResourceHandle());
        UInt8* pixels = pixelsResource.GetMutableData();
        if (pixels == 0) {
            return false;
        }

        MemoryPlatform::Set(pixels, 0U, static_cast<SizeType>(bitmap->GetWidth())* bitmap->GetHeight());

        BitmapTextRenderContext textRenderContext(GetMeasureReferenceContext());
        if (!textRenderContext.SetBitmap(bitmap)) {
            return false;
        }
        textRenderContext.SetPenColor(Color(1.F, 1.F, 1.F, 1.F));
        if (!Candera::Internal::TextNodeRendererTools::RenderText(textRenderContext,
                                               textNode,
                                               TextCoordinate(-boundRect.GetLeft(),
                                                              -boundRect.GetTop()))) {
            return false;
        }
        return m_bitmapImage->Upload(DeviceObject2D::NoHint);
    }

    class GlyphCacheAccessTextRenderContext : public TextRendering::TextRenderContext
    {
    public:
        GlyphCacheAccessTextRenderContext()
        {
        }

        virtual const TextRect& GetClipRect() const
        {
            return m_textRect;
        }

        virtual void Blit(Int16 , Int16 , const GlyphBitmap &)
        {
        }

        virtual TextRendering::GlyphCacheAccess* GetGlyphCacheAccess() const {
            return &BitmapGlyphCacheAccess::GetInstance();
        }
        TextRect m_textRect;
    };


    static const TextRendering::TextRenderContext& BitmapTextNodeRendererGetMeasureReferenceContext()
    {
        FEATSTD_SYNCED_STATIC_OBJECT(GlyphCacheAccessTextRenderContext, s_glyphCacheAccessTextRenderContext);
        return s_glyphCacheAccessTextRenderContext;
    }

    const TextRendering::TextRenderContext* BitmapTextNodeRenderer::GetMeasureReferenceContext() const
    {
        return &BitmapTextNodeRendererGetMeasureReferenceContext();
    }

    void BitmapTextNodeRenderer::RenderInternal(TextNode2D& textNode,
                                        RenderTarget2D* renderTarget,
                                        const Matrix3x2& localTransform)
    {
        if (m_bitmapImage != 0) {
            const TextRect boundRect = textNode.GetBoundingTextRectangle();
            const Vector2 postion(static_cast<Float>(boundRect.GetLeft()), static_cast<Float>(boundRect.GetTop()));
            Candera::Internal::TextNodeRendererTools::RenderImage(*m_bitmapImage,
                                               textNode,
                                               renderTarget,
                                               localTransform,
                                               postion);
        }
    }

    bool BitmapTextNodeRenderer::UploadSelf()
    {
        if (m_bitmapImage != 0) {
            // Do not upload more than once, otherwise Node2D::UploadSelf() and on-demand
            // upload by PreRenderInternal() mess up the upload reference count.
            if (!m_bitmapImage->IsUploaded()) {
                return m_bitmapImage->Upload();
            }
        }

        return true;
    }

    bool BitmapTextNodeRenderer::UnloadSelf()
    {
        if (m_bitmapImage != 0) {
            // Only unload if it is uploaded. Regular Upload/Unload cycle is broken by
            // PreRenderInteral() on-demand upload.
            if (m_bitmapImage->IsUploaded()) {
                return m_bitmapImage->Unload();
            }
        }

        return true;
    }

    /******************************************************************************
     *  Clone
     ******************************************************************************/
    BitmapTextNodeRenderer* BitmapTextNodeRenderer::Clone() const
    {
        return FEATSTD_NEW(BitmapTextNodeRenderer)(*this);
    }

    /******************************************************************************
     *  Dispose
     ******************************************************************************/
    void BitmapTextNodeRenderer::Dispose()
    {
        if (m_mustBeDisposed) {            
            FEATSTD_DELETE(this);
        }        
    }
    /******************************************************************************
     *  RTTI
     ******************************************************************************/
    FEATSTD_RTTI_DEFINITION(BitmapTextNodeRenderer, TextNodeRenderer)
}   // namespace Candera
