//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "TextNode2D.h"

#include <FeatStd/Event/EventSource.h>

#include <Candera/System/Mathematics/Rectangle.h>
#include <CanderaPlatform/Device/Common/Effects/BitmapBrush.h>
#include <Candera/TextEngine/Async/TextValidationEventSources.h>
#include <Candera/TextEngine/LayoutingOptions.h>
#include <Candera/TextEngine/TextRenderer.h>
#include <Candera/TextEngine/Internal/StyleTools.h>
#include <Candera/TextEngine/ShapingOptions.h>
#include <Candera/TextEngine/AlignTextMeasureContext.h>
#include <Candera/TextEngine/GlyphTextMeasureContext.h>
#include <Candera/TextEngine/TextProperties.h>
#include <Candera/Engine2D/Core/TextNodeRenderer/TextNodeRenderer.h>
#include <Candera/TextEngine/TextRenderContexts/NestedTextLayoutStrategy.h>
#include <Candera/TextEngine/TextRenderContexts/LineBreakLayoutStrategy.h>
#include <Candera/TextEngine/TextRenderContexts/TruncationToGlyphIteratorContext.h>
#ifdef FEATSTD_THREADSAFETY_ENABLED
#include <FeatStd/Platform/CriticalSectionLocker.h>
#endif
#if defined(CANDERA_LAYOUT_ENABLED)
#include <Candera/EngineBase/Layout/DefaultLayouter.h>
#include <Candera/EngineBase/Layout/Layouter.h>
#endif
#include <Candera/TextEngine/Async/TextRenderDispatcher.h>

#include <Candera/Engine2D/Core/Scene2D.h>
#include <Candera/EngineBase/Common/AbstractNodePointer.h>
#include <Candera/TextEngine/Async/TextValidationGroup.h>

#include <Candera/System/GlobalizationBase/CultureChangeListener.h>

namespace Candera {
    FEATSTD_LOG_SET_REALM(Diagnostics::LogRealm::CanderaEngine2D);
    using namespace Candera::Internal;
    using namespace TextRendering;
    using namespace TextRendering::Internal;

    class CustomPreprocessingResult{
    public:
        Candera::TextRendering::PreprocessingContext::Iterator m_customPreprocessingIterator;
        TextRendering::TextRect m_boundingRectangle;
        TextRendering::TextRect m_layoutRectangle;
        bool m_useCustomPreprocessingResult;
    };
    
    class Text2DCultureChangeListener : public Candera::Globalization::CultureChangeListener {
    public:

        Text2DCultureChangeListener(TextNode2D& textNode) :m_textNode(textNode)
        {
            Candera::Globalization::CultureManager::GetInstance().AddCultureChangeListener(this);
        }

        virtual ~Text2DCultureChangeListener()
        {
            bool result = Candera::Globalization::CultureManager::GetInstance().RemoveCultureChangeListener(this);
            FEATSTD_UNUSED(result);
            FEATSTD_DEBUG_ASSERT(result);
            // check whether culture change listener were added more than a single time
            FEATSTD_DEBUG_ASSERT(!Candera::Globalization::CultureManager::GetInstance().RemoveCultureChangeListener(this));
        }

    protected:
        FEATSTD_MAKE_CLASS_UNCOPYABLE(Text2DCultureChangeListener);
        TextNode2D& m_textNode;

        // post hook has to be used as text changes in OnCultureChanged 
        virtual void OnPostCultureChanged(const Candera::Globalization::Culture& /*culture*/) override
        {
            m_textNode.CheckPropertiesValidity();
        }

        virtual void OnCultureChanged(const Candera::Globalization::Culture&) override { /* Nothing to do here - everything has to happen in post hook */ }


#ifdef FEATSTD_THREADSAFETY_ENABLED
        virtual void Obtain()
        {
            (void)FeatStd::Internal::AtomicOp::Inc(m_obtainCounter);
        }

        virtual void Release()
        {
            (void)FeatStd::Internal::AtomicOp::Dec(m_obtainCounter);
        }

        virtual void WaitForRelease()
        {
            while (!FeatStd::Internal::AtomicOp::TestAndSet(m_obtainCounter, 0, 0)) {
                FeatStd::Internal::Thread::Sleep(5);
            }
        }

        FeatStd::Internal::AtomicOp::Atomic m_obtainCounter;
#endif
    };


    static void TextRectToRectangle(Rectangle& out, const TextRendering::TextRect& in)
    {
        out.SetTop(static_cast<Float>(in.GetTop()));
        out.SetLeft(static_cast<Float>(in.GetLeft()));
        out.SetWidth(static_cast<Float>(in.GetWidth()));
        out.SetHeight(static_cast<Float>(in.GetHeight()));
    }
    /******************************************************************************
    *  GetBasePoint
    ******************************************************************************/
    void TextNode2D::GetBasePoint(Vector2& basePoint) const
    {
        Rectangle boundingRectangle;
        GetComputedBoundingRectangle(boundingRectangle);
        basePoint.SetX(boundingRectangle.GetLeft() + (boundingRectangle.GetWidth() * 0.5F));
        if (!m_style.PointsToNull()) {
            const TextRendering::Metrics& metrics = m_style->GetMetrics();
            Float ascender = static_cast<Float>(metrics.ascender - 1);
            basePoint.SetY(ascender);
        }
    }

    void TextNode2D::CheckPropertiesValidity(bool invalidateScene /*= true*/)
    {
        if (m_style != 0) {
            UInt16 styleVersion = TextRendering::Internal::StyleTools::GetStyleVersion(*m_style);
            if (styleVersion != m_styleVersion) {
                InvalidateText(invalidateScene);
                m_styleVersion = styleVersion;
            }
        }

        if (m_text.HasChanged()) {
            InvalidateText(invalidateScene);
        }
    }

    void TextNode2D::PreRender()
    {
        CheckPropertiesValidity();
        if (!IsLayouterAttached()) {
            DefaultTextPlacement();
        }
        if (IsPrerenderRequired()) {
            if (m_renderer->PreRender(*this)) {
                GetActiveGlyphContainer().Reset();
                m_renderStates.SetPrerenderState(TextNode2DRenderState::Idle);
            }

        }
    }

    void TextNode2D::Render(RenderTarget2D* renderTarget, const Matrix3x2& localTransform)
    {
        if (m_renderer != 0) {
            m_renderer->Render(*this, renderTarget, localTransform);
        }
    }

    void TextNode2D::RenderImage(Image2D& image, RenderTarget2D* renderTarget, const Matrix3x2& localTransform)
    {
        Effect2D* effect = GetEffect(0);
        if (effect != 0) {
            BitmapBrush* bitmapBrush = Dynamic_Cast<BitmapBrush*>(effect->GetBrushEffect2D());
            if (bitmapBrush != 0) {
                MemoryManagement::SharedPointer<Image2D> originalImage(bitmapBrush->Image().Get());
                bitmapBrush->Image().Set(SharedPtrProperty<Image2D>::TPtr(&image));
                if (bitmapBrush->Update()) {
                    RenderNode::Render(renderTarget, localTransform);
                    bitmapBrush->Image().Set(originalImage);
                    static_cast<void>(bitmapBrush->Update());
                }
            }
        }
    }


    Candera::TextRendering::TextRect TextNode2D::GetLayoutTextRectangle() const
    {
        return GetLayoutRectangle();
    }

    Candera::TextRendering::TextRect TextNode2D::GetBoundingTextRectangle() const
    {
        return GetBoundingRectangle();
    }

    void TextNode2D::GetComputedBoundingRectangle(Rectangle& boundingRectangle, bool ignoreInvisible) const
    {
        if (ignoreInvisible && (!IsEffectiveRenderingEnabled())) {
            boundingRectangle = Rectangle();
        }
        else {
            TextRectToRectangle(boundingRectangle, GetBoundingRectangle());
#ifdef CANDERA_LAYOUT_ENABLED
            if (ignoreInvisible) {
#ifdef CANDERA_LAYOUT_CLIPPING_ENABLED
                boundingRectangle.Intersect(this->GetClippingRect());
#else
                boundingRectangle.Intersect(Layouter::GetClippingArea(*this));
#endif
            }
#endif
        }
    }

#if defined(CANDERA_LAYOUT_ENABLED)
    void TextNode2D::GetComputedLayoutRectangle(Rectangle& rectangle) const
    {
        if (!IsLayouterAttached()) {
            TextNode2D* notConstNode = const_cast<TextNode2D*>(this);
            notConstNode->m_isLayouterUsesTextNodeNoLayout = true;
            notConstNode->DefaultTextPlacement();
        }
        TextRectToRectangle(rectangle, GetLayoutRectangle());
    }

    void TextNode2D::OnBeforeLayouterSet(Layouter* newLayouter)
    {
        FEATSTD_UNUSED(newLayouter);
        InvalidateText();
    }
#endif


    void TextNode2D::SetAsyncPreRenderEnabled(bool enabled)
    {
        if (!IsCustomGlyphInformationUsed()) {
            if (m_isAsyncPreRenderEnabled != enabled) {
                m_isAsyncPreRenderEnabled = enabled;
                if (m_isAsyncPreRenderEnabled == false) {
                    m_isAsyncPreRenderChanged = true;
                }
                else {
                    m_isAsyncPreRenderChanged = false;
                }
            }
        }
        else {
            FEATSTD_LOG_WARN("The asynchronous flag for TextNode2D cannot be changed whilst custom glyph information is in use!");
        }
    }

    bool TextNode2D::AttachValidationHelper(FeatStd::ValidationHelperBase* validationHelper)
    {
        bool result = true;
        if (m_overrideValidationUser != 0) {
            result = m_overrideValidationUser->AttachValidationHelper(validationHelper);
        }
        if (result) {
            GetRenderController().SetValidationHelper(validationHelper);
        }
        return result;
    }

    bool TextNode2D::DetachValidationHelper(FeatStd::ValidationHelperBase* validationHelper)
    {
        bool result = true;
        if (m_overrideValidationUser != 0) {
            result = m_overrideValidationUser->DetachValidationHelper(validationHelper);
        }
        if (result) {
            FEATSTD_DEBUG_ASSERT(GetRenderController().GetValidationHelper() == validationHelper);
            GetRenderController().SetValidationHelper(0);
        }
        return result;
    }

    void TextNode2D::TriggerValidUpdate()
    {
        if (m_overrideValidationUser != 0) {
            m_overrideValidationUser->TriggerValidUpdate();
        }
        else {
#if defined(CANDERA_LAYOUT_ENABLED)
            if ((IsLayouterAttached()) || (m_isLayouterUsesTextNodeNoLayout == true)) {
                Candera::Scene2D * scene = GetScene();
                if (scene != 0) {
                    scene->SetLayoutInvalid();
                }
            }
#endif
           FeatStd::EventSource& evtSource = Candera::TextRendering::Internal::TextValidationEventSources::GetDefaultValidationEventSource();
           Candera::TextRendering::Internal::Node2DValidationEvent nodeEvent(this);
           evtSource.DispatchEvent(nodeEvent);
        }
    }


    /******************************************************************************
     *  Constructor
     ******************************************************************************/
    TextNode2D::TextNode2D() :
        Base(),
        m_text(),
        m_renderController(),
        m_customPreprocessingResult(0),
        m_boundingRectangle(),
        m_layoutRectangle(),
        m_activeContainer(GlyphDataContainer::Create()),
        m_style(),
        m_renderer(0),
        m_overrideValidationUser(0),
        m_cultureChangeListener(0),
        m_styleVersion(0),
        m_renderStates(),
        m_isAsyncPreRenderEnabled(false),
        m_isAsyncPreRenderChanged(false),
        m_isLayouterUsesTextNodeNoLayout(false),
        m_isPreviousRightToLeft(false)

    {
        m_cultureChangeListener = TEXTENGINE_TRANSIENT_NEW(Text2DCultureChangeListener)(*this);
        if (m_cultureChangeListener == 0) {
            FEATSTD_LOG_ERROR("Culture change listener could not be added to a TextNode2D");
        }
        m_renderController.SetOwner(this);
        static_cast<void>(TextValidationGroup::GetDefaultValidator().AttachValidationUser(this));
        if (!m_style.PointsToNull()) {
            if (!m_style->AddNotifyPropertyChangedEventListener(this)) {
                FEATSTD_LOG_ERROR("The TextNode2D was not able to register the property changed event listener to style.");
            }
        }
    }

    TextNode2D::TextNode2D(const TextNode2D& textNode2D) :
        Base(textNode2D),
        m_text(textNode2D.m_text),
        m_renderController(),
        m_customPreprocessingResult(0),
        m_boundingRectangle(textNode2D.m_boundingRectangle),
        m_layoutRectangle(textNode2D.m_layoutRectangle),
        m_activeContainer(GlyphDataContainer::Create()),
        m_style(textNode2D.m_style),
        //Renderer will be cloned by DeepTextNode2DCloneStrategy
        m_renderer(0),
        m_overrideValidationUser(textNode2D.m_overrideValidationUser),
        m_cultureChangeListener(0),
        m_styleVersion(textNode2D.m_styleVersion),
        m_renderStates(),
        m_isAsyncPreRenderEnabled(textNode2D.m_isAsyncPreRenderEnabled),
        m_isAsyncPreRenderChanged(false),
        m_isLayouterUsesTextNodeNoLayout(false),
        m_isPreviousRightToLeft(false)
    {
        m_cultureChangeListener = TEXTENGINE_TRANSIENT_NEW(Text2DCultureChangeListener)(*this);
        if (m_cultureChangeListener == 0) {
            FEATSTD_LOG_ERROR("Culture change listener could not be added to a TextNode2D");
        }
        m_renderController.SetOwner(this);
        static_cast<void>(TextValidationGroup::GetDefaultValidator().AttachValidationUser(this));

        if (!m_style.PointsToNull()) {
            if (!m_style->AddNotifyPropertyChangedEventListener(this)) {
                FEATSTD_LOG_ERROR("The TextNode2D was not able to register the property changed event listener to style.");
            }
        }
    }

    /******************************************************************************
     *  Destructor
     ******************************************************************************/
    TextNode2D::~TextNode2D()
    {
        if (!m_style.PointsToNull()) {
            if(!m_style->RemoveNotifyPropertyChangedEventListener(this)) {
                FEATSTD_LOG_ERROR("The TextNode2D was not able to unregister the property changed event listener to style.");
            }
        }
#ifdef FEATSTD_THREADSAFETY_ENABLED
        this->WaitForRelease();
#endif
        // PC-Lint warning : 
        if (m_renderer != 0) {
            CANDERA_SUPPRESS_LINT_FOR_SYMBOL(1740, Candera::TextNode2D::m_renderer, freeing renderer is done in dispose)
                m_renderer->Dispose();
        }
        m_renderer = 0;
        if (m_customPreprocessingResult != 0) {
            CANDERA_SUPPRESS_LINT_FOR_SYMBOL(1740, Candera::TextNode2D::m_customPreprocessingResult, TextNode2D is freeing and zeroing in the line below)
            FEATSTD_DELETE(m_customPreprocessingResult); m_customPreprocessingResult = 0;
        }
        if (m_cultureChangeListener != 0) {
            CANDERA_SUPPRESS_LINT_FOR_SYMBOL(1740, Candera::TextNode2D::m_cultureChangeListener, TextNode2D is freeing and zeroing in the line below)
            FEATSTD_SAFE_DELETE(m_cultureChangeListener);

        }
        static_cast<void>(TextValidationGroup::GetDefaultValidator().DetachValidationUser(this));
        CANDERA_SUPPRESS_LINT_FOR_SYMBOL(1740, Candera::TextNode2D::m_overrideValidationUser, TextNode2D is not the owner of this object)
    }

    /******************************************************************************
     *  Create
     ******************************************************************************/
    TextNode2D* TextNode2D::Create()
    {
        return FEATSTD_NEW(TextNode2D);
    }

    FEATSTD_RTTI_DEFINITION(TextNode2D, RenderNode)

        /******************************************************************************
         *  DisposeSelf
         ******************************************************************************/
         void TextNode2D::DisposeSelf()
    {
             FEATSTD_DELETE(this);
         }

    bool TextNode2D::UploadSelf()
    {
        bool success = true;
        if (0 != m_renderer) {
            success = m_renderer->UploadSelf();
        }

        return Base::UploadSelf() && success;
    }

    bool TextNode2D::UnloadSelf()
    {
        bool success = true;
        if (0 != m_renderer) {
            success = m_renderer->UnloadSelf();
            if (!m_renderer->IsTextValidAfterUnload()) {
                InvalidateText();
            }
        }

        return Base::UnloadSelf() && success;
    }

    void TextNode2D::InvalidateText(bool invalidateScene /*= true*/)
    {
        if (m_isAsyncPreRenderChanged == true) {
            GetRenderController().AbortAll();
        }
        m_isAsyncPreRenderChanged = false;
        // set first state invalid - the layout state handler has to make sure that the next
        // state is also set to invalid
        m_renderStates.SetLayoutState(TextNode2DRenderState::Invalid);
#if defined(CANDERA_LAYOUT_ENABLED)
        if (invalidateScene) {
            if ((IsLayouterAttached()) || (m_isLayouterUsesTextNodeNoLayout == true)) {
                AbstractNodePointer abstractNode(this);
                abstractNode.InvalidateLayout();
            }
        }
#else
        FEATSTD_UNUSED(invalidateScene);
#endif
    }

    void TextNode2D::SetText(const FeatStd::String& text)
    {
        m_text = text;
        InvalidateText();
    }

    void TextNode2D::SetStyle(const MemoryManagement::SharedPointer<TextRendering::SharedStyle>& style)
    {
        if (!m_style.PointsToNull()) {
            if(!m_style->RemoveNotifyPropertyChangedEventListener(this)) {
                FEATSTD_LOG_ERROR("The TextNode2D was not able to unregister the property changed event listener to style.");
            }
        }
        m_style = style;
        if (!style.PointsToNull()) {
            if(!style->AddNotifyPropertyChangedEventListener(this)) {
                FEATSTD_LOG_ERROR("The TextNode2D was not able to register the property changed event listener to style.");
            }
        }
        CheckPropertiesValidity(); // implicitly updates style version
        InvalidateText();
    }

    void TextNode2D::SetTextNodeRenderer(TextNodeRenderer* renderer)
    {
        // legacy reasons that Dispose() is not called!
        //   if (m_renderer != 0) {
        //       m_renderer->Dispose();
        //   }
        m_renderer = renderer;
        InvalidateText();
    }

    TextNode2DRenderState::Enum TextNode2D::GetPreparationState()const
    {
#ifdef FEATSTD_THREADSAFETY_ENABLED
        FeatStd::Internal::CriticalSectionLocker cs(&m_textAsyncLock);
#endif
        return m_renderStates.GetLayoutState();
    }


    Candera::TextRendering::TextRect const& TextNode2D::GetBoundingRectangle() const
    {
        if (IsCustomGlyphInformationUsed()) {
            return m_customPreprocessingResult->m_boundingRectangle;
        }
        ValidateBoundingInfo();
        return m_boundingRectangle;
    }

    void TextNode2D::SetPreparationState(TextNode2DRenderState::Enum state)
    {
#ifdef FEATSTD_THREADSAFETY_ENABLED
        FeatStd::Internal::CriticalSectionLocker cs(&m_textAsyncLock);
#endif
        m_renderStates.SetLayoutState(state);
    }

    void TextNode2D::SetPreparationFinished()
    {
#ifdef FEATSTD_THREADSAFETY_ENABLED
        FeatStd::Internal::CriticalSectionLocker cs(&m_textAsyncLock);
#endif
        if (GetRenderController().GetQueuedCount() == 0) {
            SetPreparationState(TextNode2DRenderState::Idle);
        }
        else if (GetRenderController().CheckForFinished() != 0) {
            SetPreparationState(TextNode2DRenderState::Finished);
        }
        else {
            SetPreparationState(TextNode2DRenderState::Processing);
        }
    }

    void TextNode2D::SetPreprocessedText(TextRendering::TextRenderArgs::SharedPointer& resultArgs)
    {
        FEATSTD_DEBUG_ASSERT(!IsCustomGlyphInformationUsed());
        GlyphDataContainer::SharedPointer tempDataContainer = m_activeContainer;
        SetPreprocessedText(resultArgs->GetGlyphContainer(), resultArgs->GetLayoutRectangle(), resultArgs->GetTextRectangle());
        resultArgs->SetGlyphContainer(tempDataContainer);
    }

    void TextNode2D::ArrangePreprocessedTextPosition(const TextRendering::TextRect& layoutingRectangle)
    {
        if (IsCustomGlyphInformationUsed()) {
            return;
        }
        FEATSTD_DEBUG_ASSERT((layoutingRectangle.GetWidth() >= GetLayoutTextRectangle().GetWidth()) && (layoutingRectangle.GetHeight() >= GetLayoutTextRectangle().GetHeight()));
        PixelPosition moveInX = m_layoutRectangle.GetLeft() - layoutingRectangle.GetLeft();
        PixelPosition moveInY = m_layoutRectangle.GetTop() - layoutingRectangle.GetTop();
        if (!m_activeContainer.PointsToNull()) {
            m_activeContainer->MoveInDirection(-moveInX, -moveInY);
        }
        TextRendering::TextRect diffRect(
            m_layoutRectangle.GetLeft() - m_boundingRectangle.GetLeft(),
            m_layoutRectangle.GetTop() - m_boundingRectangle.GetTop(),
            m_layoutRectangle.GetRight() - m_boundingRectangle.GetRight(),
            m_layoutRectangle.GetBottom() - m_boundingRectangle.GetBottom()
            );
        m_layoutRectangle = layoutingRectangle;
        m_boundingRectangle = TextRendering::TextRect(
            m_layoutRectangle.GetLeft() - diffRect.GetLeft(),
            m_layoutRectangle.GetTop() - diffRect.GetTop(),
            m_layoutRectangle.GetRight() - diffRect.GetRight(),
            m_layoutRectangle.GetBottom() - diffRect.GetBottom()
            );
    }

    const Candera::TextRendering::PreprocessingContext::Iterator TextNode2D::GetTextIterator() const
    {
        if (IsCustomGlyphInformationUsed()) {
            return m_customPreprocessingResult->m_customPreprocessingIterator;
        }
        return GetActiveGlyphContainer().GetIterator();
    }

    bool TextNode2D::IsTriggerUpdateForced()
    {
        if (m_overrideValidationUser != 0) {
            return m_overrideValidationUser->IsTriggerUpdateForced();
        }
        return false;
    }


    bool TextNode2D::IsCustomGlyphInformationUsed() const
    {
        return (m_customPreprocessingResult != 0) && (m_customPreprocessingResult->m_useCustomPreprocessingResult);
    }


    void TextNode2D::SetCustomGlyphInformation(Candera::TextRendering::PreprocessingContext::Iterator customIterator,
                                            const TextRendering::TextRect& layoutRectangle,
                                            const TextRendering::TextRect& boundingRectangle)
    { 
        // setting a custom glyph iterator when async rendering is enabled is not allowed and not possible due to restrictions in the design.
        FEATSTD_DEBUG_ASSERT(!IsAsyncPreRenderEnabled());
        if (m_customPreprocessingResult == 0) {
            m_customPreprocessingResult = FEATSTD_NEW(CustomPreprocessingResult)();
        }
        if (m_customPreprocessingResult != 0) {
            m_customPreprocessingResult->m_useCustomPreprocessingResult = true;
            m_customPreprocessingResult->m_customPreprocessingIterator = customIterator;
            m_customPreprocessingResult->m_layoutRectangle = layoutRectangle;
            m_customPreprocessingResult->m_boundingRectangle = boundingRectangle;
            m_renderStates.SetLayoutState(TextNode2DRenderState::Idle);
            m_renderStates.SetPrerenderState(TextNode2DRenderState::Invalid);
        }
        else {
            FEATSTD_LOG_ERROR("Error allocating memory for custom glyph information!");
        }
    }


    void TextNode2D::DisableCustomGlyphInformation()
    {
        if (IsCustomGlyphInformationUsed()) {
            FEATSTD_DELETE(m_customPreprocessingResult); m_customPreprocessingResult = 0;
            m_renderStates.SetLayoutState(TextNode2DRenderState::Invalid);
        }
        FEATSTD_DEBUG_ASSERT(!IsCustomGlyphInformationUsed());
    }


    FeatStd::EventResult::Enum TextNode2D::OnEvent(const FeatStd::Event& event)
    {
        const NotifyPropertyChangedEvent * notifyEvent = Candera::Dynamic_Cast<const NotifyPropertyChangedEvent*>(&event);
        if (notifyEvent != 0) {
            // object is of type: style
            if (notifyEvent->GetBaseObject()->GetTypeId() == Candera::TextRendering::Style::GetTypeId()) {
                CheckPropertiesValidity();
            }
        }
        return FeatStd::EventResult::Proceed;
    }

    /******************************************************************************
    *  Clone
    ******************************************************************************/
    TextNode2D* TextNode2D::Clone() const
    {
        return FEATSTD_NEW(TextNode2D)(*this);
    }

    bool TextNode2D::IsLayouterAttached() const
    {
#if defined(CANDERA_LAYOUT_ENABLED)
        return DefaultLayouter::GetInstance() != GetLayouter();
#else
        return false;
#endif
    }


    Candera::TextRendering::GlyphDataContainer& TextNode2D::GetActiveGlyphContainer()
    {
        FEATSTD_DEBUG_ASSERT(!m_activeContainer.PointsToNull());
        return *m_activeContainer;
    }


    const Candera::TextRendering::GlyphDataContainer& TextNode2D::GetActiveGlyphContainer() const
    {
        FEATSTD_DEBUG_ASSERT(!m_activeContainer.PointsToNull());
        return *m_activeContainer;
    }

    bool TextNode2D::IsPrerenderRequired()const
    {
        return (m_renderer != 0) && ((m_renderStates.GetPrerenderState() == TextNode2DRenderState::Finished) || (m_renderStates.GetPrerenderState() == TextNode2DRenderState::Invalid));
    }

    void TextNode2D::ValidateBoundingInfo() const {
        if (m_renderStates.GetLayoutState() == TextNode2DRenderState::Invalid) {
            // necessary for now, since it updates internal data 
            TextNode2D* notConstNode = const_cast<TextNode2D*>(this);

             if (IsLayouterAttached()) {
#if defined(CANDERA_LAYOUT_ENABLED)
                const AbstractNodePointer abstractNode(notConstNode);
                notConstNode->GetLayouter()->Layout(abstractNode);
#endif
            } else {
                notConstNode->DefaultTextPlacement(true);
            }           
        }

    }

    void TextNode2D::DefaultTextPlacement(bool forceSyncPreProcessOnly)
    {
        // Should only be called if there is NOT a layouter attached
        // Otherwise the layouter handles this request with its own definitions!
        FEATSTD_DEBUG_ASSERT(!IsLayouterAttached());
        if (IsLayouterAttached()) {
            return;
        }
        if (IsCustomGlyphInformationUsed()) {
            return;
        }
        TextRenderArgs::SharedPointer ptr;
        UInt8 freeIdx;
        bool renderResult;
#if defined(CANDERA_GLOBALIZATION_ENABLED)
        Globalization::Culture::SharedPointer currentCulture = Globalization::CultureManager::GetInstance().GetCurrentCulture();
        if (!currentCulture.PointsToNull()) {
            bool isRightToLeft = (currentCulture->GetTextDirection() == Candera::Globalization::RightToLeft);
            if (m_isPreviousRightToLeft != isRightToLeft) {
                m_isPreviousRightToLeft = isRightToLeft;
                InvalidateText(false);
            }
        }
#endif
        if (m_renderStates.GetLayoutState() == TextNode2DRenderState::Invalid) {
            freeIdx = GetRenderController().GetFreeArgs(ptr);
            FEATSTD_DEBUG_ASSERT(freeIdx != GetRenderController().InvalidIndex());
            if (freeIdx == GetRenderController().InvalidIndex()) {
                return;
            }
            FEATSTD_DEBUG_ASSERT(!ptr.PointsToNull());
            static_cast<void>(ptr->DefaultInit(*this));
#if defined(CANDERA_LAYOUT_ENABLED)
            const Vector2& size = Layouter::GetSize(static_cast<CanderaObject&>(*this));
            ptr->SetTextAreaSize(TextSize(static_cast<Int16>(size.GetX()), static_cast<Int16>(size.GetY())));
            if (size.GetX() > 0.0F) {
                ptr->GetLayoutingOptionsRef().SetHorizontalAlignment(HLeft);
            }
#endif

            SetPreparationState(TextNode2DRenderState::Processing);
            if ((!IsAsyncPreRenderEnabled()) || forceSyncPreProcessOnly) {
                renderResult = PreprocessTextInternal(ptr);
                GetRenderController().SetFinishedHandle(freeIdx);
                if (forceSyncPreProcessOnly) {
                    // Preprocessinng complete, so the layout state/values are updated
                    m_renderStates.SetLayoutState(TextNode2DRenderState::Finished); 
                }
            }
            else {
                GetRenderController().SetHandle(freeIdx, TextRenderDispatcher::RenderAsync(TextNode2D::PreprocessTextInternal, ptr, 1));
            }

        }
        if (m_renderStates.GetLayoutState() == TextNode2DRenderState::Finished) {
            bool retrievePendingResult = (!IsAsyncPreRenderEnabled()) ||
                (GetRenderController().GetValidationHelper() == 0) ||
                (GetRenderController().GetValidationHelper()->IsValid());
            if (retrievePendingResult) {
                freeIdx = GetRenderController().GetRenderCompletedIndex(ptr, renderResult);
                if (freeIdx != GetRenderController().InvalidIndex()) {
                    if (renderResult) {
                        FEATSTD_DEBUG_ASSERT(!ptr.PointsToNull());
                        SetPreprocessedText(ptr);
                    }
                    GetRenderController().FreeIndex(freeIdx);
                    SetPreparationFinished();
                }

                if ((IsAsyncPreRenderEnabled()) && (GetRenderController().GetValidationHelper() != 0)) {
                    GetRenderController().GetValidationHelper()->ConfirmValidHandled();
                }
            }
        }

    }

    void TextNode2D::SetPreprocessedText(GlyphDataContainer::SharedPointer activeGlyphDataContainer,
                                         const TextRendering::TextRect& layoutRectangle,
                                         const TextRendering::TextRect& boundingRectangle)
    {
        m_activeContainer = activeGlyphDataContainer;
        m_layoutRectangle = layoutRectangle;
        m_boundingRectangle = boundingRectangle;
        m_renderStates.SetLayoutState(TextNode2DRenderState::Idle);
        m_renderStates.SetPrerenderState(TextNode2DRenderState::Invalid);
    }


    bool TextNode2D::PreprocessTextInternal(TextRendering::TextRenderArgs::SharedPointer renderArgs)
    {
        if ((renderArgs.PointsToNull()) || (renderArgs->GetGlyphContainer().PointsToNull())) {
            return false;
        }
        NestedTextLayoutStrategy baseStrategy;
        renderArgs->GetGlyphContainer()->Reset();
        TextToGlyphIteratorContext context(renderArgs);


        renderArgs->GetLayoutingOptionsRef().SetLayoutStrategy(&baseStrategy);
        LineBreakLayoutStrategy lineBreakStrategy(context);
        baseStrategy.AddStrategy(&lineBreakStrategy);
        if (!TextRenderer().Render(
            context,
            renderArgs->GetLayoutingOptionsRef(),
            ShapingOptions(renderArgs->GetStyle()),
            renderArgs->GetText())) {
            return false;
        }
        static_cast<void>(renderArgs->SetResults(context.GetLayoutRectangle(), context.GetTextRectangle()));
        return true;
    }

    Candera::TextRendering::TextRect const& TextNode2D::GetLayoutRectangle() const
    {
        if (IsCustomGlyphInformationUsed()) {
            return m_customPreprocessingResult->m_layoutRectangle;
        }

        ValidateBoundingInfo();
        return m_layoutRectangle;
    }

    //////////////////////////////////////////////////////////////////////////
    // Deprecated
    //////////////////////////////////////////////////////////////////////////

    void SetPreprocessedText(const TextRendering::MinimalPreprocessingContext&,
                             const TextRendering::TextRect&,
                             const TextRendering::TextRect&)
    {
        // function is deprecated and does nothing anymore
        FEATSTD_DEBUG_FAIL();
    }
}
