//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#ifndef CANDERA_Renderer2DListener_H
#define CANDERA_Renderer2DListener_H

#include <Candera/Environment.h>

namespace Candera {

/** @addtogroup Core2D
 *  @{
 */
 
    class RenderNode;
    class Camera2D;

    /**
    * @brief A Renderer2DListener defines hooks that are called before and
    * after a Node is rendered.
    */
    class Renderer2DListener
    {
    public:
        /**
        * Constructs a Renderer2DListener object.
        */
        Renderer2DListener() {}

        /**
        * Destructs a Renderer2DListener object.
        */
        virtual ~Renderer2DListener() {}

        /**
        * Is invoked before a node is rendered.
        * @param node The next node that will render.
        */
        virtual void OnNodePreRender(RenderNode* node) = 0; 

        /**
        * Is invoked after a node is rendered.
        * @param node The most recently rendered node.
        */
        virtual void OnNodePostRender(RenderNode* node) = 0;

        /**
        *  Is invoked after a camera has been rendered.
        *  @param camera The camera that was rendered.
        */
        virtual void OnCameraPostRender(Camera2D* /*camera*/) {};

        /**
        * Is invoked before a node is rendered. The default implementation will call OnNodePreRender(RenderNode*).
        * @param node The next node that will render.
        * @param modelViewMatrix The model view matrix for the given node
        */
        virtual void OnNodePreRender(const RenderNode* node, const Matrix3x2* /*modelViewMatrix*/) { OnNodePreRender(const_cast<RenderNode*>(node)); };

        /**
        * Is invoked after a node is rendered. The default implementation will call OnNodePostRender(RenderNode*).
        * @param node The most recently rendered node.
        * @param modelViewMatrix The model view matrix for the given node
        */
        virtual void OnNodePostRender(const RenderNode* node, const Matrix3x2* /*modelViewMatrix*/) { OnNodePostRender(const_cast<RenderNode*>(node)); };

        /**
        *  Is invoked before a camera is activated.
        *  @param camera The camera that will be activated.
        */
        virtual void OnCameraPreActivate(const Camera2D* /*camera*/) {};

        /**
        *  Is invoked after a camera is activated.
        *  @param camera The camera that was activated.
        */
        virtual void OnCameraPostActivate(const Camera2D* /*camera*/) {};

        /**
        *  Is invoked before clearing the framebuffer.
        *  @param camera The camera that will be activated.
        */
        virtual void OnCameraPreClear(const Camera2D* /*camera*/) {};

        /**
        *  Is invoked after a clearing the framebuffer.
        *  @param camera The camera that was activated.
        */
        virtual void OnCameraPostClear(const Camera2D* /*camera*/) {};

    };
 
/** @} */ // end of Core2D
 
} // namespace Candera

#endif  // CANDERA_Renderer2DListener_H

