//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERA_Renderer2D_H)
    #define CANDERA_Renderer2D_H

#include <Candera/Environment.h>
#include <Candera/System/Container/Vector.h>
#include <Candera/System/Mathematics/Rectangle.h>
#include <CanderaPlatform/Device/Common/Base/RenderDevice2D.h>
#include <Candera/Engine2D/Core/DeviceObject2D.h>
#include <Candera/Macros.h>
#include <FeatStd/Platform/CriticalSection.h>

CANDERA_UNIT_TEST_CLASS_DECLARATION(Renderer2DWrapper)

namespace Candera {

/** @addtogroup Core2D
 *  @{
 */

    class Scene2D;
    class Camera2D;
    class RenderNode;
    class Renderer2DListener;
    class RenderTarget2D;
    class Camera2DRenderStrategy;
    class Camera2DListener;

    /**
     * @brief   The class Renderer2D represents the unique Candera 2D render interface to the application.
     *          Each render call shall be triggered via the Renderer2D interface.
     */
    class Renderer2D {

        friend class Camera2D;
        friend class Renderer2D3D;
        CANDERA_UNIT_TEST_CLASS_FRIEND(Renderer2DWrapper);

        public:
            /**
             *  Only the given camera is rendered
             *  @param camera           The Camera2D that shall be rendered.
             *  @param resetStatistics  Reset the accumulated statistics data of the renderer. Default value is true.
             *  @param dirtyArea        Dirty area in screen space pixels. The default values of (0,0,-1,-1) denote the full
             *                          screen area.
             */
            static void RenderCamera(Camera2D* camera, bool resetStatistics = true,
                const Rectangle& dirtyArea = Rectangle(0.0F, 0.0F, -1.0F, -1.0F));

            /**
             *  Only those cameras are rendered, which use the given render target.
             *  @param  renderTarget    0: Render all cameras; otherwise render only those cameras for the given render target.
             */
            static void RenderAllCameras(const RenderTarget2D* renderTarget = 0);

            /**
             * Determine whether the renderer is currently rendering a camera.
             * @return True if RenderCamera is currently called.
             */
            static bool IsRenderingCamera() { return s_isRenderingCamera; }

            /**
             *  Gets the currently active camera.
             *  @return     The currently active camera.
             */
            static Camera2D* GetActiveCamera() { return s_activeCamera; }

            /**
             * Retrieves the number of cameras associated with this renderer.
             * @return      The number of cameras associated with this renderer.
             */
            static UInt32 GetCameraCount();

            /**
             * Retrieves the camera with the given index. Use GetCameraCount() to
             * obtain the upper bound of valid camera indices.
             * @param   index   The index of the sought camera. The index must be in the
             *                  range [0..GetCameraCount()[
             * @return          The camera at the specified index.
             */
            static Camera2D* GetCamera(UInt32 index);

            /** Adds a listener. Event/timing of the calls: before/after a node is rendered.
             * @param listener The lifetime of the listener is managed by the calling code. Renderer does not take ownership.
             * @return true if successful, false otherwise
             */
            static bool AddRendererListener(Renderer2DListener* listener);

            /** Removes a listener.
             * @param listener specifies the RendererListener object to remove from this Renderer.
             * @return true if successful, false otherwise
             */
            static bool RemoveRendererListener(Renderer2DListener* listener);

            /**
             *  Clear the active area of the destination surface to the specified color.
             *  @param context 2D context obtained from RenderTarget2D.
             *  @param r     Red color channel.
             *  @param g     Green color channel.
             *  @param b     Blue color channel.
             *  @param a     Alpha channel.
             *  @return      true on success, false on failure.
             */
            static bool Clear(ContextHandle2D context, Float r, Float g, Float b, Float a);

            /**
             *  Culls the source surface against the destination surface of the context and uses the RenderDevice2D to perform
             *  the Blit accordingly.
             *
             *  @param context    2D context obtained from RenderTarget2D.
             *  @param forceBlit  If true, culling is skipped and the Blit is always executed.
             *  @return           true on success, false on failure.
             */
            static bool Blit(ContextHandle2D context, bool forceBlit = false);

            /**
             *  Set a transformation matrix for a surface in the RenderDevice2D.
             *  @param context   2D context obtained from RenderTarget2D.
             *  @param sidx      Surface type.
             *  @param mat       Transformation matrix.
             *  @return          true on success, false on failure.
             */
            static bool SetTransformationMatrix(ContextHandle2D context, RenderDevice2D::SurfaceType sidx, const Matrix3x2& mat);

            /**
             *  Structure to keep track of renderer statistics since the last RenderCamera() or RenderAllCameras() call.
             */
            struct Statistics {
                UInt Blits;                     ///< number of Blits submitted to the RenderDevice2D
                UInt BlitsCulled;               ///< number of Blits that were culled, thus not submitted to the RenderDevice2D
                UInt Clears;                    ///< number of Clears executed
                Float AverageDirtyAreaFactor;   ///< average dirty area factor (dirty area / renderTarget area) in the range of (0..1]

                Statistics()
                    :
                    Blits(0),
                    BlitsCulled(0),
                    Clears(0),
                    AverageDirtyAreaFactor(0.0F),
                    m_dirtyAreaCounter(0) {}

                /**
                 *  Reset the statistics.
                 */
                void Reset() {
                    Blits = 0;
                    BlitsCulled = 0;
                    Clears = 0;
                    AverageDirtyAreaFactor = 0.0F;
                    m_dirtyAreaCounter = 0;
                }

        private:
                    friend class Renderer2D;
                    UInt m_dirtyAreaCounter;
            };

            /**
             *  Get render statistics (number of Blits, etc.) since the last RenderCamera() or RenderAllCameras() call.
             *  @return  A reference to the Renderer::Statistics that were collected since the last RenderCamera()
             *           or RenderAllCameras() call.
             */
            static const Statistics& GetStatistics() { return s_statistics; }

            /**
             *  Get the camera translation offset. During rendering the camera's view matrix has to be translated by this offset
             *  to produce correct results.
             *  @return  The camera translation offset.
             */
            static const Vector2& GetCameraTranslationOffset() { return s_cameraTranslationOffset; }


            /**
            * Uploads the given BitmapImage2D to VideoMemory.
            * @param vb The BitmapImage2D to Upload.
            * @param loadingHint customized loading behavior (e.g. warn if happening during rendering or force Upload.
            * @return Whether the Upload was succesful (true) or not (false);
            */
            static bool UploadBitmapImage2D(BitmapImage2D& image, DeviceObject2D::LoadingHint loadingHint);
            /**
            * Unloads the given BitmapImage2D to VideoMemory.
            * @param vb The BitmapImage2D to Unload.
            * @param loadingHint customized unloading behavior (e.g. warn if happening during rendering or force Upload.
            * @return Whether the Unload was succesful (true) or not (false);
            */
            static bool UnloadBitmapImage2D(BitmapImage2D& image, DeviceObject2D::LoadingHint loadingHint);

            /**
             * Uploads the given VertexBuffer2D to VideoMemory.
             * @param vb The VertexBuffer2D to Upload.
             * @param loadingHint customized loading behavior (e.g. warn if happening during rendering or force Upload.
             * @return Whether the Upload was succesful (true) or not (false);
             */
            static bool UploadVertexBuffer2D(VertexBuffer2D& vb, DeviceObject2D::LoadingHint loadingHint);
            /**
            * Unloads the given VertexBuffer2D to VideoMemory.
            * @param vb The VertexBuffer2D to Unload.
            * @param loadingHint customized unloading behavior (e.g. warn if happening during rendering or force Upload.
            * @return Whether the Unload was succesful (true) or not (false);
            */
            static bool UnloadVertexBuffer2D(VertexBuffer2D& vb, DeviceObject2D::LoadingHint loadingHint);

#if defined(CANDERA_LAYOUT_CLIPPING_ENABLED)
            /**
             *  Get the pre translation of the render node after applying the clipping rectangle.
             *  @return  The render node translation coordinates.
            */
             static const Vector2& GetPreTranslate() { return s_preTranslate; }
#endif
        private:
            static Statistics s_statistics;

            static bool s_isRenderingCamera;

            typedef Internal::Vector<Camera2D*> CameraList;
            static CameraList& s_forceInitCameraList;

            static Camera2DListener* s_forceInitCamera2dListenerInstance;

            class OnNodePreRenderEvent;
            class OnNodePostRenderEvent;
            class RendererCameraListener;
            class OnCameraPostRenderEvent;
            class OnCameraPreActivateEvent;
            class OnCameraPostActivateEvent;
            class OnCameraPreClearEvent;
            class OnCameraPostClearEvent;


            static Camera2DListener* GetCamera2dListenerInstance();
            static void OnCameraRenderTargetChanged(const Camera2D* camera, RenderTarget2D* previousRenderTarget);
            static void OnCameraRenderStrategyChanged(Camera2D* camera, const Camera2DRenderStrategy* previousCameraRenderStrategy);

            static void RenderCameraInternal(Camera2D* camera, const Rectangle& dirtyArea);

            CANDERA_SUPPRESS_LINT_FOR_NEXT_EXPRESSION(1704, CANDERA_LINT_REASON_NONCOPYABLE)
            Renderer2D();
            virtual ~Renderer2D() = 0; // Provoke abstract (static) class

            /**
             * Called before a node is rendered. Notifies the RendererListeners.
             * @param node Notifies renderer listeners that node will be rendered next.
             */
            static void NotifyRendererListenersOnNodePreRender(const RenderNode* node, const Matrix3x2* modelViewMatrix);
            /**
             * Called after a node is rendered. Notifies the RendererListeners.
             * @param node Notifies renderer listeners that node was rendered last.
             */
            static void NotifyRendererListenersOnNodePostRender(const RenderNode* node, const Matrix3x2* modelViewMatrix);

            /**
             * Called after a camera is completely rendered. Notifies the RendererListeners.
             *  @param camera The camera that was rendered.
             */
            static void NotifyListenersOnCameraPostRender(Camera2D* camera);

            /**
            *  Called before a camera is activated.
            *  @param camera The camera that will be activated.
            */
            static void NotifyListenersOnCameraPreActivate(const Camera2D* camera);

            /**
            *  Is invoked after a camera is activated.
            *  @param camera The camera that was activated.
            */
            static void NotifyListenersOnCameraPostActivate(const Camera2D* camera);

            /**
            *  Is invoked before clearing the framebuffer.
            *  @param camera The camera that will be activated.
            */
            static void NotifyListenersOnCameraPreClear(const Camera2D* camera);

            /**
            *  Is invoked after a clearing the framebuffer.
            *  @param camera The camera that was activated.
            */
            static void NotifyListenersOnCameraPostClear(const Camera2D* camera);


            /**
             * Retrieve the singleton renderer camera list.
             * @return A list of all the cameras rendered by the renderer.
             */
            static CameraList& GetCameraList();

            /**
             * Retrieve a mutex to lock when accessing the CameraList
             */
            static FeatStd::Internal::CriticalSection* GetCameraListCriticalSection();

            static void PreRenderNodes(Scene2D* scene);
            static Camera2D* s_activeCamera;

            static bool IsSourceSurfaceAreaCulled(ContextHandle2D context);
            static Matrix3x2 s_sourceSurfaceTransformationMatrix;
            static Vector2 s_cameraTranslationOffset;

            static FeatStd::Internal::CriticalSection& s_cameraListCrticalSection;

#if defined(CANDERA_LAYOUT_CLIPPING_ENABLED)
            static Vector2 s_preTranslate;
            static void CalculateScreenSpaceClippingRect(const RenderNode& renderNode, const Matrix3x2& viewMatrix, Rectangle& screenSpaceClippingRect);
#endif
};

 /** @} */ // end of Core2D

}   // namespace Candera

#endif  // CANDERA_Renderer2D_H
