//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "RenderNode.h"
#include <Candera/Engine2D/Core/Scene2D.h>
#include <Candera/Engine2D/Effects/BrushEffect2D.h>
#include <Candera/Engine2D/Effects/InPlaceEffect2D.h>
#include <Candera/Engine2D/Effects/BlendEffect2D.h>
#if defined(CANDERA_LAYOUT_ENABLED)
#include <Candera/EngineBase/Layout/Layout.h>
#endif
#include <Candera/System/Monitor/PerfMonPublicIF.h>
#include <Candera/System/Mathematics/Rectangle.h>
#include <CanderaPlatform/Device/Common/Base/RenderDevice2D.h>
#include <CanderaPlatform/Device/Common/Base/RenderTarget2D.h>

namespace Candera {
    FEATSTD_LOG_SET_REALM(Diagnostics::LogRealm::CanderaEngine2D);

    /******************************************************************************
     *  Constructor
     ******************************************************************************/
    RenderNode::RenderNode() :
        Base(),
        m_snapToDevicePixel(false),
        m_effectiveRenderOrderRank(0),
        m_positionInTree(0)
#if defined(CANDERA_LAYOUT_CLIPPING_ENABLED)
        ,m_clippingRect(0.0F, 0.0F, -1.0F, -1.0F)
#endif
    {
        static_cast<void>(m_effectList.Reserve(1));
    }

    RenderNode::RenderNode(const RenderNode& renderNode) :
        Base(renderNode),
        m_snapToDevicePixel(renderNode.m_snapToDevicePixel),
        m_effectiveRenderOrderRank(renderNode.m_effectiveRenderOrderRank),
        m_positionInTree(0),
        m_effectList(renderNode.m_effectList)
#if defined(CANDERA_LAYOUT_CLIPPING_ENABLED)
        ,m_clippingRect(renderNode.m_clippingRect)
#endif
    {
    }

    /******************************************************************************
     *  Destructor
     ******************************************************************************/
    RenderNode::~RenderNode()
    {
    }

    /******************************************************************************
     *  Create
     ******************************************************************************/
    RenderNode* RenderNode::Create()
    {
        return FEATSTD_NEW(RenderNode);
    }

    /******************************************************************************
     *  AddEffect
     ******************************************************************************/
    bool RenderNode::AddEffect(Effect2D* effect, Effect2D* predecessor /*= 0*/)
    {
        const SizeType size = m_effectList.Size();
        SizeType pos = size;

        if (predecessor != 0) {
            pos = 0;
            while ((pos < size) && (m_effectList[pos] != predecessor)) {
                pos++;
            }
        }

        return InsertEffectToList(pos, effect);
    }

    /******************************************************************************
     *  GetComputedBoundingRectangle
     ******************************************************************************/
    void RenderNode::GetComputedBoundingRectangle(Rectangle& boundingRectangle, bool ignoreInvisible) const
    {
        if (ignoreInvisible && (!IsEffectiveRenderingEnabled())) {
            boundingRectangle = Rectangle();
        }
        else {
            const SizeType listSize = m_effectList.Size();
            if (listSize != 0) {
                // the first effect must hold the brush effect!
                Effect2D* effect = m_effectList[0].GetPointerToSharedInstance();
                BrushEffect2D* brush = effect->GetBrushEffect2D();
                if (brush != 0) {
                    effect->GetBoundingRectangle(boundingRectangle);
                }
                else {
                    boundingRectangle = Rectangle();
                    FEATSTD_LOG_ERROR("In order to use an effect chain, the first"
                        "effect must contain a brush. See node %s. "
                        "Bounding box set to empty.",
                        (GetName() != 0) ? GetName() : "<Unnamed>");
                }
#ifdef CANDERA_LAYOUT_ENABLED
                if (ignoreInvisible) {
#ifdef CANDERA_LAYOUT_CLIPPING_ENABLED
                    boundingRectangle.Intersect(this->GetClippingRect());
#else
                    boundingRectangle.Intersect(Layouter::GetClippingArea(*this));
#endif
                }
#endif
            }
        }
    }

#if defined(CANDERA_LAYOUT_CLIPPING_ENABLED)
    void RenderNode::SetClippingRect(const Rectangle& clippingRect)
    {
#if defined(CANDERA_LAYOUT_ENABLED)
        Vector2 position = clippingRect.GetPosition();
        Vector2 size = clippingRect.GetSize();
        Internal::LayoutEvents::ArrangeEvent arrangeEvent(Internal::LayoutEvents::LayoutEvent::ArrangeSetClippingRectangle, AbstractNodePointer(this), position, size);
        Layouter::GetEventSource().DispatchEvent(arrangeEvent);
#endif
        m_clippingRect = clippingRect;
    }
#endif
#if defined(CANDERA_LAYOUT_ENABLED)

    /******************************************************************************
     *  GetComputedLayoutRectangle
     ******************************************************************************/
    void RenderNode::GetComputedLayoutRectangle(Rectangle& rectangle) const
    {
        const SizeType listSize = m_effectList.Size();
        if (listSize != 0) {
            // the first effect must hold the brush effect!
            Effect2D* effect = m_effectList[0].GetPointerToSharedInstance();
            BrushEffect2D* brush = effect->GetBrushEffect2D();
            if (brush != 0) {
                effect->GetLayoutingRectangle(rectangle);
            }
            else {
                rectangle = Rectangle();
                FEATSTD_LOG_ERROR("In order to use an effect chain, the first"
                    "effect must contain a brush. See node %s."
                    "Layouting box set to empty.",
                    (GetName() != 0) ? GetName() : "<Unnamed>");
            }
        }
    }
#endif

    /******************************************************************************
     *  GetBoundingRectangle
     ******************************************************************************/
    void RenderNode::GetBasePoint(Vector2& basePoint) const
    {
        const SizeType listSize = m_effectList.Size();
        if (listSize != 0) {
            // the first effect must hold the brush effect!
            Effect2D* effect = m_effectList[0].GetPointerToSharedInstance();
            BrushEffect2D* brush = effect->GetBrushEffect2D();
            if (brush != 0) {
                brush->GetBasePoint(basePoint);
            }
            else {
                basePoint = Vector2();
                FEATSTD_LOG_ERROR("In order to use an effect chain, the first"
                    "effect must contain a brush. See node %s."
                    "Base point set to (0, 0).",
                    (GetName() != 0) ? GetName() : "<Unnamed>");
            }
        }
    }

    FEATSTD_RTTI_DEFINITION(RenderNode, Node2D)

    /******************************************************************************
     *  Render
     ******************************************************************************/
    void RenderNode::Render(RenderTarget2D* renderTarget, const Matrix3x2& localTransform)
    {
        CANDERA_PERF_RECORDER(Timing, (FeatStd::PerfMon::TimingRecId::RenderNode2D, this->GetName()));

        static const Matrix3x2 identity;

        const SizeType listSize = m_effectList.Size();
        bool renderTargetActive = true;

        Rectangle twoRects[2];
        UInt8 currentRect = 0;

        Rectangle bound;

        bool contextIsCoherent = true;
        for (SizeType i = 0; (i < listSize) && contextIsCoherent; i++) {
            SurfaceHandle source;
            ContextHandle2D destination;

            const Matrix3x2& transformationMatrix = (i == 0) ? localTransform : identity;

            Effect2D* effect = m_effectList[i].GetPointerToSharedInstance();

            // brush doesn't have input surface.
            if (i == 0) {
                source = 0;
            }
            else {
                source = RenderDevice2D::GetNewOffscreenSurfaceHandle(renderTarget->Get2DContextHandle());
            }

            // blend draws directly to render target.
            if (i == (listSize - 1)) {
                if (!renderTargetActive) {
                    contextIsCoherent = renderTarget->Activate();
                    renderTargetActive = true;
                }
                destination = renderTarget->Get2DContextHandle();
            }
            else {
                if (renderTargetActive) {
                    // first time compute the on screen bound of the node
                    effect->GetBoundingRectangle(bound);
                    bound.Transform(localTransform);
                }
                else {
                    // transform the bounding rectangle of the previous effect
                    bound = twoRects[currentRect];
                    effect->GetBoundingRectangle(bound);
                }
                // retrieve an off-screen bound with the correct size
                destination = RenderDevice2D::GetNewOffscreenContextHandle(renderTarget->Get2DContextHandle(), bound);
                renderTargetActive = false;
            }

            if (i == 0) {
                //for brush set filtering property.
                BrushEffect2D* brush = effect->GetBrushEffect2D();
                FEATSTD_DEBUG_ASSERT(brush != 0);
                contextIsCoherent = contextIsCoherent && RenderDevice2D::SetFilter(destination, brush->Filter());
            }
            else {
                //in-place effects usually have pixel mapping 1 to 1, so Nearest is the best filtering solution.
                contextIsCoherent = contextIsCoherent && RenderDevice2D::SetFilter(destination, RenderDevice2D::NearestFilter);
            }

            if (contextIsCoherent) {
                effect->Render(source, twoRects[currentRect], transformationMatrix, *this, destination, twoRects[1 - currentRect]);
                currentRect = 1 - currentRect;
            }
        }
    }

   /******************************************************************************
     *  UploadSelf
     ******************************************************************************/
    bool RenderNode::UploadSelf()
    {
        bool ok = true;
        const SizeType listSize = m_effectList.Size();

        for (SizeType i = 0; i < listSize; i++) {
            Effect2D* effect = m_effectList[i].GetPointerToSharedInstance();
            bool isUploaded = true;

            if (effect->GetBrushEffect2D() != 0) {
                isUploaded = effect->GetBrushEffect2D()->IsUploaded();
            }

            const UInt8 count = effect->GetInPlaceEffect2DCount();
            for (UInt8 j = 0; j < count; j++) {
                InPlaceEffect2D* inplaceEffect = effect->GetInPlaceEffect2D(j);
                isUploaded = isUploaded && (inplaceEffect != 0) && inplaceEffect->IsUploaded();
            }

            if (effect->GetBlendEffect2D() != 0) {
                isUploaded = isUploaded && effect->GetBlendEffect2D()->IsUploaded();
            }

            if ((!isUploaded) && (!effect->Upload())) {
                ok = false;
            }
        }

        return ok;
    }

    /******************************************************************************
     *  UnloadSelf
     ******************************************************************************/
    bool RenderNode::UnloadSelf()
    {
        bool ok = true;
        const SizeType listSize = m_effectList.Size();

        for (SizeType i = 0; i < listSize; i++) {
            Effect2D* effect = m_effectList[i].GetPointerToSharedInstance();
            bool isUploaded = true;

            if (effect->GetBrushEffect2D() != 0) {
                isUploaded = effect->GetBrushEffect2D()->IsUploaded();
            }

            const UInt8 count = effect->GetInPlaceEffect2DCount();
            for (UInt8 j = 0; j < count; j++) {
                InPlaceEffect2D* inplaceEffect = effect->GetInPlaceEffect2D(j);
                isUploaded = isUploaded && (inplaceEffect != 0) && inplaceEffect->IsUploaded();
            }

            if (effect->GetBlendEffect2D() != 0) {
                isUploaded = isUploaded && effect->GetBlendEffect2D()->IsUploaded();
            }

            if (isUploaded && (!effect->Unload())) {
                ok = false;
            }
        }

        return ok;
    }

    /******************************************************************************
     *  Clone
     ******************************************************************************/
    RenderNode* RenderNode::Clone() const
    {
        return FEATSTD_NEW(RenderNode)(*this);
    }

    /******************************************************************************
     *  DisposeSelf
     ******************************************************************************/
    void RenderNode::DisposeSelf()
    {
        m_effectList.Clear();
        FEATSTD_DELETE(this);
    }

    /******************************************************************************
     *  OnAncestorAdded
     ******************************************************************************/
    void RenderNode::OnAncestorAdded(Scene2D* scene)
    {
        if (scene != 0) {
            AddToRenderOrderList(scene);
        }
    }

    /******************************************************************************
     *  OnAncestorRemoved
     ******************************************************************************/
    void RenderNode::OnAncestorRemoved(Scene2D* scene)
    {
        if (scene != 0) {
            RemoveFromRenderOrderList(scene);
        }
    }

    /******************************************************************************
     *  AddToRenderOrderList
     ******************************************************************************/
    void RenderNode::AddToRenderOrderList(Scene2D* scene)
    {
        if (scene != 0) {
            static_cast<void>(scene->GetRenderOrder().AddNode(this));
        }
    }

    /******************************************************************************
     *  RemoveFromRenderOrderList
     ******************************************************************************/
    void RenderNode::RemoveFromRenderOrderList(Scene2D* scene) const
    {
        if (scene != 0) {
            static_cast<void>(scene->GetRenderOrder().RemoveNode(this));
        }
    }

    /******************************************************************************
     *  InsertEffect2DPtrToList
     ******************************************************************************/
    bool RenderNode::InsertEffectToList(SizeType pos, Effect2D* effect)
    {
        Effect2D::SharedPointer ptr(effect);
        return m_effectList.Insert(pos, ptr);
    }

    /******************************************************************************
     *  RemoveEffectFromList
     ******************************************************************************/
    bool RenderNode::RemoveEffectFromList(Effect2D* effect)
    {
        SizeType pos = 0;
        const SizeType size = m_effectList.Size();

        while ((pos < size) && (m_effectList[pos].GetPointerToSharedInstance() != effect)) {
            pos++;
        }

        if (pos < size) {
            const BrushEffect2D* brush = effect->GetBrushEffect2D();
#if defined(CANDERA_LAYOUT_ENABLED)
            if (brush != 0) {
                SetLayouter(DefaultLayouter::GetInstance());
            }
#else 
FEATSTD_UNUSED(brush);
#endif

            return m_effectList.Remove(pos);
        }

        return false;
    }
}   // namespace Candera
