//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERA_RenderDirectionHelper_H)
#define CANDERA_RenderDirectionHelper_H

#include <Candera/EngineBase/DynamicProperties/DynamicPropertyHost.h>
#include <Candera/Engine2D/Core/RenderNode.h>
#include <Candera/Engine2D/Effects/InPlaceEffect2D.h>
#include <FeatStd/Util/StaticObject.h>

namespace Candera {
namespace Internal {

/**
 * The RenderDirectionHelper is used to build a bridge between layout direction information and effects that have to use this node related information.
 * The Layouter will provide the information of a right to left layout direction as dynamic property on RenderNodes.
 */
class RenderDirectionHelper : public DynamicProperties::DynamicPropertyHost
{
public:
    /**
     * Set the RightToLeft setting of the current RenderNode for further processing in the effect.
     */
    static void SetCurrentRightToLeft(bool enabled)
    {
        CurrentRightToLeft() = enabled;
    }

    /**
     * Return the RightToLeft setting of the current RenderNode for further processing in the effect.
     */
    static bool IsCurrentRightToLeft()
    {
        return CurrentRightToLeft();
    }

    /**
     * Set the RightToLeft setting of the given node RenderNode for processing in the effect.
     */
    static void SetRightToLeft(RenderNode& node, bool enabled)
    {
        static_cast<void>(node.SetValue(CdaDynamicPropertyInstance(RightToLeft), enabled));
    }

    /**
     * Return the RightToLeft setting of the given node RenderNode for processing in the effect.
     */
    static bool IsRightToLeft(const RenderNode& node)
    {
        return node.GetValue(CdaDynamicPropertyInstance(RightToLeft));
    }

    /**
     * Clear the RightToLeft setting of the given node RenderNode.
     */
    static void ClearRightToLeft(RenderNode& node)
    {
        static_cast<void>(node.ClearValue(CdaDynamicPropertyInstance(RightToLeft)));
    }

    /**
     * Set the RightToLeft awareness setting of the given node InPlaceEffect2D for processing in the effect.
     */
    static void SetUseNodeLayoutDirection(InPlaceEffect2D& inPlaceEffect2D, bool enabled)
    {
        static_cast<void>(inPlaceEffect2D.SetValue(CdaDynamicPropertyInstance(RightToLeft), enabled));
    }

    /**
     * Return the RightToLeft awareness setting of the given node InPlaceEffect2D for processing in the effect.
     */
    static bool GetUseNodeLayoutDirection(const InPlaceEffect2D& inPlaceEffect2D)
    {
        return inPlaceEffect2D.GetValue(CdaDynamicPropertyInstance(RightToLeft));
    }

    /**
     * Clear the RightToLeft awareness setting of the given node InPlaceEffect2D.
     */
    static void ClearUseNodeLayoutDirection(InPlaceEffect2D& inPlaceEffect2D)
    {
        static_cast<void>(inPlaceEffect2D.ClearValue(CdaDynamicPropertyInstance(RightToLeft)));
    }
private:
    friend class Candera::DynamicProperties::DynamicPropertyHost;

    static bool& CurrentRightToLeft()
    {
        FEATSTD_UNSYNCED_STATIC_OBJECT(bool, s_currentRightToLeft, false);
        return s_currentRightToLeft;
    }

    static const Candera::DynamicProperties::DynamicPropertyHost* ParentProvider(const Candera::DynamicProperties::DynamicPropertyHost* host) {
        CANDERA_SUPPRESS_LINT_FOR_NEXT_EXPRESSION(1774, type check above ensures cast is safe)
            return static_cast<const Node2D*>(host)->GetParent();
    }

    CdaDynamicProperties(Candera::Internal::RenderDirectionHelper, Candera::DynamicProperties::DynamicPropertyHost);
        CdaDynamicProperty(RightToLeft, bool);
        CdaDynamicPropertyEnd();
    CdaDynamicPropertiesEnd();
};

}   // namespace Internal
}   // namespace Candera

#endif  // CANDERA_Renderer2D_H
