//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERA_PROXY_IMAGE_2D_H)
#define CANDERA_PROXY_IMAGE_2D_H

#include <Candera/Engine2D/Core/Image2D.h>
#include <Candera/Engine2D/Core/Renderer2DListener.h>
#include <Candera/System/MemoryManagement/MemoryManagement.h>

namespace Candera {

    /** @addtogroup Core2D
     *  @{
     */

    class ImageSource2D;

    /**
     * @brief  Supplies an Image2D interface to an ImageSource2D.
     */
    class ProxyImage2D : public Image2D {
        FEATSTD_TYPEDEF_BASE(Image2D);

        public:
            FEATSTD_RTTI_DECLARATION();

            /**
             *  Creates an instance of this class.
             *  Use Dispose() to delete the instance and possible children, if any.
             *  @return MemoryManagement::SharedPointer to the created object
             */
            static MemoryManagement::SharedPointer<ProxyImage2D> Create();

            /**
             *  Destructor
             */
            ~ProxyImage2D();

            /**
             *  Assign an image source.
             *  @param source        Assigned image source.
             */
            void SetImageSource(ImageSource2D* source) {
                m_listener.m_source = source;
            }

            /**
             *  Retrieves the surface handle provided by the image source.
             *  @return The surface handle to the uploaded data
             */
            virtual SurfaceHandle Get2DSurfaceHandle() const override;

            /**
             *  Retrieves the image source associated with object.
             *  @return A pointer to the ImageSource2D object.
             *
             */
            ImageSource2D* GetImageSource() {
                return m_listener.m_source;
            }

            /**
             *  Get the attached surface Width.
             *  @return The width of the associated surface
             */
            virtual UInt32 GetWidth() const override;

            /**
             *  Get the attached surface Height.
             *  @return The height of the associated surface
             */
            virtual UInt32 GetHeight() const override;

            /**
             * Activates the current image to be used as a certain surface type within
             * the provided context. Overrides method from Image2D.
             * In addition to activation, if synchronization is enabled, this method
             * also calls ImageSource2D::Sync.
             * @param context The context in which the image will be bound.
             * @param surfaceType The usage this image will have within the context.
             * @return  True, if activation of the image succeeded, false otherwise.
             */
            FEATSTD_LINT_NEXT_EXPRESSION(1735,"Default argument on virtual function matches the base-class argument it overrides. (See Image2D.h)")
            virtual bool Activate(ContextHandle2D context, RenderDevice2D::SurfaceType surfaceType = RenderDevice2D::SourceSurface) override;

            /**
             * Sets synchronization enabled or not. Synchronization is enabled by default.
             * @param enable defines if synchronization shall be enabled or not.
             */
            void SetSyncEnabled(bool enable) {
                m_listener.m_isSyncEnabled = enable;
            }

            /**
             * Retrieves if synchronization is enabled or not.
             * @return True if synchronization is enabled, false otherwise.
             */
            bool IsSyncEnabled() const {
                return m_listener.m_isSyncEnabled;
            }

        protected:
            ProxyImage2D();

            virtual bool UploadInternal(LoadingHint /*loadingHint*/) override {
                return true;
            }

            virtual bool UnloadInternal(LoadingHint /*loadingHint*/) override {
                return true;
            }

            virtual void DisposeInternal() override {}

        private:

            class ProxyImageRendererListener : public Renderer2DListener
            {
            public:
                ProxyImageRendererListener();
                ~ProxyImageRendererListener();

                virtual void OnNodePreRender(RenderNode* node) override; 
                virtual void OnNodePostRender(RenderNode* node) override;
                virtual void OnNodePreRender(const RenderNode* node, const Matrix3x2* modelViewMatrix) override;
                virtual void OnNodePostRender(const RenderNode* node, const Matrix3x2* modelViewMatrix) override;

                bool m_isSyncEnabled;
                ImageSource2D* m_source;
            };

            ProxyImageRendererListener m_listener;
    };

    /** @} */ // end of Core2D

}

#endif //CANDERA_PROXY_IMAGE_2D_H
