//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERA_NODE2D_H)
#define CANDERA_NODE2D_H

#include <Candera/Engine2D/Core/2DStringBufferAppenders.h>
#include <Candera/Engine2D/Core/Transformable2D.h>
#include <Candera/EngineBase/EngineBase.h>
#include <Candera/System/Container/LinkedList.h>
#include <Candera/System/Mathematics/Rectangle.h>
#include <Candera/System/Mathematics/MathDataTypes.h>

#include <Candera/EngineBase/Common/ListenerContainer.h>
#include <Candera/Engine2D/Core/Node2DListener.h>

#if defined (CANDERA_LAYOUT_ENABLED)
#include <Candera/EngineBase/Layout/Margin.h>
#include <Candera/EngineBase/Layout/LayoutDataTypes.h>
#include <Candera/EngineBase/Layout/LayoutTypes.h>
#endif

namespace Candera {

/** @addtogroup Core2D
 *  @{
 */

    class Scene2D;
    class Camera2D;
    class RenderTarget2D;
    class Layouter;

    /**
     *  @brief  The class Node2D is an abstract base class for all 2D scene graph nodes.
     *
     *  Each node defines a local coordinate system relative to the coordinate system of the parent node.
     *  The functionality to define the local coordinate system is derived from Transformable2D and consists of
     *  following components: translation, rotation, scale and a generic matrix called transform matrix.
     *  If the node is transformed from the local coordinate system to the world's coordinate system, then
     *  the node's local transformation is multiplied with all its parents transformations.
     *  A Node2D can also store a set of nodes as its children but a node can at most have one parent at a time.
     *  Cycles are prohibited.
     */
    class Node2D : public Transformable2D {

        FEATSTD_TYPEDEF_BASE(Transformable2D);

    public:
        /**
         *  @deprecated
         *  Traversing specifies the method used to traverse attached descendants starting from this Node2D.
         */
        enum Traversing {
            Flat,       ///< Only this node.
            Deep        ///< This node and all its descendants.
        };

        /**
         * Enumerator that defines possible options for the picking point relativity to the screen space.
         * @see IsPickIntersectingBoundingRectangle
         */
        enum PickPointCoordinateSpace
        {
            Absolute, ///< Point coordinates are absolute in screen space
            Relative, ///< Point coordinates are relative to window space
            Automatic ///< Relative if the plaform has a Window Manager, absolute otherwise. @see PlatformDescriptor
        };

        /**
         *  Provides a shallow clone of this node. The new node is not attached
         *  to any node graph, and any other naked pointers, such as strategies
         *  or listeners are not liked to the clone. Strings (objects or type 
         *  const Char*) are an exception from this rule.
         *  @remark               It is not possible to clone Candera::Scene2D.
         *  @return               The cloned Node if successful, null otherwise.
         */
        virtual Node2D* Clone() const = 0;

        /**
         *  Disposes this node and all descendant children Nodes if any.
         */
        void Dispose();

        /**
         *  Retrieves the scene graph parent of this Node
         *  @return         Pointer to parent Node, or null if there is no parent available
         */
        Node2D* GetParent() { return m_parent; }
        /**
         *  Retrieves the scene graph parent of this Node
         *  @return         Pointer to parent Node, or null if there is no parent available
         */
        const Node2D* GetParent() const { return m_parent; }

        /**
         *  Adds a Node2D as child to this node. If the parent node already has child nodes, the node will be
         *  added at the end of the list.
         *  @param  child   The Node2D to add to this node must not form a loop.
         *  @return         true if the adding has been successful. false is returned if child is null,
         *                  the child is this Node, or the child already has a parent,
         *                  or if the Node is created without a NodeContainer and therefore cannot add children.
         */
        bool AddChild(Node2D* child) { return InsertBeforeChild(child, 0); }

        /**
         *  Adds a Node2D as child to this node. The position will be before the child 'beforeChild'.
         *  @param  newChild    The Node2D to add to this node must not form a loop.
         *  @param  beforeChild newChild will be added before beforeChild in the list of child nodes.
         *  @return             true if the adding has been successful. false is returned if child is null,
         *                      the child is this Node, or the child already has a parent,
         *                      or if the Node is created without a NodeContainer and therefore cannot add children.
         */
        bool InsertBeforeChild(Node2D* newChild, Node2D* beforeChild);

        /** Removes the child.
         *  @param   child   The child node that shall be removed
         *  @return  true on success; InvalidParamError if child is 0; UnspecifiedError if the child was not in fact a child of this node.
         */
        bool RemoveChild(Node2D* child);

        /**
         *  Gets a child Node by name.
         *  @param   name     Name of the the child Node to get.
         *  @return           The child node with the given name, or null if there is no child node
         *                   with this name or there is no child attached to this Node.
         */
        Node2D* GetChild(const Char* name);
        /**
         *  Gets a child Node by name.
         *  @param   name     Name of the the child Node to get.
         *  @return           The child node with the given name, or null if there is no child node
         *                   with this name or there is no child attached to this Node.
         */
        const Node2D* GetChild(const Char* name) const;

        /**
         *  Gets a child Node by id.
         *  @param   id       Id of the the child Node to get.
         *  @return           The child node with the given id, or null if there is no child node
         *                    with this id or there is no child attached to this Node.
         */
        Node2D* GetChild(Id id);

        /**
         *  Gets a child Node by id.
         *  @param   id       Id of the the child Node to get.
         *  @return           The child node with the given id, or null if there is no child node
         *                    with this id or there is no child attached to this Node.
         */
        const Node2D* GetChild(Id id) const;

        /**
         *  Retrieves the number of children attached to this Node.
         *  @return  The number of children that are directly attached to this Node.
         */
        UInt32 GetChildCount() const { return static_cast<UInt32>(m_childCount); }


        /**
         *  Gets the first child node (0 if no children). Use together with GetNextSibling to iterate like this:
         *  for (Node2D* child = m_firstChild; child != 0; child = child->GetNextSibling()) {
         *      child->Render();
         *  }
         *  @remark     The order of the returned child is undefined.
         *  @return     Pointer to child node, or null if there is no child available
         */
        Node2D* GetFirstChild() { return m_firstChild; }
        /**
         *  Gets the first child node (0 if no children). Use together with GetNextSibling to iterate like this:
         *  for (Node2D* child = m_firstChild; child != 0; child = child->GetNextSibling()) {
         *      child->Render();
         *  }
         *  @remark     The order of the returned child is undefined.
         *  @return     Pointer to child node, or null if there is no child available
         */
        const Node2D* GetFirstChild() const { return m_firstChild; }

        /**
         *  Gets the next sibling (0 if no more siblings).
         *  @remark     The order of the returned sibling is undefined.
         *  @return     Pointer to sibling node, or null if there is no sibling available
         */
        CANDERA_SUPPRESS_LINT_FOR_SYMBOL(1762, Candera::Node2D::GetNextSibling, CANDERA_LINT_REASON_NONCONSTMETHOD)
        Node2D* GetNextSibling() { return m_childListNode.GetNext(); }
        /**
         *  Gets the next sibling (0 if no more siblings).
         *  @remark     The order of the returned sibling is undefined.
         *  @return     Pointer to sibling node, or null if there is no sibling available
         */
        const Node2D* GetNextSibling() const { return m_childListNode.GetNext(); }

        /**
         *  Retrieve the single Scene2D node that is ancestor to this node.
         *  @return Scene2D node if available, 0 otherwise.
         */
        virtual Scene2D* GetScene() const;

        /**
         *  Defines the local setting if rendering shall be enabled or not.
         *  This boolean value influences this node's descendants. See function IsEffectiveRenderingEnabled() for further details.
         *  @param enable true if rendering shall be enabled
         */
        void SetRenderingEnabled(bool enable) {m_isRenderingEnabled = enable; }

        /**
         *  Retrieve this node's local setting, if rendering is enabled or disabled.
         *  To compute if effective rendering is enabled in dependency to this node's ancestors, see
         *  function IsEffectiveRenderingEnabled() for further details.
         *  @return true if rendering is enabled
         */
        bool IsRenderingEnabled() const { return m_isRenderingEnabled; }

        /**
         *  Retrieve whether rendering is enabled or not in dependency to this node's ancestors settings.
         *  @return      true if this node AND all ancestor nodes have set rendering enabled.
         *              false if either this node OR any ancestor node have set rendering disabled.
         */
        bool IsEffectiveRenderingEnabled() const;

        /**
         *  Retrieves the world transformation matrix of this node.
         *  Multiplies local transformation with transformations of all ancestors to this node.
         *  As internal cache is used, recalculation of world transform occurs only, if local transform or any transform of an ancestor to this node changes.
         *  @return The world transformation matrix of this node
         */
        const Matrix3x2& GetWorldTransform() const;

        /**
         *  Returns the position of this node in world coordinates.
         *  @return     The position of this node in world coordinates.
         */
        Vector2 GetWorldPosition() const;

        /**
         *  Determines if the given point is inside the bounding rectangle associated to the node.
         *  @param   point       Specifies a point in world space coordinates.
         *  @return              true if the given point is inside the bounding rectangle, false otherwise.
         */
        bool IsInsideBoundingRectangle(const Vector2& point, bool ignoreInvisible = false) const;

        /**
         *  Determines if the given point is inside the area associated to the node.
         *  @param  camera      The view camera.
         *  @param  point       Defines the position in screen/window space coordinates (depends on next parameter).
         *  @param  space       Specifies if the point position is given in screen or window coordinates.
         *  @return             true if the given point is inside the area associated to the node
         */
        bool IsPickIntersectingBoundingRectangle(const Camera2D& camera, const Vector2& point, PickPointCoordinateSpace space = Automatic, bool ignoreInvisible = false) const;

        /**
         *  Gets the axis-aligned bounding rectangle in world coordinate space.
         *  @param  boundingRectangle   Out parameter for the axis-aligned bounding rectangle in world coordinate space.
         *  @param  traverse            Determines whether to get the union of all child node's bounding rectangles (Deep) or
         *                              only the bounding rectangle of that node (Flat).
         */
        void GetWorldAxisAlignedBoundingRectangle(Rectangle& boundingRectangle, Traversing traverse = Flat) const;

        /**
         *  Get oriented bounding rectangle in world coordinate space.
         *  Only the current node is considered and will not include child nodes.
         *  @param   points  Out parameter containing the 4 points of the bounding rectangle in world space.
         */
        void GetWorldOrientatedBoundingRectangle(Vector2 points[4]) const;

        /**
         *  Gets the base point in local coordinate space. The base point is typically in the center of a bitmap,
         *  or on the base line of a text.
         *  @param  basePoint   Out parameter for the base point in local coordinate space.
         */
        virtual void GetBasePoint(Vector2& basePoint) const;

        /**
         *  Upload asset data attached to this node to video memory.
         *  @param  scopeMask   Only nodes that are in scope of the given ScopeMask will be uploaded.
         *  @param  traverse    Determines whether upload shall be done for this node only or for all its descendants also.
         *  @return             true if upload succeeded and false if upload failed.
         */
        bool Upload(const ScopeMask& scopeMask = ScopeMask(), Traversing traverse = Flat);

        /**
         *  Unload asset data attached to this node from video memory.
         *  @param  scopeMask   Only nodes that are in scope of the given ScopeMask will be unloaded.
         *  @param  traverse    Determines whether unload shall be done for this node only or for all its descendants also.
         *  @return             true if unload succeeded and false if unload failed.
         */
        bool Unload(const ScopeMask& scopeMask = ScopeMask(), Traversing traverse = Flat);

        /**
         *  Sets ScopeMask that defines all scope memberships of this node independent from the scene graph hierarchy.
         *  If deep traversing is specified the scope mask is set also of all descendants of this node.
         *  @param   scopeMask   Specifies the entire scope mask of a node, thus all scopes a node belongs to.
         *  @param   traverse    Tells if the scope mask shall be set for this node only or for all its descendants also.
         */
        void SetScopeMask(const ScopeMask& scopeMask, Traversing traverse = Flat);

        /**
         *  Retrieves ScopeMask of this node.
         *  @return      The ScopeMask defining all scope memberships of this node.
         */
        const ScopeMask& GetScopeMask() const { return m_scopeMask; }

        /**
         *  Sets dedicated scope of this node and - if deep traversing is specified - also of all descendants of this node.
         *  @param   scopeIndex  Specifies the scope to set.
         *  @param   enable      Defines if scope is enabled or not. Thus, if the node belongs to the scope specified or not.
         *  @param   traverse    Tells if the scope shall be set for this Node only or for all its descendants also.
         *  @return true if scope index is within allowed indices (for details, see class ScopeMask), false otherwise.
         */
        bool SetScopeEnabled(UInt scopeIndex, bool enable, Traversing traverse = Flat);

        /**
         *  Retrieves if this Node belongs to a certain scope or not.
         *  @param scopeIndex specifies to scope to interrogate.
         *  @return true if Node belongs to the scope specified and false otherwise.
         */
        bool IsScopeEnabled(UInt scopeIndex) const { return m_scopeMask.IsScopeEnabled(scopeIndex); }

        /**
         *  Retrieves if this Node is in Scope of the scope mask given. Thus, at least one scope of this Node is enabled that is
         *  also enabled in the scope mask passed.
         *  @param scopeMask defines all scopes to test this Node's scopes against.
         *  @return true if this Node is in scope of the scopes defined by the scope mask given, false otherwise.
         */
        bool IsInScopeOf(const ScopeMask& scopeMask) const { return m_scopeMask.OverlapsWith(scopeMask); }

#if defined(CANDERA_LAYOUT_ENABLED)
        /**
         *  Sets a layout strategy for the child nodes, e.g. a grid or stack layout.
         *  The attached layouter is Disposed when the node is disposed.
         *  @param layouter Layout strategy for child nodes.
         */
        void SetLayouter(Layouter* layouter);

        /**
         *  Gets the layout strategy for the child nodes.
         *  @return Layouter layout the child nodes.
         */
        Layouter* GetLayouter() const;

        /**
         *  Starts the layout for this node, if a layouter has been assigned with SetLayouter().
         */
        void Layout();
#endif

        /**
         *  Sets the local alpha value for this node.
         *  @param  alphaValue      Values allowed between [0.0F, 1.0F] at which 0.0F means fully transparent (invisible), 1.0F means fully opaque.
         *                          This can be used to define translucency of groups.
         *                          It is the responsibility of Node specializations to actually use the alpha value; also, see GetEffectiveAlphaValue().
         */
        void SetAlphaValue(Float alphaValue);

        /**
         *  Gets the local alpha value for this node.
         *  @return     The alpha value of this node: [0.0F, 1.0F]; 0.0F means fully transparent (invisible), 1.0F means fully opaque.
         *              The alpha value returned is the local alpha value and therefore not multiplied with the alpha values of any ancestor to this node.
         */
        Float GetAlphaValue() const;

        /**
         *  Gets the product of the alpha values of this Node and all ancestors.
         *  @return     Product of the alpha values of this Node and all ancestors.
         */
        Float GetEffectiveAlphaValue() const;

        /**
         *  Sets the bounding rectangle for this node to a fixed value.
         *  As long as the bounding rectangle for the node is set manually with this method, it is no longer calculated depending on its content dimensions.
         *  @param  boundingRectangle   The bounding rectangle to be set.
         */
        void SetBoundingRectangle(const Rectangle& boundingRectangle);

        /**
         *  Retrieves the manually set bounding rectangle.
         *  @return     The set bounding rectangle.
         */
        const Rectangle& GetBoundingRectangle() const;

        /**
         *  Check if there is a bounding rectangle set for this node.
         *  @return     True if the bounding rectangle is set.
         */
        bool IsBoundingRectangleSet() const;

        /**
         *  Clears the set bounding rectangle.
         *  From now on the bounding rectangle will be calculated depending on its content dimensions.
         *  @return     true if successful, false otherwise.
         */
        bool ClearBoundingRectangle();

        /**
         *  Gets the axis-aligned bounding rectangle in local coordinate space.
         *  If the bounding rectangle has been set with SetBoundingRectangle() then this rectangle is returned, otherwise
         *  the bounding rectangle is computed by its actual content (e.g. boundaries of the image, text...) by calling
         *  GetComputedBundingRectangle() internally.
         *  @param[out] boundingRectangle   Out parameter for the axis-aligned bounding rectangle in local coordinate space.
         */
        void GetEffectiveBoundingRectangle(Rectangle& boundingRectangle, bool ignoreInvisible = false) const;

        /**
         *  Gets the axis-aligned bounding rectangle in local coordinate space which is computed by its
         *  actual content (e.g. boundaries of the image, text...).
         *  @param   boundingRectangle  Out parameter for the axis-aligned bounding rectangle in local coordinate space.
         */
        FEATSTD_LINT_NEXT_EXPRESSION(1735, "Virtual function does not override a base-class function.")
        virtual void GetComputedBoundingRectangle(Rectangle& boundingRectangle, bool ignoreInvisible = false) const
        {
            boundingRectangle = Rectangle();
            FEATSTD_UNUSED(ignoreInvisible);
        }

#if defined(CANDERA_LAYOUT_ENABLED)
        /**
         *  Retrieves the manually set layout rectangle.
         *  @return     The set layout rectangle.
         */
        CANDERA_DEPRECATED_3_2_0(
            "The layout rectangle is made obsolete by the introduction of TextNode2DLayouter. "
            "Use the dynamic property Size, accessed through Layouter::GetSize(const Node2D&)",
            const Rectangle& GetLayoutingRectangle() const);

        /**
         *  Gets the axis-aligned layout rectangle in local coordinate space.
         *  If the layout rectangle has been set with SetLayoutingRectangle()
         *  then this rectangle is returned, otherwise the layout rectangle
         *  is computed by calling GetComputedBundingRectangle() internally.
         *  @param rectangle  [out] Out parameter for the axis-aligned
         *                      layout rectangle in local coordinate space.
         */
        CANDERA_DEPRECATED_3_2_0(
            "Use Layouter::GetEffectiveLayoutRectangle(Node2D&) instead.",
            void GetEffectiveLayoutingRectangle(Rectangle& rectangle) const);

        /**
         *  Gets the layout rectangle in local coordinate space.
         *  The layout rectangle may be different from the bounding box when
         *  effects like text, glowing or shadow are involved. This is the
         *  rectangle that should be used when determining the position of this
         *  node.
         *  @param  rectangle  Out parameter for the layout rectangle.
         */
        CANDERA_DEPRECATED_3_2_0(
            "Use GetComputedLayoutRectangle(Rectangle&) instead.",
            virtual void GetComputedLayoutingRectangle(Rectangle& rectangle) const);

        virtual void GetComputedLayoutRectangle(Rectangle& rectangle) const
        {
            GetComputedBoundingRectangle(rectangle);
        }

#ifdef CANDERA_DEPRECATED_3_2_0
        /// Deprecated function, added to allow implementation of other deprecated functions.
        const Rectangle& GetLayoutingRectangleDeprecated() const;
        /// Deprecated function, added to allow implementation of other deprecated functions.
        bool IsLayoutingRectangleSetDeprecated() const;
        /// Deprecated function, added to allow implementation of other deprecated functions.
        void GetEffectiveLayoutingRectangleDeprecated(Rectangle& rectangle, bool ignoreInvisible = false) const;
#else
#error Remove this!
#endif
#endif
        /**
         *  The render order rank defines the relative value within render order. Nodes with lower render order rank are rendered 
         *  before nodes with higher value, and therefore appear in the background. The effective render order rank is the sum of 
         *  this node's and all its ancestor nodes render order rank values.
         *  @param  renderOrderRank     Specifies the rank within the render order.
         */
        void SetRenderOrderRank(Int renderOrderRank);

        /**
         *  The render order rank defines the relative value within render order. Nodes with lower render order rank are rendered 
         *  before nodes with higher value, and therefore appear in the background. The effective render order rank is the sum of 
         *  this node's and all its ancestor nodes render order rank values.
         *  @return     Specifies the rank within the render order.
         */
        Int GetRenderOrderRank() const;

        /**
         *  @return     The effective render order rank is the sum of this node's and all its ancestor nodes render order rank values.
         */
        Int GetEffectiveRenderOrderRank() const;

        /**
        *  Set the render measure value of this Node2D, which is a benchmark related to rendering time, typically calculated by
        *  external tool and utilized by a camera render strategy (see class Camera2DRenderStrategy).
        *  @param benchmark Render measure value of this Node2D to be set.
        */
        void SetRenderBenchmark(Float benchmark);

        /**
        *  Retrieves the render measure value of this Node2D, which is a benchmark related to rendering time, typically calculated by
        *  external tool and utilized by a camera render strategy (see class Camera2DRenderStrategy).
        *  @return The render measure value of this Node2D.
        */
        Float GetRenderBenchmark() const;

        /**
        *  Adds a Node2DListener object.
        *  @param listener The lifetime of the listener is managed by the calling code. Node does not take ownership.
        *  @return True if successful, false otherwise.
        */
        bool AddNodeListener(Node2DListener* listener);

        /**
        *  Removes a NodeListener object.
        *  @param listener Specifies the NodeListener object to remove from this Node.
        *  @return True if successful, false otherwise.
        */
        bool RemoveNodeListener(Node2DListener* listener);

        FEATSTD_RTTI_DECLARATION();

        /**
         *  Retrieves a descendant Node by name.
         *  A depth search explores all Nodes and descendants to this Node.
         *  In other words, a Node can be found that is either a child of this Node or any other Node that is descendant to this Node.
         *  @param  id      Identifier of Node.
         *  @return         The first Node found with the name given. Returns null if no attached Node matches the name given.
         */
        CANDERA_SUPPRESS_LINT_FOR_SYMBOL(1762, Candera::Node2D::GetDescendant, CANDERA_LINT_REASON_NONCONSTMETHOD)
        Node2D* GetDescendant(const Char* id);
        /**
         *  Retrieves a descendant Node by name.
         *  A depth search explores all Nodes and descendants to this Node.
         *  In other words, a Node can be found that is either a child of this Node or any other Node that is descendant to this Node.
         *  @param  id      Identifier of Node.
         *  @return         The first Node found with the name given. Returns null if no attached Node matches the name given.
         */
        const Node2D* GetDescendant(const Char* id) const;

        /**
         *  Provides the parent for dynamic property scene graph inheritance.
         *  @param node The node, whose parent is sought.
         *  @return The parent of node if available, null otherwise.
         */
        static const Candera::DynamicProperties::DynamicPropertyHost* ParentProvider(const Candera::DynamicProperties::DynamicPropertyHost *node) {
            CANDERA_SUPPRESS_LINT_FOR_NEXT_EXPRESSION(1774, type check above ensures cast is safe)
            return static_cast<const Node2D*>(node)->GetParent();
        }

    protected:
        template<typename T> friend class TreeTraverserBase;

        Node2D();
        explicit Node2D(const Node2D& node);

        /**
         *  Destructor
         */
        virtual ~Node2D() override;

        /**
         *  When Upload is called, traversing through all child nodes calls this method for each node.
         *  @return             Pure virtual function that must be overridden in each object derived from Node2D.
         *                      Derived classes return whether uploading was successful or not.
         */
        virtual bool UploadSelf() { return true; }

        /**
         *  When Unload is called, traversing through all child nodes calls this method for each node.
         *  @return             Pure virtual function that must be overridden in each object derived from Node2D.
         *                      Derived classes return whether uploading was successful or not.
         */
        virtual bool UnloadSelf() { return true; }

        Node2D* CloneTraversing(TraverseOperation traverseOperation, CloneOperation cloneOperation) const;

        /**
         *  Disposes this node only - not descendant children nodes.
         *  Pure virtual function that must be overridden in each object derived from Node2D.
         */
        virtual void DisposeSelf() = 0;

        /**
         * Override from Transformable2D. Invalidates the world transform cache of all descendants. Do not override.
         */
        virtual void OnCompositeTransformChanged() override;

        /** 
         *  Reacts to a transformation change further up in the tree. The reaction should be local/flat, i.e. limited
         *  to this Node2D (not propagated recursively to descendants).
         */
        virtual void OnAncestorTransformChanged();

        /**
         *  Notification that an ancestor to this node has been added.
         *  An ancestor is each node up the tree hierarchy that has a parent relationship to this node.
         *  An ancestor can therefore be reached via GetParent(), recursively.
         *  Any derived class might override this function in order to react to this event.
         *  At the time the notification OnAncestorAdded is invoked, the parent of this Node is already set and the
         *  node's WorldTransformCache is invalidated.
         *  @param  scene       The scene node which became root of this node's scene graph.
         */
        virtual void OnAncestorAdded(Scene2D* scene) {
            FEATSTD_UNUSED(scene);
        }

        /**
         *  Notification that an ancestor to this node has been removed.
         *  An ancestor is each node up the tree hierarchy that has a parent relationship to this node.
         *  An ancestor can therefore be reached via GetParent(), recursively.
         *  Any derived class might override this function in order to react to this event.
         *  At the time the notification OnAncestorRemoved is invoked, the ancestor of this node is already
         *  disconnected to the scene.
         *  @param  scene       The scene node which previously had been the root of the scene graph this node was
         *                      connected with.
         */
        virtual void OnAncestorRemoved(Scene2D* scene) {
            FEATSTD_UNUSED(scene);
        }

        /**
         *  Notification that a descendant of this Node2D has been removed.
         *  Any derived class might override this function in order to react to this event.
         *  At the time the notification OnDescendantRemoved is invoked, the descendant of this node is already
         *  disconnected from the node.
         *  @param descendant The descendant Node2D that was removed.
         */
        virtual void OnDescendantRemoved(Node2D* descendant) {
            FEATSTD_UNUSED(descendant);
        }

    private:
        friend class InvalidateWorldTransformCacheTraverser2D;
        friend class InvalidateEffectiveAlphaValueCacheTraverser2D;
        friend class UploadTraverser2D;
        friend class UnloadTraverser2D;
        friend class OnAncestorAddedTraverser2D;
        friend class OnAncestorRemovedTraverser2D;
        friend class CloneTraverser2D;
        friend class Renderer2D;
        friend class Layouter;
        friend class Node2DDynamicProperties;

        mutable bool m_isWorldTransformCacheValid : 1;
        mutable bool m_isEffectiveAlphaValueCacheValid : 1;
        bool m_isRenderingEnabled : 1;
        FEATSTD_LINT_NEXT_EXPRESSION(1960, "MISRA 9-6-2: Used type is a typedef to explicitly signed/unsigned base type.")
        UInt16 m_childCount : 16;
        Node2D* m_parent;   // parent to this Node2D.
        Node2D* m_firstChild;
        mutable Matrix3x2 m_worldTransformCache;
        mutable Float m_effectiveAlphaValueCache;
        ScopeMask m_scopeMask;
        // intrusive linked list node for membership in the list of children of the parent
        Internal::LinkedListNode<Node2D> m_childListNode;

        /**
         * Sets the scene graph parent of this Node.
         * @param parent Scene graph parent of this Node to be set.
         */
        void SetParent(Node2D* parent);

        /**
         * Invalidates current world transform cache.
         * Forces this node to recalculate its world transform and to update its internal world transform cache the next time the function
         * GetWorldTransform() is invoked.
         */
        void InvalidateWorldTransformCache() { m_isWorldTransformCacheValid = false; }
        /**
         * Returns whether the world transform cache is valid
         * @return true if cache is valid
         */
        bool IsWorldTransformCacheValid() const { return m_isWorldTransformCacheValid; }

        /**
         * Invalidates current alpha value cache.
         * Forces this node to recalculate its effective alpha value and to update its internal alpha value cache the next time the function
         * GetEffectiveAlphaValue() is invoked.
         */
        void InvalidateEffectiveAlphaValueCache() const { m_isEffectiveAlphaValueCacheValid = false; }
        /**
         * Returns whether the alpha value cache is valid
         * @return true if cache is valid
         */
        bool IsAlphaValueCacheValid() const { return m_isEffectiveAlphaValueCacheValid; }

#if defined(CANDERA_LAYOUT_ENABLED)
        friend class AbstractNodePointer;
        void SetPreferredSize(const Vector2& preferredSize);
        const Vector2& GetPreferredSize() const;
        bool ClearLayouter();
        virtual void OnBeforeLayouterSet(Layouter*) {}
#endif

        typedef Internal::ListenerContainer<Node2DListener> Node2DListenerContainer;
        Node2DListenerContainer m_Node2DListeners;
        class OnNodeRemovedEvent;
        class OnTransformChangeEvent;
        class OnNodeAddedEvent;
        class OnChildAddedEvent;
        class OnChildRemovedEvent;

        void NotifyListenersOnNodeRemoved(Node2D* parent);
        void NotifyListenersOnTransformChange(Node2D* node);

        // Notifies all listeners that a parent has been set for this 2D node.
        void NotifyListenersOnNodeAdded(Node2D* parent);

        // Notifies all listeners that a new child has been added to this 2D node.
        void NotifyListenersOnChildAdded(Node2D* child);

        // Notifies all listeners that a child has been removed from this 2D node.
        void NotifyListenersOnChildRemoved(Node2D* child);

        CdaDynamicPropertiesDeclaration();
    };

 /** @} */ // end of Core2D

}   // namespace Candera

#endif  // CANDERA_NODE2D_H
