//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERA_MESH2D_H)
#define CANDERA_MESH2D_H

#include <Candera/Engine2D/Core/RenderNode.h>
#include <Candera/Engine2D/Core/VertexBuffer2D.h>

namespace Candera {

    /** @addtogroup Core2D
     *  @{
     */

     /**
      * @brief   The class Mesh2D represents a two-dimensional rigid body object defined by polygons. The Mesh2D's polygonal surface is defined by its VertexBuffer2D.
      *          A Mesh2D inherits from RenderNode and can be applied to any effect.
      *          Note: NinePatch images are not compatible with the Mesh2D and are used just as normal textures.
      */
    class Mesh2D : public RenderNode {

        FEATSTD_TYPEDEF_BASE(RenderNode);

    public:
        /**
         *  Creates an instance of this class.
         *  Use Dispose() to delete the instance and possible children, if any.
         *  @return Pointer to the created object
         */
        CANDERA_SUPPRESS_LINT_FOR_SYMBOL(1511, Candera::Mesh2D::Create, CANDERA_LINT_REASON_EXPLICITHIDING)
            static Mesh2D* Create();

        /**
         *  Destructor.
         */
        virtual ~Mesh2D() override;

        /**
         *  Clones this Mesh2D only and creates a shallow clone.
         *  In case VertexBuffer2D needs to be cloned, it can be cloned separatedly using the VertexBuffer2D::Clone function.
         *  Attached Node resources like  VertexBuffer2D are not deep-copied but referenced.
         *  @return  The pointer to the cloned Mesh2D if successful, otherwise NULL.
         */
        virtual Mesh2D* Clone() const override;

        /**
         *  Sets the VertexBuffer2D that will be assigned to this Mesh2D, which defines the geometry attributes.
         *  @param vertexBuffer2D The VertexBuffer2D that is set.
         */
        void SetVertexBuffer2D(VertexBuffer2D::SharedPointer vertexBuffer2D) {
            m_vertexBuffer2D = vertexBuffer2D;
        }

        /**
         *  Retrieves the VertexBuffer2D assigned to this Mesh2D, which defines the geometry attributes.
         *  @return VertexBuffer2D assigned to this Mesh2D.
         */
        VertexBuffer2D::SharedPointer GetVertexBuffer2D() const { return m_vertexBuffer2D; }

        /**
         *  Gets the axis-aligned bounding rectangle in local coordinate space which is computed by its
         *  VertexBuffer2D. An attached effect cannot change the bounding rectangle,as it is completely determined by the VertexBuffer2D.
         *  @param   boundingRectangle  Out parameter for the axis-aligned bounding rectangle in local coordinate space.
         *  @param ignoreInvisible
         */
        FEATSTD_LINT_NEXT_EXPRESSION(1735, "Virtual function uses same default parameter as abstract base class.")
        virtual void GetComputedBoundingRectangle(Rectangle& boundingRectangle, bool ignoreInvisible = false) const override;

#if defined(CANDERA_LAYOUT_ENABLED)
        /**
        *  Returns the computed bounding rectangle as layout rectangle.
        *  An attached effect cannot change the bounding rectangle,as it is completely determined by the VertexBuffer2D.
        *  Note: The bounding rectangle can only be calculated once if the VertexBuffer2D is uploaded with the DisposedAfterUpload flag enabled.
        *  @param rectangle The computed bounding rectangle as layout rectangle.
        */
        virtual void GetComputedLayoutRectangle(Rectangle& rectangle) const override;
#endif


        FEATSTD_RTTI_DECLARATION();

    protected:
        // Explicit protected Constructor and Copy-Constructor, use Create() to create an instance of this object.
        Mesh2D();
        explicit Mesh2D(const Mesh2D& rhs);
        Mesh2D& operator = (const Mesh2D& rhs);

        virtual void Render(RenderTarget2D* renderTarget, const Matrix3x2& localTransform) override;

        virtual bool UploadSelf() override;
        virtual bool UnloadSelf() override;

        virtual void DisposeSelf() override;

    private:
        VertexBuffer2D::SharedPointer m_vertexBuffer2D;


        CdaDynamicProperties(Candera::Mesh2D, Candera::RenderNode);
        CdaDynamicPropertiesEnd();
    };


    /** @} */ // end of Core2D

}   // namespace Candera

#endif  // CANDERA_MESH2D_H
