//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "Mesh2D.h"
#include <Candera/Engine2D/Core/VertexGeometry2D.h>
#include <CanderaPlatform/Device/Common/Base/RenderDevice2D.h>
#include <CanderaPlatform/Device/Common/Base/RenderTarget2D.h>

namespace Candera {
    FEATSTD_LOG_SET_REALM(Diagnostics::LogRealm::CanderaEngine2D);

    Mesh2D::Mesh2D():
        Base(),
        m_vertexBuffer2D(0)
    {

    }

    Mesh2D::Mesh2D(const Mesh2D& rhs):
        Base(rhs),
        m_vertexBuffer2D(rhs.m_vertexBuffer2D)
    {

    }


    Mesh2D::~Mesh2D()
    {

    }

    Mesh2D* Candera::Mesh2D::Create()
    {
        return FEATSTD_NEW(Mesh2D);
    }

    Mesh2D * Mesh2D::Clone() const
    {
        return FEATSTD_NEW(Mesh2D)(*this);
    }


    void Mesh2D::GetComputedBoundingRectangle(Rectangle& boundingRectangle, bool ignoreInvisible) const
    {
        if (ignoreInvisible && (!IsEffectiveRenderingEnabled())) {
            boundingRectangle = Rectangle();
        }
        else {

            if (!m_vertexBuffer2D.PointsToNull()) {
                boundingRectangle = m_vertexBuffer2D->GetBoundingRectangle();
            }
            else {
                boundingRectangle = Rectangle();
                 FEATSTD_LOG_ERROR("In order to use a Mesh2D a VertexBuffer2D must be set."
                "Bounding box and layout rectangle set to empty. See Mesh2D: ",
                (GetName() != 0) ? GetName() : "<Unnamed>");
            }

#ifdef CANDERA_LAYOUT_ENABLED
            if (ignoreInvisible) {
#ifdef CANDERA_LAYOUT_CLIPPING_ENABLED
                boundingRectangle.Intersect(this->GetClippingRect());
#else
                boundingRectangle.Intersect(Layouter::GetClippingArea(*this));
#endif
            }
#endif
        }
    }

#if defined(CANDERA_LAYOUT_ENABLED)

    void Mesh2D::GetComputedLayoutRectangle(Rectangle& rectangle) const
    {
        Rectangle rect;
        GetComputedBoundingRectangle(rect);
        rectangle = rect;
    }

#endif

    void Mesh2D::Render(RenderTarget2D * renderTarget, const Matrix3x2 & localTransform)
    {
        Effect2D* effect = GetEffect(0);
        if (0 != effect) {
            if(effect->IsMesh2DSupported()) {
                if ((!m_vertexBuffer2D.PointsToNull()) && (renderTarget != 0)) {
                    if (m_vertexBuffer2D->IsUploaded()) {
                        GeometryHandle handle = m_vertexBuffer2D->Get2DGeometryHandle();
                        if (0 != handle) {
                            static_cast<void>(RenderDevice2D::SetGeometry(renderTarget->Get2DContextHandle(), handle, m_vertexBuffer2D->GetBoundingRectangle())); //Activate Geometry
                            Base::Render(renderTarget, localTransform);
                            static_cast<void>(RenderDevice2D::SetGeometry(renderTarget->Get2DContextHandle(), 0));      //Deactivate Geometry
                        }
                    }
                }
            }
            else {
                FEATSTD_LOG_INFO("Effect: %s is not supported to be rendered with Mesh2D. \n", effect->GetDynamicTypeId());
            }
        }
    }

    bool Mesh2D::UploadSelf()
    {
        bool result = false;
        if (!m_vertexBuffer2D.PointsToNull()) {
            result = Base::UploadSelf();
            result = result && m_vertexBuffer2D->Upload();
        }
        return result;
    }

    bool Mesh2D::UnloadSelf()
    {
        bool result = false;
        if (!m_vertexBuffer2D.PointsToNull()) {
            result = Base::UnloadSelf();
            result = result && m_vertexBuffer2D->Unload();
        }
        return result;
    }

    void Mesh2D::DisposeSelf()
    {
        FEATSTD_DELETE(this);
    }

    FEATSTD_RTTI_DEFINITION(Mesh2D, RenderNode)
}
