//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERA_DEVICEOBJECT2D_H)
    #define CANDERA_DEVICEOBJECT2D_H

//this class has been customized by Bosch
#define CANDERA_CUSTOMIZATION_DEVICE_OBJECT_2D


#include <FeatStd/Container/SingleLinkedList.h>

#include <Candera/EngineBase/Common/CanderaObject.h>

namespace Candera {

    class DeviceObject2DListener;

    /** @addtogroup CommonBase
     *  @{
     */

    /**
     * @brief DeviceObject2D is the pure abstract base class for all objects which are able to
     *        upload or unload itself from Video Device Memory. (VRAM)
     *        DeviceObject2D encapsulates also the upload reference counting mechanism.
     */
    class DeviceObject2D : public CanderaObject {
        FEATSTD_TYPEDEF_BASE(CanderaObject);
    
        public:

            /**
            * Loading hint for uploading and unloading.
            */
            enum LoadingHint
            {
                WarnWhileRendering,   ///< Up-/Unloading while rendering throws a warning.
                NoHint,               ///< Up-/Unloading while rendering does not throw a warning.
                Force                 ///< Force an unload from the currently active context resource pool, regardless of the current state of the object, the reference count will be reset.
            };

            /**
             *  Constructor.
             */
            DeviceObject2D();

            /**
             *  Destructor.
             */
            virtual ~DeviceObject2D() override;

#ifdef CANDERA_CUSTOMIZATION_DEVICE_OBJECT_2D
            UInt GetUploadReferenceCount() const
            {
               return m_uploadReferenceCount;
            }
#endif

            /**
             *  Uploads the data to Video Memory. (VRAM) The upload functionality is reference counted.
             *  See details in the overridden implementation of UploadInternal.
             *  @return true if object is successfully uploaded, false otherwise.
             */
            bool Upload(LoadingHint loadingHint = WarnWhileRendering);

            /**
             *  Unloads the data to Video Memory. (VRAM) The unload functionality is reference counted.
             *  See details in the overridden implementation of UnloadInternal.
             *  @return true if unload count has been decreased or object is successfully unloaded, false if object cannot be unloaded.
             */
            bool Unload(LoadingHint loadingHint = WarnWhileRendering);

            /**
            *   Check whether the object was uploaded to the current active context resource pool.
            *   @return true if the object was successfully uploaded to VRAM, false otherwise.
            */
            bool IsUploaded() const {
                return m_uploadReferenceCount > 0;
            }

            /**
             *  Sets the following behavior: After a successful upload, the asset data (often residing in system memory) is disposed.
             *  By default, this behavior is NOT enabled.
             *  Use this to free system memory if you are sure that only the representation uploaded to VRAM is needed any more.
             *  @param disposeAfterUpload Enables(true)/Disables(false) the behavior described above.
             */
            void SetDisposedAfterUpload(bool disposeAfterUpload) {
                m_isDisposedAfterUpload = disposeAfterUpload;
            }

            /**
             *  Retrieves whether the asset data is disposed after upload or not.
             *  @return Boolean whether the asset data is disposed after upload or not.
             */
            bool IsDisposedAfterUpload() const {
                return m_isDisposedAfterUpload;
            }

            /**
             * @brief Register an action listener to this device object.
             *
             * The listener will be notified when the DeviceObject is uploaded,
             *  unloaded or destroyed.
             *
             * @param deviceObject2DListener Listener that will receive
             *  action notifications.
             * @return true if listener was successful registered and will
             *  receive notifications.
             * @return false if listener could not be registered.
             *
             * @see DeviceObject2DListener
             */
            bool AddListener(DeviceObject2DListener* deviceObject2DListener) {
                return m_listenerList.Append(deviceObject2DListener);
            }

            /**
             * @brief Unregister an action listener of this device object.
             *
             * The listener will not receive action notifications anymore.
             *
             * @param deviceObject2DListener Listener that will not receive
             *  notifications anymore.
             * @return true if listener was successful unregistered.
             * @return false if listener could not be unregistered.
             *
             * @see DeviceObject2DListener
             */
            bool RemoveListener(DeviceObject2DListener* deviceObject2DListener) {
                return m_listenerList.Remove(deviceObject2DListener);
            }

            FEATSTD_RTTI_DECLARATION();
        protected:
            /**
             *  Implementation for child classes where the actual upload happens.
             *  @return     true if the object was successfully uploaded to VRAM, false otherwise.
             */
            virtual bool UploadInternal(LoadingHint loadingHint) = 0;

            /**
             *  Implementation for child classes where the actual unload happens.
             *  @return     true if the object was successfully unloaded from VRAM, false otherwise.
             */
            virtual bool UnloadInternal(LoadingHint loadingHint) = 0;

            /**
             *  Implementation for child classes where the data are disposed after upload if DisposeAfterUpload is set.
             *  Derived classes must override pure virtual function to define whether or how data is disposed.
             */
            virtual void DisposeInternal() = 0;

        private:
            typedef FeatStd::Internal::SingleLinkedList<DeviceObject2DListener*> ListenerList;

            // Reference counters for VRAM upload.
            UInt m_uploadReferenceCount : 16;
            bool m_isDisposedAfterUpload : 1;
            ListenerList m_listenerList;

            void NotifyListenersOnPreUpload();
            void NotifyListenersOnPostUpload();
            void NotifyListenersOnPreUnload();
            void NotifyListenersOnPostUnload();
            void NotifyListenersOnDestruction();
    };

    /** @} */ // end of CommonBase
} // namespace Candera

#endif    // CANDERA_DEVICEOBJECT2D_H
