//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#ifndef Candera_Camera2DRenderStrategy_h
    #define Candera_Camera2DRenderStrategy_h

#include <Candera/Environment.h>
#include <Candera/System/Rtti/Rtti.h>

namespace Candera {

/** @addtogroup Core2D
 *  @{
 */

// Forward declaration.
class Node2D;
namespace Monitor {
    class DebugRenderStrategy2D;
}

/**
 *  @brief The abstract class Camera2DRenderStrategy is intended to be derived in order to tell the Renderer2D, if rendering shall proceed, pause, or stop
 *  for a given node in a camera's render pass.
 *  Purpose: Spreading rendering of Nodes over several calls to Renderer2D::RenderCamera (when processing all of them in one batch would take too long).
 *  A Camera2DRenderStrategy is assigned to a camera and affects only the processing of that camera and its nodes.
 *
 *  Caution: Switching a camera from single-batch to many-batches may have unintended side effects. In particular, the
 *  following topics should be considered carefully:
 *   - swapping (avoid swapping of incomplete scenes)
 *   - camera/render listeners
 *   - render state changes
 *   - dynamics of the scene (avoid creating inconsistent states by changing state in the middle of a render cycle)
 *   - relationships with other scenes
 */
class Camera2DRenderStrategy {

    public:
        /**
         *  The RenderPassAction describes the action the Renderer shall process when attempting to render a Node in a Camera's render pass.
         *  This enumerator is returned by function GetRenderPassAction, which must be implemented by derived class.
         */
        enum RenderPassAction {
            ProceedRenderPass,  ///< Proceed rendering. Rendering is not interrupted.
            PauseRenderPass,    ///< Pause rendering. The next time the Camera attempts to render, it restarts from the node that has been paused.
            StopRenderPass,     ///< Stop rendering. The next time the Camera attempts to render, it restarts from the very first node in render order.
            SkipNode            ///< Proceed rendering, but skip the current node.
        };

        /**
         *  Constructor
         */
        Camera2DRenderStrategy();

        /**
         *  Destructor
         */
        virtual ~Camera2DRenderStrategy() { m_previousNodeRendered = 0; }

        FEATSTD_RTTI_DECLARATION();
        
        /**  Returns true if all of the Nodes enumerated in the Camera's RenderOrder have been rendered.
          *  This function is intended to be used to determine if the Camera's output is complete and ready for swapping.
          *  Note: When a cycle has been completed, the next call to Renderer::RenderCamera will start a new cycle,
          *  so the appropriate time to call this function is between calls to RenderCamera.
          *  @return true if camera's render pass is completed, false if render pass has been paused and will be resumed the next time the camera is rendered.
          */
        bool IsRenderPassCompleted() const;

    protected:
        /**
         *  Tells the Renderer if rendering of Nodes shall continue.
         *  @param nextNode specifies the next Node in the render order that shall be rendered.
         *  @return RenderPassAction to define how the Renderer shall proceed:
         *          ProceedRenderPass allows the Renderer to render the node given and to proceed with next node.
         *          PauseRenderPass pauses the Renderer before the given node is rendered. The Renderer resumes at the node given the next time the Camera is rendered.
         *          StopRenderPass stops the Renderer before the given node is rendered. The Renderer restarts at the first node in render order, the next time the Camera is rendered.
         */
        virtual RenderPassAction GetRenderPassAction(const Node2D* nextNode) = 0;

        /**
         *  Called by Renderer at the beginning of each Camera's render pass. This is when previous Camera's render pass has been completed or stopped
         *  before (see GetRenderPassAction and RenderPassAction enumerator StopRenderPass).
         */
        virtual void OnRenderPassBegins() {}

        /**
         *  Called by Renderer each time the camera's render pass resumes from the previous node paused (see GetRenderAction and RenderPassAction
         *  enumerator PauseRenderPass).
         */
        virtual void OnRenderPassResumes() {}

    private:
        friend class Renderer2D;
        friend class Monitor::DebugRenderStrategy2D;

        bool m_isRenderPassCompleted;
        const Node2D* m_previousNodeRendered;

        // called by Renderer
        void SetRenderPassCompleted(bool complete);

        // called by Renderer
        void SetPreviousNodeRendered(const Node2D* node) { m_previousNodeRendered = node; }
        const Node2D* GetPreviousNodeRendered() const { return m_previousNodeRendered; }
};

/** @} */ // end of Core2D

} // namespace Candera

#endif // Candera_Camera2DRenderStrategy_h
