//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERA_Camera2DListener_H)
    #define CANDERA_Camera2DListener_H

#include <Candera/Environment.h>

namespace Candera {

/** @addtogroup Core2D
 *  @{
 */

class Camera2D;

/**
* @brief    A Camera2DListener defines hooks that are called before or after a Camera2D is rendered.
*           In order to register a Camera2DListener to a Camera2D simply derive from Camera2DListener and override pure virtual functions with custom code.
*/
class Camera2DListener
{
    public:
        /**
          * Constructs a Camera2DListener object.
          */
        Camera2DListener() {}

        /**
          * Destructs a Camera2DListener object.
          */
        virtual ~Camera2DListener() {}

        /**
          * Is invoked before a camera is rendered.
          * At the time when this function is called, the scene is already activated.
          * @param camera Camera2D to listen to.
          */
        virtual void OnPreRender(Camera2D* camera) = 0;

        /**
          * Is invoked after a camera is rendered.
          * @param camera Camera2D to listen to.
          */
        virtual void OnPostRender(Camera2D* camera) = 0;

        /**
         * Is invoked after the RenderTarget2D of the Camera was changed.
         * Use Camera::GetRenderTarget() to retrieve the new associated RenderTarget2D.
         * @param camera Camera that triggered the notification.
         * @param previousRenderTarget RenderTarget2D that was replaced.
         */
        virtual void OnRenderTargetChanged(Camera2D* camera, RenderTarget2D* previousRenderTarget);

        /**
         * Is invoked after the Camera2DRenderStrategy of the Camera was changed.
         * Use Camera::GetCamera2DRenderStrategy() to retrieve the new associated Camera2DRenderStrategy.
         * @param camera Camera that triggered the notification.
         * @param previousCameraRenderStrategy Camera2DRenderStrategy that was replaced.
         */
        virtual void OnCameraRenderStrategyChanged(Camera2D* camera, Camera2DRenderStrategy* previousCameraRenderStrategy);

        /**
        *  Is invoked before a camera is activated.
        *  @param camera The camera that will be activated.
        */
        virtual void OnPreActivate(const Camera2D* camera);

        /**
        *  Is invoked after a camera is activated.
        *  @param camera The camera that was activated.
        */
        virtual void OnPostActivate(const Camera2D* camera);

        /**
        *  Is invoked before clearing the framebuffer.
        *  @param camera The camera that will be activated.
        */
        virtual void OnPreClear(const Camera2D* camera);

        /**
        *  Is invoked after a clearing the framebuffer.
        *  @param camera The camera that was activated.
        */
        virtual void OnPostClear(const Camera2D* camera);


    protected:
        Camera2DListener(const Camera2DListener&);
        Camera2DListener& operator=(const Camera2DListener&);
};

inline void Camera2DListener::OnRenderTargetChanged(Camera2D* /*camera*/, RenderTarget2D* /*previousRenderTarget*/)
{
}

inline void Camera2DListener::OnCameraRenderStrategyChanged(Camera2D* /*camera*/, Camera2DRenderStrategy* /*previousCameraRenderStrategy*/)
{
}

inline void Camera2DListener::OnPreActivate(const Camera2D* /*camera*/)
{
}

inline void Camera2DListener::OnPostActivate(const Camera2D* /*camera*/)
{
}

inline void Camera2DListener::OnPreClear(const Camera2D* /*camera*/)
{
}

inline void Camera2DListener::OnPostClear(const Camera2D* /*camera*/)
{
}

/** @} */ // end of Core2D

} // namespace Candera

#endif  // CANDERA_Camera2DListener_H
