//########################################################################
// (C) Candera GmbH
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Candera GmbH.
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERA_Camera2D_H)
    #define CANDERA_Camera2D_H

#include <Candera/Engine2D/Core/2DStringBufferAppenders.h>
#include <Candera/Engine2D/Core/Node2D.h>
#include <Candera/EngineBase/Common/Color.h>
#include <Candera/System/Container/Vector.h>
#include <Candera/System/Mathematics/Rectangle.h>
#include <Candera/System/Container/SingleLinkedList.h>

namespace Candera {

/** @addtogroup Core2D
 *  @{
 */

    class RenderTarget2D;
    class Camera2DRenderStrategy;
    class Camera2DListener;

    /**
     *  @brief A camera defines the view to the scene. At least one Camera2D has to be added to a 2D scene graph and a render
     *  target has to be assigned to the camera.
     */
    class Camera2D : public Node2D {

        FEATSTD_TYPEDEF_BASE(Node2D);

    public:
        /**
         *  Creates an instance of this class.
         *  Use Dispose() to delete the instance.
         *  @return     A pointer to the created object.
         */
        static Camera2D* Create();

        /**
         *  Destructor
         */
        virtual ~Camera2D() override;

        virtual Camera2D* Clone() const override;

        /**
         *  Sets the Render Target for this camera.
         *
         * If the render pass of the render strategy of this camera is not yet
         *  completed the RenderTarget should not be changed. A listener should be
         *  used to check when the RenderTarget can be changed in case of multi
         *  pass rendering.
         *
         * @param renderTarget Render target of this camera to be set.
         */
        void SetRenderTarget(RenderTarget2D* renderTarget);

        /**
         *  Gets the render target of the camera.
         *  @return A pointer to the render target.
         */
        RenderTarget2D* GetRenderTarget() { return m_renderTarget; }
        const RenderTarget2D* GetRenderTarget() const { return m_renderTarget; }

        FEATSTD_RTTI_DECLARATION();

        /**
         *  @brief Assigns a viewport to the camera, where the viewport is specified as a Rectangle in pixels.
         *
         *  It is possible to specify negative viewport dimensions, which define that the viewport dimensions from the
         *  RenderTarget. If additionally negative viewport coordinates are set, the viewport will not include the entire
         *  render target as the dimensions remain the same. If the render target should be included entirely, the dimensions
         *  need to be set manually.
         *
         *  The default viewport Rectangle is (0, 0, -1, -1), which will include the entire render target.
         *
         *  @param viewport The viewport that is assigned with coordinates and dimensions in pixels.
         */
        void SetViewport(const Rectangle& viewport) { m_viewport = viewport; }

        /**
         *  Gets the viewport of the camera.
         *  @return The viewport.
         */
        const Rectangle& GetViewport() const { return m_viewport; }

        /**
         *  Assigns the clear color enabled value to the camera.
         *  @param isEnabled The value that is assigned.
         */
        void SetClearColorEnabled(bool isEnabled) { m_isClearColorEnabled = isEnabled; }

        /**
         *  Gets if the clear color is enabled or not.
         *  @return The clear color state.
         */
        bool IsClearColorEnabled() const { return m_isClearColorEnabled; }

        /**
         *  Assigns the swap enabled value to the camera.
         *  @param isEnabled The value that is assigned.
         */
        void SetSwapEnabled(bool isEnabled) { m_isSwapEnabled = isEnabled; }

        /**
         *  Gets if the swap is enabled or not.
         *  @return The swap state.
         */
        bool IsSwapEnabled() const { return m_isSwapEnabled; }

        /**
         *  Assigns the clear color to the camera.
         *  @param clearColor The clear color that is assigned.
         */
        void SetClearColor(const Color& clearColor) { m_clearColor = clearColor; }

        /**
         *  Gets the clear color of the camera.
         *  @return The clear color.
         */
        const Color& GetClearColor() const { return m_clearColor; }

        /**
         *  The scissor rectangle specifies a rectangular region within the Camera's RenderTarget.
         *  The scissor rectangles width and height must be greater than 0.
         *  A scissor rectangle that intersects the RenderTarget is allowed.
         *  If scissoring is enabled (see IsScissoringIsEnabled) pixels outside of the intersection area of viewport and scissor
         *  rectangle are clipped.
         *  @param scissorRectangle Scissor rectangle that is assigned.
         */
        void SetScissorRectangle(const Rectangle& scissorRectangle) { m_scissorRectangle = scissorRectangle; }

        /**
         *  Retrieves the scissor rectangle of the camera.
         *  @return Scissor rectangle of this camera.
         */
        const Rectangle& GetScissorRectangle() const { return m_scissorRectangle; }

        /**
         *  Defines if scissoring is enabled or not. Per default, scissoring is disabled.
         *  @param  enable      true, to enable scissoring.
         */
        void SetScissoringEnabled(bool enable) { m_isScissoringEnabled = enable; }

        /**
         *  Gets whether scissoring is enabled or not.
         *  @return The scissoring sate.
         */
        bool IsScissoringEnabled() const { return m_isScissoringEnabled; }

        /**
         *  Sets render sequence of this camera in an ascending order, used if Renderer::RenderAllCameras is invoked.
         *  @param sequenceNumber Sequence of number of this camera to be set.
         */
        void SetSequenceNumber(Int16 sequenceNumber);

        /**
         *  Retrieves render sequence of this camera.
         *  @return Render sequence of this camera.
         */
        Int16 GetSequenceNumber() const { return m_sequenceNumber; }

        /**
         *  Gets view matrix of this Camera.
         *  @return View matrix of this Camera.
         */
        const Matrix3x2& GetViewMatrix() const;

        /**
         *  Assigns a Camera2DRenderStrategy to the camera. The camera does not take ownership.
         *  @param cameraRenderStrategy specifies the Camera2DRenderStrategy object assigned to this camera. A Camera2DRenderStrategy
         *  may interrupt one camera's render pass for special use cases like partitioning a camera's render pass into
         *  multiple sub render passes.
         *  If no Camera2DRenderStrategy object is assigned the camera attempts to render all nodes within its parent scene in one render pass.
         */
        void SetCameraRenderStrategy(Camera2DRenderStrategy* cameraRenderStrategy);

        /**
         *  Retrieves the camera render strategy for this camera.
         *  @return The camera render strategy for this camera.
         */
        CANDERA_SUPPRESS_LINT_FOR_SYMBOL(1763, Candera::Camera2D::GetCameraRenderStrategy, CANDERA_LINT_REASON_NONCONST)
        Camera2DRenderStrategy* GetCameraRenderStrategy() const { return m_cameraRenderStrategy; }

        /**
         *  Adds a Camera2DListener object. Event/timing of the calls: before/after this Camera2D is rendered.
         *  @param listener The lifetime of the listener is managed by the calling code. Camera2D does not take ownership.
         *  @return True if successful, false otherwise.
         */
        bool AddCameraListener(Camera2DListener* listener);

        /**
         *  Removes a Camera2DListener object.
         *  @param listener Specifies the Camera2DListener object to remove from this Camera2D.
         *  @return True if successful, false otherwise.
         */
        bool RemoveCameraListener(Camera2DListener* listener);

        /**
         * Check if camera setting were updated: scissoring, clear color, viewport
         */
        bool WasUpdated() const;


        /**
        * Sets whether the camera effective alpha is multiplied to the effective alpha of a Node when being rendered.
        * @param enable Enables or disables the camera effective alpha.
        */
        void SetCameraEffectiveAlphaEnabled(bool enable) { m_isCameraEffectiveAlphaEnabled = enable; }

        /**
        * Gets whether the camera effective alpha is multiplied to the effective alpha of a Node when being rendered.
        * @return Wehther the camera effective alpha is enabled (true) or disabled (false).
        */
        bool IsCameraEffectiveAlphaEnabled() const { return m_isCameraEffectiveAlphaEnabled; }

    protected:
        friend class Renderer2D;

        // Explicit protected Constructor and Copy-Constructor, use Create() to create an instance of this object.
        Camera2D();
        explicit Camera2D(const Camera2D& camera);
        Camera2D& operator=(const Camera2D& camera);

        virtual void DisposeSelf() override;

        virtual void OnCompositeTransformChanged() override;

        /**
         *  Reacts to a transformation change further up in the tree. The reaction should be local/flat, i.e. limited
         *  to this Node2D (not propagated recursively to descendants).
         */
        virtual void OnAncestorTransformChanged();

        /**
         *  Called before this Camera2D is rendered. Notifies the Camera2DListeners.
         */
        void NotifyListenersOnPreRender();

        /**
         *  Called after this Camera2D is rendered. Notifies the Camera2DListeners.
         */
        void NotifyListenersOnPostRender();

        /**
         *  Called after this Camera2D's RenderTarget2D was changed. Notifies the RenderCameraListeners.
         */
        void NotifyListenersOnRenderTargetChanged(RenderTarget2D* previousRenderTarget);

        /**
         *  Called after Camera2D's Camera2DRenderStrategy changed. Notifies the CameraListeners.
         */
        void NotifyListenersOnRenderStrategyChanged(Camera2DRenderStrategy* previousRenderStrategy);

        /**
        *  Called before the camera is activated.
        */
        void NotifyListenersOnPreActivate();

        /**
        *  Called after the camera has been activated.
        */
        void NotifyListenersOnPostActivate();

        /**
        *  Called before the framebuffer is cleared.
        */
        void NotifyListenersOnPreClear();

        /**
        *  Called after the framebuffer clear call has been sent to the device.
        */
        void NotifyListenersOnPostClear();

        /**
         * Compute the internal hash from the camera settings
         */
        bool ComputeHash() const;

    private:
        mutable bool m_isViewMatrixCacheValid : 1;

        bool m_isClearColorEnabled : 1;
        bool m_isSwapEnabled : 1;
        bool m_isScissoringEnabled : 1;
        bool m_isCameraEffectiveAlphaEnabled : 1;

        FEATSTD_LINT_NEXT_EXPRESSION(1960, "MISRA 9-6-2: Used type is a typedef to explicitly signed/unsigned base type.")
        Int16 m_sequenceNumber : 16;
        RenderTarget2D* m_renderTarget;
        Camera2DRenderStrategy* m_cameraRenderStrategy;

        Color m_clearColor;

        Rectangle m_viewport;
        Rectangle m_scissorRectangle;

        mutable Matrix3x2 m_viewMatrixCache;

        typedef Internal::SingleLinkedList<Camera2DListener*> Camera2DListenerContainer;
        Camera2DListenerContainer m_cameraListeners;

        mutable UInt32 m_hash;
    };

 /** @} */ // end of Core2D

}   // namespace Candera

#endif  // CANDERA_Camera2D_H
