//########################################################################
// (C) Candera GmbH
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Candera GmbH.
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "Camera2D.h"

#include <Candera/Engine2D/Core/Camera2DListener.h>
#include <Candera/Engine2D/Core/Camera2DRenderStrategy.h>
#include <Candera/Engine2D/Core/Renderer2D.h>
#include <Candera/Engine2D/Mathematics/Math2D.h>
#include <Candera/System/Diagnostics/Log.h>
#include <FeatStd/Platform/CriticalSectionLocker.h>
#include <FeatStd/Util/Hash.h>

namespace Candera {
    using namespace Diagnostics;
    using namespace Internal;

    /**
     * CameraLessBySequenceNumber is a function object to compare Cameras by sequence number.
     */
    class CameraLessBySequenceNumber {
    public:
        bool operator()(const Camera2D* a, const Camera2D* b) const
        {
            return a->GetSequenceNumber() < b->GetSequenceNumber();
        }
    };

    /******************************************************************************
     *  Constructor
     ******************************************************************************/
    Camera2D::Camera2D() :
        Base(),
        m_isViewMatrixCacheValid(false),
        m_isClearColorEnabled(false),
        m_isSwapEnabled(false),
        m_isScissoringEnabled(false),
        m_isCameraEffectiveAlphaEnabled(false),
        m_sequenceNumber(0),
        m_renderTarget(0),
        m_cameraRenderStrategy(0),
        m_viewport(0.0F, 0.0F, -1.0F, -1.0F),
        m_scissorRectangle(0.0F, 0.0F, -1.0F, -1.0F),
        m_hash(FeatStd::Hash::cInvalidHashValue)
    {
#ifdef FEATSTD_THREADSAFETY_ENABLED
        FeatStd::Internal::CriticalSectionLocker cameraListLock(Renderer2D::GetCameraListCriticalSection());
#endif
        // because the default value for m_sequenceNumber is 0, it will be inserted at the front of the list
        static_cast<void>(Renderer2D::GetCameraList().Insert(0, this));
        static_cast<void>(AddCameraListener(Renderer2D::GetCamera2dListenerInstance()));

        ComputeHash();
    }

    Camera2D::Camera2D(const Camera2D& camera) :
        CANDERA_SUPPRESS_LINT_FOR_SYMBOL(1554, Candera::Camera2D::m_renderTarget, CANDERA_LINT_REASON_ASSOCIATION)
        CANDERA_SUPPRESS_LINT_FOR_SYMBOL(1554, Candera::Camera2D::m_cameraRenderStrategy, CANDERA_LINT_REASON_ASSOCIATION)
        Base(camera),
        m_isViewMatrixCacheValid(false),
        m_isClearColorEnabled(camera.m_isClearColorEnabled),
        m_isSwapEnabled(camera.m_isSwapEnabled),
        m_isScissoringEnabled(camera.m_isScissoringEnabled),
        m_isCameraEffectiveAlphaEnabled(camera.m_isCameraEffectiveAlphaEnabled),
        m_sequenceNumber(camera.m_sequenceNumber),
        m_renderTarget(0),
        m_cameraRenderStrategy(0),
        m_clearColor(camera.m_clearColor),
        m_viewport(camera.m_viewport),
        m_scissorRectangle(camera.m_scissorRectangle),
        m_hash(camera.m_hash)
    {
#ifdef FEATSTD_THREADSAFETY_ENABLED
        FeatStd::Internal::CriticalSectionLocker cameraListLock(Renderer2D::GetCameraListCriticalSection());
#endif
        static_cast<void>(Renderer2D::GetCameraList().Insert(0, this));
        static_cast<void>(AddCameraListener(Renderer2D::GetCamera2dListenerInstance()));
    }

    /******************************************************************************
     *  Destructor
     ******************************************************************************/
    Camera2D::~Camera2D()
    {
#ifdef FEATSTD_THREADSAFETY_ENABLED
        FeatStd::Internal::CriticalSectionLocker cameraListLock(Renderer2D::GetCameraListCriticalSection());
#endif
        Renderer2D::CameraList& cameraList = Renderer2D::GetCameraList();
        SizeType i = 0;
        const SizeType size = cameraList.Size();
        while ((i < size) && (cameraList[i] != this)) {
            i++;
        }

        if (i < size) {
            static_cast<void>(cameraList.Remove(i));
            if (0 == cameraList.Size()) {
                cameraList.Free();
            }
        }

        m_renderTarget = 0;
        m_cameraRenderStrategy = 0;
    }

    /******************************************************************************
     *  Create
     ******************************************************************************/
    Camera2D* Camera2D::Create()
    {
        return FEATSTD_NEW(Camera2D);
    }

    FEATSTD_RTTI_DEFINITION(Camera2D, Node2D)

    /******************************************************************************
     *  Clone
     ******************************************************************************/
    Camera2D* Camera2D::Clone() const
    {
        return FEATSTD_NEW(Camera2D)(*this);
    }

    /******************************************************************************
     *  DisposeSelf
     ******************************************************************************/
    void Camera2D::DisposeSelf()
    {
        FEATSTD_DELETE(this);
    }

    /******************************************************************************
     *  OnCompositeTransformChange
     ******************************************************************************/
    void Camera2D::OnCompositeTransformChanged()
    {
        m_isViewMatrixCacheValid = false;
        Base::OnCompositeTransformChanged();
    }

    /******************************************************************************
     *  OnAncestorTransformChanged
     ******************************************************************************/
    void Camera2D::OnAncestorTransformChanged()
    {
        m_isViewMatrixCacheValid = false;
    }

    /******************************************************************************
     *  SetSequenceNumber
     ******************************************************************************/
    void Camera2D::SetSequenceNumber(Int16 sequenceNumber)
    {
        if (sequenceNumber != GetSequenceNumber()) {
            m_sequenceNumber = sequenceNumber;
            CANDERA_SUPPRESS_LINT_FOR_SYMBOL(1502, objSort, CameraLessBySequenceNumber only defines the comparison operator for sorting the list)
            CameraLessBySequenceNumber objSort;
#ifdef FEATSTD_THREADSAFETY_ENABLED
            FeatStd::Internal::CriticalSectionLocker cameraListLock(Renderer2D::GetCameraListCriticalSection());
#endif
            Renderer2D::GetCameraList().Sort(objSort);
        }
    }

    /******************************************************************************
     *  GetViewMatrix
     ******************************************************************************/
    const Matrix3x2& Camera2D::GetViewMatrix() const
    {
        if (m_isViewMatrixCacheValid == false) {
            m_isViewMatrixCacheValid = true;
            m_viewMatrixCache = GetWorldTransform();
            m_viewMatrixCache.Inverse();
        }

        return m_viewMatrixCache;
    }

    /******************************************************************************
     *  AddCameraListener
     ******************************************************************************/
    bool Camera2D::AddCameraListener(Camera2DListener* listener)
    {
        bool isSuccessful = false;
        if (listener != 0) {
            isSuccessful = m_cameraListeners.Append(listener);
        }
        return isSuccessful;
    }

    /******************************************************************************
     *  RemoveCameraListener
     ******************************************************************************/
    bool Camera2D::RemoveCameraListener(Camera2DListener* listener)
    {
        bool isSuccessful = false;
        if (listener != 0) {
            isSuccessful = m_cameraListeners.Remove(listener);
        }
        return isSuccessful;
    }

    /******************************************************************************
     *  NotifyListenersOnPreRender
     ******************************************************************************/
    void Camera2D::NotifyListenersOnPreRender()
    {
        for (Camera2DListenerContainer::Iterator it = m_cameraListeners.Begin(); it != m_cameraListeners.End(); ++it) {
            (*it)->OnPreRender(this);
        }
    }

    /******************************************************************************
     *  NotifyListenersOnPostRender
     ******************************************************************************/
    void Camera2D::NotifyListenersOnPostRender()
    {
        for (Camera2DListenerContainer::Iterator it = m_cameraListeners.Begin(); it != m_cameraListeners.End(); ++it) {
            (*it)->OnPostRender(this);
        }
    }

    void Camera2D::NotifyListenersOnRenderTargetChanged(RenderTarget2D* previousRenderTarget)
    {
        for (Camera2DListenerContainer::Iterator it = m_cameraListeners.Begin(); it != m_cameraListeners.End(); ++it) {
            Camera2DListener* cameraListener = *it;
            cameraListener->OnRenderTargetChanged(this, previousRenderTarget);
        }
    }

    void Camera2D::NotifyListenersOnRenderStrategyChanged(Camera2DRenderStrategy* previousRenderStrategy)
    {
        for (Camera2DListenerContainer::Iterator it = m_cameraListeners.Begin(); it != m_cameraListeners.End(); ++it) {
            Camera2DListener* cameraListener = *it;
            cameraListener->OnCameraRenderStrategyChanged(this, previousRenderStrategy);
        }
    }

    void Camera2D::NotifyListenersOnPreActivate()
    {
        for (Camera2DListenerContainer::Iterator it = m_cameraListeners.Begin(); it != m_cameraListeners.End(); ++it) {
            Camera2DListener* cameraListener = *it;
            cameraListener->OnPreActivate(this);
        }
    }

    void Camera2D::NotifyListenersOnPostActivate()
    {
        for (Camera2DListenerContainer::Iterator it = m_cameraListeners.Begin(); it != m_cameraListeners.End(); ++it) {
            Camera2DListener* cameraListener = *it;
            cameraListener->OnPostActivate(this);
        }
    }

    void Camera2D::NotifyListenersOnPreClear()
    {
        for (Camera2DListenerContainer::Iterator it = m_cameraListeners.Begin(); it != m_cameraListeners.End(); ++it) {
            Camera2DListener* cameraListener = *it;
            cameraListener->OnPreClear(this);
        }
    }

    void Camera2D::NotifyListenersOnPostClear()
    {
        for (Camera2DListenerContainer::Iterator it = m_cameraListeners.Begin(); it != m_cameraListeners.End(); ++it) {
            Camera2DListener* cameraListener = *it;
            cameraListener->OnPostClear(this);
        }
    }

    bool Camera2D::WasUpdated() const
    {
        return ComputeHash();
    }

    bool Camera2D::ComputeHash() const
    {
       using namespace FeatStd;

        UInt32 hash = Hash::cInitialHashValue;
        UInt32 seed = Hash::cInitialSeed;

        do {
            if (m_isClearColorEnabled)
            {
                // Add clear color
                const Color::Data &data = m_clearColor.GetData();
                Hash::UpdateBinBufferHash(reinterpret_cast<const UInt8*>(&data),
                    reinterpret_cast<const UInt8*>(&data) + sizeof(data),
                    hash);
            }

            // Add viewport rectangle
            Hash::UpdateBinBufferHash(reinterpret_cast<const UInt8*>(&m_viewport),
                reinterpret_cast<const UInt8*>(&m_viewport) + sizeof(m_viewport),
                hash);

            // Add scissoring rectangle
            if (m_isScissoringEnabled)
            {
                Hash::UpdateBinBufferHash(reinterpret_cast<const UInt8*>(&m_scissorRectangle),
                    reinterpret_cast<const UInt8*>(&m_scissorRectangle) + sizeof(m_scissorRectangle),
                    hash);
            }
        } while (!Hash::FinishBinBufferHash(hash, seed));

        if (hash != m_hash)
        {
            m_hash = hash;
            return true;
        }

        return false;
    }

    void Camera2D::SetRenderTarget(RenderTarget2D* renderTarget)
    {
        RenderTarget2D* previousRenderTarget = m_renderTarget;
        m_renderTarget = renderTarget;
        NotifyListenersOnRenderTargetChanged(previousRenderTarget);
    }

    void Camera2D::SetCameraRenderStrategy(Camera2DRenderStrategy* cameraRenderStrategy)
    {
        if (m_cameraRenderStrategy != cameraRenderStrategy) {
            Camera2DRenderStrategy* previousRenderStrategy = m_cameraRenderStrategy;
            m_cameraRenderStrategy = cameraRenderStrategy;
            NotifyListenersOnRenderStrategyChanged(previousRenderStrategy);
        }
    }
}   // namespace Candera
