//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#ifndef Candera_BreakNodeCamera2DRenderStrategy_h
    #define Candera_BreakNodeCamera2DRenderStrategy_h

#include <Candera/Engine2D/Core/Camera2DRenderStrategy.h>
#include <Candera/System/Container/SingleLinkedList.h>
#include <Candera/System/Rtti/Rtti.h>

namespace Candera {

/** @addtogroup Core2D
 *  @{
 */

// Forward declaration.
class Node2D;

/**
 * @brief The BreakNodeCamera2DRenderStrategy is an implementation of the abstract class Camera2DRenderStrategy.
 * The rendering of the nodes is done until it reaches a node that is in the list of break nodes (See function AddBreakNode).
 */
class BreakNodeCamera2DRenderStrategy: public Camera2DRenderStrategy
{
    FEATSTD_TYPEDEF_BASE(Camera2DRenderStrategy);

    public:

        BreakNodeCamera2DRenderStrategy();
        virtual ~BreakNodeCamera2DRenderStrategy() override {}

        /**
         *  Adds a break node to the break node list, which causes the camera render pass to stop or pause if encountered during rendering.
         *  If a break node triggers the render pass to stop or pause is specified by the render pass action set (see SetRenderPassActionOnBreakNode).
         *  @param node Node to be added to break node list.
         */
        void AddBreakNode(const Node2D* node);

        /**
         *  Removes a break node from the break node list.
         *  @param node Node to be removed from break node list.
         */
        void RemoveBreakNode(const Node2D* node);

        /**
         *  Sets the render pass action indicated, when a break node is encountered during rendering.
         *  Reasonable values are PauseRenderPass or StopRenderPass.
         *  @param action Render pass action to be set.
         */
        void SetRenderPassActionOnBreakNode(Camera2DRenderStrategy::RenderPassAction action) { m_actionOnBreakNode = action; }

        /**
         *  Retrieves the render pass action.
         *  @return The render pass action.
         */
        Camera2DRenderStrategy::RenderPassAction GetRenderPassActionOnBreakNode() const { return m_actionOnBreakNode; }

        FEATSTD_RTTI_DECLARATION();

    protected:
        /**
         *  Tells the Renderer if rendering of Nodes shall continue.
         *  The rendering of the nodes is done until the node given is contained in the list of break nodes (see function AddBreakNode).
         *  At this point the rendering is paused or stopped, depending on the render pass action set. (see SetRenderPassActionOnBreakNode).
         *  @param nextNode Next node to be rendered. Based on this node the next RenderPassAction is chosen.
         *  @return         RenderPassAction to be executed on this node.
         */
        virtual RenderPassAction GetRenderPassAction(const Node2D* nextNode) override;

        /**
         *  Called by Renderer at the beginning of each Camera's render pass.
         *  Render until the first break node matches a node that shall be rendered.
         */
        virtual void OnRenderPassBegins() override;

        /**
         *  Called by Renderer each time the camera's render pass resumes from the previous node paused.
         *  The node that caused the cameras render pass to pause, will be rendered in the subsequent render pass.
         */
        virtual void OnRenderPassResumes() override;

    private:
        bool m_isRenderPassResuming;            // Indicates if a render pass resumes with the break node. If so, the break node proceeds rendering in resuming pass.
        Internal::SingleLinkedList <Node2D* > m_breakNodeList;  // Holds a list of all break nodes added.
        RenderPassAction m_actionOnBreakNode;   // The indicated render pass action if node to render matches a break node. Default value is PauseRenderPass.
};

/** @} */ // end of Core2D

} // namespace Candera

#endif // Candera_BreakNodeCamera2DRenderStrategy_h
