//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#include "BitmapImage2D.h"
#include <Candera/System/Diagnostics/VideoMemoryStatistic.h>
#include <Candera/Engine2D/Core/Renderer2D.h>

namespace Candera {

    FEATSTD_RTTI_DEFINITION(BitmapImage2D, Base)

    BitmapImage2D::BitmapImage2D() :
        m_isMipMappingEnabled(false),
        m_memoryPool(VideoMemory),
        m_surfaceHandle(0),
        m_bitmap(0)
    {
    }

    BitmapImage2D::~BitmapImage2D()
    {
        static_cast<void>(Unload(Force));
        m_surfaceHandle = 0;
    }

    MemoryManagement::SharedPointer<BitmapImage2D> BitmapImage2D::Create()
    {
        return MemoryManagement::SharedPointer<BitmapImage2D>(FEATSTD_NEW(BitmapImage2D));
    }

    bool BitmapImage2D::SetBitmap(const Bitmap::SharedPointer& bitmap)
    {
        if (m_surfaceHandle != 0) {
            return false; // bitmap data is already uploaded
        }

        m_bitmap = bitmap;
        return true;
    }

    Bitmap::SharedPointer BitmapImage2D::GetMutableBitmap()
    {
        return ((m_bitmap != 0) && (m_bitmap->GetPixelsResourceHandle().m_isMutable == 1)) ? m_bitmap : Bitmap::SharedPointer();
    }

    UInt32 BitmapImage2D::GetWidth() const
    {
        return (m_bitmap != 0) ? static_cast<UInt32>(m_bitmap->GetWidth()) : 0;
    }

    UInt32 BitmapImage2D::GetHeight() const
    {
        return (m_bitmap != 0) ? static_cast<UInt32>(m_bitmap->GetHeight()) : 0;
    }

    UInt BitmapImage2D::GetSize() const
    {
        UInt32 imageSize = 0;
        if (m_bitmap != 0) {
            //Calculate bytes per color channel (texel) in bytes based on type. (All texel formats are assembled into an RGBA element by GL.)
            imageSize = RenderDevice2D::GetSize(m_bitmap.GetPointerToSharedInstance(), m_memoryPool);

            if (IsMipMappingEnabled() && RenderDevice2D::IsMipMappingSupported()) {
                if (m_bitmap->HasNext() != 0) {
                    Bitmap::SharedPointer bitmap = m_bitmap->GetNext();
                    while (bitmap != 0) {
                        imageSize += RenderDevice2D::GetSize(m_bitmap.GetPointerToSharedInstance(), m_memoryPool);
                        bitmap = bitmap->GetNext();
                    }
                } else {
                    if (!m_bitmap->IsCompressed()) {
                        // If imageSize is null the next calculation would underflow UInt32.
                        if (imageSize != 0) {
                            // A mipmap chain must conclude with a 1x1 sized level in OpenGL ES and has max. 1/3 more memory consumption than the greatest image.
                            imageSize = ((4 * imageSize) - 1) / 3;
                        }
                    }
                }
            }
        }
        return static_cast<UInt>(imageSize);
    }

    bool BitmapImage2D::SetMipMappingEnabled(bool enableMipMapping)
    {
        bool isNotUploaded = !IsUploaded();
        if (isNotUploaded) {
            m_isMipMappingEnabled = enableMipMapping;
        }
        return isNotUploaded;
    }

    bool BitmapImage2D::UploadInternal(LoadingHint loadingHint)
    {
        Diagnostics::VideoMemoryStatistic::SetTextureImageStatisticLocked(true);
        bool success = Renderer2D::UploadBitmapImage2D(*this, loadingHint);
        Diagnostics::VideoMemoryStatistic::SetTextureImageStatisticLocked(false);

        if (success) {
            //As various 2D render devices exist, this is the logical place to count the VRAM statistics.
            Diagnostics::VideoMemoryStatistic::OnBitmapImage2DUploaded(*this);
        }

        return success;
    }

    bool BitmapImage2D::UnloadInternal(LoadingHint loadingHint)
    {
        bool success =  Renderer2D::UnloadBitmapImage2D(*this, loadingHint);
        Diagnostics::VideoMemoryStatistic::OnBitmapImage2DUnloaded(*this);
        return success;
    }

    void BitmapImage2D::DisposeInternal()
    {
        if ((m_bitmap != 0) && (m_memoryPool == VideoMemory)) {
            m_bitmap->DisposePixels();
        }
    }

    bool BitmapImage2D::Update()
    {
        if ((!IsUploaded()) || (m_bitmap == 0)) {
            return false;
        }

        Bitmap::PixelsResource pixelsResource(m_bitmap->GetPixelsResourceHandle());
        if (pixelsResource.GetData() == 0) {
            return false;
        }

        return RenderDevice2D::UpdateSubImage(m_surfaceHandle, 0, 0, m_bitmap->GetWidth(), m_bitmap->GetHeight(), pixelsResource.GetData());
    }

    bool BitmapImage2D::Update(Int xOffset, Int yOffset, UInt width, UInt height, const UInt8* data) const
    {
        bool result = false;

        if (IsUploaded() && (data != 0)) {
            result = RenderDevice2D::UpdateSubImage(m_surfaceHandle, xOffset, yOffset, width, height, data);
        }
        return result;
    }   
}
