//########################################################################
// (C) Socionext Embedded Software Austria GmbH (SESA)
// All rights reserved.
// -----------------------------------------------------
// This document contains proprietary information belonging to
// Socionext Embedded Software Austria GmbH (SESA).
// Passing on and copying of this document, use and communication
// of its contents is not permitted without prior written authorization.
//########################################################################

#if !defined(CANDERA_Transformable2DRelativeScalePropertySetter_H)
#define CANDERA_Transformable2DRelativeScalePropertySetter_H

#include <Candera/Engine2D/AnimationPropertySetters/BaseTransformable2DRelativePropertySetter.h>
#include <Candera/System/Mathematics/Vector2.h>

namespace Candera { namespace Animation {

    /**
     *  @addtogroup AnimationPropertySetters2D
     *  @{
     */

    /**
     *  @brief Relative Scale AnimationPropertySetter.
     *
     *  The property setter applies a scaling to the associated Transformable2D object, relative to the object scaling
     *   cached at PropertySetter initialization time.
     */
    class Transformable2DRelativeScalePropertySetter: public BaseTransformable2DRelativePropertySetter {
        FEATSTD_TYPEDEF_BASE(BaseTransformable2DRelativePropertySetter);

        public:
            FEATSTD_TYPEDEF_SHARED_POINTER(Transformable2DRelativeScalePropertySetter);

            FEATSTD_SHARED_POINTER_CREATE_DECLARATION();

            FEATSTD_RTTI_DECLARATION();

            Transformable2DRelativeScalePropertySetter() : m_scale(Vector2(1.0F, 1.0F)) {}
            virtual ~Transformable2DRelativeScalePropertySetter() {}

            /**
             * Set the relative scaling for the the Transformable2D object.
             *
             * @param value     Pointer to a float array that represents the offsets that needs to be multiplied with
             *      the cached Transformable2D scale value (value[0] for X axis, value[1] for Y axis).
             */
            virtual void Set(const Float* value) override;

            /**
             * Initialize the PropertySetter.
             *
             * The PropertySetter caches the scale value of the current associated Transformable2D object, to
             *  be used by future Set calls. If an Transformable2D object is not associated at initialization time,
             *  the cached value is set to a default scaling of (1.0F, 1.0F).
             */
            virtual void Initialize() override;

        private:
            Vector2 m_scale;
    };

    /**
     *  @brief Relative Scale AnimationPropertySetter on X axis.
     *
     *  The property setter applies a scaling on the X axis to the associated Transformable2D object, relative
     *   to the object scaling cached at PropertySetter initialization time.
     */
    class Transformable2DRelativeScaleXPropertySetter : public BaseTransformable2DRelativePropertySetter {
            typedef BaseTransformable2DRelativePropertySetter Base;

        public:
            FEATSTD_TYPEDEF_SHARED_POINTER(Transformable2DRelativeScaleXPropertySetter);

            FEATSTD_SHARED_POINTER_CREATE_DECLARATION();

            FEATSTD_RTTI_DECLARATION();

            Transformable2DRelativeScaleXPropertySetter():m_scaleX(1.0F) {}
            virtual ~Transformable2DRelativeScaleXPropertySetter() override {}

            /**
             * Set the relative scaling on the X axis for the the Transformable2D object.
             *
             * @param value     Pointer to a float value that represents the offset that needs to be multiplied with
             *      the cached Transformable2D X axis scale value.
             */
            virtual void Set(const Float* value) override;

            /**
             * Initialize the PropertySetter.
             *
             * The PropertySetter caches the scale value on the X axis of the current associated Transformable2D object, to
             *  be used by future Set calls. If an Transformable2D object is not associated at initialization time,
             *  the cached value is set to a default scaling of 1.0F.
             */
            virtual void Initialize() override;

        private:
            Float m_scaleX;
    };

    /**
     *  @brief Relative Scale AnimationPropertySetter on Y axis.
     *
     *  The property setter applies a scaling on the Y axis to the associated Transformable2D object, relative
     *   to the object scaling cached at PropertySetter initialization time.
     */
    class Transformable2DRelativeScaleYPropertySetter : public BaseTransformable2DRelativePropertySetter {
            typedef BaseTransformable2DRelativePropertySetter Base;

        public:
            FEATSTD_TYPEDEF_SHARED_POINTER(Transformable2DRelativeScaleYPropertySetter);

            FEATSTD_SHARED_POINTER_CREATE_DECLARATION();

            FEATSTD_RTTI_DECLARATION();

            Transformable2DRelativeScaleYPropertySetter():m_scaleY(1.0F) {}
            virtual ~Transformable2DRelativeScaleYPropertySetter() override {}

            /**
             * Set the relative scaling on the Y axis for the the Transformable2D object.
             *
             * @param value     Pointer to a float value that represents the offset that needs to be multiplied with
             *      the cached Transformable2D Y axis scale value.
             */
            virtual void Set(const Float* value) override;

            /**
             * Initialize the PropertySetter.
             *
             * The PropertySetter caches the scaling value on the Y axis of the current associated Transformable2D object, to
             *  be used by future Set calls. If an Transformable2D object is not associated at initialization time,
             *  the cached value is set to a default scaling of 1.0F.
             */
            virtual void Initialize() override;

        private:
            Float m_scaleY;
    };

    /**
     * @}
     */ // end of addtogroup AnimationPropertySetters2D
    } // namespace Animation

}   // namespace Candera

#endif  // CANDERA_Transformable2DRelativeScalePropertySetter_H
