#pragma once

template<typename T>
class WeakRef;

template <typename T>
class Ref {
    public:
        Ref() :
            m_ref(0)
        {
        }

        Ref(T* r) :
            m_ref(0)
        {
            Assign(r);
        }

        Ref(const Ref& r) :
            m_ref(0)
        {
            Assign(r.m_ref);
        }

        ~Ref()
        {
            Release();
        }

        bool IsValid() const
        {
            return 0 != m_ref;
        }

        Ref& operator=(const Ref& r)
        {
            Assign(r.m_ref);
            return *this;
        }

        Ref& operator=(T* r)
        {
            Assign(r);
            return *this;
        }

        bool operator==(const Ref& r)
        {
            return m_ref == r.m_ref;
        }

        bool operator==(const T* r)
        {
            return m_ref == r;
        }

        bool operator!=(const Ref* r)
        {
            return m_ref != r->m_ref;
        }

        bool operator!=(const T* r)
        {
            return m_ref != r;
        }

        T* operator->()
        {
            return m_ref;
        }

        T* operator->() const
        {
            return m_ref;
        }

        T& operator*()
        {
            return *m_ref;
        }

        T& operator*() const
        {
            return *m_ref;
        }

        operator bool() const
        {
            return IsValid();
        }

    private:
        template<typename WeakRef_T> friend class ::WeakRef;

        void Assign(T* r)
        {
            if (r != m_ref) {
                if (0 != r) {
                    r->Obtain();
                }
                Release();
                m_ref = r;
            }
        }

        void Release()
        {
            if (0 != m_ref) {
                m_ref->Release();
                m_ref = 0;
            }
        }

        T* m_ref;
};

namespace RefInternal {
    class RefCounter {
        public:
            RefCounter() :
                m_refCounter(0)
            {
            }

            size_t& operator++()
            {
                return ++m_refCounter;
            }

            size_t& operator--()
            {
                return --m_refCounter;
            }

            bool operator>(size_t value)
            {
                return m_refCounter > value;
            }

            bool operator==(size_t value)
            {
                return m_refCounter == value;
            }

        private:
            size_t m_refCounter;
    };
}

#define DefineShareable() \
    template<typename T> friend class ::Ref; \
    \
    void Obtain() \
    { \
        REF_CS_SCOPE scope(&m_criticalSection); \
        ++m_refCounter; \
    } \
    \
    void Release() \
    { \
        REF_CS_SCOPE scope(&m_criticalSection); \
        if (m_refCounter > 0) { \
            if (--m_refCounter == 0) { \
                scope.Release(); \
                LITEHTML_DELETE(this); \
            } \
        } \
    } \
    \
    REF_CS m_criticalSection; \
    RefInternal::RefCounter m_refCounter; 
