/*!
*******************************************************************************
* @file             : URI_DBusHandler.c
*******************************************************************************
*  - PROJECT:       : ECore
*  - SW-COMPONENT   : Gateway
*  - DESCRIPTION    : Implementation of Handler functions for URI
*  - COPYRIGHT      : &copy; 2015 Robert Bosch Engineering & Business Solutions
*  - Documents      : Give link of relevant documents
*  - HISTORY
*
*  Date     | Name          |  Version | Modification
* ----------|---------------|--------------------------|-----------------------
*  21.03.2016 | Ashish(RBEI/ECO2) | 1.0.0 | methods for URI DBus Server
*  26-05-2017 |RVH5KOR(RBEI/ECO22)| 1.0.1 | DLT Logs Changes
******************************************************************************/
#ifndef VARIANT_S_FTR_ENABLE_FEAT_GW_PSARCC
#include"URI_DBusHandler.h"
#include "dlt/dlt.h"

#ifdef VARIANT_S_FTR_ENABLE_FEAT_GW_RNAIVI
#include "routing-request.h"
#include "BeepHandler.h"
#endif //VARIANT_S_FTR_ENABLE_FEAT_GW_RNAIVI

#include "Utility.h"
#include "Phone_ClientInterface.h"
#include "ap-navigation-enums.h"

//Dlt Context
DLT_IMPORT_CONTEXT(AGW_URI);


typedef struct
{
  URI *object;
  GDBusMethodInvocation *invocation;
#ifdef VARIANT_S_FTR_ENABLE_FEAT_GW_RNAIVI
  TrpRoutingRequest *routing_request;
#endif //VARIANT_S_FTR_ENABLE_FEAT_GW_RNAIVI
  gchar *uri;
} MethodRequestData;

static void
parse_uri (MethodRequestData *request_data);

#ifdef VARIANT_S_FTR_ENABLE_FEAT_GW_RNAIVI
/*****************************************************************************/
/* URI-NAVI STATIC FUNCTION DECLARATION*/
/*****************************************************************************/
static void
parse_nav_uri (MethodRequestData *request_data);
static void
insert_way_points (MethodRequestData *request_data);
static void
method_request_data_free (MethodRequestData *request_data);
static void
vStartGuidance (MethodRequestData *request_data, gdouble dLongitude,
                gdouble dLatitude);
static void
start_guidance_cb (GDBusProxy *proxy, GAsyncResult *res, gpointer user_data);
static void 
vSetRouteCriterion( MethodRequestData *request_data, 
                    const gchar *chRouteCriteria);
static void set_route_criterion_cb ( GDBusProxy *proxy,
                                     GAsyncResult *res,
 									 gpointer user_data);
static void vParseAndSetDestinationAndWPCoord(MethodRequestData *request_data);
/*****************************************************************************/
/* URI-NAVI STATIC FUNCTION ENDS*/
/*****************************************************************************/

static void vParseBeepURI(const char* arg_uri);
#endif //VARIANT_S_FTR_ENABLE_FEAT_GW_RNAIVI

static void vParseSMSURI(const char* arg_uri);



/******************************************************************************
* Function:      method_request_data_free
* Description:   function to free method request data
* Parameters:    Method request data
* Return:        void
*****************************************************************************/
static void
method_request_data_free (MethodRequestData *request_data)
{
  if (request_data == NULL)
    return;

  g_clear_object (&request_data->object);
  g_clear_object (&request_data->invocation);
#ifdef VARIANT_S_FTR_ENABLE_FEAT_GW_RNAIVI
  g_clear_object (&request_data->routing_request);
#endif //VARIANT_S_FTR_ENABLE_FEAT_GW_RNAIVI
  g_free (request_data->uri);
  g_free (request_data);
}

/******************************************************************************
 * Function:      uri_error_quark
 * Description:   function to return GQuark
 * Parameters:    void
 * Return:        GQuark
 *****************************************************************************/
GQuark
uri_error_quark (void)
{
  static gsize id = 0;

  if (g_once_init_enter (&id))
    {
      g_once_init_leave (&id, g_quark_from_static_string ("uri-error-quark"));
    }

  return (GQuark) id;
}

/******************************************************************************
 * Function:      handle_delegate_uri
 * Description:   Method call handler Function
 * Parameters:    Proxy object, invocation pointer, uri
 * Return:        boolean
 *****************************************************************************/
gboolean
handle_delegate_uri (URI *object, GDBusMethodInvocation *invocation,
                     const gchar *arg_uri)
{
  MethodRequestData *request_data = NULL;
  DLT_LOG(AGW_URI,DLT_LOG_INFO,DLT_STRING( "URI : - METHOD HANDLER\t "),
                                             DLT_STRING( __FUNCTION__));
  DLT_LOG(AGW_URI,DLT_LOG_INFO,DLT_STRING( "URI : - URI RECEIVED\t "),
                                                  DLT_STRING( arg_uri));

  request_data = g_new0 (MethodRequestData, 1);
  request_data->object = g_object_ref (object);
  request_data->invocation = g_object_ref (invocation);
  request_data->uri = g_strdup (arg_uri);
  parse_uri (request_data);
  return TRUE;
}

/******************************************************************************
 * Function:      parse_uri
 * Description:   Method to parse URI to get scheme
 * Parameters:    MethodRequestData *request_data
 * Return:        void
 *****************************************************************************/
static void
parse_uri (MethodRequestData *request_data)
{
  g_autofree gchar *scheme;
  DLT_LOG(AGW_URI,DLT_LOG_INFO,DLT_STRING( "URI : - "),
                              DLT_STRING( __FUNCTION__));

  scheme = g_uri_parse_scheme (request_data->uri);
  if (NULL == scheme)
    {
      GError *error = NULL;
      DLT_LOG(AGW_URI,DLT_LOG_INFO,DLT_STRING( "URI : - INVALID STRING"));
      g_set_error (&error, uri_error_quark (), URI_ERROR_UNKNOWN_SCHEME,
                   "Unknown scheme");
      g_dbus_method_invocation_take_error (request_data->invocation, error);
      method_request_data_free (request_data);
      return;
    }
  if (!g_strcmp0 (scheme, "tel"))
    {
      DLT_LOG(AGW_URI,DLT_LOG_INFO,DLT_STRING( "URI : - TELEPHONE SCHEME -> "),
	                                                      DLT_STRING( scheme));
      DLT_LOG(AGW_URI,DLT_LOG_INFO,DLT_STRING("URI : - TELEPHONE SCHEME -> "),
	                                                       DLT_STRING(scheme));
      char **arr = g_strsplit (request_data->uri,":",2);
      DLT_LOG(AGW_URI,DLT_LOG_INFO,DLT_STRING("URI : - TOKEN VALUE  -> "),
	                                                       DLT_STRING(arr[1]));
	  vDialCall_Method(arr[1]);
      uri_complete_delegate_uri (request_data->object,
                                 request_data->invocation);
      method_request_data_free (request_data);
	  g_strfreev (arr);
    }
  else if (!g_strcmp0(scheme, "sms"))
    {
	  DLT_LOG(AGW_URI,DLT_LOG_INFO,DLT_STRING( "URI : - SMS SCHEME -> "),
	                                                DLT_STRING( scheme));
	  vParseSMSURI(request_data->uri);
	  uri_complete_delegate_uri(request_data->object,
		  request_data->invocation);
	  method_request_data_free(request_data);
    }
#ifdef VARIANT_S_FTR_ENABLE_FEAT_GW_RNAIVI
  else if (!g_strcmp0 (scheme, "nav"))
    {
      DLT_LOG(AGW_URI,DLT_LOG_INFO,DLT_STRING( "URI : NAVIGATION SCHEME -> "),
	                                                     DLT_STRING( scheme));
      parse_nav_uri (request_data);
    }
  else if (!g_strcmp0 (scheme, "beep"))
    {
      DLT_LOG(AGW_URI,DLT_LOG_INFO,DLT_STRING( "URI : Beep SCHEME -> "),DLT_STRING( scheme));
      vParseBeepURI (request_data->uri);
      uri_complete_delegate_uri(request_data->object,
		  request_data->invocation);
	   method_request_data_free(request_data);
    }
#endif
  else
    {
      GError *error = NULL;
      DLT_LOG(AGW_URI,DLT_LOG_INFO,DLT_STRING( "URI : - INVALID SCHEME TYPE-> "),
	                                                        DLT_STRING( scheme));
      g_set_error (&error, uri_error_quark (), URI_ERROR_UNKNOWN_SCHEME,
                   "Unknown scheme");
      g_dbus_method_invocation_take_error (request_data->invocation, error);
      method_request_data_free (request_data);
    }

}

#ifdef VARIANT_S_FTR_ENABLE_FEAT_GW_RNAIVI
/******************************************************************************
 * Function:      parse_nav_uri
 * Description:   Method to parse nav URI using TrpRoutingRequest utility
 * Parameters:    Method request data
 * Return:        void
 *****************************************************************************/
static void
parse_nav_uri (MethodRequestData *request_data)
{
	DLT_LOG(AGW_URI,DLT_LOG_INFO,DLT_STRING( "URI : - "),
	                               DLT_STRING( __FUNCTION__));
  GError *error = NULL;
  
  const gchar *l_chRouteCriteria = NULL;

  request_data->routing_request = trp_routing_request_new_for_uri (
                                                          request_data->uri,
                                                          TRP_URI_FLAGS_NONE,
                                                          &error   );
  if (error) 
    {
      g_dbus_method_invocation_take_error (request_data->invocation, error);
      method_request_data_free (request_data);
      return;
    }
  l_chRouteCriteria = trp_routing_request_get_parameter_value(
                                  request_data->routing_request, "optimize");
  if(l_chRouteCriteria != NULL)
  {
	  vSetRouteCriterion(request_data, l_chRouteCriteria);
  }
  else
  {
	  vParseAndSetDestinationAndWPCoord(request_data);
  }
}

/******************************************************************************
 * Function:	 vParseAndSetDestinationAndWPCoord
 * Description:  Function to parse and set destination and waypoint coordinates.
 * Parameters:	 const gchar *
 * Return:		 void
 ******************************************************************************/
static void vParseAndSetDestinationAndWPCoord(MethodRequestData *request_data)
{
	DLT_LOG(AGW_URI,DLT_LOG_INFO,DLT_STRING( "URI : - "),
	                          DLT_STRING( __FUNCTION__));
	GError *error = NULL;
	TrpPlace *destination;
	gdouble latitude, longitude;
	destination = 
	        trp_routing_request_get_destination (request_data->routing_request);
	latitude = trp_place_get_latitude (destination);
	longitude = trp_place_get_longitude (destination);
	g_object_unref (destination);

	if (!_trp_place_is_latitude (latitude) ||
		!_trp_place_is_longitude (longitude))
	{
		g_set_error (&error, TRP_PLACE_ERROR, TRP_PLACE_ERROR_OUT_OF_RANGE,
			"Unknown latitude or longitude");
		g_dbus_method_invocation_take_error (request_data->invocation, error);
		method_request_data_free (request_data);
		return;
	}
	vStartGuidance (request_data, longitude, latitude);
}

/******************************************************************************
 * Function:	 vSetRouteCriterion
 * Description:  Function to set routeCriteria i.e. ECO, FAST or SHORT
 * Parameters:	 const gchar *
 * Return:		 void
 ******************************************************************************/
static void vSetRouteCriterion( MethodRequestData *request_data,
                                const gchar *chRouteCriteria )
{
	DLT_LOG(AGW_URI,DLT_LOG_INFO,DLT_STRING( "URI : -"),
	                          DLT_STRING( __FUNCTION__));
	DLT_LOG(AGW_URI,DLT_LOG_INFO,DLT_STRING(
       	   "URI : Route criteria : -"),DLT_STRING( chRouteCriteria));
	guint32 l_u32RouteCriteria = 0;

	if(!g_strcmp0 (chRouteCriteria,"eco"))
	{
		l_u32RouteCriteria = (guint32) AP_NAVIGATION_ROUTE_CRITERIA_ECO;
	}
	else if(!g_strcmp0 (chRouteCriteria,"fast"))
	{
		l_u32RouteCriteria = (guint32) AP_NAVIGATION_ROUTE_CRITERIA_FAST;
	}
	else if(!g_strcmp0 (chRouteCriteria,"short"))
	{
		l_u32RouteCriteria = (guint32) AP_NAVIGATION_ROUTE_CRITERIA_SHORT;
	}
	else
	{
            g_dbus_method_invocation_return_error(
     			         request_data->invocation,
                         G_DBUS_ERROR,
                         G_DBUS_ERROR_NOT_SUPPORTED,
                         "Currently only eco,fast and short are supported" ); 		
	}

	DLT_LOG(AGW_URI,DLT_LOG_INFO,DLT_STRING( "URI : Route criteria : -"),
	                                      DLT_UINT( l_u32RouteCriteria));
	if(poGetNaviProxy())
	{
		g_dbus_proxy_call (
		                   poGetNaviProxy(),
 						   "setRouteCriterion",
           				   g_variant_new ("(u)", l_u32RouteCriteria),
						   G_DBUS_CALL_FLAGS_NONE,
						   -1, 
						   NULL,
						  (GAsyncReadyCallback)
 						      set_route_criterion_cb, request_data
						 );
	}
	else
	{
	   DLT_LOG(AGW_URI,DLT_LOG_ERROR,DLT_STRING("fatal:Invalid proxy pointer"));
	}
}

/******************************************************************************
 * Function:    	start_guidance_cb
 * Description:		call back function for startguidance
 * Parameters:      proxy object, Async result object, user data
 * Return:			void
 ******************************************************************************/
static void
set_route_criterion_cb(GDBusProxy *proxy, GAsyncResult *res, gpointer user_data)
{
	DLT_LOG(AGW_URI,DLT_LOG_INFO,DLT_STRING( "URI : - "),
	                           DLT_STRING( __FUNCTION__));
	MethodRequestData *request_data = user_data;
	GVariant *l_pResult;
	GError *l_poError = NULL;
	
	l_pResult = g_dbus_proxy_call_finish (proxy, res, &l_poError);

	if (l_poError)
	{
		DLT_LOG(AGW_URI,DLT_LOG_ERROR,DLT_STRING( 
		    "Failed to set route critera :  "),DLT_STRING( l_poError->message));
		g_dbus_method_invocation_take_error ( request_data->invocation,
                                  		      l_poError);
		g_clear_error(&l_poError);
	}
	else
	{
		DLT_LOG(AGW_URI,DLT_LOG_INFO,
		          DLT_STRING("Route Criteria set successfully"));
		vParseAndSetDestinationAndWPCoord(request_data);
		g_variant_unref (l_pResult);
	}
}

/******************************************************************************
 * Function:	 vStartGuidance
 * Description:  Function to connect navi servcie
                 and invokes method(startguidance)
 * Parameters:	 Method request data, longitude, latitude
 * Return:		 void
 ******************************************************************************/
static void
vStartGuidance (MethodRequestData *request_data, gdouble dLongitude,
                gdouble dLatitude)
{
  g_autofree gchar *log_string;
  GVariant *l_ptrLongLatVal;

  DLT_LOG(AGW_URI,DLT_LOG_INFO,DLT_STRING("URI: -"),DLT_STRING( __FUNCTION__));
  DLT_LOG(AGW_URI,DLT_LOG_INFO,DLT_STRING("Latitude: -"),DLT_FLOAT64(dLatitude));
  DLT_LOG(AGW_URI,DLT_LOG_INFO,DLT_STRING("Longitude: -"),
                                               DLT_FLOAT64(dLongitude)); 

  l_ptrLongLatVal = g_variant_new ("(dd)", dLongitude, dLatitude);
  GVariant *l_ptrLongLatValRef = g_variant_ref_sink(l_ptrLongLatVal);
  g_dbus_proxy_call (poGetNaviProxy (), "startGuidanceToPosWGS84",
                     g_variant_new ("(@(dd))", l_ptrLongLatValRef),
                     G_DBUS_CALL_FLAGS_NONE, -1, NULL,
                     (GAsyncReadyCallback) start_guidance_cb, request_data);
   g_variant_unref(l_ptrLongLatValRef);
}

/******************************************************************************
 * Function:    	start_guidance_cb
 * Description:		call back function for startguidance
 * Parameters:      proxy object, Async result object, user data
 * Return:			void
 ******************************************************************************/
static void
start_guidance_cb (GDBusProxy *proxy, GAsyncResult *res, gpointer user_data)
{
  MethodRequestData *request_data = user_data;
  GVariant *l_pResult;
  GError *l_poError = NULL;

  DLT_LOG(AGW_URI,DLT_LOG_INFO,DLT_STRING( "NAVI : - "),
                             DLT_STRING( __FUNCTION__));
  l_pResult = g_dbus_proxy_call_finish (proxy, res, &l_poError);

  if (l_poError)
    {
      DLT_LOG(AGW_URI,DLT_LOG_INFO,DLT_STRING("Failed to start guidance :  "),
	                                         DLT_STRING( l_poError->message));
      g_dbus_method_invocation_take_error (request_data->invocation, l_poError);
    }
  else
    {
      uri_complete_delegate_uri (request_data->object,
                                 request_data->invocation);
	  insert_way_points (request_data);
      g_variant_unref (l_pResult);
    }

  method_request_data_free(request_data);
}

/******************************************************************************
* Function:    	insert_way_points
* Description:	function to insert way points
* Parameters:   Method request data
* Return:		void
*******************************************************************************/
static void
insert_way_points (MethodRequestData *request_data)
{
  TrpRoutingRequest *routing_request = request_data->routing_request;
  GPtrArray *points;
  guint counter;
  GVariantBuilder builder;
  guint way_point_counter = 0;
  GVariant *way_point_list = NULL;

  points = trp_routing_request_get_points (routing_request);

  if(points != NULL)
  {
      DLT_LOG(AGW_URI,DLT_LOG_INFO,DLT_STRING(
  	      "NAVI:- insert_way_points points length is: "),DLT_UINT(points->len));
      if (points->len == 1) /* No way points ,1 indicates destination*/
      {
          g_ptr_array_free (points, TRUE);
          return ;
      }

      DLT_LOG(AGW_URI,DLT_LOG_INFO,DLT_STRING(
	     "NAVI: - insert_way_points points length is: "),DLT_UINT(points->len));
      g_variant_builder_init (&builder, G_VARIANT_TYPE ("a(dd)"));
      for (counter = 0; counter < (points->len - 1); counter++)
        {
          TrpRoutingParameter *p = g_ptr_array_index (points, counter);
          DLT_LOG(AGW_URI,DLT_LOG_INFO,DLT_STRING(
 			            "NAVI : parameter name is : "),
					     DLT_STRING(trp_routing_parameter_get_name (p)));
          DLT_LOG(AGW_URI,DLT_LOG_INFO,DLT_STRING("NAVI: parameter value is: "),
		           DLT_STRING(trp_routing_parameter_get_value (p)));
          if (!g_strcmp0 (trp_routing_parameter_get_name (p),
                          TRP_ROUTING_PARAMETER_WAYPOINT))
          {
              TrpPlace *way_place = trp_routing_parameter_get_place (p);
              gdouble latitude, longitude;

              latitude = trp_place_get_latitude (way_place);
              longitude = trp_place_get_longitude (way_place);

              if (_trp_place_is_latitude (latitude)
                  && _trp_place_is_longitude (longitude))
             {
                  g_variant_builder_add (&builder, "(dd)",
                                         latitude,
                                         longitude);
                  way_point_counter++;
             }
          }

        }
      g_ptr_array_free (points, TRUE);
      way_point_list = g_variant_builder_end (&builder);
  }

  if (way_point_counter > 0)
      vInitWaypointOp(way_point_list);
}

/******************************************************************************
* Function:    	    vParseBeepURI
* Description:		Parses the uri scheme and extracts the beep value.
* Parameters:       Beep URI
* Return:			void
********************************************************************************************/
static void vParseBeepURI(const char* arg_uri)
{
    DLT_LOG(AGW_URI,DLT_LOG_INFO,DLT_STRING( "URI : - HMI SCHEME -> "),
            DLT_STRING( arg_uri));
    //"beep:4"
    char **splitForBeepValue = g_strsplit(arg_uri, ":", 2);
    DLT_LOG(AGW_URI,DLT_LOG_INFO,DLT_STRING( "URI : - TOKEN VALUE  -> "),
            DLT_STRING( splitForBeepValue[1]));
     
    if(splitForBeepValue[1] != NULL)
    {
        DLT_LOG(AGW_URI,DLT_LOG_INFO,DLT_STRING( "URI : - TOKEN VALUE  -> "),
                DLT_STRING( splitForBeepValue[1]));
        gint iBeepType = atoi (splitForBeepValue[1]);
        if(iBeepType > 0)
        {
            vForwardBeep((ApBeepType)iBeepType);
        }
    }
    g_strfreev(splitForBeepValue);     
}
#endif

/*********************************************************************************************
* Function:    	    vParseSMSURI
* Description:		Parses the uri scheme and extracts the sms message.
* Parameters:       SMS URI
* Return:			void
*******************************************************************************/

static void vParseSMSURI(const char* arg_uri)
{
   DLT_LOG(AGW_URI,DLT_LOG_INFO,DLT_STRING("URI : - SMS SCHEME -> "),
                                               DLT_STRING( arg_uri));
   char **arr = g_strsplit(arg_uri, ":", 2);
   DLT_LOG(AGW_URI,DLT_LOG_INFO,DLT_STRING( "URI : - TOKEN VALUE  -> "),
                                                    DLT_STRING( arr[1]));
   
   // sample string for the sms.
   //"sms:+15105550101?body=welcome%20to%20bosch%20";
   char* smsTextMessage = NULL;
   char* phoneNumber = NULL;

   DLT_LOG(AGW_URI,DLT_LOG_INFO,DLT_STRING("first: "),DLT_STRING( arr[0]));
   DLT_LOG(AGW_URI,DLT_LOG_INFO,DLT_STRING("second: "),DLT_STRING( arr[1]));

   char **splitForPhoneNumber = g_strsplit(arr[1], "?", 2);
   DLT_LOG(AGW_URI,DLT_LOG_INFO,DLT_STRING("first: "),
                         DLT_STRING( splitForPhoneNumber[0]));
   DLT_LOG(AGW_URI,DLT_LOG_INFO,DLT_STRING("second: "),
                         DLT_STRING( splitForPhoneNumber[1]));
   phoneNumber = splitForPhoneNumber[0];

   char **splitForSMS = g_strsplit(splitForPhoneNumber[1], "=", 2);
   DLT_LOG(AGW_URI,DLT_LOG_INFO,DLT_STRING("first: "),
                                     DLT_STRING( splitForSMS[0]));
   DLT_LOG(AGW_URI,DLT_LOG_INFO,DLT_STRING( "second: "),
                                    DLT_STRING( splitForSMS[1]));
   
   //string is in url encoded format. Need to convert to normal form.
   smsTextMessage = g_uri_unescape_string(splitForSMS[1], NULL);

   DLT_LOG(AGW_URI,DLT_LOG_INFO,DLT_STRING("sms "),DLT_STRING( smsTextMessage));

   gsize strLen = strlen(smsTextMessage);

   if (strLen > 160)
   {
	   DLT_LOG(AGW_URI,DLT_LOG_INFO,DLT_STRING( 
	                      "cannot send messagees length more than 160"));
	   return;
   }

   vCreateMessageMethod(phoneNumber, smsTextMessage);
   g_strfreev(arr);
   g_strfreev(splitForPhoneNumber);
   g_strfreev(splitForSMS);
   g_free(smsTextMessage);
}
#endif//VARIANT_S_FTR_ENABLE_FEAT_GW_PSARCC