/******************************************************************************
*
* FILE:          SPDBusHandler.c
*
* PROJECT:       FC_GENERIC_GATEWAY
*
* DESCRIPTION:   DBUS service handler for SP.
*
* AUTHOR:        RVH5KOR(RBEI/ECO2)
*
* COPYRIGHT:     (c) 2017 Robert Bosch GmbH, Hildesheim
*
******************************************************************************/
#include <gio/gio.h>
#include <dlfcn.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include "SP_ClientInterface.h"
#include "SPDbusHandler.h"
#include "dlt/dlt.h"
#include "AutomDBusServer.h"
#include "ap-sp-error-enums.h"

DLT_DECLARE_CONTEXT(AGW_SPS);
static GDBusConnection* g_poconnection_object = NULL;
static SeamlessPairingServer* pSPProxyObj = NULL;
static guint  iStartSPServer = 0;
static guint  iStopSPServer = 0;
static guint  iSendMessageToApp = 0;

static void  vHandle_start_sp_server (GDBusConnection *connection,
                                        const gchar *sender_name,
                                        const gchar *object_path,
                                        const gchar *interface_name,
                                        const gchar *signal_name,
                                        GVariant *parameters,
                                        gpointer user_data);

static void  vHandle_stop_sp_server (GDBusConnection *connection,
                                           const gchar *sender_name,
                                           const gchar *object_path,
                                           const gchar *interface_name,
                                           const gchar *signal_name,
                                           GVariant *parameters,
                                           gpointer user_data);

static void  vHandle_send_message_to_app (GDBusConnection *connection,
                                           const gchar *sender_name,
                                           const gchar *object_path,
                                           const gchar *interface_name,
                                           const gchar *signal_name,
                                           GVariant *parameters,
                                           gpointer user_data);

/******************************************************************************
* Function:    poGetSPProxyObj
* Description: Gives the Proxy object of SP Handler
* Parameters:  void
* Return:      SeamlessPairingServer *
*****************************************************************************/
SeamlessPairingServer *poGetSPProxyObj()
{
    DLT_LOG(AGW_SPS, DLT_LOG_INFO, DLT_STRING("+"),DLT_STRING(__FUNCTION__));
    if(pSPProxyObj == NULL)
    {
        pSPProxyObj = seamless_pairing_server_skeleton_new();
    }
    return pSPProxyObj;
}

/******************************************************************************
* Function:    bHandle_process_app_message
* Description: used to push the certificate
* Parameters:  proxy object, invocation pointer, user data
* Return:      gboolean
*****************************************************************************/
gboolean
bHandle_process_app_message (SeamlessPairingServer *object,
                                                GDBusMethodInvocation *invocation,
                                                const gchar *key,
                                                GVariant *value,
                                            gpointer user_data)
{
    DLT_LOG(AGW_SPS, DLT_LOG_INFO, DLT_STRING("+"),DLT_STRING(__FUNCTION__));
    user_data = invocation;
    //Call SPClientHandler method here
    if(!bProcessAppMsg(key,value,user_data))
    {
        //ERROR
        DLT_LOG(AGW_SPS, DLT_LOG_ERROR, DLT_STRING(__FUNCTION__),
             DLT_STRING("Error in bHandle_process_app_message"));
        g_dbus_method_invocation_return_error (invocation,
                                                AP_SP_ERROR,
                                                AP_SP_ERROR_SPMW_UNAVAILABLE,
                                                "Seamless midw proxy not available");
        return FALSE;
    }
    seamless_pairing_server_complete_proccess_app_message(object, invocation);
    return TRUE;
}

/******************************************************************************
* Function:    bHandle_SPSConnection_State
* Description: Handler for SPS Connection state
* Parameters:  proxy object, invocation pointer, user data
* Return:      gboolean
*****************************************************************************/
gboolean
bHandle_SPSConnection_State( SeamlessPairingServer *object,
                             GDBusMethodInvocation *invocation,
                             guint16 connState,
                             gpointer user_data )
{
    DLT_LOG(AGW_SPS, DLT_LOG_INFO, DLT_STRING("+"),DLT_STRING(__FUNCTION__));
    user_data = invocation;
    //Call SPClientHandler method here
    if(!bHandleSPSConnectionState(connState, user_data))
    {
        //ERROR
        DLT_LOG(AGW_SPS, DLT_LOG_ERROR, DLT_STRING(__FUNCTION__),
            DLT_STRING("Error in bHandleSPSConnState"));
        g_dbus_method_invocation_return_error ( invocation,
                                                AP_SP_ERROR,
                                                AP_SP_ERROR_SPMW_UNAVAILABLE,
                                                "Seamless midw proxy not available");

	return FALSE;
    }
    seamless_pairing_server_complete_spsconnection_state(object, invocation);
    return TRUE;
}

/******************************************************************************
* Function:    bInitialize
* Description: Initializes the plugin by creating skeleton object
* Parameters:  void
* Return:      gboolean - TRUE when skeleteon creation is success,
                          FALSE when skeleton creation fails
*****************************************************************************/
gboolean bInitializeSP()
{
    DLT_REGISTER_CONTEXT(AGW_SPS,"GWSS","Gateway seamlesspairing"
                          "sps context for DLT Logging");

    DLT_LOG(AGW_SPS, DLT_LOG_INFO, DLT_STRING("+"),DLT_STRING(__FUNCTION__));

    if(sp_service_watch())
    {
        DLT_LOG(AGW_SPS, DLT_LOG_INFO, DLT_STRING("sp_service_watch success"));
        return TRUE;
    }
    else
    {
        DLT_LOG(AGW_SPS, DLT_LOG_INFO, DLT_STRING("sp_service_watch failed"));
        return FALSE;
    }
}

/******************************************************************************
* Function:    vUninitializeSP
* Description: This method unintializes the SP gateway component
* Parameters:  void
* Return:      void
*****************************************************************************/
void vUninitializeSP()
{
    DLT_LOG(AGW_SPS, DLT_LOG_INFO, DLT_STRING("+"),DLT_STRING(__FUNCTION__));
    //Call service UNwatch method here
    sp_service_unwatch();
    DLT_UNREGISTER_CONTEXT(AGW_SPS);
}

/*******************************************************************************
 * Function:     vExportAGWSPInterface
 * Description:  This method exposes the gateway SP methods
 * Parameters:   void
 * Return:       void
 ******************************************************************************/
void vExportAGWSPInterface()
{
    DLT_LOG(AGW_SPS, DLT_LOG_INFO, DLT_STRING("+"),DLT_STRING(__FUNCTION__));
    if (poGetSPProxyObj() == NULL)
    {
        DLT_LOG(AGW_SPS, DLT_LOG_ERROR, DLT_STRING(__FUNCTION__),
                   DLT_STRING("(): poGetSPProxyObj failed"));
        return;
    }

    GError *error = NULL;
    GDBusConnection* connection = NULL;
    gboolean bRet = FALSE;
    if((connection = poGetGDBusConnection()) == NULL)
    {
        DLT_LOG(AGW_SPS, DLT_LOG_ERROR, DLT_STRING(__FUNCTION__),
                   DLT_STRING("(): poGetGDBusConnection failed"));
        return;
    }


    if (!g_dbus_interface_skeleton_export(
                          G_DBUS_INTERFACE_SKELETON(poGetSPProxyObj()),
                          connection,
                          "/com/bosch/AutomotiveProxy/SeamlessPairingServer",
                          &error) )
    {
        DLT_LOG(AGW_SPS, DLT_LOG_ERROR, DLT_STRING(__FUNCTION__),
               DLT_STRING("error in export"));

        if (error)
        {
            DLT_LOG(AGW_SPS, DLT_LOG_ERROR, DLT_STRING(__FUNCTION__),
               DLT_STRING("():Error msg: "), 
                  DLT_STRING(error->message ? error->message : "")); 
            g_clear_error(&error);
        }
        return;
    }
    else
    {
        DLT_LOG(AGW_SPS, DLT_LOG_ERROR, DLT_STRING(__FUNCTION__),
            DLT_STRING("No Error in vExportAGWSPInterface"));
    }

    g_signal_connect (poGetSPProxyObj(), "handle-proccess-app-message",
                     G_CALLBACK (bHandle_process_app_message), NULL);

    g_signal_connect (poGetSPProxyObj(), "handle-spsconnection-state",
                     G_CALLBACK (bHandle_SPSConnection_State), NULL);

    bRet = bRegisterSPServiceProperties();
	if(bRet == FALSE)
	{
		 DLT_LOG(AGW_SPS, DLT_LOG_ERROR, DLT_STRING(__FUNCTION__),
            DLT_STRING("Signal Subscription Failed"));	
	}	
}

/*******************************************************************************
  * Function:    vUnSubscribeSignals
  * Description: unexports the SP interface
  * Parameters:  void
  * Return:      void
*******************************************************************************/
void vUnSubscribeSignals()
{
  if(g_poconnection_object != NULL )
	{
		if(iStartSPServer != 0)
		{
			g_dbus_connection_signal_unsubscribe(g_poconnection_object,
																iStartSPServer);
			iStartSPServer = 0;
		}
		if(iStopSPServer != 0)
		{
			g_dbus_connection_signal_unsubscribe(g_poconnection_object,
																iStopSPServer);
			iStopSPServer = 0;
		}
		if(iSendMessageToApp != 0)
		{
			g_dbus_connection_signal_unsubscribe(g_poconnection_object,
															iSendMessageToApp);
			iSendMessageToApp = 0;
		}
	}
}
/*******************************************************************************
  * Function:    vUnexportAGWSPInterface
  * Description: unexports the SP interface
  * Parameters:  void
  * Return:      void
*******************************************************************************/
void vUnexportAGWSPInterface()
{
    DLT_LOG(AGW_SPS, DLT_LOG_INFO, DLT_STRING("+"),DLT_STRING(__FUNCTION__));
	
	vUnSubscribeSignals();

	
	gboolean bExported = FALSE;
	
	
	if(pSPProxyObj != NULL )
     {
		 bExported = g_dbus_interface_skeleton_has_connection(G_DBUS_INTERFACE_SKELETON(pSPProxyObj),poGetSessionBusConnection());
		 if(bExported == TRUE)
		 { 
         g_dbus_interface_skeleton_unexport(
                           G_DBUS_INTERFACE_SKELETON(pSPProxyObj));
         }
		 g_clear_object(&pSPProxyObj);
		 
     }
	if(g_poconnection_object != NULL)
	{
		 g_object_unref(g_poconnection_object);
		 g_poconnection_object = NULL;
	}	
}

/*******************************************************************************
 * Function:    bRegisterSPServiceProperties
 * Description:
 * Parameters: void
 * Return: gboolean
 ******************************************************************************/
gboolean bRegisterSPServiceProperties()
{
    DLT_LOG(AGW_SPS, DLT_LOG_INFO, DLT_STRING("+"),DLT_STRING(__FUNCTION__));

    gboolean bRetVal = false;
    GError *l_poerror = NULL;
    g_poconnection_object = g_bus_get_sync(G_BUS_TYPE_SESSION
                                                            , NULL, &l_poerror);
    if(g_poconnection_object)
    {
        /* connect to signal StartSPServer */
        iStartSPServer = g_dbus_connection_signal_subscribe(g_poconnection_object
                                           , SP_CONNECTED_SERVICE_BUS_NAME
                                           , SP_CONNECTED_SERVICE_INTERFACE_NAME
                                           , "StartSPServer"
                                           , SP_CONNECTED_SERVICE_OBJ_PATH
                                           , NULL
                                           , G_DBUS_SIGNAL_FLAGS_NONE
                                           , vHandle_start_sp_server
                                           , NULL
                                           , NULL);
		if(iStartSPServer == 0)
		{
			DLT_LOG(AGW_SPS, DLT_LOG_ERROR, DLT_STRING(__FUNCTION__),
            DLT_STRING("Subscription of signal StartSPServer failed"));
			bRetVal = false;
		}
        /* connect to signal StopSPServer */
       iStopSPServer = g_dbus_connection_signal_subscribe(g_poconnection_object
                                           , SP_CONNECTED_SERVICE_BUS_NAME
                                           , SP_CONNECTED_SERVICE_INTERFACE_NAME
                                           , "StopSPServer"
                                           , SP_CONNECTED_SERVICE_OBJ_PATH
                                           , NULL
                                           , G_DBUS_SIGNAL_FLAGS_NONE
                                           , vHandle_stop_sp_server
                                           , NULL
                                           , NULL);
		                                  
		if(iStopSPServer == 0)
		{
			DLT_LOG(AGW_SPS, DLT_LOG_ERROR, DLT_STRING(__FUNCTION__),
            DLT_STRING("Subscription of signal StopSPServer failed"));
			bRetVal = false;	
		}
        /* connect to signal SendMessageToApp */
       iSendMessageToApp = g_dbus_connection_signal_subscribe(g_poconnection_object
                                           , SP_CONNECTED_SERVICE_BUS_NAME
                                           , SP_CONNECTED_SERVICE_INTERFACE_NAME
                                           , "SendMessageToApp"
                                           , SP_CONNECTED_SERVICE_OBJ_PATH
                                           , NULL
                                           , G_DBUS_SIGNAL_FLAGS_NONE
                                           , vHandle_send_message_to_app
                                           , NULL
                                           , NULL);
       
	if(iSendMessageToApp == 0)
	{
         DLT_LOG(AGW_SPS, DLT_LOG_ERROR, DLT_STRING(__FUNCTION__),
         DLT_STRING("Subscription of signal StopSPServer failed"));
		 bRetVal = false;		
	}
    if(iStartSPServer && iStopSPServer && iSendMessageToApp)	
	{
	bRetVal = true;
    }
	}
    else
    {
        bRetVal = false;
    }
    return bRetVal;
}

/*******************************************************************************
 * Function:		vHandle_start_sp_server
 * Description:		Invoked from DBus Client
 * Parameters:
 * Return:			void
 ******************************************************************************/
static void  vHandle_start_sp_server (GDBusConnection *connection,
                                        const gchar *sender_name,
                                        const gchar *object_path,
                                        const gchar *interface_name,
                                        const gchar *signal_name,
                                        GVariant *parameters,
                                        gpointer user_data)
{
     DLT_LOG(AGW_SPS, DLT_LOG_INFO, DLT_STRING("+"),DLT_STRING(__FUNCTION__));
     if(!parameters)
     {
        DLT_LOG(AGW_SPS, DLT_LOG_ERROR, DLT_STRING(__FUNCTION__),
                DLT_STRING("(): Null params"));
        return;
     }
     GVariant *connInfo = NULL;
     g_variant_get(parameters, "(@a{sv})", &connInfo);
     if(!connInfo)
     {
         DLT_LOG(AGW_SPS, DLT_LOG_ERROR, DLT_STRING(__FUNCTION__),
                DLT_STRING("(): Null connection info"));
         return;
     } 
     
    //Read the data & Emit Signal to SP Server
    if(poGetSPProxyObj())
    {
        seamless_pairing_server_emit_start_spserver(poGetSPProxyObj(), connInfo);
    }
    else
    {
       DLT_LOG(AGW_SPS, DLT_LOG_ERROR,DLT_STRING(__FUNCTION__),
                 DLT_STRING(": Null proxy"));
    }
}

/*******************************************************************************
 * Function:		vHandle_stop_sp_server
 * Description:		Invoked from DBus Client
 * Parameters:
 * Return:			void
 ******************************************************************************/
static void  vHandle_stop_sp_server (GDBusConnection *connection,
                                        const gchar *sender_name,
                                        const gchar *object_path,
                                        const gchar *interface_name,
                                        const gchar *signal_name,
                                        GVariant *parameters,
                                        gpointer user_data)
{
    DLT_LOG(AGW_SPS, DLT_LOG_INFO, DLT_STRING("+"),DLT_STRING(__FUNCTION__));
    guint16 pairStatus = 0;
    gchar*  certPath;

    if((!poGetSPProxyObj()) || (!parameters))
    {
         DLT_LOG(AGW_SPS, DLT_LOG_ERROR,DLT_STRING(__FUNCTION__),
                 DLT_STRING(": Null proxy or params"));
         return;
    }
    else
    {
        g_variant_get(parameters, "(qs)", &pairStatus, &certPath);

        DLT_LOG(AGW_SPS, DLT_LOG_INFO,DLT_STRING(__FUNCTION__),
                 DLT_UINT16(pairStatus), DLT_STRING(certPath));

        seamless_pairing_server_emit_stop_spserver( poGetSPProxyObj(), 
                                                    pairStatus,
                                                    certPath 
                                                  );
        g_free(certPath);  
    }
    DLT_LOG(AGW_SPS, DLT_LOG_INFO, DLT_STRING("-"),DLT_STRING(__FUNCTION__));
}

/*******************************************************************************
 * Function:		vHandle_send_message_to_app
 * Description:		Invoked from DBus Client
 * Parameters:
 * Return:			void
 ******************************************************************************/
static void  vHandle_send_message_to_app (GDBusConnection *connection,
                                        const gchar *sender_name,
                                        const gchar *object_path,
                                        const gchar *interface_name,
                                        const gchar *signal_name,
                                        GVariant *parameters,
                                        gpointer user_data)
{
     DLT_LOG(AGW_SPS, DLT_LOG_INFO, DLT_STRING("+"),DLT_STRING(__FUNCTION__));
     if(!parameters)
     {
        DLT_LOG(AGW_SPS, DLT_LOG_ERROR, DLT_STRING(__FUNCTION__),
                DLT_STRING("(): Null params"));
        return;
     }
     GVariant *message = NULL;
     g_variant_get(parameters, "(@a{ss})", &message);
     if(!message)
     {
         DLT_LOG(AGW_SPS, DLT_LOG_ERROR, DLT_STRING(__FUNCTION__),
                DLT_STRING("(): Null message"));
         return;
     }

     if(poGetSPProxyObj())
     {
         seamless_pairing_server_emit_send_message_to_app(poGetSPProxyObj(), message);
     }
     else
     {
         DLT_LOG(AGW_SPS, DLT_LOG_INFO,DLT_STRING(__FUNCTION__),
                  DLT_STRING(": Null proxy"));
     }
     DLT_LOG(AGW_SPS, DLT_LOG_INFO, DLT_STRING("-"),DLT_STRING(__FUNCTION__));
}
