/*!
********************************************************************************
* @file             : SPClientHandler.c
********************************************************************************
*  - PROJECT:       : Methods to connect to SP
*                       and call API's
*  - SW-COMPONENT   : IPCM
*  - DESCRIPTION    : D-Bus client to connect to SP
*  - COPYRIGHT      : &copy; 2017 Robert Bosch Engineering & Business Solutions
*  - Documents      : -
*  - HISTORY
*
*  Date     | Name          |  Version | Modification
* ----------|---------------|--------------------------|------------------------
*02.01.2018 | RVH5KOR(RBEI/ECO2) | 1.0.0 | Dbus client impl. to connect to SP
*******************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include "SPClientHandler.h"
#include "SP_ClientInterface.h"
#include "dlt/dlt.h"

DLT_IMPORT_CONTEXT(AGW_SPS);

static GDBusProxy*  pSPClientProxyObj = NULL;
static guint        iSPWatcherID      = 0;

/*******************************************************************************
 * Function:    sp_service_watch
 * Description: This methods intiates the watch for SP service
 * Parameters:  void
 * Return:      gboolean
 ******************************************************************************/
gboolean sp_service_watch()
{
    DLT_LOG(AGW_SPS, DLT_LOG_INFO, DLT_STRING("+"),DLT_STRING(__FUNCTION__));
    //DLT_LOG(AGW_ECM, DLT_LOG_INFO, DLT_STRING("+"),DLT_STRING(__FUNCTION__));
    /*iSPWatcherID = g_bus_watch_name ( G_BUS_TYPE_SESSION,
                                    "com.escrypt.ecm",
                                    G_BUS_NAME_WATCHER_FLAGS_NONE,
                                    sp_service_appeared,
                                    sp_service_vanished,
                                    NULL,
                                    NULL );*/

    iSPWatcherID = g_bus_watch_name ( G_BUS_TYPE_SESSION,
                                    SP_CONNECTED_SERVICE_BUS_NAME,
                                    G_BUS_NAME_WATCHER_FLAGS_NONE,
                                    sp_service_appeared,
                                    sp_service_vanished,
                                    NULL,
                                    NULL);

     //DLT_LOG(AGW_ECM, DLT_LOG_INFO, DLT_STRING("-"),DLT_STRING(__FUNCTION__));
     return (iSPWatcherID ? TRUE : FALSE );
}

/*******************************************************************************
 * Function:    sp_service_unwatch
 * Description: This methods uninitializes the SP service
 * Parameters:  void
 * Return:      void
 ******************************************************************************/
void sp_service_unwatch()
{
    //DLT_LOG(AGW_ECM, DLT_LOG_INFO, DLT_STRING("+"),DLT_STRING(__FUNCTION__));
    DLT_LOG(AGW_SPS, DLT_LOG_INFO, DLT_STRING("+"),DLT_STRING(__FUNCTION__));
    if(iSPWatcherID)
    {
       g_bus_unwatch_name(iSPWatcherID);
       iSPWatcherID = 0;
    }
    clearGatewaySPClientProxy();
    vUnexportAGWSPInterface();
    //DLT_LOG(AGW_ECM, DLT_LOG_INFO, DLT_STRING("-"),DLT_STRING(__FUNCTION__));
}

/******************************************************************************
* Function:    bProcessAppMsg
* Description: This method provides the SP client proxy object
* Parameters:  None
* Return:      gboolean
*******************************************************************************/
gboolean bProcessAppMsg(const gchar *key,
                         GVariant *value,
                      gpointer user_data)
{
    DLT_LOG(AGW_SPS, DLT_LOG_INFO, DLT_STRING("+"),DLT_STRING(__FUNCTION__));
    if(pSPClientProxyObj)
    {
        //FORWORD THE CALL TO SPMW
        g_dbus_proxy_call (pSPClientProxyObj,
                        "ProccessAppMessage",
                   g_variant_new ("(s@a(y))",
                                         key,
                                      value),
                      G_DBUS_CALL_FLAGS_NONE,
                                          -1,
                                        NULL,
                                        NULL,
                                  user_data);
    }
    else
    {
        //ERROR
        return FALSE;
    }
    return TRUE;
}

/******************************************************************************
* Function:    bHandleSPSConnectionState
* Description: This method is to forward SPS Connection state
* Parameters:  ErrorCode, User_data
* Return:      gboolean
*******************************************************************************/
gboolean  bHandleSPSConnectionState( guint16 errorCode,
                                     gpointer user_data)
{
    DLT_LOG(AGW_SPS, DLT_LOG_INFO, DLT_STRING("+"),DLT_STRING(__FUNCTION__));
    if(pSPClientProxyObj)
    {
        //FORWORD THE CALL TO SPMW
        g_dbus_proxy_call (pSPClientProxyObj,
          "SPSConnectionState",
          g_variant_new ("(q)",
                         errorCode),
          G_DBUS_CALL_FLAGS_NONE,
          -1,
          NULL,
          NULL,
          user_data);
    }
    else
    {
        //ERROR
        return FALSE;
    }
    return TRUE;
}

/****************************************************************************
 * Function:    sp_service_appeared
 * Description: Name appeared call back for SP
 * Parameters:  connection , service_name , owner_name, userdata
 * Return:      void
 *****************************************************************************/
void
sp_service_appeared (GDBusConnection *connection, const gchar *name,
               const gchar *name_owner, gpointer user_data)
{
    DLT_LOG(AGW_SPS, DLT_LOG_INFO, DLT_STRING("+"),DLT_STRING(__FUNCTION__));
    if(connection)
    {
        g_dbus_proxy_new ( connection,
                           G_DBUS_PROXY_FLAGS_DO_NOT_LOAD_PROPERTIES,
                           NULL,
                           SP_CONNECTED_SERVICE_BUS_NAME,
                           SP_CONNECTED_SERVICE_OBJ_PATH,
                           SP_CONNECTED_SERVICE_INTERFACE_NAME,
                           NULL,
                           vSPClientProxyCallBack,
                           NULL );
    }
}

/******************************************************************
 * Function:     vSPClientProxyCallBack
 * Description:  This is call back function for SP proxy
                 object creation
 * Parameters:   source_object, res, user_data
 * Return:       void
 ******************************************************************/
void vSPClientProxyCallBack( GObject *source_object,
                        GAsyncResult *res,
                        gpointer user_data )
{
    DLT_LOG(AGW_SPS, DLT_LOG_INFO, DLT_STRING("+"),DLT_STRING(__FUNCTION__));
    GError *error = NULL;
    pSPClientProxyObj = g_dbus_proxy_new_finish (res, &error);

    if(pSPClientProxyObj)
    {
        /*DLT_LOG(AGW_ECM, DLT_LOG_INFO,DLT_STRING(__FUNCTION__), DLT_STRING("():"
               "ECM Proxy object created"));*/
        vExportAGWSPInterface();
    }
    else
    {
        if(error != NULL)
        {
           g_clear_error (&error);
        }
    }
}

/****************************************************************************
 * Function:    sp_service_vanished
 * Description: Name disappeared call back for ECM service
 * Parameters:  connection, service_name, user_data
 * Return:      void
 *****************************************************************************/
void
sp_service_vanished (GDBusConnection *connection,
                      const gchar *service_name,
                      gpointer user_data)
{
    DLT_LOG(AGW_SPS, DLT_LOG_INFO, DLT_STRING("+"),DLT_STRING(__FUNCTION__));
    clearGatewaySPClientProxy();
    vUnexportAGWSPInterface();
}

/******************************************************************************
 * Function:      clearGatewaySPClientProxy
 * Description:   This function clears the AGW SP Proxy
 * Parameters:    void
 * Return:        void
 ******************************************************************************/
void clearGatewaySPClientProxy()
{
    DLT_LOG(AGW_SPS, DLT_LOG_INFO, DLT_STRING("+"),DLT_STRING(__FUNCTION__));
    if(pSPClientProxyObj)
    {
       g_clear_object(&pSPClientProxyObj);
       
    }
}
