/******************************************************************************
*
* FILE:          MediaPlayerDbusHandler.c
*   
* PROJECT:       Generic gateway
*
* DESCRIPTION:   Dbus server implemenatation for MediaPlayer
*
* AUTHOR:        
*
* COPYRIGHT:     (c) 2017 Robert Bosch GmbH, Hildesheim
*
*****************************************************************************/

#ifndef VARIANT_S_FTR_ENABLE_FEAT_GW_PSARCC
#include <gio/gio.h>
#include <stdlib.h>

#include "MediaPlayerDbusHandler.h"
#include "AutomDBusServer.h"
#include "DBusToMPlayProxyWrapper.h"
#include "ap-media-player-error-enums.h"

//Dlt
#include "dlt/dlt.h"
//Dlt Context
DLT_IMPORT_CONTEXT(AGW_MediaPlayer);

static GDBusMethodInvocation *g_invocation_mediaplay = NULL;
static GDBusMethodInvocation *g_invocation_mediastop = NULL;
static GDBusMethodInvocation *g_invocation_mediapause = NULL;
static GDBusMethodInvocation *g_invocation_mediaseek = NULL;
static GDBusMethodInvocation *g_invocation_medianext = NULL;
static GDBusMethodInvocation *g_invocation_mediaprev = NULL;
static GDBusMethodInvocation *g_invocation_currenttrackinfo = NULL;


/******************************************************************************
* Function:    handle_media_play
*
* Description: Handler for Play method
*
* Parameters:  MediaPlayerPlayer object,invocation pointer
*
* Return:      bool
******************************************************************************/
gboolean handle_media_play (MediaPlayerPlayer *object,
                       GDBusMethodInvocation *invocation)
{
	DLT_LOG(AGW_MediaPlayer, DLT_LOG_INFO, DLT_STRING("[FUNC] entered"),
										   DLT_STRING(__FUNCTION__));
	
	g_invocation_mediaplay =  g_object_ref(invocation);    

	bool bRet = bMediaInterface_play();
    if(bRet)
	{
       DLT_LOG(AGW_MediaPlayer,DLT_LOG_INFO,DLT_STRING(
	                "getCurrentPlayingTrack invoked successfully"));
	   return bRet;
	}else{
		vRequestPlayMError(AP_MPLAYER_ERROR_FUNC_NOT_SUPPORTED);
		return TRUE;
	}
	DLT_LOG(AGW_MediaPlayer, DLT_LOG_INFO, DLT_STRING("[FUNC] left"),
								   DLT_STRING(__FUNCTION__));
}

/******************************************************************************
* Function:   handle_media_pause
*
* Description: Handler for Pause method
*
* Parameters:  MediaPlayerPlayer object,invocation pointer
*
* Return:      bool
*******************************************************************************/
gboolean handle_media_pause (MediaPlayerPlayer *object,
                       GDBusMethodInvocation *invocation)
{
	DLT_LOG(AGW_MediaPlayer, DLT_LOG_INFO, DLT_STRING("[FUNC] entered"),
										   DLT_STRING(__FUNCTION__));
	g_invocation_mediapause =  g_object_ref(invocation);    

	bool bRet = bMediaInterface_pause();
    if(bRet)
	{
       DLT_LOG(AGW_MediaPlayer,DLT_LOG_INFO,DLT_STRING(
	            "getCurrentPlayingTrack invoked successfully"));
	   return bRet;
	}else{
		vRequestPauseMError(AP_MPLAYER_ERROR_FUNC_NOT_SUPPORTED);
		return TRUE;
	}    
	DLT_LOG(AGW_MediaPlayer, DLT_LOG_INFO, DLT_STRING("[FUNC] left"),
								   DLT_STRING(__FUNCTION__));
}

/******************************************************************************
* Function:    handle_media_stop
*
* Description: Handler for Stop method
*
* Parameters:  MediaPlayerPlayer object,invocation pointer
*
* Return:      bool
*******************************************************************************/
gboolean handle_media_stop (MediaPlayerPlayer *object,
                       GDBusMethodInvocation *invocation)
{
	DLT_LOG(AGW_MediaPlayer, DLT_LOG_INFO, DLT_STRING("[FUNC] entered"),
										   DLT_STRING(__FUNCTION__));
	g_invocation_mediastop =  g_object_ref(invocation);    

	bool bRet = bMediaInterface_stop();    
    if(bRet)
	{
       DLT_LOG(AGW_MediaPlayer,DLT_LOG_INFO,DLT_STRING(
	            "getCurrentPlayingTrack invoked successfully"));
	   return bRet;
	}else{
		vRequestStopMError(AP_MPLAYER_ERROR_FUNC_NOT_SUPPORTED);
		return TRUE;
	}
	DLT_LOG(AGW_MediaPlayer, DLT_LOG_INFO, DLT_STRING("[FUNC] left"),
								   DLT_STRING(__FUNCTION__));
}

/******************************************************************************
* Function:    handle_media_seek 
*
* Description: Handler for seek method
*
* Parameters:  MediaPlayerPlayer object,invocation pointer
*
* Return:      bool
*******************************************************************************/
gboolean handle_media_seek (MediaPlayerPlayer *object,
                       GDBusMethodInvocation *invocation,
                       gint64 arg_offset)
{
	DLT_LOG(AGW_MediaPlayer, DLT_LOG_INFO, DLT_STRING("[FUNC] entered"),
										   DLT_STRING(__FUNCTION__));
	g_invocation_mediaseek =  g_object_ref(invocation);    

	bool bRet = bMediaInterface_seekTo((int)arg_offset);
    if(bRet)
	{
       DLT_LOG(AGW_MediaPlayer,DLT_LOG_INFO,DLT_STRING(
	             "getCurrentPlayingTrack invoked successfully"));
	   return bRet;
	}else{
		vSeekToMError(AP_MPLAYER_ERROR_FUNC_NOT_SUPPORTED);
		return TRUE;
	}
	DLT_LOG(AGW_MediaPlayer, DLT_LOG_INFO, DLT_STRING("[FUNC] left"),
								   DLT_STRING(__FUNCTION__));
}

/******************************************************************************
* Function:    handle_media_next
*
* Description: Handler to skip to next song method
*
* Parameters:  MediaPlayerPlayer object,invocation pointer
*
* Return:      bool
*******************************************************************************/
gboolean handle_media_next (MediaPlayerPlayer *object,
                       GDBusMethodInvocation *invocation)
{
	DLT_LOG(AGW_MediaPlayer, DLT_LOG_INFO, DLT_STRING("[FUNC] entered"),
										   DLT_STRING(__FUNCTION__));
	g_invocation_medianext =  g_object_ref(invocation);    

	bool bRet = bMediaInterface_next();  
    if(bRet)
	{
       DLT_LOG(AGW_MediaPlayer,DLT_LOG_INFO,DLT_STRING(
	             "getCurrentPlayingTrack invoked successfully"));
	   return bRet;
	}else{
		vRequestNextMError(AP_MPLAYER_ERROR_FUNC_NOT_SUPPORTED);
		return TRUE;
	}
	DLT_LOG(AGW_MediaPlayer, DLT_LOG_INFO, DLT_STRING("[FUNC] left"),
								   DLT_STRING(__FUNCTION__));
}

/******************************************************************************
* Function:    handle_media_prev
*
* Description: Handler to skip to next song method
*
* Parameters:  MediaPlayerPlayer object,invocation pointer
*
* Return:      bool
*******************************************************************************/
gboolean handle_media_prev (MediaPlayerPlayer *object,
                       GDBusMethodInvocation *invocation)
{
	DLT_LOG(AGW_MediaPlayer, DLT_LOG_INFO, DLT_STRING("[FUNC] entered"),
										   DLT_STRING(__FUNCTION__));
	g_invocation_mediaprev =  g_object_ref(invocation);    

	bool bRet = bMediaInterface_prev();  

    if(bRet)
	{
       DLT_LOG(AGW_MediaPlayer,DLT_LOG_INFO,DLT_STRING(
	               "getCurrentPlayingTrack invoked successfully"));
	   return bRet;
	}else{
		vRequestPrevMError(AP_MPLAYER_ERROR_FUNC_NOT_SUPPORTED);
		return TRUE;
	}
	DLT_LOG(AGW_MediaPlayer, DLT_LOG_INFO, DLT_STRING("[FUNC] left"),
								   DLT_STRING(__FUNCTION__));
}

/******************************************************************************
* Function:    handle_media_get_current_playing_track
*
* Description: Handler to skip to next song method
*
* Parameters:  MediaPlayerPlayer object,invocation pointer,
*
* Return:      bool
*******************************************************************************/
gboolean handle_media_get_current_playing_track (MediaPlayerPlayer *object,
                       GDBusMethodInvocation *invocation)
{
	DLT_LOG(AGW_MediaPlayer, DLT_LOG_INFO, DLT_STRING("[FUNC] entered"),
										   DLT_STRING(__FUNCTION__));
	g_invocation_currenttrackinfo =  g_object_ref(invocation);    

//	bool bRet = bMediaInterface_getCurrentPlayingTrack();    
	return TRUE;
	DLT_LOG(AGW_MediaPlayer, DLT_LOG_INFO, DLT_STRING("[FUNC] left"),
								   DLT_STRING(__FUNCTION__));
}

/******************************************************************************
* Function:   	vRequestPlayMResult
*
* Description: Play method result call back
*
* Parameters:  void 
*
* Return:      void
******************************************************************************/
void vRequestPlayMResult()  
{
	DLT_LOG(AGW_MediaPlayer, DLT_LOG_INFO, DLT_STRING("[FUNC] entered"),
										   DLT_STRING(__FUNCTION__));
   if(g_invocation_mediaplay !=NULL)
   {
      media_player_player_complete_play(
	   (MediaPlayerPlayer*)poGetMediaPlayerProxyObj(),
		g_invocation_mediaplay);
      g_object_unref(g_invocation_mediaplay);
      g_invocation_mediaplay = NULL;

   }
   else{
      DLT_LOG(AGW_MediaPlayer,DLT_LOG_INFO,DLT_STRING("Invocation is NULL"));
   }

   DLT_LOG(AGW_MediaPlayer,DLT_LOG_INFO,DLT_STRING(
                 "The playback action invoked successfully"));
	DLT_LOG(AGW_MediaPlayer, DLT_LOG_INFO, DLT_STRING("[FUNC] left"),
								   DLT_STRING(__FUNCTION__));			 
}  

/******************************************************************************
* Function:    vRequestPauseMResult
*
* Description: Pause action method result call back
*
* Parameters:  void 
*
* Return:      void
******************************************************************************/
void vRequestPauseMResult()  
{
	DLT_LOG(AGW_MediaPlayer, DLT_LOG_INFO, DLT_STRING("[FUNC] entered"),
										   DLT_STRING(__FUNCTION__));
   if(g_invocation_mediapause !=NULL)
   {   
      media_player_player_complete_pause(
	   (MediaPlayerPlayer*)poGetMediaPlayerProxyObj(),
		g_invocation_mediapause);
      g_object_unref(g_invocation_mediapause);
      g_invocation_mediapause = NULL;
   }
   else{
      DLT_LOG(AGW_MediaPlayer,DLT_LOG_INFO,DLT_STRING("Invocation is NULL"));
   }
   DLT_LOG(AGW_MediaPlayer,DLT_LOG_INFO,DLT_STRING(
                         "The pause invoked successfully"));
	DLT_LOG(AGW_MediaPlayer, DLT_LOG_INFO, DLT_STRING("[FUNC] left"),
								   DLT_STRING(__FUNCTION__));	

}  

/******************************************************************************
* Function:     vRequestStopMResult
*
* Description: Stop action method result call back
*
* Parameters:  void 
*
* Return:      void
******************************************************************************/
void vRequestStopMResult()  
{
	DLT_LOG(AGW_MediaPlayer, DLT_LOG_INFO, DLT_STRING("[FUNC] entered"),
										   DLT_STRING(__FUNCTION__));
   if(g_invocation_mediastop !=NULL)
   {
      media_player_player_complete_stop(
	   (MediaPlayerPlayer*)poGetMediaPlayerProxyObj(),
		g_invocation_mediastop);
      g_object_unref(g_invocation_mediastop);
      g_invocation_mediastop = NULL;

   }
   else{
      DLT_LOG(AGW_MediaPlayer,DLT_LOG_INFO,DLT_STRING("Invocation is NULL"));
   }      
   DLT_LOG(AGW_MediaPlayer,DLT_LOG_INFO,DLT_STRING(
                   "The stop action invoked successfully"));
	DLT_LOG(AGW_MediaPlayer, DLT_LOG_INFO, DLT_STRING("[FUNC] left"),
								   DLT_STRING(__FUNCTION__));			   

}  

/******************************************************************************
* Function:    vRequestNextMResult
*
* Description: Next  method result call back
*
* Parameters:  void 
*
* Return:      void
******************************************************************************/
void vRequestNextMResult()  
{
   DLT_LOG(AGW_MediaPlayer, DLT_LOG_INFO, DLT_STRING("[FUNC] entered"),
										   DLT_STRING(__FUNCTION__));
   if(g_invocation_medianext !=NULL)
   {   
      media_player_player_complete_next(
	   (MediaPlayerPlayer*)poGetMediaPlayerProxyObj(),
		g_invocation_medianext);
      g_object_unref(g_invocation_medianext);
      g_invocation_medianext = NULL;
   }
   else
   {
      DLT_LOG(AGW_MediaPlayer,DLT_LOG_INFO,DLT_STRING("Invocation is NULL"));
   }
   DLT_LOG(AGW_MediaPlayer,DLT_LOG_INFO,DLT_STRING(
                       "The skip to next song invoked successfully"));
	DLT_LOG(AGW_MediaPlayer, DLT_LOG_INFO, DLT_STRING("[FUNC] left"),
								   DLT_STRING(__FUNCTION__));	

}  

/******************************************************************************
* Function:    vRequestPrevMResult
*
* Description: Prev method result call back
*
* Parameters:  void 
*
* Return:      void
******************************************************************************/
void vRequestPrevMResult()  
{
	DLT_LOG(AGW_MediaPlayer, DLT_LOG_INFO, DLT_STRING("[FUNC] entered"),
										   DLT_STRING(__FUNCTION__));
   if(g_invocation_mediaprev !=NULL)
   {
      media_player_player_complete_previous(
	   (MediaPlayerPlayer*)poGetMediaPlayerProxyObj(),
		g_invocation_mediaprev);
      g_object_unref(g_invocation_mediaprev);
      g_invocation_mediaprev = NULL;
   }
   else{
         DLT_LOG(AGW_MediaPlayer,DLT_LOG_INFO,DLT_STRING("Invocation is NULL"));
   }
   DLT_LOG(AGW_MediaPlayer,DLT_LOG_INFO,DLT_STRING(
                        "The Prev action invoked successfully"));
	DLT_LOG(AGW_MediaPlayer, DLT_LOG_INFO, DLT_STRING("[FUNC] left"),
								   DLT_STRING(__FUNCTION__));						

}  

/******************************************************************************
* Function:     vSeekToMResult
*
* Description: Pseek method result call back
*
* Parameters:  void 
*
* Return:      void
******************************************************************************/
void vSeekToMResult()
{
	DLT_LOG(AGW_MediaPlayer, DLT_LOG_INFO, DLT_STRING("[FUNC] entered"),
										   DLT_STRING(__FUNCTION__));
   if(g_invocation_mediaseek !=NULL)
   {
      media_player_player_complete_seek(
	   (MediaPlayerPlayer*)poGetMediaPlayerProxyObj(),
		g_invocation_mediaseek);
		g_object_unref(g_invocation_mediaseek);
		g_invocation_mediaseek = NULL;
   }else{
	DLT_LOG(AGW_MediaPlayer,DLT_LOG_INFO,DLT_STRING(
	                      "The seek method called successfully"));
   }
   DLT_LOG(AGW_MediaPlayer, DLT_LOG_INFO, DLT_STRING("[FUNC] left"),
								   DLT_STRING(__FUNCTION__));	
}

/******************************************************************************
* Function:    vRequestPlayMError
*
* Description: Play Method error call back
*
* Parameters:  ApMplayerError
*
* Return:      void
******************************************************************************/
void vRequestPlayMError(ApMplayerError eState)
{
	DLT_LOG(AGW_MediaPlayer, DLT_LOG_INFO, DLT_STRING("[FUNC] entered"),
										   DLT_STRING(__FUNCTION__));


    if(g_invocation_mediaplay){
    g_dbus_method_invocation_return_error (g_invocation_mediaplay,
									  AP_MPLAYER_ERROR,
									  eState,
									  "Error in Mediaplayer Play");       
    }
	DLT_LOG(AGW_MediaPlayer, DLT_LOG_INFO, DLT_STRING("[FUNC] left"),
								   DLT_STRING(__FUNCTION__));	
}

/******************************************************************************
* Function:    vRequestPauseMError
*
* Description: Pause Method error call back
*
* Parameters:  ApMplayerError
*
* Return:      void
******************************************************************************/
void vRequestPauseMError(ApMplayerError eState)
{
	DLT_LOG(AGW_MediaPlayer, DLT_LOG_INFO, DLT_STRING("[FUNC] entered"),
										   DLT_STRING(__FUNCTION__));


    if(g_invocation_mediapause)
    {
        g_dbus_method_invocation_return_error (g_invocation_mediapause,
									  AP_MPLAYER_ERROR,
									  eState,
									  "Error in Pause method");
       
    }
	DLT_LOG(AGW_MediaPlayer, DLT_LOG_INFO, DLT_STRING("[FUNC] left"),
								   DLT_STRING(__FUNCTION__));
}

/******************************************************************************
* Function:    vRequestStopMError
*
* Description: Stop Method error call back
*
* Parameters:  ApMplayerError
*
* Return:      void
******************************************************************************/
void vRequestStopMError(ApMplayerError eState)
{
	DLT_LOG(AGW_MediaPlayer, DLT_LOG_INFO, DLT_STRING("[FUNC] entered"),
										   DLT_STRING(__FUNCTION__));


    if(g_invocation_mediastop)
    {
        g_dbus_method_invocation_return_error (g_invocation_mediastop,
									  AP_MPLAYER_ERROR,
									  eState,
									  "Error in Stop method");
        
    }
	DLT_LOG(AGW_MediaPlayer, DLT_LOG_INFO, DLT_STRING("[FUNC] left"),
								   DLT_STRING(__FUNCTION__));
}

/******************************************************************************
* Function:    vRequestNextMError
*
* Description: Next Method error call back
*
* Parameters:  ApMplayerError
*
* Return:      void
******************************************************************************/
void vRequestNextMError(ApMplayerError eState)
{
	DLT_LOG(AGW_MediaPlayer, DLT_LOG_INFO, DLT_STRING("[FUNC] entered"),
										   DLT_STRING(__FUNCTION__));
    if(g_invocation_medianext)
    {
        g_dbus_method_invocation_return_error (g_invocation_medianext,
									  AP_MPLAYER_ERROR,
									  eState,
									  "Error in skip to next song");
        
    }
	DLT_LOG(AGW_MediaPlayer, DLT_LOG_INFO, DLT_STRING("[FUNC] left"),
								   DLT_STRING(__FUNCTION__));
}

/******************************************************************************
* Function:    vRequestPrevMError
*
* Description: Prev Method error call back
*
* Parameters:  ApMplayerError
*
* Return:      void
******************************************************************************/
void vRequestPrevMError(ApMplayerError eState)
{
	DLT_LOG(AGW_MediaPlayer, DLT_LOG_INFO, DLT_STRING("[FUNC] entered"),
										   DLT_STRING(__FUNCTION__));

    if(g_invocation_mediaprev)
    {
        g_dbus_method_invocation_return_error (g_invocation_mediaprev,
									  AP_MPLAYER_ERROR,
									  eState,
									  "Error in skip to Prev song");
       
    }
	DLT_LOG(AGW_MediaPlayer, DLT_LOG_INFO, DLT_STRING("[FUNC] left"),
								   DLT_STRING(__FUNCTION__));
}

/******************************************************************************
* Function:    vSeekToMError
*
* Description: Method error call back
*
* Parameters:  ApMplayerError 
*
* Return:      void
******************************************************************************/
void vSeekToMError(ApMplayerError eState)
{
	DLT_LOG(AGW_MediaPlayer, DLT_LOG_INFO, DLT_STRING("[FUNC] entered"),
										   DLT_STRING(__FUNCTION__));
    if(g_invocation_mediaseek)
    {
        g_dbus_method_invocation_return_error (g_invocation_mediaseek,
									  AP_MPLAYER_ERROR,
									  eState,
									  "Error in Seek method");
		
    }
	DLT_LOG(AGW_MediaPlayer, DLT_LOG_INFO, DLT_STRING("[FUNC] left"),
								   DLT_STRING(__FUNCTION__));

}

/******************************************************************************
* Function:    vPlaytimeOpMError
*
* Description: Method error call back
*
* Parameters:  ApMplayerError
*
* Return:      void
******************************************************************************/
void vPlaytimeOpMError(ApMplayerError eState)
{
	DLT_LOG(AGW_MediaPlayer, DLT_LOG_INFO, DLT_STRING("[FUNC] entered"),
										   DLT_STRING(__FUNCTION__));

	g_dbus_method_invocation_return_error (g_invocation_mediaseek,
									  AP_MPLAYER_ERROR,
									  eState,
									  "Error in Position property Update");
									  
	
	DLT_LOG(AGW_MediaPlayer, DLT_LOG_INFO, DLT_STRING("[FUNC] left"),
								   DLT_STRING(__FUNCTION__));
								  
}

/******************************************************************************
* Function:    vPlaybackStateOpMError
*
* Description: Method error call back
*
* Parameters:  ApMplayerError
*
* Return:      void
******************************************************************************/
void vPlaybackStateOpMError(ApMplayerError eState)
{
	DLT_LOG(AGW_MediaPlayer, DLT_LOG_INFO, DLT_STRING("[FUNC] entered"),
										   DLT_STRING(__FUNCTION__));

	g_dbus_method_invocation_return_error (
	                                 g_invocation_mediaseek,
									 AP_MPLAYER_ERROR,
									 eState,
									 "Error in Playbackstatus property Update");
	
		
	DLT_LOG(AGW_MediaPlayer, DLT_LOG_INFO, DLT_STRING("[FUNC] left"),
								   DLT_STRING(__FUNCTION__));	
}

/******************************************************************************
* Function:    vPlaybackModeOpMError
*
* Description: Method error call back
*
* Parameters:  ApMplayerError 
*
* Return:      void
******************************************************************************/
void vPlaybackModeOpMError(ApMplayerError eState)
{
	DLT_LOG(AGW_MediaPlayer, DLT_LOG_INFO, DLT_STRING("[FUNC] entered"),
										   DLT_STRING(__FUNCTION__));

	g_dbus_method_invocation_return_error (g_invocation_mediaseek,
									  AP_MPLAYER_ERROR,
									  eState,
									  "Error in Shuffle property Update");
			
	
	DLT_LOG(AGW_MediaPlayer, DLT_LOG_INFO, DLT_STRING("[FUNC] left"),
								   DLT_STRING(__FUNCTION__));						  
}

/******************************************************************************
* Function:    vRepeatModeOpMError
*
* Description: Method error call back
*
* Parameters:  ApMplayerError
*
* Return:      void
******************************************************************************/
void vRepeatModeOpMError(ApMplayerError eState)
{
	DLT_LOG(AGW_MediaPlayer, DLT_LOG_INFO, DLT_STRING("[FUNC] entered"),
										   DLT_STRING(__FUNCTION__));

	g_dbus_method_invocation_return_error (g_invocation_mediaseek,
									  AP_MPLAYER_ERROR,
									  eState,
									  "Error in Loopstatus property Update");
								  
									  
	DLT_LOG(AGW_MediaPlayer, DLT_LOG_INFO, DLT_STRING("[FUNC] left"),
								   DLT_STRING(__FUNCTION__));
}

/******************************************************************************
* Function:    vSendPlayTimeStatus
*
* Description: Sends the position of the current playing song
*
* Parameters:  unsigned int 
*
* Return:      void
******************************************************************************/
void vSendPlayTimeStatus(unsigned int currentPlaytime)
{
	DLT_LOG(AGW_MediaPlayer, DLT_LOG_INFO, DLT_STRING("[FUNC] entered"),
										   DLT_STRING(__FUNCTION__));
#if 0
    DLT_LOG(AGW_MediaPlayer,DLT_LOG_INFO,DLT_STRING(
             "MediaPlayer:sendPlayTimeStatus : "),DLT_UINT32(currentPlaytime));
#endif
	gint64 playtime = (gint64)currentPlaytime * 1000000;
   if((MediaPlayerPlayer*)poGetMediaPlayerProxyObj()!=NULL)
	media_player_player_set_position(
             (MediaPlayerPlayer*)poGetMediaPlayerProxyObj(),playtime);
	DLT_LOG(AGW_MediaPlayer, DLT_LOG_INFO, DLT_STRING("[FUNC] left"),
								   DLT_STRING(__FUNCTION__));
}

/******************************************************************************
* Function:    sendPlaybackStatus
*
* Description: Sends the playback status
*
* Parameters:  ApMPlayerState 
*
* Return:      void
******************************************************************************/
void vSendPlaybackStatus(ApMPlayerState estate)
{
	DLT_LOG(AGW_MediaPlayer, DLT_LOG_INFO, DLT_STRING("[FUNC] entered"),
										   DLT_STRING(__FUNCTION__));
	DLT_LOG(AGW_MediaPlayer,DLT_LOG_INFO,DLT_STRING(
	             "MediaPlayer:sendPlaybackStatus : "),DLT_INT(estate));
   if((MediaPlayerPlayer*)poGetMediaPlayerProxyObj()!=NULL)
	media_player_player_set_playback_status(
                (MediaPlayerPlayer*)poGetMediaPlayerProxyObj(),estate);
	DLT_LOG(AGW_MediaPlayer, DLT_LOG_INFO, DLT_STRING("[FUNC] left"),
								   DLT_STRING(__FUNCTION__));
}

/******************************************************************************
* Function:    sendShuffleStatus
*
* Description: Sends the order of the songs being played
*
* Parameters:  gboolean 
*
* Return:      void
******************************************************************************/
void vSendShuffleStatus(gboolean status)
{
	DLT_LOG(AGW_MediaPlayer, DLT_LOG_INFO, DLT_STRING("[FUNC] entered"),
										   DLT_STRING(__FUNCTION__));
   if((MediaPlayerPlayer*)poGetMediaPlayerProxyObj()!=NULL)
	media_player_player_set_shuffle(
                (MediaPlayerPlayer*)poGetMediaPlayerProxyObj(),
				status );
	DLT_LOG(AGW_MediaPlayer, DLT_LOG_INFO, DLT_STRING("[FUNC] left"),
								   DLT_STRING(__FUNCTION__));
}

/******************************************************************************
* Function:    vSendLoopStatus
*
* Description: Sends the status of the track/playlist repeated
*
* Parameters:  ApMPlayerLoopStatus 
*
* Return:      void
******************************************************************************/
void vSendLoopStatus(ApMPlayerLoopStatus estatus)
{
	DLT_LOG(AGW_MediaPlayer, DLT_LOG_INFO, DLT_STRING("[FUNC] entered"),
										   DLT_STRING(__FUNCTION__));
	DLT_LOG(AGW_MediaPlayer,DLT_LOG_INFO,DLT_STRING(
	            "MediaPlayer:sendShuffleStatus : "),DLT_INT(estatus));
    if((MediaPlayerPlayer*)poGetMediaPlayerProxyObj()!=NULL)
	media_player_player_set_loop_status(
                (MediaPlayerPlayer*)poGetMediaPlayerProxyObj(),
				estatus);
	DLT_LOG(AGW_MediaPlayer, DLT_LOG_INFO, DLT_STRING("[FUNC] left"),
								   DLT_STRING(__FUNCTION__));
}

/******************************************************************************
* Function:    vNotifyCurrentFolderPath
*
* Description: Sends the current playing track folder path to client.
*
* Parameters:  char *, int
*
* Return:      void
******************************************************************************/
void vNotifyCurrentFolderPath(char * currFolderPath, int iListHandle)
{
	DLT_LOG(AGW_MediaPlayer, DLT_LOG_INFO, DLT_STRING("[FUNC] entered"),
										   DLT_STRING(__FUNCTION__));
	DLT_LOG(AGW_MediaPlayer,DLT_LOG_INFO,DLT_STRING(
	         "MediaPlayer:vNotifyCurrentFolderPath : currentFolderPath:"
			 "  & Listhandle : %d"),DLT_STRING(currFolderPath));

   if ((currFolderPath != NULL) && (currFolderPath[0] == '\0')) {
      DLT_LOG(AGW_MediaPlayer,DLT_LOG_INFO,DLT_STRING(
	                  "currFolderPath is Empty/NULL "));
      return;
   }
	if((MediaPlayerPlayer*)poGetMediaPlayerProxyObj()!=NULL)
	{
		media_player_player_set_get_current_folder_path((
		 MediaPlayerPlayer*)poGetMediaPlayerProxyObj(),(gchar *)currFolderPath);
	}else
	{
		DLT_LOG(AGW_MediaPlayer,DLT_LOG_INFO,DLT_STRING(
		                          "MediaPlayer instance is NULL "));
      return;
	}
	DLT_LOG(AGW_MediaPlayer, DLT_LOG_INFO, DLT_STRING("[FUNC] left"),
								   DLT_STRING(__FUNCTION__));
}

/******************************************************************************
* Function:    vNotifyTrackChanged
*
* Description: Notifies client when the track is changed to new track.
*
* Parameters:  int 
*
* Return:      void
******************************************************************************/
void vNotifyTrackChanged(int iTrackId)
{
	DLT_LOG(AGW_MediaPlayer, DLT_LOG_INFO, DLT_STRING("[FUNC] entered"),
										   DLT_STRING(__FUNCTION__));
	DLT_LOG(AGW_MediaPlayer,DLT_LOG_INFO,DLT_STRING(
	          "MediaPlayer:vNotifyTrackChanged : "),DLT_INT(iTrackId));
	if((MediaPlayerPlayer*)poGetMediaPlayerProxyObj()!=NULL)
	{
		DLT_LOG(AGW_MediaPlayer,DLT_LOG_INFO,DLT_STRING(
		          "Emit media_player_player_emit_track_changed"));
		media_player_player_emit_track_changed(
		      (MediaPlayerPlayer*)poGetMediaPlayerProxyObj(),(guchar)iTrackId);
		DLT_LOG(AGW_MediaPlayer,DLT_LOG_INFO,DLT_STRING(
		       "Emitted media_player_player_emit_track_changed"));
	}else
	{
		DLT_LOG(AGW_MediaPlayer,DLT_LOG_INFO,
		          DLT_STRING("MediaPlayer instance is NULL "));
      return;
	}
	DLT_LOG(AGW_MediaPlayer, DLT_LOG_INFO, DLT_STRING("[FUNC] left"),
								   DLT_STRING(__FUNCTION__));
}

/******************************************************************************
 * Function:    vNotifyCurrentPlayingInfo
 *
 * Description: sends the current playing song info.
 *
 * Parameters:  Gvariant
 *
 * Return:      void
 ******************************************************************************/
void vNotifyCurrentPlayingInfo(GVariant *currentTrackInfo)
{
	DLT_LOG(AGW_MediaPlayer, DLT_LOG_INFO, DLT_STRING("[FUNC] entered"),
										   DLT_STRING(__FUNCTION__));
	DLT_LOG(AGW_MediaPlayer,DLT_LOG_INFO,
	               DLT_STRING("MediaPlayer:vNotifyCurrentPlayingInfo "));
	if(!currentTrackInfo)
	{
		DLT_LOG(AGW_MediaPlayer,DLT_LOG_INFO,DLT_STRING( 
		              "null result in vNotifyCurrentPlayingInfo"));
		return;
	}
	else
	{
		gchar* sTrackInfo = g_variant_print (currentTrackInfo, FALSE);
		DLT_LOG(AGW_MediaPlayer,DLT_LOG_INFO,DLT_STRING(
		        " vNotifyCurrentPlayingInfo ::"),DLT_STRING(sTrackInfo));
		g_free(sTrackInfo);		
	}

	if((g_invocation_currenttrackinfo!=NULL) && 
      ((MediaPlayerPlayer*)poGetMediaPlayerProxyObj()!=NULL))
	{
	   DLT_LOG(AGW_MediaPlayer,DLT_LOG_INFO,DLT_STRING(
       	                  "calling dbus-method currenttrackinfo."));
	   media_player_player_complete_get_current_playing_track(
	                          (MediaPlayerPlayer*)poGetMediaPlayerProxyObj(),
                              g_invocation_currenttrackinfo,currentTrackInfo);
		g_object_unref(g_invocation_currenttrackinfo);
		g_invocation_currenttrackinfo = NULL;
		
	}
	else
	{
		DLT_LOG(AGW_MediaPlayer,DLT_LOG_INFO,DLT_STRING(
            		"g_invocation_currenttrackinfo IS NULL"));
	}
           
}

/******************************************************************************
 * Function:    vCurrentPlayingTrackInfoError
 *
 * Description: sends the current playing song info.
 *
 * Parameters:  ApMplayerError
 *
 * Return:      void
 ******************************************************************************/

void vCurrentPlayingTrackInfoError(ApMplayerError eState)
{
	DLT_LOG(AGW_MediaPlayer, DLT_LOG_INFO, DLT_STRING("[FUNC] entered"),
										   DLT_STRING(__FUNCTION__));
	DLT_LOG(AGW_MediaPlayer,DLT_LOG_INFO,DLT_STRING(
	                "MediaPlayer:vCurrentPlayingTrackInfoError"));

   if(g_invocation_currenttrackinfo!=NULL)
   {
      g_dbus_method_invocation_return_error (g_invocation_currenttrackinfo,
									  AP_MPLAYER_ERROR,
									  eState,
									  "Error in CurrentPlayingTrackInfo method");
   }
   else
	{
		DLT_LOG(AGW_MediaPlayer,DLT_LOG_INFO,DLT_STRING( 
		                        "g_invocation_currenttrackinfo IS NULL"));
	}
	DLT_LOG(AGW_MediaPlayer, DLT_LOG_INFO, DLT_STRING("[FUNC] left"),
								   DLT_STRING(__FUNCTION__));
}

/*****************************************************************************
 * Function:    vExportAGWMediaPlayerInterface
 *
 * Description: function to expose media player interface
 *
 * Parameters:  void
 *
 * Return:      void
 *****************************************************************************/
void vExportAGWMediaPlayerInterface()
{
	DLT_LOG(AGW_MediaPlayer, DLT_LOG_INFO, DLT_STRING("[FUNC] entered"),
										   DLT_STRING(__FUNCTION__));
	
   GDBusConnection* poConn = poGetGDBusConnection();
   MediaPlayerPlayer* poMPalyerProxyObj = poGetMediaPlayerProxyObj();
   GError *error = NULL;
   if((NULL == poConn) || (NULL == poMPalyerProxyObj))
   {
       DLT_LOG(AGW_MediaPlayer,DLT_LOG_ERROR,DLT_STRING(
	           "Connection ptr or MPlayer proxy obj is NULL "));
       return;
   }
   if(!g_dbus_interface_skeleton_export (
                              G_DBUS_INTERFACE_SKELETON (poMPalyerProxyObj),
                              poConn,
                              "/com/bosch/AutomotiveProxy/MediaPlayer/Player",
                              &error))
   {
      if(error != NULL)
      {
         DLT_LOG(AGW_MediaPlayer,DLT_LOG_ERROR,DLT_STRING("MediaPlayerPlayer :"
               " -ERROR-- Interface skeleton export failed -"),
			   DLT_STRING(error->message));
         g_clear_error (&error);
      }
      return;
   }
   else
   {
      DLT_LOG(AGW_MediaPlayer,DLT_LOG_INFO,DLT_STRING("MediaPlayerPlayer "
                                        ": - Object created & exported"));
   }

   g_signal_connect (poMPalyerProxyObj,
                      "handle-play",
                      G_CALLBACK (handle_media_play), NULL);
   g_signal_connect (poMPalyerProxyObj,
                      "handle-pause",
                      G_CALLBACK (handle_media_pause), NULL);
   g_signal_connect (poMPalyerProxyObj,
                      "handle-stop",
                      G_CALLBACK (handle_media_stop), NULL);
   g_signal_connect (poMPalyerProxyObj,
                      "handle-seek",
                      G_CALLBACK (handle_media_seek), NULL);
   g_signal_connect (poMPalyerProxyObj,
                      "handle-next",
                      G_CALLBACK (handle_media_next), NULL);
   g_signal_connect (poMPalyerProxyObj,
                      "handle-previous",
                      G_CALLBACK (handle_media_prev), NULL);
   g_signal_connect ( poMPalyerProxyObj,
                      "handle-get-current-playing-track",
                      G_CALLBACK (handle_media_get_current_playing_track), 
					  NULL);
   DLT_LOG(AGW_MediaPlayer, DLT_LOG_INFO, DLT_STRING("[FUNC] left"),
								   DLT_STRING(__FUNCTION__));		
}

/******************************************************************************
 * Function:    vUnexportAGWMediaPlayerInterface
 *
 * Description: function to unexport media player interface
 *
 * Parameters:  void
 *
 * Return:      void
 ******************************************************************************/
void vUnexportAGWMediaPlayerInterface()
{
   DLT_LOG(AGW_MediaPlayer, DLT_LOG_INFO, DLT_STRING("[FUNC] entered"),
										   DLT_STRING(__FUNCTION__));
    MediaPlayerPlayer* poMPalyerProxyObj = poGetMediaPlayerProxyObj();
    if(poMPalyerProxyObj)
    {
        DLT_LOG(AGW_MediaPlayer,DLT_LOG_INFO,DLT_STRING("MediaPlayerPlayer "
                                                     ":unexporting interface"));
        g_dbus_interface_skeleton_unexport(G_DBUS_INTERFACE_SKELETON
														   (poMPalyerProxyObj));
    }

   DLT_LOG(AGW_MediaPlayer, DLT_LOG_INFO, DLT_STRING("[FUNC] left"),
								   DLT_STRING(__FUNCTION__));		
}
#endif//VARIANT_S_FTR_ENABLE_FEAT_GW_PSARCC