/******************************************************************************
*
* FILE:          ECMDBusHandler.c
*
* PROJECT:       FC_GENERIC_GATEWAY
*
* DESCRIPTION:   DBUS service handler for ECM.
*
* AUTHOR:        IPD5KOR(RBEI/ECO2)
*
* COPYRIGHT:     (c) 2017 Robert Bosch GmbH, Hildesheim
*
******************************************************************************/

#include <gio/gio.h>
#include <dlfcn.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include "ECMDbusHandler.h"
#include "ECM_ClientInterface.h"
#include "ap-ecm-enums.h"
#include "dlt/dlt.h"
#include "AutomDBusServer.h"

DLT_DECLARE_CONTEXT(AGW_ECM);
static CertificateManager* pCertManagerProxyObj = NULL;

/******************************************************************************
* Function:    handle_push_certificate
* Description: used to push the certificate
* Parameters:  proxy object, invocation pointer, certBuffer, certSize , user data
* Return:      gboolean
*****************************************************************************/
gboolean 
handle_push_certificate (CertificateManager *object,
                         GDBusMethodInvocation *invocation,
                         GVariant* certBuffer,
                         guint32   certSize,
                         gpointer user_data)
{
    DLT_LOG(AGW_ECM, DLT_LOG_INFO, DLT_STRING("+"),DLT_STRING(__FUNCTION__)); 
    user_data = invocation; 
    if(certBuffer == NULL)
    {
         g_dbus_method_invocation_return_error( invocation,
                                                AP_ECM_ERROR,
                                                AP_ECM_ERROR_UNKNOWN,
                                               "Unknown Error" );
         return TRUE;
    }
     
    if(!bPushCertificate(certBuffer, certSize, user_data))
    {
        DLT_LOG(AGW_ECM, DLT_LOG_ERROR, DLT_STRING(__FUNCTION__), 
		      DLT_STRING("(): bPushCertificate returned bad result"));
	    g_dbus_method_invocation_return_error( invocation,
                                               AP_ECM_ERROR,
                                               AP_ECM_ERROR_UNKNOWN,
                                               "Unknown Error" );
        return TRUE;
    }
    DLT_LOG(AGW_ECM, DLT_LOG_INFO, DLT_STRING("-"),DLT_STRING(__FUNCTION__));
    return TRUE;
}

/******************************************************************************
* Function:    send_push_cert_result
* Description: This function handles the result of 'handle_push_certificate'
* Parameters:  result->0 indicates success & 1 indicates failure
*              error, errorDesc, user_data
* Return:      void
*****************************************************************************/
void
send_push_cert_result(guint16 result,
                      gboolean error,
                      gchar* errorDesc,
                      gpointer user_data)
{
	DLT_LOG(AGW_ECM, DLT_LOG_INFO, DLT_STRING("+"),DLT_STRING(__FUNCTION__));
        if (user_data == NULL)
        {
           DLT_LOG(AGW_ECM, DLT_LOG_ERROR,DLT_STRING(__FUNCTION__),
  		        DLT_STRING("(): g_invocation_push_cert is NULL"));
        }
        else
        {
           if(error)
           {
              g_dbus_method_invocation_return_error( 
			                (GDBusMethodInvocation*)user_data,
                            AP_ECM_ERROR,
				            AP_ECM_ERROR_UNKNOWN,
                            "Unknown Error" );         
           }
           else
           {
              DLT_LOG(AGW_ECM, DLT_LOG_INFO,DLT_STRING(__FUNCTION__),
  			        DLT_STRING("():result is"), DLT_UINT16(result));
              if(pCertManagerProxyObj != NULL)
              {
                certificate_manager_complete_push_certificate(
                                     pCertManagerProxyObj,
                                     (GDBusMethodInvocation*)user_data,																
                                     result,
 							         errorDesc?errorDesc:"");
              }
              else{
                  DLT_LOG(AGW_ECM, DLT_LOG_INFO,DLT_STRING(__FUNCTION__),
                        DLT_STRING("pCertManagerProxyObj is null,"
                                   " push cert result not sent"));
                  g_dbus_method_invocation_return_error(
                                (GDBusMethodInvocation*)user_data,
                                AP_ECM_ERROR,
                                AP_ECM_ERROR_UNKNOWN,
                                "Unknown Error" );
              }
           }
        }
        if(errorDesc) free(errorDesc);
		DLT_LOG(AGW_ECM, DLT_LOG_INFO,DLT_STRING("-"),DLT_STRING(__FUNCTION__));
}


/******************************************************************************
* Function:    handle_push_crl
* Description: used to push the revocation list into header unit
* Parameters:  proxy object, invocation pointer, crl, crlSize, user data
* Return:      gboolean
*****************************************************************************/
gboolean
handle_push_crl ( CertificateManager    *object,
                  GDBusMethodInvocation *invocation,
                  GVariant *crl,
                  guint32 crlSize,
                  gpointer user_data )
{
	DLT_LOG(AGW_ECM, DLT_LOG_INFO, DLT_STRING("+"),DLT_STRING(__FUNCTION__));
     user_data = invocation; 
     if(crl == NULL)
     {
          g_dbus_method_invocation_return_error( invocation,
                                                AP_ECM_ERROR,
                                                AP_ECM_ERROR_UNKNOWN,
                                               "Unknown Error" );
          return TRUE;
     }

     if(!bPushCrl(crl, crlSize, user_data))
     {
        DLT_LOG(AGW_ECM, DLT_LOG_ERROR,DLT_STRING(__FUNCTION__),DLT_STRING("():"
		   "bPushCrl returned bad result"));
        g_dbus_method_invocation_return_error( invocation,
                                               AP_ECM_ERROR,
                                               AP_ECM_ERROR_UNKNOWN,
                                               "Unknown Error" );
        return TRUE;
     }
	 DLT_LOG(AGW_ECM, DLT_LOG_INFO, DLT_STRING("-"),DLT_STRING(__FUNCTION__));
     return TRUE;
}

/******************************************************************************
* Function:    send_push_crl_result
* Description: Handler to send the response for 'handle_push_crl_result'
* Parameters:  result - 0 indicates success and 1 indicates failure
*              error, errorDesc, user_data
* Return:      void
*****************************************************************************/
void
send_push_crl_result(guint16 result, 
                     gboolean error,
                     gchar* errorDesc,
                     gpointer user_data)
{
	DLT_LOG(AGW_ECM, DLT_LOG_INFO, DLT_STRING("+"),DLT_STRING(__FUNCTION__));
        if (user_data == NULL)
        {
             DLT_LOG(AGW_ECM, DLT_LOG_INFO,DLT_STRING(__FUNCTION__),
             DLT_STRING("():"
			  "g_invocation_get_configuration_data is NULL"));
        }
        else
        {
           if(error)
           {
              g_dbus_method_invocation_return_error( 
			            (GDBusMethodInvocation*)user_data,
                        AP_ECM_ERROR,
                        AP_ECM_ERROR_UNKNOWN,
                        "Unknown Error" );  
           }
           else
           {
             DLT_LOG(AGW_ECM, DLT_LOG_INFO,DLT_STRING(__FUNCTION__), 
			     DLT_STRING("(): result is"), DLT_UINT16(result));
             if(pCertManagerProxyObj != NULL)
             {
                certificate_manager_complete_push_crl(
                                  pCertManagerProxyObj,
                                  (GDBusMethodInvocation*)user_data,
					              result,
						          errorDesc?errorDesc:"");
             }
             else{
                 DLT_LOG(AGW_ECM, DLT_LOG_INFO,DLT_STRING(__FUNCTION__),
                       DLT_STRING("pCertManagerProxyObj is null,"
                                  " push crl result not sent"));
                 g_dbus_method_invocation_return_error(
                           (GDBusMethodInvocation*)user_data,
                           AP_ECM_ERROR,
                           AP_ECM_ERROR_UNKNOWN,
                           "Unknown Error" );
             }
           }
        }
        if(errorDesc) free(errorDesc);
		DLT_LOG(AGW_ECM, DLT_LOG_INFO,DLT_STRING("-"),DLT_STRING(__FUNCTION__));
}

/******************************************************************************
* Function:    handle_get_crlvalidity
* Description: Gives the validity state of a CRL.
* Parameters:  proxy object, invocation pointer, crl_type, user data
* Return:      gboolean
*****************************************************************************/
gboolean
handle_get_crlvalidity (CertificateManager *object,
                        GDBusMethodInvocation *invocation,
                        guchar crl_type,
                        gpointer user_data)
{
    DLT_LOG(AGW_ECM, DLT_LOG_INFO, DLT_STRING("+"),DLT_STRING(__FUNCTION__));
    gchar crl_identifier[10] =  { '\0' };

    user_data = invocation;
	
    if(crl_type == AP_ECM_CRL_ROOT)
       strncpy(crl_identifier, "ROOT", 4 );
    else if(crl_type == AP_ECM_CRL_IC)
       strncpy(crl_identifier, "IC", 2);
    else 
	{
        g_dbus_method_invocation_return_error( invocation,
                                              AP_ECM_ERROR,
                                              AP_ECM_ERROR_UNKNOWN,
                                              "Invalid input" );
        return TRUE;
    }  
     
     
    if(!bGetCrlNextUpd(crl_identifier, user_data)) 
    {
            g_dbus_method_invocation_return_error(   invocation,
                                                     AP_ECM_ERROR,
                                                     AP_ECM_ERROR_UNKNOWN,
                                                     "Unknown Error" );
	    return TRUE;
    } 
    DLT_LOG(AGW_ECM, DLT_LOG_INFO, DLT_STRING("-"),DLT_STRING(__FUNCTION__));	
	return TRUE;
}

/******************************************************************************
* Function:    send_get_crl_validity_result
* Description: Sends the result of handle_get_crlvalidity
* Parameters:  result - 0 indicates success and 1 indicates failure,
*              validity, error, errorDesc, user_data
* Return:      void
*****************************************************************************/
void
send_get_crl_validity_result( guint16 result,
                              guint64 validity,
                              gboolean error,
                              gchar* errorDesc,
                              gpointer user_data)
{
    if(user_data == NULL)
       DLT_LOG(AGW_ECM, DLT_LOG_ERROR, DLT_STRING(__FUNCTION__),
          DLT_STRING("(): g_invocation_get_crl_validity is NULL"));
    else
    {
           if(error)
           {              
              g_dbus_method_invocation_return_error(
      			                (GDBusMethodInvocation*)user_data,
                                AP_ECM_ERROR,
				                AP_ECM_ERROR_UNKNOWN,
                                "Unknown Error" );
              
           }
           else
           {    
               certificate_manager_complete_get_crlvalidity(
                           			   poGetCertManagerProxyObj(),
                                       (GDBusMethodInvocation*)user_data,
                                       validity,
                                       result,
  								       errorDesc ? errorDesc: "");
           }
    } 
    if(errorDesc) free(errorDesc);
	DLT_LOG(AGW_ECM, DLT_LOG_INFO, DLT_STRING("-"),DLT_STRING(__FUNCTION__));
}

/******************************************************************************
* Function:    handle_get_device_cert
* Description: This method provides certificate
               for the device specific key in PEM format
* Parameters:  proxy object, invocation pointer, certBuffer,
*              certSize, user data
* Return:      gboolean
*****************************************************************************/

gboolean
handle_get_device_cert ( CertificateManager *object,
                         GDBusMethodInvocation *invocation,
                         GVariant* certBuffer,
                         guint32   certSize,
                         gpointer user_data)
{
    DLT_LOG(AGW_ECM, DLT_LOG_INFO, DLT_STRING("+"),DLT_STRING(__FUNCTION__));
	return TRUE;
}

/******************************************************************************
* Function:    handle_check_cert
* Description: This method takes a DER encoded certificate and 
               check if it`s valid and if  the chain is valid 
               up to the root certifcate.
* Parameters:  proxy object, invocation pointer, certBuffer,
*              certSize , user data
* Return:      gboolean
*****************************************************************************/

gboolean
handle_check_cert (CertificateManager *object,
                   GDBusMethodInvocation *invocation,
                   GVariant* certBuffer,
                   guint32   certSize,
                   gpointer user_data)
{
    DLT_LOG(AGW_ECM, DLT_LOG_INFO, DLT_STRING("+"),DLT_STRING(__FUNCTION__));
    user_data = invocation;
    if(certBuffer == NULL)
    {
          g_dbus_method_invocation_return_error( invocation,
                                                AP_ECM_ERROR,
                                                AP_ECM_ERROR_UNKNOWN,
                                               "Unknown Error" );
          return TRUE;
    }


    if(!bCheckCertificate(certBuffer, certSize, user_data))
    {
       DLT_LOG(AGW_ECM, DLT_LOG_ERROR,DLT_STRING(__FUNCTION__),
           DLT_STRING("():Error in checkCert"));
       g_dbus_method_invocation_return_error(  invocation,
                                               AP_ECM_ERROR,
                                               AP_ECM_ERROR_UNKNOWN,
                                               "Unknown Error" );

       return TRUE;
    }
	DLT_LOG(AGW_ECM, DLT_LOG_INFO, DLT_STRING("-"),DLT_STRING(__FUNCTION__));
    return TRUE; 
}

/******************************************************************************
* Function:    send_check_cert_result
* Description: Sends the result of handle_check_cert
* Parameters:  certValidity, error, errorDesc, user_data
* Return:      void
*****************************************************************************/
void
send_check_cert_result ( guint16  certValidity,
                         gboolean error,
                         gchar* errorDesc,
                         gpointer user_data)
{
    DLT_LOG(AGW_ECM, DLT_LOG_INFO, DLT_STRING("+"),DLT_STRING(__FUNCTION__));

    if((user_data != NULL) && (poGetCertManagerProxyObj() != NULL))
    {
           if(error)
           {     
               g_dbus_method_invocation_return_error(
          			            (GDBusMethodInvocation*)user_data,
                                AP_ECM_ERROR,
				                AP_ECM_ERROR_UNKNOWN,
                               "Unknown Error");
           }
           else
           {						   
              certificate_manager_complete_check_cert(
          			                     poGetCertManagerProxyObj(),
                                         (GDBusMethodInvocation*)user_data, 
                                         certValidity,
	                                     errorDesc ? errorDesc: "");
           }
    }
    else 
    {
       DLT_LOG(AGW_ECM, DLT_LOG_ERROR,DLT_STRING(__FUNCTION__),
	       DLT_STRING("(): NULL Invocation pointer"));
    } 
    if(errorDesc) free(errorDesc);
	DLT_LOG(AGW_ECM, DLT_LOG_INFO, DLT_STRING("-"),DLT_STRING(__FUNCTION__));
}

/******************************************************************************
* Function:    poGetCertManagerProxyObj
* Description: Creates and returns CertificateManager proxy object
* Parameters:  void
* Return:      CertificateManager* 
*****************************************************************************/
CertificateManager* poGetCertManagerProxyObj()
{

   if(pCertManagerProxyObj == NULL)
       pCertManagerProxyObj = certificate_manager_skeleton_new();
    return pCertManagerProxyObj;
} 


/******************************************************************************
 * Function:    isValidECMError
 * Description: This method checks whether error code is part
                of ApEcmError
 * Parameters:  errorCode
 * Returns:     gboolean
 ******************************************************************************/
 gboolean isValidECMError(gint errorCode)
 {
    switch(errorCode)
    {
      case AP_ECM_NO_ERROR:
      case AP_ECM_CERT_SUCCESS:
      case AP_ECM_CERT_PUSH_SUCCESS:
      case AP_ECM_CRL_PUSH_SUCCESS:
      case AP_ECM_SECURE_TIME_UNAVAILABLE:
      case AP_ECM_NOT_CERTIFICATE:
      case AP_ECM_NOT_CRL:
      case AP_ECM_CERT_NOT_PRESENT:
      case AP_ECM_CERT_NOT_VALID_YET:
      case AP_ECM_CERT_EXPIRED:
      case AP_ECM_CERT_INVALID:
      case AP_ECM_CERT_REVOKED:
      case AP_ECM_CRL_NOT_VALID_YET:
      case AP_ECM_CRL_EXPIRED:
      case AP_ECM_CRL_INVALID:

                       return TRUE;
      default:
                       return FALSE;
    }
 } 


/******************************************************************************
* Function:    bInitializeECM
* Description: This method intializes the ECM gateway component
* Parameters:  void
* Return:      gboolean
******************************************************************************/
gboolean bInitializeECM()
{
   DLT_REGISTER_CONTEXT(AGW_ECM,"GWECM","Gateway framework "
                         "ECM context for DLT Logging");
						 
   DLT_LOG(AGW_ECM, DLT_LOG_INFO, DLT_STRING("+"),DLT_STRING(__FUNCTION__));
   
   if (poGetCertManagerProxyObj() == NULL)
   {
	   DLT_LOG(AGW_ECM, DLT_LOG_ERROR, DLT_STRING(
	        "CertManager proxy object is NULL, failed to initializeECM"));
	   return FALSE;
   }

   if(ecm_service_watch()) 
   {
      DLT_LOG(AGW_ECM, DLT_LOG_INFO,DLT_STRING(__FUNCTION__), DLT_STRING(
	      "(): ecm service watch success"));
      return TRUE;
   }
   else
   {
      DLT_LOG(AGW_ECM, DLT_LOG_ERROR,DLT_STRING(__FUNCTION__), DLT_STRING(
	         "():ecm service watch failure"));
      return FALSE;
   }
   DLT_LOG(AGW_ECM, DLT_LOG_INFO, DLT_STRING("-"),DLT_STRING(__FUNCTION__));
}

/*******************************************************************************
 * Function:     vExportAGWECMInterface
 * Description:  This method exposes the gateway ECM methods
 * Parameters:   void
 * Return:       void
 ******************************************************************************/
void vExportAGWECMInterface()
{ 
    DLT_LOG(AGW_ECM, DLT_LOG_INFO, DLT_STRING("+"),DLT_STRING(__FUNCTION__));

    if (poGetCertManagerProxyObj() == NULL)
    {
		DLT_LOG(AGW_ECM, DLT_LOG_ERROR, DLT_STRING(__FUNCTION__), DLT_STRING(
		    "():CertManager proxy object is NULL, failed to exposeECM"));
		return;
    }

    GError *error = NULL;
    GDBusConnection* connection = NULL;

    if((connection = poGetGDBusConnection()) == NULL)
    {
        DLT_LOG(AGW_ECM, DLT_LOG_ERROR, DLT_STRING(__FUNCTION__),
            DLT_STRING("(): poGetGDBusConnection returned NULL"));
        return;
    }

    if (!g_dbus_interface_skeleton_export(
	                  G_DBUS_INTERFACE_SKELETON(poGetCertManagerProxyObj()),
                          connection,
                          "/com/bosch/AutomotiveProxy/CertificateManager",
                          &error) )
    {
        if (error)
        {
            DLT_LOG(AGW_ECM, DLT_LOG_ERROR, DLT_STRING(__FUNCTION__), 
			    DLT_STRING("():Certificate interface skeleton export error: "),
 				DLT_STRING(error->message));

            g_clear_error(&error);
        }
        else DLT_LOG(AGW_ECM, DLT_LOG_ERROR, DLT_STRING(__FUNCTION__), 
		    DLT_STRING("():Certificate interface skeleton export error"));
        return;
    }

    else 
    {
        DLT_LOG(AGW_ECM, DLT_LOG_INFO, DLT_STRING(__FUNCTION__), DLT_STRING(
                 "():Certificate interface skeleton exported successfully"));
    }

    g_signal_connect (poGetCertManagerProxyObj(), "handle-push-certificate",
                     G_CALLBACK (handle_push_certificate), NULL);

    g_signal_connect (poGetCertManagerProxyObj(), "handle-push-crl",
                     G_CALLBACK (handle_push_crl), NULL);

    g_signal_connect (poGetCertManagerProxyObj(), "handle-get-crlvalidity",
                      G_CALLBACK (handle_get_crlvalidity), NULL);

    g_signal_connect (poGetCertManagerProxyObj(), "handle-check-cert",
                     G_CALLBACK (handle_check_cert), NULL);

   DLT_LOG(AGW_ECM, DLT_LOG_INFO, DLT_STRING("-"),DLT_STRING(__FUNCTION__));
}

/******************************************************************************
* Function:    uninitialize
* Description: This method unintializes the ECM gateway component
* Parameters:  void
* Return:      void
*****************************************************************************/
void uninitializeECM()
{
  DLT_LOG(AGW_ECM, DLT_LOG_INFO, DLT_STRING("+"),DLT_STRING(__FUNCTION__));

  ecm_service_unwatch();
  
  DLT_LOG(AGW_ECM, DLT_LOG_INFO, DLT_STRING("-"),DLT_STRING(__FUNCTION__));
  DLT_UNREGISTER_CONTEXT(AGW_ECM);
}    

/*******************************************************************************
  * Function:    vUnexportAGWECMInterface
  * Description: unexports ECM interface
  * Parameters:  void
  * Return:      void
*******************************************************************************/
void vUnexportAGWECMInterface()
{
     DLT_LOG(AGW_ECM, DLT_LOG_INFO, DLT_STRING("+"),
                  DLT_STRING(__FUNCTION__));
	gboolean bExported = FALSE;
	
     if(pCertManagerProxyObj != NULL)
     {
		 bExported = g_dbus_interface_skeleton_has_connection(G_DBUS_INTERFACE_SKELETON(pCertManagerProxyObj),poGetSessionBusConnection());
		 if(bExported == TRUE)
		 { 
         g_dbus_interface_skeleton_unexport(
                           G_DBUS_INTERFACE_SKELETON(pCertManagerProxyObj));
         }
		 g_clear_object(&pCertManagerProxyObj);
     }
     DLT_LOG(AGW_ECM,DLT_LOG_INFO, DLT_STRING("-"),
                DLT_STRING(__FUNCTION__));
}

