/***********************************************************************
*FILE:         ConfigMgmtDBusHandler.c
*SW-COMPONENT: Internet
*DESCRIPTION:  This file provides the method definitions for 
*              the config management interface
*COPYRIGHT: Ã© 2017 Robert Bosch GmbH
*
*The reproduction, distribution and utilization of this file as
*well as the communication of its contents to others without express
*authorization is prohibited. Offenders will be held liable for the
*payment of damages. All rights reserved in the event of the grant
*of a patent, utility model or design.
***********************************************************************/

/* Header Inclusions */
#include <glib.h>
#include <gio/gio.h>
#include "dlt/dlt.h"
#include "ConfigMgmtDBusHandler.h"
#include "ap-cm-enums.h"
#include "AutomDBusServer.h"

/* Global Declarations */

DLT_DECLARE_CONTEXT(AGW_CONFIGMGR);

#define AGW_CONFIG_MANAGEMENT_OBJECT_PATH "/com/bosch/AutomotiveProxy/ConfigurationManagement"

static ConfigurationManagement* poConfigMgmtProxy = NULL;

 /* Function  Definitions */

 /******************************************************************************
 * Function:      poGetConfigMgmtProxy
 * Description:   Gets the config management proxy object.
 * Parameters:    void
 * Return:        ConfigurationManagement*
 *****************************************************************************/
 ConfigurationManagement*  poGetConfigMgmtProxy()
 {
    if(poConfigMgmtProxy == NULL)
    {
       poConfigMgmtProxy = configuration_management_skeleton_new();
    }
    return poConfigMgmtProxy;
 }


 /******************************************************************************
 * Function:    vExportConfigMgmtInterface
 * Description: Exposes the config management
                interface of gateway
 * Parameters:  GDBusConnection* 
 * Returns:     void
 *****************************************************************************/
 void vExportConfigMgmtInterface(GDBusConnection* pConnection)
 {
     DLT_LOG(AGW_CONFIGMGR, DLT_LOG_INFO,DLT_STRING("+"),
                         DLT_STRING(__FUNCTION__));
     GError *error = NULL;
     GDBusConnection* connection = NULL;

     if((connection = poGetGDBusConnection()) == NULL)
     {
          DLT_LOG(AGW_CONFIGMGR,DLT_LOG_ERROR,DLT_STRING(__FUNCTION__),
              DLT_STRING("(): poGetGDBusConnection returned NULL"));
          return;
     }

     if (poGetConfigMgmtProxy() == NULL)   
     {
         DLT_LOG(AGW_CONFIGMGR,DLT_LOG_ERROR,DLT_STRING(__FUNCTION__),
              DLT_STRING("(): null config mgmt proxy"));
         return;
     }
     if (!g_dbus_interface_skeleton_export (
                             G_DBUS_INTERFACE_SKELETON (
                             (ConfigurationManagement*) poGetConfigMgmtProxy()),
                             connection,
                             AGW_CONFIG_MANAGEMENT_OBJECT_PATH,
                             &error) )
     {
         if(error != NULL)
         {
             DLT_LOG(AGW_CONFIGMGR, DLT_LOG_ERROR, DLT_STRING(__FUNCTION__),
               DLT_STRING("Skeleton export error:"),DLT_STRING(error->message));
             g_clear_error (&error);
         }
     }
     else
     {
         g_signal_connect ( poGetConfigMgmtProxy(),
                    		"handle-register-public-config-item",
                            G_CALLBACK (handle_register_public_item),
							NULL );

         g_signal_connect ( poGetConfigMgmtProxy(),
                		    "handle-unregister-public-config-item",
                            G_CALLBACK (handle_unregister_public_item),
							NULL );

         g_signal_connect ( poGetConfigMgmtProxy(),
 		                    "handle-register-private-config-item",
                            G_CALLBACK (handle_register_private_item),
							NULL );

         g_signal_connect ( poGetConfigMgmtProxy(),
 		                    "handle-unregister-private-config-item",
                            G_CALLBACK (handle_unregister_private_item),
							NULL );

         g_signal_connect ( poGetConfigMgmtProxy(),
   		                    "handle-set-public-item",
                            G_CALLBACK (handle_set_public_item),
							NULL ); 

         g_signal_connect ( poGetConfigMgmtProxy(),
   		                    "handle-get-public-item",
                            G_CALLBACK (handle_get_public_item),
							NULL );

     }
     DLT_LOG(AGW_CONFIGMGR, DLT_LOG_INFO,DLT_STRING("-"),
                           DLT_STRING(__FUNCTION__));
 }

 /******************************************************************************
 * Function:    vUnExportConfigMgmtInterface
 * Description: unexports the config management
                interface of gateway
 * Parameters:  None
 * Returns:     void
 *****************************************************************************/
 void vUnExportConfigMgmtInterface()
 {
      DLT_LOG(AGW_CONFIGMGR, DLT_LOG_INFO,DLT_STRING("+"),
                       DLT_STRING(__FUNCTION__));

      if(poConfigMgmtProxy)
      {
           g_dbus_interface_skeleton_unexport(
               G_DBUS_INTERFACE_SKELETON(poConfigMgmtProxy));           
 
           g_clear_object(&poConfigMgmtProxy);
      }

      DLT_LOG(AGW_CONFIGMGR, DLT_LOG_INFO,DLT_STRING("-"),
                       DLT_STRING(__FUNCTION__));
 }
            

 /******************************************************************************
 * Function:    bHandleRegisterConfigRequest
 * Description: Handles the Registration request for config items
 * Parameters:  proxyObj, invocation, itemName, 
                dbusName, format, user_data, ConfigItemType
 * Returns:     gboolean
 ******************************************************************************/
 gboolean bHandleRegisterConfigRequest( ConfigurationManagement* proxyObj,
                                        GDBusMethodInvocation* invocation,
                                        gchar*         itemName,
                                        gchar*         dbusName,
                                        guint          format,
                                        gpointer       user_data,
                                        ConfigItemType itemType )
 {
     DLT_LOG(AGW_CONFIGMGR, DLT_LOG_INFO,DLT_STRING("+"),
                      DLT_STRING(__FUNCTION__));

     user_data = g_object_ref(invocation);
     if((!itemName) || (!dbusName))
     {
         DLT_LOG(AGW_CONFIGMGR,DLT_LOG_INFO,DLT_STRING(__FUNCTION__),
            DLT_STRING("(): Null inputs"));
         configuration_management_complete_register_public_config_item (
                                                                proxyObj,
                                                              invocation,
                                    AP_CONFIG_REG_STATUS_INVALID_INPUT );
         return TRUE;
     }

     DLT_LOG(AGW_CONFIGMGR,DLT_LOG_INFO,DLT_STRING(
	      "Request to register config item: "), DLT_STRING(itemName),
		  DLT_STRING(" from "), DLT_STRING(dbusName), DLT_STRING(
                                    " format: "), DLT_UINT(format) ); 
     vRegisterConfigItem( itemName, dbusName, format, itemType, user_data );
     g_object_unref(user_data);
     return TRUE;
 }

 /*****************************************************************************
 * Function:    vHandleRegisterConfigResult
 * Description: Sends the result of Config Registration
 * Parameters:  status, itemType, user_data
 * Returns:     void
 *****************************************************************************/
 void vHandleRegisterConfigResult ( guint          status,
                                    ConfigItemType type,
                                    gpointer       user_data )
 {
     DLT_LOG(AGW_CONFIGMGR,DLT_LOG_INFO,DLT_STRING("+"),
                                 DLT_STRING(__FUNCTION__));
     if(!user_data)
     {
         DLT_LOG(AGW_CONFIGMGR,DLT_LOG_ERROR,DLT_STRING(__FUNCTION__),
                               DLT_STRING("(): Null invocation"));
         return;
     }

     if( status > AP_CONFIG_REG_STATUS_INTERNAL_ERROR )
     {
         DLT_LOG(AGW_CONFIGMGR,DLT_LOG_ERROR,DLT_STRING(__FUNCTION__),
                DLT_STRING("(): Status is not a valid ApConfigRegStatus"));
         status = AP_CONFIG_REG_STATUS_INTERNAL_ERROR;
     }

     if(type == CONFIG_PUBLIC_ITEM)
     { 
          configuration_management_complete_register_public_config_item ( 
                                                   poGetConfigMgmtProxy(),
                                        (GDBusMethodInvocation*)user_data,
                                            (ApConfigRegStatus)  status );
     }
     else
     {
          configuration_management_complete_register_private_config_item (
                                                    poGetConfigMgmtProxy(),
                                         (GDBusMethodInvocation*)user_data,
                                             (ApConfigRegStatus)  status );
     }

     DLT_LOG(AGW_CONFIGMGR,DLT_LOG_INFO,DLT_STRING("-"),
                               DLT_STRING(__FUNCTION__));
 }    
   
 /**************************************************************************
 * Function:    handle_register_public_item
 * Description: Registers for the requested
                public config item
 * Parameters:  proxyObj, invocation, itemName,
                format, dbusName, user_data
 * Returns:     gboolean
 ***************************************************************************/
 gboolean handle_register_public_item(  ConfigurationManagement* proxyObj,
                                        GDBusMethodInvocation* invocation,
                                        gchar*   itemName,
                                        guint    format,
                                        gchar*   dbusName,
                                        gpointer user_data )
 {
     DLT_LOG(AGW_CONFIGMGR,DLT_LOG_INFO,DLT_STRING("+"),
                                 DLT_STRING(__FUNCTION__));

     return bHandleRegisterConfigRequest( proxyObj,
                                   invocation,
                                   itemName,
                                   dbusName,
                                   format,
                                   user_data,
                                   CONFIG_PUBLIC_ITEM );

 }

 /**************************************************************************
 * Function:    handle_register_private_item
 * Description: Registers for the requested
                private config item
 * Parameters:  proxyObj, invocation, itemName,
                dbusName,format, user_data
 * Returns:     gboolean
 ***************************************************************************/
 gboolean handle_register_private_item( ConfigurationManagement* proxyObj,
                                        GDBusMethodInvocation* invocation,
                                        gchar*   itemName,
                                        gchar*   dbusName,
                                        guint    format,
                                        gpointer user_data )
 {
     DLT_LOG(AGW_CONFIGMGR,DLT_LOG_INFO,DLT_STRING("+"),
                                 DLT_STRING(__FUNCTION__));

     return bHandleRegisterConfigRequest( proxyObj,
                                          invocation,
                                          itemName,
                                          dbusName,
                                          format,
                                          user_data,
                                          CONFIG_PRIVATE_ITEM );
 }

 /****************************************************************************
 * Function:    bHandleUnregisterConfigRequest
 * Description: Handles the config unregistration request 
 * Parameters:  proxyObj, invocation, itemName, 
                dbusName, user_data, itemType
 * Returns:     gboolean
 ****************************************************************************/
 gboolean bHandleUnregisterConfigRequest( ConfigurationManagement* proxyObj,
                                          GDBusMethodInvocation* invocation,
                                          gchar*         itemName,
                                          gchar*         dbusName,
                                          gpointer       user_data,
                                          ConfigItemType itemType )
 {
     DLT_LOG(AGW_CONFIGMGR,DLT_LOG_INFO,DLT_STRING("+"),
                                 DLT_STRING(__FUNCTION__));
     user_data = invocation;

     if( (!itemName) || (!dbusName) ) 
     {
         DLT_LOG(AGW_CONFIGMGR,DLT_LOG_INFO,DLT_STRING(__FUNCTION__),
           DLT_STRING("Null input"));
         configuration_management_complete_unregister_public_config_item (
                                                                 proxyObj,
                                                               invocation,
                                     AP_CONFIG_REG_STATUS_INVALID_INPUT );
         return TRUE;
     }


     DLT_LOG(AGW_CONFIGMGR,DLT_LOG_INFO,DLT_STRING(
	     "Request to unregister config item: "), DLT_STRING(itemName),
 		 DLT_STRING(" from service: "), DLT_STRING(dbusName));

     vUnregisterConfigItem(itemName, dbusName, itemType, user_data);

     DLT_LOG(AGW_CONFIGMGR,DLT_LOG_INFO,DLT_STRING("-"),
                                 DLT_STRING(__FUNCTION__));
     return TRUE; 
 }

 /*****************************************************************************
 * Function:    vHandleUnregisterConfigResult
 * Description: Sends the result of Config unregistration
 * Parameters:  status, itemType, user_data
 * Returns:     void
 ******************************************************************************/
 void vHandleUnregisterConfigResult ( guint          status,
                                      ConfigItemType type,
                                      gpointer       user_data )
 {

     DLT_LOG(AGW_CONFIGMGR,DLT_LOG_INFO,DLT_STRING("+"),
                                 DLT_STRING(__FUNCTION__));
     if(!user_data)
     {
         DLT_LOG(AGW_CONFIGMGR,DLT_LOG_ERROR,DLT_STRING(__FUNCTION__),
                               DLT_STRING("(): Null invocation"));
         return;
     }

     if( status > AP_CONFIG_REG_STATUS_INTERNAL_ERROR )
     {
         DLT_LOG(AGW_CONFIGMGR,DLT_LOG_ERROR,DLT_STRING(__FUNCTION__),
                DLT_STRING("(): Status is not a valid ApConfigRegStatus"));
         status = AP_CONFIG_REG_STATUS_INTERNAL_ERROR;
     }

     if(type == CONFIG_PUBLIC_ITEM)
     {
          configuration_management_complete_unregister_public_config_item (
                                                   poGetConfigMgmtProxy(),
                                        (GDBusMethodInvocation*)user_data,
                                            (ApConfigRegStatus)  status );
     }
     else
     {
          configuration_management_complete_unregister_private_config_item (
                                                   poGetConfigMgmtProxy(),             
                                        (GDBusMethodInvocation*)user_data,
                                             (ApConfigRegStatus)  status );
     }

     DLT_LOG(AGW_CONFIGMGR,DLT_LOG_INFO,DLT_STRING("-"),
                               DLT_STRING(__FUNCTION__));
 }

 /***************************************************************************
 * Function:    handle_unregister_public_item
 * Description: Registers for the requested
                public config item
 * Parameters:  proxyObj, invocation, itemName,
                dbusName, user_data
 * Returns:     gboolean
 ***************************************************************************/
 gboolean handle_unregister_public_item( ConfigurationManagement* proxyObj,
                                         GDBusMethodInvocation *invocation,
                                         gchar*         itemName,
                                         gchar*         dbusName,
                                         gpointer       user_data )
 {
     DLT_LOG(AGW_CONFIGMGR,DLT_LOG_INFO,DLT_STRING("+"),
                               DLT_STRING(__FUNCTION__));

     return bHandleUnregisterConfigRequest( proxyObj,
                                            invocation,
                                            itemName,
                                            dbusName,
                                            user_data,
                                            CONFIG_PUBLIC_ITEM );
 }

 /***************************************************************************
 * Function:    handle_unregister_private_item
 * Description: Registers for the requested
                private config item
 * Parameters:  proxyObj, invocation, itemName,
                dbusName, user_data
 * Returns:     gboolean
 ***************************************************************************/
 gboolean handle_unregister_private_item( ConfigurationManagement* proxyObj,
                                          GDBusMethodInvocation *invocation,
                                          gchar*   itemName,
                                          gchar*   dbusName,
                                          gpointer user_data )
 {
     DLT_LOG(AGW_CONFIGMGR,DLT_LOG_INFO,DLT_STRING("+"),
                               DLT_STRING(__FUNCTION__));

     return bHandleUnregisterConfigRequest( proxyObj,
                                     invocation,
                                     itemName,
                                     dbusName,
                                     user_data,
                                     CONFIG_PRIVATE_ITEM );     
 }

 /*********************************************************
 * Function:    handle_set_public_item
 * Description: Sets the public config item 
 * Parameters:  proxyObj, invocation, itemName,
                item, user_data
 * Returns:     gboolean
 *********************************************************/
 gboolean handle_set_public_item( ConfigurationManagement* proxyObj,
                                  GDBusMethodInvocation *invocation,
                                  gchar*    itemName,
                                  GVariant* item,
                                  gpointer  user_data )
 {
     DLT_LOG(AGW_CONFIGMGR,DLT_LOG_INFO,DLT_STRING("+"),
                              DLT_STRING(__FUNCTION__));

     user_data = invocation;

     if( (!itemName) || (!item) )
     {
         DLT_LOG(AGW_CONFIGMGR,DLT_LOG_INFO,DLT_STRING(__FUNCTION__),
                                      DLT_STRING("(): Null input"));
         configuration_management_complete_set_public_item ( 
                                                    proxyObj,
                                                  invocation,
                     AP_CONFIG_ACTION_STATUS_INVALID_INPUT );
         return TRUE;
     }
   
     vSetPublicItem(itemName, item, user_data); 

     DLT_LOG(AGW_CONFIGMGR,DLT_LOG_INFO,DLT_STRING("-"),
                             DLT_STRING(__FUNCTION__));  
     return TRUE; 
 }

 /*********************************************************
 * Function:    vHandleSetPublicItemResult
 * Description: Sends the result of set public config item
 * Parameters:  status, user_data
 * Returns:     gboolean
 *********************************************************/
 void  vHandleSetPublicItemResult ( guint    status,
                                    gpointer user_data )
 {
     DLT_LOG(AGW_CONFIGMGR,DLT_LOG_INFO,DLT_STRING("+"),
                               DLT_STRING(__FUNCTION__));
     if(!user_data)
     {
         DLT_LOG( AGW_CONFIGMGR, DLT_LOG_ERROR, DLT_STRING(__FUNCTION__),
                  DLT_STRING("(): Null invocation") );
        return;  
     }
     if( status > AP_CONFIG_ACTION_STATUS_INTERNAL_ERROR )
     {
          DLT_LOG(AGW_CONFIGMGR,DLT_LOG_ERROR,DLT_STRING(__FUNCTION__),
               DLT_STRING("():Status is not a valid ApConfigDataActionStatus"));
          status = AP_CONFIG_ACTION_STATUS_INTERNAL_ERROR;
     }

     configuration_management_complete_set_public_item ( 
                                       poGetConfigMgmtProxy(),
                            (GDBusMethodInvocation*)user_data,
                                               status  );
     DLT_LOG(AGW_CONFIGMGR,DLT_LOG_INFO,DLT_STRING("-"),
                               DLT_STRING(__FUNCTION__));      
 }     

 /********************************************************************
 * Function:    handle_get_public_item
 * Description: Gets the public config item
 * Parameters:  proxyObj, invocation, itemName, user_data
 * Returns:     gboolean
 ********************************************************************/
 gboolean handle_get_public_item( ConfigurationManagement* proxyObj,
                                  GDBusMethodInvocation* invocation,
                                  gchar*   itemName,
                                  gpointer user_data )
 {
     DLT_LOG(AGW_CONFIGMGR,DLT_LOG_INFO,DLT_STRING("+"),
                              DLT_STRING(__FUNCTION__));

     user_data = invocation;

     if( !itemName )
     {
         DLT_LOG(AGW_CONFIGMGR,DLT_LOG_INFO,DLT_STRING(__FUNCTION__),
            DLT_STRING("(): Null input"));
        GVariantBuilder builder;
        g_variant_builder_init (&builder, G_VARIANT_TYPE ("a(ss)"));
        g_variant_builder_add(&builder, "(ss)", "", "");
        GVariant *item = g_variant_builder_end(&builder);

         configuration_management_complete_get_public_item ( proxyObj,
                                                           invocation,
                                                                 item,
                                AP_CONFIG_ACTION_STATUS_INVALID_INPUT );
         return TRUE;
     }

     vGetPublicItem(itemName, user_data);
 
     DLT_LOG(AGW_CONFIGMGR, DLT_LOG_INFO, DLT_STRING("-"),
                              DLT_STRING(__FUNCTION__));
     return TRUE;
 }

 /*********************************************************
 * Function:    vHandleGetPublicItemResult
 * Description: Handles the result of GetPublicItem
 * Parameters:  item, status, user_data
 * Returns:     gboolean
 *********************************************************/
 void  vHandleGetPublicItemResult ( GVariant* item, 
                                    guint     status,
                                    gpointer  user_data )
 {
     DLT_LOG(AGW_CONFIGMGR,DLT_LOG_INFO,DLT_STRING("+"),
                               DLT_STRING(__FUNCTION__));
     if(!user_data)
     {
          DLT_LOG( AGW_CONFIGMGR, DLT_LOG_ERROR, DLT_STRING(__FUNCTION__),
                   DLT_STRING("(): Null invocation") );
          return;
     }
     if( status > AP_CONFIG_ACTION_STATUS_INTERNAL_ERROR )
     {
          DLT_LOG(AGW_CONFIGMGR, DLT_LOG_ERROR, DLT_STRING(__FUNCTION__),
               DLT_STRING("():Status is not a valid ApConfigDataActionStatus"));
          status = AP_CONFIG_ACTION_STATUS_INTERNAL_ERROR;
     }
     if(item == NULL)
     {
         DLT_LOG( AGW_CONFIGMGR, DLT_LOG_ERROR, DLT_STRING(__FUNCTION__),
                                  DLT_STRING("(): NULL item recieved"));
        GVariantBuilder builder;
        g_variant_builder_init (&builder, G_VARIANT_TYPE ("a(ss)"));
        g_variant_builder_add(&builder, "(ss)", "", "");
        item = g_variant_builder_end(&builder);
     }        
        
     configuration_management_complete_get_public_item (
                                 poGetConfigMgmtProxy(),
                      (GDBusMethodInvocation*)user_data,
                                                   item,
                                                status );

     DLT_LOG(AGW_CONFIGMGR,DLT_LOG_INFO,DLT_STRING("-"),
                               DLT_STRING(__FUNCTION__));
 }

 /*************************************************************************
 * Function:    initializeConfigMgmt
 * Descriprion: Initializes the gateway config manager interface
 * Parameters:  void
 * Returns:     gboolean
 ************************************************************************/
 gboolean initializeConfigMgmt()
 {
     DLT_REGISTER_CONTEXT(AGW_CONFIGMGR,"GWCM","AGW Config Mgr DLT Logging");
     DLT_LOG(AGW_CONFIGMGR, DLT_LOG_INFO, DLT_STRING("+"),
                              DLT_STRING(__FUNCTION__));

     gboolean bReturnVal = watch_config_mgmt();

     DLT_LOG(AGW_CONFIGMGR, DLT_LOG_INFO, DLT_STRING("-"),
                             DLT_STRING(__FUNCTION__));
     return bReturnVal; 
 }

 /*************************************************************************
 * Function:    uninitializeConfigMgmt
 * Descriprion: uninitializes the gateway config management interface
 * Parameters:  void
 * Returns:     void
 ************************************************************************/
 void uninitializeConfigMgmt()
 {
     DLT_LOG(AGW_CONFIGMGR, DLT_LOG_INFO, DLT_STRING("+"),
                             DLT_STRING(__FUNCTION__));

     unwatch_config_mgmt();

     DLT_LOG(AGW_CONFIGMGR, DLT_LOG_INFO, DLT_STRING("-"),
                             DLT_STRING(__FUNCTION__));
 }
