/***********************************************************************
*FILE: ConfigClientBusMap.cpp
*SW-COMPONENT: Internet
*DESCRIPTION:  Stores the dbus services
               registered for configurations items
*COPYRIGHT: Ã© 2017 Robert Bosch GmbH
*
*The reproduction, distribution and utilization of this file as
*well as the communication of its contents to others without express
*authorization is prohibited. Offenders will be held liable for the
*payment of damages. All rights reserved in the event of the grant
*of a patent, utility model or design.
***********************************************************************/

#include <iostream>
#include <map>
#include <string>
#include "dlt/dlt.h"
#include "ConfigMgmtClientInterface.h"

DLT_IMPORT_CONTEXT(AGW_CONFIGMGR);

using namespace std;
 
static map<string, string> publicConfigClientBusMap;
static map<string, string> privateConfigClientBusMap; 

#ifdef __cplusplus
extern "C"
{
#endif

 /*************************************************************************
 * Function:    AddConfigClientBusPair
 * Description: Maps the client bus with its registered config item
 * Parameters:  item name, bus name, item type
 * Returns:     void
 **************************************************************************/
 void AddConfigClientBusPair( char*          itemName,
                              char*          dbusName,
                              ConfigItemType itemType )     
 {
     DLT_LOG(AGW_CONFIGMGR, DLT_LOG_INFO,DLT_STRING("+"),
                          DLT_STRING(__FUNCTION__));
     
     if((!itemName) || (!dbusName))
     {
        DLT_LOG(AGW_CONFIGMGR, DLT_LOG_ERROR,DLT_STRING(__FUNCTION__),
            DLT_STRING("Null input(s)"));
        return;
     }
     DLT_LOG( AGW_CONFIGMGR, DLT_LOG_INFO,DLT_STRING(__FUNCTION__),
         DLT_STRING("Item name: "), DLT_STRING(itemName), DLT_STRING(
            "dbusName: "), DLT_STRING(dbusName));
     string s_itemName = string((const char*)itemName);
     string s_dbusName = string((const char*)dbusName);
     if(itemType == CONFIG_PUBLIC_ITEM)
     {
         DLT_LOG(AGW_CONFIGMGR, DLT_LOG_INFO,DLT_STRING("Add public"));
         if(publicConfigClientBusMap.find(s_itemName) 
			         != publicConfigClientBusMap.end())             
         {
             DLT_LOG(AGW_CONFIGMGR, DLT_LOG_INFO, DLT_STRING(__FUNCTION__),
                      DLT_STRING("Item is already registered with bus: "),
                      DLT_STRING(publicConfigClientBusMap[s_itemName].c_str()));
             return;
         }
         publicConfigClientBusMap[s_itemName] = s_dbusName;
     }
     else
     {
          DLT_LOG(AGW_CONFIGMGR, DLT_LOG_INFO,DLT_STRING("Add private"));
          if(privateConfigClientBusMap.find(s_itemName) != 
		                    privateConfigClientBusMap.end())
          {
               DLT_LOG( AGW_CONFIGMGR, DLT_LOG_INFO, DLT_STRING(__FUNCTION__),
                  DLT_STRING("Item is already registered with bus: "),
                  DLT_STRING( privateConfigClientBusMap[s_itemName].c_str()) );
               return;
          }
          privateConfigClientBusMap[s_itemName] = s_dbusName; 
    }
     DLT_LOG(AGW_CONFIGMGR, DLT_LOG_INFO,DLT_STRING("-"),
                          DLT_STRING(__FUNCTION__));
 }

 /*************************************************************************
 * Function:    RemoveConfigClientBusPair
 * Description: Removes the client bus mapping with
                its unregistered config item
 * Parameters:  item name, bus name, item type
 * Returns:     void
 **************************************************************************/
 void RemoveConfigClientBusPair( char*          itemName,
                                 char*          dbusName,
                                 ConfigItemType itemType )
 {
     DLT_LOG(AGW_CONFIGMGR, DLT_LOG_INFO,DLT_STRING("+"),
                          DLT_STRING(__FUNCTION__));
     if((!itemName) || (!dbusName))
     {
         DLT_LOG(AGW_CONFIGMGR, DLT_LOG_ERROR,DLT_STRING(__FUNCTION__),
             DLT_STRING("Null input(s)"));
         return;
     }

     DLT_LOG( AGW_CONFIGMGR, DLT_LOG_INFO,DLT_STRING(__FUNCTION__),
         DLT_STRING("Item name: "), DLT_STRING(itemName), DLT_STRING(
            "dbusName: "), DLT_STRING(dbusName));
     string s_itemName = string((const char*)itemName);

     if(itemType == CONFIG_PUBLIC_ITEM)
     {
         if(publicConfigClientBusMap.find(s_itemName) !=
                  		    publicConfigClientBusMap.end())
         {
             publicConfigClientBusMap.erase(s_itemName); 
             DLT_LOG(AGW_CONFIGMGR, DLT_LOG_INFO, DLT_STRING(__FUNCTION__),
                DLT_STRING("():Successfuly removed Config - bus pair"));
         }
         else  
             DLT_LOG(AGW_CONFIGMGR, DLT_LOG_INFO, DLT_STRING(__FUNCTION__),
                  DLT_STRING("():Config - Bus pair not found"));
     }
     else
     {
         if(privateConfigClientBusMap.find(s_itemName) != 
		                    privateConfigClientBusMap.end())
         {
             privateConfigClientBusMap.erase(s_itemName);
             DLT_LOG(AGW_CONFIGMGR, DLT_LOG_INFO, DLT_STRING(__FUNCTION__),
                  DLT_STRING("():Successfuly removed Config - bus pair"));
         }
         else
             DLT_LOG(AGW_CONFIGMGR, DLT_LOG_INFO, DLT_STRING(__FUNCTION__),
                  DLT_STRING("():Config - Bus pair not found"));
     }
     DLT_LOG(AGW_CONFIGMGR, DLT_LOG_INFO,DLT_STRING("-"),
                          DLT_STRING(__FUNCTION__));
 }
 
 /****************************************************************************
 * Function:    FindClientBusName
 * Description: Provides the client dbus name for the
                requested config item
 * Parameters:  item name, item type
 * Returns:     char*->dbus name if present, else NULL
 ****************************************************************************/
 char* FindClientBusName( char*          itemName,
                          ConfigItemType itemType )
 {
     DLT_LOG(AGW_CONFIGMGR, DLT_LOG_INFO,DLT_STRING("+"),
                          DLT_STRING(__FUNCTION__));
     if(!itemName)
     {
         DLT_LOG(AGW_CONFIGMGR, DLT_LOG_ERROR,DLT_STRING(__FUNCTION__),
             DLT_STRING("Null input(s)"));
         return NULL;
     }
     string s_itemName = string((const char*)itemName);
     if(itemType == CONFIG_PUBLIC_ITEM)
     {
          DLT_LOG(AGW_CONFIGMGR, DLT_LOG_INFO,DLT_STRING("Find public"));
         if(publicConfigClientBusMap.find(s_itemName) != 
		                       publicConfigClientBusMap.end())
         {
             return (char*)publicConfigClientBusMap[s_itemName].c_str();
         } 
         else 
             return NULL;
     }
     else
     {
          DLT_LOG(AGW_CONFIGMGR, DLT_LOG_INFO,DLT_STRING("find private"));
         if(privateConfigClientBusMap.find(s_itemName) != 
		                     privateConfigClientBusMap.end())
         {
             return (char*)privateConfigClientBusMap[s_itemName].c_str();
         }
         else
             return NULL;
     }
     DLT_LOG(AGW_CONFIGMGR, DLT_LOG_INFO,DLT_STRING("-"),
                          DLT_STRING(__FUNCTION__));
 }

#ifdef __cplusplus
}
#endif

                                 
