/***********************************************************************
*FILE: ConfigMgmtClientInterface.h
*SW-COMPONENT: Internet
*DESCRIPTION: Interface for the config
*             management client plugin 
*COPYRIGHT: Ã© 2017 Robert Bosch GmbH
*
*The reproduction, distribution and utilization of this file as
*well as the communication of its contents to others without express
*authorization is prohibited. Offenders will be held liable for the
*payment of damages. All rights reserved in the event of the grant
*of a patent, utility model or design.
***********************************************************************/

#ifndef CONFIG_MGMT_CLIENT_INTERFACE_H
#define CONFIG_MGMT_CLIENT_INTERFACE_H

#include <gio/gio.h>
#include <glib.h>
#include "ap-cm-enums.h"

typedef enum
{
  CONFIG_PUBLIC_ITEM  = 0,
  CONFIG_PRIVATE_ITEM = 1
} ConfigItemType;

typedef struct
{
   gchar*                   itemName;
   gchar*                   dbusName;
   guint                    format;
   guint                    itemType;
   GDBusMethodInvocation*   invocation;
} RegisterRequestDetails;

 /***************************************************************
 * Function:    watch_config_mgmt
 * Description: Connects to Config management Dbus service
 * Parameters:  void
 * Returns:     gboolean
 ***************************************************************/
 gboolean watch_config_mgmt();

 /***************************************************************
 * Function:    unwatch_config_mgmt
 * Description: Disconnects to Config management Dbus service
 * Parameters:  void
 * Returns:     void
 ***************************************************************/
 void unwatch_config_mgmt();

 /***************************************************************
 * Function:    vRegisterConfigItem
 * Description: Registers with Config Manager for
                the requested config item
 * Parameters:  item name, dbus name, format,
                type, user_data
 * Returns:     void
 ***************************************************************/
 void vRegisterConfigItem( gchar*         itemName,
                           gchar*         dbusName,
                           guint          format,
                           ConfigItemType type,
                           gpointer       user_data );

 /***************************************************************
 * Function:    vUnregisterConfigItem
 * Description: De-Registers the requested config item
 * Parameters:  item name, dbus name,
                type, user_data
 * Returns:     void
 ***************************************************************/
 void vUnregisterConfigItem( gchar*         itemName,
                             gchar*         dbusName,
                             ConfigItemType type,
                             gpointer       user_data );


 /***************************************************************
 * Function:    vGetPublicItem
 * Description: Provides the public config item value
 * Parameters:  itemName, user_data
 * Returns:     void
 ***************************************************************/
 void vGetPublicItem( gchar*   itemName,
                      gpointer user_data );

 /***************************************************************
 * Function:    vSetPublicItem
 * Description: Sets the requested public config item 
 * Parameters:  itemName, item, user_data
 * Returns:     void
 ***************************************************************/
 void vSetPublicItem( gchar*    itemName,
                      GVariant* item,
                      gpointer  user_data );

/******************************************************************
 * Function:    vExportConfigMgmtInterface
 * Description: Exposes the config management
                interface of gateway
 * Parameters:  void
 * Returns:     void
 ******************************************************************/
void vExportConfigMgmtInterface();

/******************************************************************
 * Function:    vUnExportConfigMgmtInterface
 * Description: Unexports the config management
                interface of gateway
 * Parameters:  void
 * Returns:     void
 ******************************************************************/
void vUnExportConfigMgmtInterface();

/* Call backs from client */

/******************************************************************************
 * Function:    vHandleRegisterConfigResult
 * Description: Sends the result of Config Registration
 * Parameters:  status, type, user_data
 * Returns:     void
 *****************************************************************************/
 void vHandleRegisterConfigResult ( guint          status,
                                    ConfigItemType type,
                                    gpointer       user_data );

 /******************************************************************************
 * Function:    vHandleUnregisterConfigResult
 * Description: Sends the result of Config unregistration
 * Parameters:  status, type, user_data
 * Returns:     void
 ******************************************************************************/
 void vHandleUnregisterConfigResult ( guint          status,
                                      ConfigItemType type,
                                      gpointer       user_data );

 /*********************************************************
 * Function:    vHandleSetPublicItemResult
 * Description: Handle the result of SetPublicItem
 * Parameters:  status, user_data
 * Returns:     gboolean
 *********************************************************/
 void  vHandleSetPublicItemResult ( guint    status,
                                    gpointer user_data );

/*********************************************************
 * Function:    vHandleGetPublicItemResult
 * Description: Handles the result of GetPublicItem
 * Parameters:  item, status, user_data
 * Returns:     gboolean
 *********************************************************/
 void  vHandleGetPublicItemResult ( GVariant* item, 
                                    guint     status,
                                    gpointer  user_data );

#endif
