/*!
*******************************************************************************
* @file             : bluetooth-manager.c
*******************************************************************************
*  - PROJECT:       : Automotive D-Bus server
*  - SW-COMPONENT   : Gateway
*  - DESCRIPTION    : Manager of Bluetooth D-Bus gateway interfaces Impl.
*  - COPYRIGHT      : &copy; 2016 Robert Bosch Engineering & Business Solutions
*  - Documents      : Give link of relevant documents
*  - HISTORY
*
*  Date     | Name          |  Version | Modification
* ----------|---------------|--------------------------|-----------------------
* 02.12.2016 | Nagaraj      | 1.0.0    | methods for start Bluetooth DBus
******************************************************************************/
#ifndef VARIANT_S_FTR_ENABLE_FEAT_GW_PSARCC
#include "bluetooth-manager.h"
#include "Utility.h"
#include "AutomDBusServer.h"

//Dlt
#include "dlt/dlt.h"
//Dlt Context
DLT_IMPORT_CONTEXT(AGW_Bluetooth);

static BluetoothManager* bluetooth_manager = NULL;
static GDBusMethodInvocation* g_BtGetServicesInvocationptr = NULL;
static GDBusMethodInvocation* g_SrcGetSourcesInvocationptr = NULL;
static GDBusConnection*   poConnectionObj = NULL;

/******************************************************************************
* Function:   poGetBtManager
* Description: Gets the BtManager proxy object
* Parameters:  void
* Return:      void
*****************************************************************************/
BluetoothManager* poGetBtManager()
{
	if (bluetooth_manager == NULL)
	{
		bluetooth_manager = bluetooth_manager_skeleton_new();
	}

	return bluetooth_manager;
}


/******************************************************************************
* Function:    handle_get_services_cb
* Description: handler for get services method
* Parameters:  BluetoothManager, invocation pointer, user_data
* Return:      gboolean
*****************************************************************************/
gboolean handle_get_services_cb(BluetoothManager *object,
		GDBusMethodInvocation *invocation, gpointer user_data) 
{
	DLT_LOG(AGW_Bluetooth,DLT_LOG_INFO,DLT_STRING( " \n"),
	                           DLT_STRING( __FUNCTION__));
	g_BtGetServicesInvocationptr = invocation;
	if(!bGetServicesMethod())
   {
      vGetServiceMethodError(AP_BLUETOOTH_ERROR_INTERNAL_ERROR);
   }
	return TRUE;
}

/******************************************************************************
* Function:    emit_device_added_signal
* Description: emits device added signal
* Parameters:  device_id, GVariant*
* Return:      void
*****************************************************************************/
void
emit_device_added_signal(const gchar *device_id, GVariant *properties)
{
	DLT_LOG(AGW_Bluetooth,DLT_LOG_INFO,DLT_STRING( " \n"),
	                             DLT_STRING( __FUNCTION__));
	g_return_if_fail (bluetooth_manager != NULL);
	g_return_if_fail (properties != NULL);
	g_return_if_fail (device_id != NULL);
	bluetooth_manager_emit_device_added (bluetooth_manager, 
	                                     device_id,
 										 properties);
}

/******************************************************************************
* Function:    emit_device_removed_signal
* Description: emits device removed signal
* Parameters:  device_id
* Return:      void
*****************************************************************************/
void
emit_device_removed_signal(const gchar *device_id)
{
	DLT_LOG(AGW_Bluetooth,DLT_LOG_INFO,DLT_STRING( "\n"),
	                           DLT_STRING( __FUNCTION__));
	g_return_if_fail (bluetooth_manager != NULL);
	g_return_if_fail (device_id != NULL);
	bluetooth_manager_emit_device_removed (bluetooth_manager, device_id);
}

/******************************************************************************
* Function:    emit_properties_changed_signal
* Description: emits properties changed signal
* Parameters:  device_id, GVariant*
* Return:      void
*****************************************************************************/
void
emit_properties_changed_signal(gchar *deviceHandle, GVariant *properties)
{
	DLT_LOG(AGW_Bluetooth,DLT_LOG_INFO,DLT_STRING( "\n"),
	                           DLT_STRING( __FUNCTION__));
	g_return_if_fail (bluetooth_manager != NULL);
	g_return_if_fail (properties != NULL);
	g_return_if_fail (deviceHandle != NULL);
	bluetooth_manager_emit_properties_changed (bluetooth_manager,
                                           	   deviceHandle,
 											   properties);
}

/******************************************************************************
* Function:    initialise_bluetooth_manager
* Description: initialises bluetooth-manager on startup 
* Parameters:  GDBusConnection*, gpointer
* Return:      void
*****************************************************************************/
void
initialise_bluetooth_manager(GDBusConnection *connection, gpointer user_data)
{
   g_return_if_fail (connection != NULL);
   poConnectionObj = connection;
   DLT_LOG(AGW_Bluetooth,DLT_LOG_INFO,DLT_STRING( "+"),
                            DLT_STRING( __FUNCTION__));
}

/*******************************************************************************
 * Function:     vExportAGWBluetoothInterface()
 * Description:  exposes the bluetooth skeleton object
 * Parameters:   None
 * Return:       void
 ******************************************************************************/
void vExportAGWBluetoothInterface()
{
    DLT_LOG(AGW_Bluetooth,DLT_LOG_INFO,DLT_STRING( "+"),
	                         DLT_STRING( __FUNCTION__));
    GError* error = NULL;
    gsize handler = 0;
    poConnectionObj =  poGetGDBusConnection();
    if(poConnectionObj == NULL)
    {
       DLT_LOG(AGW_Bluetooth,DLT_LOG_INFO,
	        DLT_STRING( "connection pointer is NULL"));
       return;
    }
	
    if(!bluetooth_manager) bluetooth_manager = bluetooth_manager_skeleton_new();
  
    if(bluetooth_manager)
    {
        handler = g_signal_connect ( bluetooth_manager,
                           		     "handle-get-services",
 									 G_CALLBACK (handle_get_services_cb),
									 NULL);

        if(!g_dbus_interface_skeleton_export (
		             G_DBUS_INTERFACE_SKELETON (bluetooth_manager),
                     poConnectionObj,
                     "/com/bosch/AutomotiveProxy/BluetoothManager",
                     &error))
        {
              DLT_LOG(AGW_Bluetooth,DLT_LOG_ERROR,DLT_STRING(
			     "BM : - ERROR-- Interface skeleton export failed - "),
			                              DLT_STRING( error->message));
              g_clear_error (&error); // clear not required
              g_signal_handler_disconnect(bluetooth_manager, handler);
        }    
        else
        {
              DLT_LOG(AGW_Bluetooth,DLT_LOG_INFO,
			          DLT_STRING("CM : - Object created & exported"));
        }
    }
    else DLT_LOG(AGW_Bluetooth,DLT_LOG_ERROR,
	       DLT_STRING("Error getting skeleton proxy object"));
}
     
/******************************************************************************
* Function:    un_initialise_bluetooth_manager
* Description: uninitialises bluetooth-manager on shutdown 
* Parameters:  GDBusConnection*, gpointer
* Return:      void
*****************************************************************************/
void
un_initialise_bluetooth_manager()
{
	DLT_LOG(AGW_Bluetooth,DLT_LOG_INFO,DLT_STRING( "\n"),
	                          DLT_STRING( __FUNCTION__));
	DLT_UNREGISTER_CONTEXT(AGW_Bluetooth);	
        if(bluetooth_manager) 
   	   g_clear_object(&bluetooth_manager);
}

/*******************************************************************************
 * Function:     vUnexportAGWBluetoothInterface()
 * Description:  unexports the bluetooth interface
 * Parameters:   None
 * Return:       void
 ******************************************************************************/
void vUnexportAGWBluetoothInterface()
{
   g_dbus_interface_skeleton_unexport(
               G_DBUS_INTERFACE_SKELETON (bluetooth_manager));
}

/******************************************************************************
* Function:    vGetServicesMethodResult
* Description: sends get services method result 
* Parameters:  GVariant*
* Return:      void
*****************************************************************************/
void vGetServicesMethodResult(GVariant* result)
{
	//push the complete to D-Bus.
	if (g_BtGetServicesInvocationptr != NULL)
	{
		bluetooth_manager_complete_get_services(
		             (BluetoothManager*)poGetBtManager(),
			         g_BtGetServicesInvocationptr, result
					                           );
	}
	else
	{
		DLT_LOG(AGW_Bluetooth,DLT_LOG_INFO,DLT_STRING(
                    		"g_BtGetServicesInvocationptr IS NULL"));
	}
}

/******************************************************************************
* Function:    vGetServiceMethodError
* Description: sends get services method error 
* Parameters:  APBluetoothError
* Return:      void
*****************************************************************************/
void vGetServiceMethodError(APBluetoothError eState)
{

	DLT_LOG(AGW_Bluetooth,DLT_LOG_INFO,DLT_STRING(
            	      "Telephone :vGetServiceMethodError"));
	if (!g_BtGetServicesInvocationptr)
	{
		DLT_LOG(AGW_Bluetooth,DLT_LOG_INFO,DLT_STRING(
                      		 "g_BtGetServicesInvocationptr NULL"));
		return;
	}

	g_dbus_method_invocation_return_error(g_BtGetServicesInvocationptr,
		AP_BLUETOOTH_ERROR,
		(int)eState,
		"Error in bt manager");

}

/******************************************************************************
* Function:    handle_get_sources
* Description: handler for get sources method
* Parameters:  SourceManager object, invocation pointer
* Return:      gboolean
*****************************************************************************/
gboolean handle_get_sources (SourceManager *object,
                             GDBusMethodInvocation *invocation) 
{
   DLT_LOG(AGW_Bluetooth,DLT_LOG_INFO,DLT_STRING("+\n"),
                              DLT_STRING( __FUNCTION__));
   g_SrcGetSourcesInvocationptr = g_object_ref(invocation);
   if(!bGetSources())
   {
      vGetSourcesMethodError(AP_BLUETOOTH_ERROR_INTERNAL_ERROR);
   }
   return TRUE;   
      
} 

/******************************************************************************
* Function:    vGetSourcesMethodError
* Description: sends get sources method error
* Parameters:  APContactError enum
* Return:      void
*****************************************************************************/
void vGetSourcesMethodError(APContactError eState)
{
   DLT_LOG(AGW_Bluetooth,DLT_LOG_INFO,DLT_STRING(
                     "SourceManager :vGetSourcesMethodError"));
   if (!g_SrcGetSourcesInvocationptr)
   {
      DLT_LOG(AGW_Bluetooth,DLT_LOG_INFO,DLT_STRING(
                   	     "g_SrcGetSourcesInvocationptr NULL"));
      return;
   }

   g_dbus_method_invocation_return_error(g_SrcGetSourcesInvocationptr,
      AP_CONTACT_ERROR,
      (int)eState,
      "Error in getting sources");
      
   g_clear_object(&g_SrcGetSourcesInvocationptr);
}

/******************************************************************************
* Function:    vGetSourcesMethodResult
* Description: sends get sources method result
* Parameters:  GVariant*
* Return:      void
*****************************************************************************/
void vGetSourcesMethodResult(GVariant* result)
{
   DLT_LOG(AGW_Bluetooth,DLT_LOG_INFO,DLT_STRING("+\n"),
                              DLT_STRING( __FUNCTION__));
   SourceManager* l_poSourceManager = poGetSourceManager();
   g_return_if_fail (l_poSourceManager != NULL);
   
   if (g_SrcGetSourcesInvocationptr != NULL)
   {
      source_manager_complete_get_sources(l_poSourceManager,
         g_SrcGetSourcesInvocationptr, result);
         
      g_clear_object(&g_SrcGetSourcesInvocationptr);
   }
   else
   {
      DLT_LOG(AGW_Bluetooth,DLT_LOG_INFO,DLT_STRING(
      	           "g_SrcGetSourcesInvocationptr IS NULL"));
   }
}

/******************************************************************************
* Function:    emit_source_removed_signal
* Description: emits source removed signal
* Parameters:  gchar*
* Return:      void
*****************************************************************************/
void
emit_source_removed_signal(const gchar *device_id)
{
   DLT_LOG(AGW_Bluetooth,DLT_LOG_INFO,DLT_STRING( "\n"),
                             DLT_STRING( __FUNCTION__));
   SourceManager* l_poSourceManager = poGetSourceManager();
   g_return_if_fail (l_poSourceManager != NULL);
   g_return_if_fail (device_id != NULL);
   source_manager_emit_source_removed (l_poSourceManager, device_id);
}

/******************************************************************************
* Function:    emit_source_added_signal
* Description: emits source added signal
* Parameters:  gchar*, GVariant*
* Return:      void
*****************************************************************************/
void
emit_source_added_signal(const gchar *device_id, GVariant *properties)
{
   DLT_LOG(AGW_Bluetooth,DLT_LOG_INFO,DLT_STRING( "\n"),
                             DLT_STRING( __FUNCTION__));
   SourceManager* l_poSourceManager = poGetSourceManager();
   g_return_if_fail (l_poSourceManager != NULL);
   g_return_if_fail (properties != NULL);
   g_return_if_fail (device_id != NULL);
   DLT_LOG(AGW_Bluetooth,DLT_LOG_INFO,DLT_STRING("Device id :"),
                                          DLT_STRING(device_id));
   DLT_LOG(AGW_Bluetooth,DLT_LOG_INFO,DLT_STRING("Properties:"));
   PRINT_GVARIANT(properties, AGW_Bluetooth);
   source_manager_emit_source_added (l_poSourceManager, device_id, properties);
}
#endif//VARIANT_S_FTR_ENABLE_FEAT_GW_PSARCC