/*!
*******************************************************************************
* @file             : Utility.h
*******************************************************************************
*  - PROJECT:       : Automotive Gateway
*  - SW-COMPONENT   : Utility Functiond
*  - DESCRIPTION    : Utility functions which can be used by multiple components.
*  - COPYRIGHT      : &copy; 2017 Robert Bosch Engineering & Business Solutions
*  - Documents      : Give link of relevant documents
*  - HISTORY
*
*  Date     | Name          |  Version | Modification
* ----------|---------------|--------------------------|-----------------------
* 8.3.2017 | BCP9KOR(RBEI/ECO2) | 1.0.0 | methods for Utility. 
******************************************************************************/
#include "Utility.h"

//Dlt
#include "dlt/dlt.h"
#include <string.h>
#include <ctype.h> //isspace
#include <stdlib.h>
#include <sys/wait.h>

#define UTIL_DEFUSEPNG "/opt/bosch/base/bin/defusePNG"

DLT_IMPORT_CONTEXT(AGW_framework);


/*****************************************************************************
 * Function:    trim_leading_space
 * Description: Trims out the leading white spaces on both
                sides of the input string.
 * Parameters:  input string to be trimmed
 * Return:      TRUE,  when input in trimmed
                FALSE, when input is not altered.
******************************************************************************/
gboolean trim_leading_space(gchar* input)
{
    guint   loopIter =0;
//    gchar* ref_str = NULL;
    int startIndex = 0, endIndex = 0;
    gchar* output = NULL;
    if(input == NULL)
    {
        DLT_LOG(AGW_framework,DLT_LOG_INFO, DLT_STRING(__FUNCTION__),
                DLT_STRING("(): Null input"));
        return FALSE;
    }
    /* Trim left */
    for(loopIter = 0; loopIter < strlen(input); loopIter++)
    {
        if(!isspace(input[loopIter]))
            break;
        startIndex++;
    }

    if(input[startIndex] == '\0')
    {
        /* All are white spaces */
        return FALSE;
    }

    /* Trim Right */
/*    for( ref_str = input_str + strlen(input_str) - 1;
         (ref_str > input_str ); ref_str-- )
    {
        if(!isspace((unsigned char)*ref_str))
            break;
    }
    *(ref_str+1) = '\0';
*/

    for(endIndex = (int)(strlen(input) - 1); endIndex > 0;
               endIndex -- )
    {
        if(!isspace(input[endIndex]))
            break;
    }  
    
    if((endIndex < (int)(strlen(input) - 1)) || (startIndex > 0))
    {
        DLT_LOG(AGW_framework,DLT_LOG_INFO, DLT_STRING(__FUNCTION__),
               DLT_STRING("(): edge space detected"));
        output = malloc((size_t)(endIndex - startIndex + 2));
        if(!output)
        {
            DLT_LOG(AGW_framework, DLT_LOG_ERROR, DLT_STRING(__FUNCTION__),  
                 DLT_STRING("(): malloc failed"));
            return FALSE;
        }
        int i = 0, j = 0;
        for(i = startIndex, j = 0; i <= endIndex; i++, j++)
        {
           output[j] = input[i];
        }
        output[j] = '\0';
        strcpy(input, output);
        free(output);
        return TRUE;
    }
    return FALSE;
}

/*****************************************************************************
 * Function:    DecimalToHexValue
 * Description: Convert the decimal value to hex
 * Parameters:  int
 * Return:      void
 *****************************************************************************/
void DecimalToHexValue(int n)
{
	DLT_LOG(AGW_framework, DLT_LOG_INFO, DLT_STRING("[FUNC] entered"),
				DLT_STRING(__FUNCTION__));
	char hexaDeciNum[100];
	char value[100];
	int i = 0, k = 0;
	while (n != 0)
	{
		int temp = 0;
		temp = n % 16;
		if (temp < 10)
		{
			hexaDeciNum[i] = (char) (temp + 48);
			i++;
		}
		else
		{
			hexaDeciNum[i] = (char) (temp + 55);
			i++;
		}
		n = n / 16;
	}
	for (int j = i - 1; j >= 0; j--)
	{
		value[k] = hexaDeciNum[j];
		k++;
	}
	value[k] = '\0';
	DLT_LOG(AGW_framework, DLT_LOG_INFO, DLT_STRING("Signal Id :"), DLT_STRING(value));
}

/*****************************************************************************
 * Function:    GetDoubleFromVariant
 * Description: Finds out the type of Variant and returns the value in 
 				gdouble
 * Parameters:  GVariant*
 * Return:      gdouble
 *****************************************************************************/
gdouble GetDoubleFromVariant(GVariant *variant)
{
	if (NULL != variant)
	{
		GVariantClass type;
		type = g_variant_classify(variant);

		switch (type)
		{
		case G_VARIANT_CLASS_BYTE:
			return g_variant_get_byte(variant);
		case G_VARIANT_CLASS_DOUBLE:
			return g_variant_get_double(variant);
		case G_VARIANT_CLASS_INT16:
			return g_variant_get_int16(variant);
		case G_VARIANT_CLASS_INT32:
			return g_variant_get_int32(variant);
		case G_VARIANT_CLASS_INT64:
			return (gdouble)g_variant_get_int64(variant);
		case G_VARIANT_CLASS_UINT16:
			return g_variant_get_uint16(variant);
		case G_VARIANT_CLASS_UINT32:
			return g_variant_get_uint32(variant);
		case G_VARIANT_CLASS_UINT64:
			return (gdouble)g_variant_get_uint64(variant);
		case G_VARIANT_CLASS_BOOLEAN:
		case G_VARIANT_CLASS_HANDLE:
		case G_VARIANT_CLASS_STRING:
		case G_VARIANT_CLASS_OBJECT_PATH:
		case G_VARIANT_CLASS_SIGNATURE:
		case G_VARIANT_CLASS_VARIANT:
		case G_VARIANT_CLASS_MAYBE:
		case G_VARIANT_CLASS_ARRAY:
		case G_VARIANT_CLASS_TUPLE:
		case G_VARIANT_CLASS_DICT_ENTRY:
		default:
			DLT_LOG(AGW_framework, DLT_LOG_ERROR, 
						DLT_STRING("Variant type not found"));
		}
	}
	else
	{
		return 0;
	}
	return 0;
}

/******************************************************************************
 * Function:       checkIfFileExists
 * Description:    Function to check if file exists
 * Parameters:     gchar*, file path
 * Return:         gboolean
 *****************************************************************************/
gboolean checkIfFileExists(gchar* fname)
{
    DLT_LOG(AGW_framework,DLT_LOG_INFO,DLT_STRING("+ "),
            DLT_STRING( __FUNCTION__));
    DLT_LOG(AGW_framework,DLT_LOG_INFO,DLT_STRING("filepath is "),
            DLT_STRING(fname));
    if( access( fname, F_OK ) != -1 ) {
        DLT_LOG(AGW_framework,DLT_LOG_INFO,DLT_STRING("File exists"));
        return TRUE;
    } else {
        DLT_LOG(AGW_framework,DLT_LOG_INFO,DLT_STRING("File doesn't exist"));
        return FALSE;
    }
}

/******************************************************************************
 * Function:       DefusePngchildCreation
 * Description:    Function to create child of defuse png
 * Parameters:     char*
 * Return:         int
 *****************************************************************************/
 int DefusePngchildCreation(char* in_arg[])
 {
    DLT_LOG(AGW_framework,DLT_LOG_INFO,DLT_STRING("+ "),
            DLT_STRING( __FUNCTION__));
    int l_iDefuserStatus = 0;
    int l_iRet = 1;
    pid_t wait_ret;
    pid_t pid = fork();
    // fork failure
    if (pid == -1)
    {
        DLT_LOG(AGW_framework,DLT_LOG_INFO,DLT_STRING("fork error"));
    }
    /* Parent waits for child process to exit*/
    else if (pid > 0)
    {
		DLT_LOG(AGW_framework,DLT_LOG_INFO,DLT_STRING("pid > 0"),DLT_INT(getppid()));
        do{
            wait_ret = waitpid(pid, &l_iDefuserStatus, 0);			
            if (wait_ret == -1)
            {
				
                 DLT_LOG(AGW_framework,DLT_LOG_INFO,DLT_STRING("waitpid failure = "),DLT_INT32(wait_ret));
                 break;
            }
            if (WIFEXITED(l_iDefuserStatus))
            {
                l_iRet = (int)WEXITSTATUS(l_iDefuserStatus);
                DLT_LOG(AGW_framework,DLT_LOG_INFO,DLT_STRING("waiting for child, exited , status= "),DLT_INT(l_iRet));

            }
            else if (WIFSIGNALED(l_iDefuserStatus))
            {
                l_iRet = (int)WIFSIGNALED(l_iDefuserStatus);
                DLT_LOG(AGW_framework,DLT_LOG_INFO,
                        DLT_STRING("killed by signal = "),
                        DLT_INT((int)l_iRet));
            }
        } while (WIFEXITED(!l_iDefuserStatus) && !WIFSIGNALED(l_iDefuserStatus));
    }
    /* Child process calls PNGDefuser*/
    else
    {
        int ret = execv(UTIL_DEFUSEPNG, &in_arg[0]);
        if(ret < 0)
        {
			_exit(1);
			
			
        }
    }
    return l_iRet;
 }