/*!
*******************************************************************************
* @file             : ReadConfig.c
*******************************************************************************
*  - PROJECT:       : Automotive D-Bus server
*  - SW-COMPONENT   : Gateway
*  - DESCRIPTION    : Implemenatation to read AGW configuration file
*  - COPYRIGHT      : &copy; 2015 Robert Bosch Engineering & Business Solutions
*  - Documents      : Give link of relevant documents
*  - HISTORY
*
*  Date     | Name          |  Version | Modification
* ----------|---------------|--------------------------|-----------------------
* 05.16.2017 | sxn5kor(RBEI/ECO2) | 1.0.0 | read AGW configuration file
* 08.18.2017 | sxn5kor(RBEI/ECO2) | 1.0.1 | coverity fix
******************************************************************************/
#include <glib.h>
#include "ReadConfig.h"
#include <string.h>
#include <stdlib.h>
#include <gio/gio.h>

//Dlt
#include "dlt/dlt.h"
DLT_IMPORT_CONTEXT(AGW_framework);

#define CONFIG_FILE "/etc/AGW.conf"
#define WAIT_TIME 300

#ifdef  ENABLE_CGROUP_IMPL
#include <sys/eventfd.h>
#include <errno.h>
#include "AutomDBusServer.h"
#include <unistd.h>
#include <pthread.h>
#include <sys/types.h>
#include <sys/syscall.h>
#import <sys/epoll.h>
//#define CGRP_EVENT "/sys/fs/cgroup/memory/system.slice/rbcm-procgenericgateway.service/cgroup.event_control"
//#define CGRP_MEM_USAGE "/sys/fs/cgroup/memory/system.slice/rbcm-procgenericgateway.service/memory.usage_in_bytes"
//#define CGRP_TASKS "/sys/fs/cgroup/memory/system.slice/rbcm-procgenericgateway.service/tasks"
#define CGRP_EVENT "/sys/fs/cgroup/memory/gateway.slice/cgroup.event_control"
#define CGRP_MEM_USAGE "/sys/fs/cgroup/memory/gateway.slice/memory.usage_in_bytes"
#define CGRP_TASKS "/sys/fs/cgroup/memory/gateway.slice/tasks"
//#define CGRP_OOM_CONTROL "/sys/fs/cgroup/memory/system.slice/rbcm-procgenericgateway.service/memory.oom_control"
#define BUFSIZE 256
#define READBUF 50

static int event_fd;
static int epollfd;
#endif  //ENABLE_CGROUP_IMPL

static GKeyFile *keyfile = NULL;

bool loadConfig()
{
    DLT_LOG(AGW_framework,DLT_LOG_INFO,DLT_STRING(
                "Entered "),DLT_STRING( __FUNCTION__));
    GKeyFileFlags flags;
    GError *error = NULL;

    // Create a new GKeyFile object
    keyfile = g_key_file_new ();
    if(NULL == keyfile)
    {
        DLT_LOG(AGW_framework,DLT_LOG_INFO,DLT_STRING("keyfile is NULL"));
        return false;
    }
    flags = G_KEY_FILE_KEEP_COMMENTS | G_KEY_FILE_KEEP_TRANSLATIONS;
    // Load the GKeyFile from keyfile.conf or return.
    if (!g_key_file_load_from_file (keyfile, CONFIG_FILE, flags, &error))
    {
        DLT_LOG(AGW_framework,DLT_LOG_ERROR,
                DLT_STRING("error in loading config file: "),
                DLT_STRING(error->message));
        g_clear_error (&error);
        return false;
    }
    return true;
}

guchar readBusConfig()
{
    DLT_LOG(AGW_framework,DLT_LOG_INFO,DLT_STRING(
                "Entered "),DLT_STRING( __FUNCTION__));
    char* sBusType = "";
    GError *error = NULL;
    guchar type = 0;
    if(NULL == keyfile)
    {
        DLT_LOG(AGW_framework,DLT_LOG_INFO,DLT_STRING("keyfile is NULL"));
        return DBUS_TYPE_SESSION;
    }
    sBusType = g_key_file_get_string(keyfile,"bus_config","bustype",&error);
    if(error != NULL)
    {
        DLT_LOG(AGW_framework,DLT_LOG_ERROR,
                DLT_STRING("error in reading config file: "),
                DLT_STRING(error->message));
        g_clear_error (&error);
    }

    if(sBusType){
        DLT_LOG(AGW_framework,DLT_LOG_ERROR,
                DLT_STRING("readBusConfig: busType =  "),
                DLT_STRING(sBusType));
        if(strncmp(sBusType, "TCPIP", strlen("TCPIP")) == 0)
        {
            type = DBUS_TYPE_TCPIP;
        }
        else if(strncmp(sBusType, "IDDBUS", strlen("IDDBUS")) == 0)
        {
            type = DBUS_TYPE_IDDBUS;
        }
        else{
            type = DBUS_TYPE_SESSION;
        }

        free(sBusType);
        sBusType = NULL;

        return type;
    }
    return DBUS_TYPE_SESSION;
}

gint readWaitParam()
{
    DLT_LOG(AGW_framework,DLT_LOG_INFO,DLT_STRING(
                 "Entered "),DLT_STRING( __FUNCTION__));
	gint wait_value;
	GError *error = NULL;

	if(NULL == keyfile)
    {
        DLT_LOG(AGW_framework,DLT_LOG_INFO,DLT_STRING("keyfile is NULL"));
		return WAIT_TIME;
    }
	wait_value = g_key_file_get_integer(keyfile,
										"bus_config",
										"wait",
										&error);
	if(error != NULL)
    {
     DLT_LOG(AGW_framework,DLT_LOG_ERROR,DLT_STRING("error in reading wait value: "),
             DLT_STRING(error->message));
     g_clear_error (&error);
    }
	return wait_value;
}

char* readPrivateBusAddress()
{
    DLT_LOG(AGW_framework,DLT_LOG_INFO,DLT_STRING(
                "Entered "),DLT_STRING( __FUNCTION__));

    char* busAddress = "";
    GError *error = NULL;
    if(NULL == keyfile)
    {
        DLT_LOG(AGW_framework,DLT_LOG_INFO,DLT_STRING("keyfile is NULL"));
        return NULL;
    }
    busAddress = g_key_file_get_string(keyfile,"bus_config",
                                       "connectionAddress",
                                       &error);
    if(error != NULL)
    {
        DLT_LOG(AGW_framework,DLT_LOG_ERROR,
                DLT_STRING("error in reading config file: "),
                DLT_STRING(error->message));
        g_clear_error (&error);
        return NULL;
    }
    DLT_LOG(AGW_framework,DLT_LOG_ERROR,
            DLT_STRING("readBusConfig: busAddress =  "),
            DLT_STRING(busAddress));
    return busAddress;
}

gint readIncomingLimit()
{
    gint limit = 0;
    GError *error = NULL;
    char* limitStr = "";
    if(NULL == keyfile)
    {
        DLT_LOG(AGW_framework,DLT_LOG_ERROR,DLT_STRING(__func__),
                DLT_STRING("keyfile is NULL"));
        return limit;
    }
    limitStr = g_key_file_get_string(keyfile,
                                     "message_limit",
                                     "maxIncomingRequestsPerSecond",
                                     &error);
    if(error)
    {
        DLT_LOG(AGW_framework,DLT_LOG_ERROR,
                DLT_STRING("error in reading config file: "),
                DLT_STRING(error->message));
        return 0;
    }
    if((!limitStr) || (strcmp(limitStr,"") == 0))
    {
        DLT_LOG(AGW_framework,DLT_LOG_ERROR,DLT_STRING(__func__),
                DLT_STRING("Error fetching msg limit from config"));
        return 0;
    }
    limit = atoi(limitStr);
	g_free(limitStr);
    return limit;
}      

gboolean readGLogSettings()
{
    GError *error = NULL;
    gboolean bEnableLogging = false;

    if(NULL == keyfile)
    {
        DLT_LOG(AGW_framework, DLT_LOG_ERROR, DLT_STRING(__func__)
                , DLT_STRING("keyfile is NULL"));
        return NULL;
    }

    bEnableLogging = g_key_file_get_boolean (keyfile, "glog_info"
                                             , "enableGLibLogging", &error);
    if(error)
    {
        DLT_LOG(AGW_framework, DLT_LOG_ERROR
                , DLT_STRING("error in reading config file: ")
                , DLT_STRING(error->message));
        g_clear_error (&error);
        return false;
    }

    return bEnableLogging;
}

void unloadConfig()
{
    DLT_LOG(AGW_framework,DLT_LOG_INFO,DLT_STRING(
                "Entered "),DLT_STRING( __FUNCTION__));
    if(keyfile != NULL)
    {
        g_key_file_free (keyfile);
    }
}

#ifdef  ENABLE_CGROUP_IMPL
void* monitor_event(void* var)
{
    DLT_LOG(AGW_framework,DLT_LOG_INFO,
            DLT_STRING("Inside monitor_event"));
    DLT_LOG(AGW_framework,DLT_LOG_INFO,
            DLT_STRING("monitor_event thread ID = "),
            DLT_INT(syscall(SYS_gettid)));
    struct epoll_event events;

    while(1) {
        int c = epoll_wait(epollfd, &events, 1, -1);

        if(c != -1) {
            DLT_LOG(AGW_framework,DLT_LOG_INFO,
                    DLT_STRING("Notification recieved "), DLT_INT(c));
            DLT_LOG(AGW_framework,DLT_LOG_INFO,
                    DLT_STRING("Memory threshhold reached,"
                               "unexport all interfaces"));
            unexportAllInterfaces();
            break;
        }
    }
    return;

}

void read_memory_threshold()
{

    DLT_LOG(AGW_framework,DLT_LOG_INFO,DLT_STRING("In read_memory_threshold"));
    char buf[16];
    int n;
    DLT_LOG(AGW_framework,DLT_LOG_INFO,DLT_STRING("getpid = "),
            DLT_INT(getpid()));
    n = snprintf(buf, sizeof(buf), "%d", getpid());
    int fd_task = open(CGRP_TASKS, O_WRONLY);
    int write_ret = write(fd_task, buf, (size_t)n);

    DLT_LOG(AGW_framework,DLT_LOG_INFO,DLT_STRING("write_ret = "),
            DLT_INT(write_ret));
    if(write_ret == -1)
    {
        DLT_LOG(AGW_framework,DLT_LOG_ERROR,
                DLT_STRING("write event ctl Errorno "),DLT_INT(errno));
        DLT_LOG(AGW_framework,DLT_LOG_ERROR,
                DLT_STRING("write event ctl Error str "),
                DLT_STRING(strerror(errno)));
    }


    int fd_event_cntl, fd_mem_usage;
    char buffer[BUFSIZE];
    int cx;
    event_fd = eventfd(0, 0);
    DLT_LOG(AGW_framework,DLT_LOG_INFO,
            DLT_STRING("event_fd = "),DLT_INT(event_fd));

    if(event_fd == -1)
    {
        DLT_LOG(AGW_framework,DLT_LOG_ERROR,
                DLT_STRING("event fd Errorno "),DLT_INT(errno));
        DLT_LOG(AGW_framework,DLT_LOG_ERROR,
                DLT_STRING("event fd ctl Error str "),
                DLT_STRING(strerror(errno)));
    }


    fd_event_cntl = open(CGRP_EVENT, O_WRONLY);
    DLT_LOG(AGW_framework,DLT_LOG_INFO,
            DLT_STRING("fd_event_cntl = "),DLT_INT(fd_event_cntl));
    if(fd_event_cntl == -1)
    {
        DLT_LOG(AGW_framework,DLT_LOG_ERROR,
                DLT_STRING("Open event ctl Errorno "),DLT_INT(errno));
        DLT_LOG(AGW_framework,DLT_LOG_ERROR,
                DLT_STRING("Open event ctl Error str "),
                DLT_STRING(strerror(errno)));
    }

    fd_mem_usage = open(CGRP_MEM_USAGE, O_RDONLY);

    DLT_LOG(AGW_framework,DLT_LOG_INFO,
            DLT_STRING("fd_mem_usage = "),
            DLT_INT(fd_mem_usage));

    if(fd_mem_usage == -1)
    {
        DLT_LOG(AGW_framework,DLT_LOG_ERROR,
                DLT_STRING("Open mem usage Errorno "),DLT_INT(errno));
        DLT_LOG(AGW_framework,DLT_LOG_ERROR,
                DLT_STRING("Open mem usage Error str "),
                DLT_STRING(strerror(errno)));
    }

    cx = snprintf(buffer, BUFSIZE, "%d %d 9000000", event_fd, fd_mem_usage);
    DLT_LOG(AGW_framework,DLT_LOG_INFO,
            DLT_STRING("buffer = "), DLT_STRING(buffer));
    DLT_LOG(AGW_framework,DLT_LOG_INFO,
            DLT_STRING("cx = "),DLT_INT(cx));

    write_ret = write(fd_event_cntl, buffer, (size_t)cx);
    DLT_LOG(AGW_framework,DLT_LOG_INFO,
            DLT_STRING("write_ret = "),DLT_INT(write_ret));

    if(write_ret == -1)
    {
        DLT_LOG(AGW_framework,DLT_LOG_ERROR,
                DLT_STRING("write event ctl Errorno "),DLT_INT(errno));
        DLT_LOG(AGW_framework,DLT_LOG_ERROR,
                DLT_STRING("write event ctl Error str "),
                DLT_STRING(strerror(errno)));
    }


    int close_ret = close(fd_event_cntl);
    if(close_ret == -1)
    {
        DLT_LOG(AGW_framework,DLT_LOG_ERROR,
                DLT_STRING("close event ctl Errorno "),DLT_INT(errno));
        DLT_LOG(AGW_framework,DLT_LOG_ERROR,
                DLT_STRING("close event ctl Error str "),
                DLT_STRING(strerror(errno)));
    }

    close_ret = close(fd_mem_usage);
    if(close_ret == -1)
    {
        DLT_LOG(AGW_framework,DLT_LOG_ERROR,
                DLT_STRING("close mem usage Errorno "),DLT_INT(errno));
        DLT_LOG(AGW_framework,DLT_LOG_ERROR,
                DLT_STRING("close mem usage Error str "),
                DLT_STRING(strerror(errno)));
    }

    DLT_LOG(AGW_framework,DLT_LOG_INFO,
            DLT_STRING("main thread ID = "),DLT_INT(syscall(SYS_gettid)));
    epollfd = epoll_create(1);

    struct epoll_event evnt = { 0 };
    evnt.data.fd = event_fd;
    evnt.events = EPOLLIN | EPOLLET;

    epoll_ctl(epollfd, EPOLL_CTL_ADD, event_fd, &evnt);

    pthread_t read_efd_thread;
    if(pthread_create(&read_efd_thread, NULL, monitor_event, NULL))
    {
        DLT_LOG(AGW_framework,DLT_LOG_ERROR,
                DLT_STRING("Error creating thread"));
    }
    else
    {
        DLT_LOG(AGW_framework,DLT_LOG_INFO,DLT_STRING("created new thread"));
    }

    sleep(1);
#ifdef ENABLE_FOR_CGROUP_TESTING
    memHog();
#endif//ENABLE_FOR_CGROUP_TESTING
}

#endif //ENABLE_CGROUP_IMPL


#ifdef ENABLE_FOR_CGROUP_TESTING
void memHog()
{
    DLT_LOG(AGW_framework,DLT_LOG_INFO,
            DLT_STRING("memHog thread ID = "),DLT_INT(syscall(SYS_gettid)));
    DLT_LOG(AGW_framework,DLT_LOG_INFO,DLT_STRING("In memHog"));

    char *p;

    int count = 70;
    while(count)
    {
        DLT_LOG(AGW_framework,DLT_LOG_INFO,DLT_STRING("I'm here"));
        p = (char *)malloc(1000000);
        if(p == NULL)
            DLT_LOG(AGW_framework,DLT_LOG_ERROR,
                    DLT_STRING("mem not allocated"));

        //memset(p, 0, 1000000);
        DLT_LOG(AGW_framework,DLT_LOG_INFO,DLT_STRING("reduce count"));
        count--;
        sleep(1);
    }
    DLT_LOG(AGW_framework,DLT_LOG_INFO,DLT_STRING("moving out of memHog"));
}
#endif //ENABLE_FOR_CGROUP_TESTING
