/*!
*******************************************************************************
* @file             : AutomDBusServer.c
*******************************************************************************
*  - PROJECT:       : Automotive D-Bus server
*  - SW-COMPONENT   : Gateway
*  - DESCRIPTION    : D-Bus server for automotive API's
*  - COPYRIGHT      : &copy; 2015 Robert Bosch Engineering & Business Solutions
*  - Documents      : Give link of relevant documents
*  - HISTORY
*
*  Date     | Name          |  Version | Modification
* ----------|---------------|--------------------------|-----------------------
* 11.30.2015 | DJS5KOR(RBEI/ECO2) | 1.0.0 | methods for Automotive D-Bus Server
* 11.30.2015 | RHK6KOR(RBEI/ECO2) | 2.0.0 | methods for Automotive D-Bus Server
* 1.31.2017  | SXN5KOR(RBEI/ECO2) | 2.1.0 | AudioSource implementation
* 02.07.2017 | SXN5KOR(RBEI/ECO2) | 3.0.0 | compiler switches added to segregate
*                                           AIVI and PSA features
* 05.26.2017 | RVH5KOR(RBEI/ECO2) | 3.0.1 | DLT Logs changes for SPM, NAVI & URI
* 06.15.2017 | SXN5KOR(RBEI/ECO2) | 4.0.0 | Implemenatation to configure bus type
* 18.07.2017 | Deepa Jose(RBEI/ECO2)| 5.0.0 | RouteList.Insert,Delete and replace 
                                              waypoints signal connect
*                                           | Navigation.IsAvaiable implementation
******************************************************************************/
#include "AutomDBusServer.h"
#include "ReadConfig.h"
#include "Utility.h"

#include <stdbool.h>
#include <malloc.h>
//Dlt
#include "dlt/dlt.h"
#include "Timer_Gateway.h"
#include <sys/stat.h>
#include "ConfigMgmtDBusHandler.h"
#include "ConfigDataDBusHandler.h"

#if defined(VARIANT_COMPILER_ENV_GEN3ARM) || defined(VARIANT_COMPILER_ENV_GEN4RCAR)
#include "SDC_DbusHandler.h"
#endif//VARIANT_COMPILER_ENV_GEN3ARM || VARIANT_COMPILER_ENV_GEN4RCAR

#ifdef ENABLE_CGROUP_IMPL
#include "DBusToBTProxyWrapper.h"
#include "ECM_ClientInterface.h"
#include "DBusToMPlayProxyWrapper.h"
#include "PersonalizationClientHandler.h"
#include "DBusToPhoneProxyWrapper.h"
#include "DBusToSpmDataProxyWrapper.h"
#include "TextToSpeech_ClientInterface.h"
#endif//ENABLE_CGROUP_IMPL

#include "ManAppsListHandler.h"

#ifdef VARIANT_S_FTR_ENABLE_FEAT_GW_RNAIVI
//E2E navigation headers
#include "E2ENaviDBusHandler.h"
#include "NaviDBusHandler.h"
#include "PDE_DbusHandler.h"
#include "FC_Gateway_MetaDataClientHandler.h"
#include "MetaDataDbusHandler.h"
#include "VehicleData_DbusHandler.h"
#include "VehicleData_ConfigurationInterface.h"
#endif //VARIANT_S_FTR_ENABLE_FEAT_GW_RNAIVI

#include "SPDbusHandler.h"

#if ENABLE_CSM_IN_GATEWAY
static bool bDoVIMSetup(GDBusConnection *pConnection);
#endif // ENABLE_CSM_IN_GATEWAY


DLT_IMPORT_CONTEXT(AGW_framework);
DLT_DECLARE_CONTEXT(AGW_DataNotifier); //DataNotifier
DLT_DECLARE_CONTEXT(AGW_Bluetooth);
DLT_DECLARE_CONTEXT(AGW_URI); //uri
DLT_DECLARE_CONTEXT(AGW_VehicleData);

static guint              iIDDBusWatcherID = 0;
static guint              isystemWatcherID = 0;
static guint              iSessionWatcherID = 0;

static GDBusConnection*  poSystemBusConn =  NULL;
static GDBusConnection*  poSessionBusConn = NULL;
static GDBusConnection*  poBusConn = NULL;
static guint    igatewayMsgCounter = 0;
static gint    igatewayMsgLimitPerSec = 500;
static gboolean g_blockGatewayIncoming = FALSE;
static guint    check_limit_timeout_id = 0;
static guint    msg_filter_id = 0;

static guchar uBusType = 0;
static gint iSubscription_Wait_Time = 0;

//#define GATEWAY_CONNECTION_ADDRESS "unix:path=/tmp/shared/iddbus/lxcdbus"

#ifdef ENABLE_NON_CMD_IN_GATEWAY
#if defined(VARIANT_S_FTR_ENABLE_FEAT_GW_RNAIVI) || defined(VARIANT_S_FTR_ENABLE_FEAT_GW_PSARCC)
static SettingsProviderFactoryReset* poFactoryReset = NULL;
#endif
#endif //ENABLE_NON_CMD_IN_GATEWAY


#ifndef VARIANT_S_FTR_ENABLE_FEAT_GW_PSARCC
static TunerControl* poTunerControlProxyObj = NULL;
static TunerInfo* poTunerInfoProxyObj = NULL;

static AudioManager*     poAudioManagerProxy  = NULL;
static URI*              URIObj = NULL;
static CallManager*      pCallManager = NULL;
static SmsManager*       pSMSManager = NULL;
static ContactManager*   pContactManager = NULL;
static SourceManager*    pSourceManager = NULL;
static MediaPlayerPlayer* MediaPlayerProxyObj = NULL;
gulong uri_handler_id;
#endif

#ifdef VARIANT_S_FTR_ENABLE_FEAT_GW_RNAIVI
static CarConfiguration* poCarConfiguration = NULL;
static SettingsProvider* poSettingsProvider = NULL;
static GDBusObjectManagerServer *pNaviRouteObjManager = NULL;
static GDBusObjectManagerServer *pNaviGeoCodeObjManager = NULL;
static GDBusObjectManagerServer *pNaviGeoCodeLocationObjManager = NULL;
static SettingsProviderMultimediaState* poMultimediaStateProxyObj = NULL;
GDBusConnection*         poNaviConnection;

static bool vExportE2ENaviInterface();
static GDBusProxy *poAGWComponentProxy = NULL;


#endif //VARIANT_S_FTR_ENABLE_FEAT_GW_RNAIVI

#ifdef VARIANT_S_FTR_ENABLE_FEAT_GW_PSARCC
static DataNotifier* poDataNotifierProxyObj = NULL;
#endif //VARIANT_S_FTR_ENABLE_FEAT_GW_PSARCC

#if ENABLE_CSM_IN_GATEWAY
static VehicleInfoManager *pVehicleInfoManagerObj = NULL;
#endif //ENABLE_CSM_IN_GATEWAY

static GDBusServer *server;
static gchar *guid;
static GDBusServerFlags server_flags;
static gchar *opt_address = "tcp:host=172.17.0.1,port=4589";

static gboolean
on_new_connection (GDBusServer *server,
                   GDBusConnection *pConnection,
                   gpointer user_data);

#ifdef ENABLE_NON_CMD_IN_GATEWAY
#if defined(VARIANT_S_FTR_ENABLE_FEAT_GW_RNAIVI) || defined(VARIANT_S_FTR_ENABLE_FEAT_GW_PSARCC)
/******************************************************************************
 * Function:       poGetFactoryResetProxy
 * Description:    Gets the factory reset proxy pointer.
 * Parameters:     void
 * Return:         SettingsProviderFactoryReset
 *****************************************************************************/
SettingsProviderFactoryReset*  poGetFactoryResetProxy()
{
    if(poFactoryReset == NULL)
    {
        poFactoryReset = settings_provider_factory_reset_skeleton_new ();
    }

    return poFactoryReset;
}


/******************************************************************************
 * Function:      bDoFactoryResetDBusSetup
 * Description:   Performs the factory reset skeleton export and
                  registrations.
 * Parameters:    GDBusConnection*
 * Return:        bool
 *****************************************************************************/
bool bDoFactoryResetDBusSetup(GDBusConnection *pConnection)
{
    GError *error = NULL;
    SettingsProviderFactoryReset* l_poFactoryReset = poGetFactoryResetProxy();
    if(l_poFactoryReset != NULL)
    {
        if (!g_dbus_interface_skeleton_export (
                    G_DBUS_INTERFACE_SKELETON (
                        l_poFactoryReset),
                    pConnection,
                    "/com/bosch/AutomotiveProxy/SettingsProvider/FactoryReset",
                    &error))
        {
            if(error != NULL)
            {
                DLT_LOG(AGW_framework,DLT_LOG_ERROR,
                        DLT_STRING("FactoryReset Interface "
                                   "skeleton export error:"),
                        DLT_STRING(error->message));

                g_clear_error (&error);
            }
            return FALSE;
        }

        /* method signal connection for FactoryReset*/
        g_signal_connect (l_poFactoryReset,
                          "handle-register-for-factory-reset",
                          G_CALLBACK (handle_register_for_factory_reset),
                          NULL);

        g_signal_connect (l_poFactoryReset,
                          "handle-factory-reset-request-complete",
                          G_CALLBACK (handle_factory_reset_request_complete),
                          NULL);
        return TRUE;
    }
    else
    {
        DLT_LOG(AGW_framework,DLT_LOG_ERROR,DLT_STRING("Proxy object is NULL"));
        return FALSE;
    }

}
#endif
#endif //ENABLE_NON_CMD_IN_GATEWAY


#ifndef VARIANT_S_FTR_ENABLE_FEAT_GW_PSARCC
/******************************************************************************
 * Function:    poGetAudioProxyObj
 * Description: Gets the audio proxy object
 * Parameters:  void
 * Return:      void
 *****************************************************************************/

AudioManager*  poGetAudioProxyObj()
{
    if(poAudioManagerProxy == NULL)
    {
        poAudioManagerProxy = audio_manager_skeleton_new ();
    }

    return poAudioManagerProxy;
}

/******************************************************************************
 * Function:    	bDoAudioSetup
 * Description:	    Performs the audio skeleton export and registrations.
 * Parameters:		void
 * Return:			bool
 *****************************************************************************/

bool bDoAudioSetup(GDBusConnection *pConnection)
{
    GError *error = NULL;
    poAudioManagerProxy = poGetAudioProxyObj();
    DLT_LOG(AGW_framework,DLT_LOG_INFO,DLT_STRING( "+bDoAudioSetup"));
    if(poAudioManagerProxy != NULL)
    {

        if (!g_dbus_interface_skeleton_export (
                    G_DBUS_INTERFACE_SKELETON (poAudioManagerProxy),
                    pConnection,
                    "/com/bosch/AutomotiveProxy/AudioManager",
                    &error))
        {
            if(error != NULL)
            {
                DLT_LOG(AGW_framework,DLT_LOG_ERROR,
                        DLT_STRING("Audio Interface"
                                   " skeleton export error:"),
                        DLT_STRING(error->message));

                g_clear_error (&error);
            }
            return FALSE;
        }

        /* signal connection for Audio */
        g_signal_connect (poAudioManagerProxy,
                          "handle-source-availability",
                          G_CALLBACK (handle_source_availability_method),NULL);
        g_signal_connect (poAudioManagerProxy,
                          "handle-audio-route-request",
                          G_CALLBACK (handle_audio_route_request_method),NULL);
        g_signal_connect (poAudioManagerProxy,
                          "handle-mute-state",
                          G_CALLBACK (handle_mute_state_method), NULL);
        g_signal_connect (poAudioManagerProxy,
                          "handle-source-activity-result",
                          G_CALLBACK (handle_source_activity_result_method),
                          NULL);

        DLT_LOG(AGW_framework,DLT_LOG_INFO,DLT_STRING("-bDoAudioSetup"));

        return TRUE;
    }
    else
    {
        DLT_LOG(AGW_framework,DLT_LOG_ERROR,DLT_STRING("Proxy object is NULL"));
        return FALSE;
    }

}

/*******************************************************************************
* Function:   poGetSMSManager
* Description: Gets the SMSManager proxy object
* Parameters:  void
* Return:      void
*****************************************************************************/

SmsManager*  poGetSMSManager()
{
    if (pSMSManager == NULL)
    {
        pSMSManager = sms_manager_skeleton_new();
    }

    return pSMSManager;
}

/*******************************************************************************
 * Function:   poGetCallManager
 * Description: Gets the CallManager proxy object
 * Parameters:  void
 * Return:      void
 *****************************************************************************/

CallManager* poGetCallManager()
{
    if(pCallManager == NULL)
    {
        pCallManager = call_manager_skeleton_new ();
    }

    return pCallManager;
}

/******************************************************************************
 * Function:    	poGetContactManager
 * Description:	    Gets the contact manager pointer.
 * Parameters:		void
 * Return:			GDBusProxy
 *****************************************************************************/
ContactManager* poGetContactManager()
{
    if(pContactManager == NULL)
    {
        pContactManager = contact_manager_skeleton_new ();
    }

    return pContactManager;
}

/******************************************************************************
 * Function:    	poGetSourceManager
 * Description:	Gets the source manager pointer.
 * Parameters:		void
 * Return:			GDBusProxy
 *****************************************************************************/
SourceManager* poGetSourceManager()
{
    if(pSourceManager == NULL)
    {
        pSourceManager = source_manager_skeleton_new ();
    }

    return pSourceManager;
}



/*******************************************************************************
 * Function:    poGetMediaPlayerProxyObj
 * Description: Gets the Mediaplayer proxy object
 * Parameters:  void
 * Return:      MediaPlayerPlayer*
 ******************************************************************************/

MediaPlayerPlayer* poGetMediaPlayerProxyObj()
{
    if(MediaPlayerProxyObj == NULL)
    {
        MediaPlayerProxyObj = media_player_player_skeleton_new ();
    }


    return MediaPlayerProxyObj;
}

/*******************************************************************************
 * Function:    URI
 * Description: Gets the URI proxy object
 * Parameters:  void
 * Return:      URI
 *****************************************************************************/

URI* poGetURIProxyObj()
{
    if(URIObj == NULL)
    {
        URIObj = uri_skeleton_new ();
    }

    return URIObj;
}

/******************************************************************************
 * Function:    	bDoUriSetup
 * Description:	    Performs the uri skeleton export and registrations.
 * Parameters:		void
 * Return:			bool
 *****************************************************************************/

bool bDoUriSetup(GDBusConnection *pConnection)
{
    GError *error = NULL;
    DLT_REGISTER_CONTEXT(AGW_URI, "GWURI", "URI "
                         "context for DLT Logging");
    //URI
    if(!g_dbus_interface_skeleton_export (G_DBUS_INTERFACE_SKELETON (
                                              (URI*)poGetURIProxyObj()),
                                          pConnection,
                                          "/com/bosch/AutomotiveProxy/URI",
                                          &error))
    {
        if(error != NULL)
        {
            DLT_LOG(AGW_framework,DLT_LOG_ERROR,
                    DLT_STRING("URI : ERROR- Interface skeleton export failed"),
                    DLT_STRING(error->message));
            g_clear_error (&error);
        }
        return FALSE;
    }
    else
    {
        DLT_LOG(AGW_framework,DLT_LOG_INFO,
                DLT_STRING("URI :Object created & exported"));
    }

    /* Signal connect for URI */
    uri_handler_id = g_signal_connect (
                (URI*)poGetURIProxyObj(),
                "handle-delegate-uri",
                G_CALLBACK (handle_delegate_uri), NULL);

    DLT_LOG(AGW_framework,DLT_LOG_INFO,DLT_STRING(
                "Delegate URI g_signal_connect handler "),
            DLT_UINT(uri_handler_id));

    return TRUE;
}

/******************************************************************************
 * Function:    	bDoSrcMngrDBusSetup
 * Description:	    Performs the source manager skeleton export and
                    registrations.
 * Parameters:		void
 * Return:			bool
 *****************************************************************************/

bool bDoSrcMngrDBusSetup(GDBusConnection *pConnection)
{
    GError *error = NULL;
    SourceManager* l_pSourceManager = poGetSourceManager();
    if(l_pSourceManager != NULL)
    {

        if (!g_dbus_interface_skeleton_export (
                    G_DBUS_INTERFACE_SKELETON (
                        l_pSourceManager),
                    pConnection,
                    "/com/bosch/AutomotiveProxy/SourceManager",
                    &error))
        {
            if(error != NULL)
            {
                DLT_LOG(AGW_framework,DLT_LOG_ERROR,
                        DLT_STRING("Source manager "
                                   "Interface skeleton export error:"),
                        DLT_STRING(error->message));

                g_clear_error (&error);
            }
            return FALSE;
        }

        /* method signal connection for contact */
        g_signal_connect (l_pSourceManager,
                          "handle-get-sources",
                          G_CALLBACK (handle_get_sources), NULL);
        return TRUE;
    }
    else
    {
        DLT_LOG(AGW_framework,DLT_LOG_ERROR,
                DLT_STRING("source manager "
                           "set up failed. Proxy object is NULL!!"));
        return FALSE;
    }
}
#endif
#ifdef VARIANT_S_FTR_ENABLE_FEAT_GW_RNAIVI

#endif // VARIANT_S_FTR_ENABLE_FEAT_GW_RNAIVI


#ifndef VARIANT_S_FTR_ENABLE_FEAT_GW_PSARCC
/*******************************************************************************
 * Function:    poGetTunerControlProxyObj
 * Description: Gets the Tuner control proxy object
 * Parameters:  void
 * Return:      TunerControl*
 ******************************************************************************/

TunerControl* poGetTunerControlProxyObj()
{
    if(poTunerControlProxyObj == NULL)
    {
        poTunerControlProxyObj = tuner_control_skeleton_new ();
    }

    return poTunerControlProxyObj;
}

/*******************************************************************************
 * Function:    poGetTunerInfoProxyObj
 * Description: Gets the Tuner control proxy object
 * Parameters:  void
 * Return:      TunerInfo*
 ******************************************************************************/

TunerInfo* poGetTunerInfoProxyObj()
{
    if(poTunerInfoProxyObj == NULL)
    {
        poTunerInfoProxyObj = tuner_info_skeleton_new ();
    }

    return poTunerInfoProxyObj;
}

/******************************************************************************
 * Function:    	bDoTunerSetup
 * Description:	    Performs the Tuner skeleton export and registrations.
 * Parameters:		void
 * Return:			bool
 *****************************************************************************/

bool bDoTunerSetup(GDBusConnection *pConnection)
{
    GError *error = NULL;
    poTunerControlProxyObj = poGetTunerControlProxyObj();
    if(NULL != poTunerControlProxyObj)
    {
        if(!g_dbus_interface_skeleton_export (
                    G_DBUS_INTERFACE_SKELETON (poTunerControlProxyObj),
                    pConnection,
                    "/com/bosch/AutomotiveProxy/Tuner/Control",
                    &error))
        {
            if(error != NULL)
            {
                DLT_LOG(AGW_framework,DLT_LOG_ERROR,
                        DLT_STRING("TunerControl"\
                                   "ERROR- Interface skeleton export failed -"),
                        DLT_STRING(error->message));

                g_clear_error (&error);
            }
            return FALSE;
        }
        else
        {
            DLT_LOG(AGW_framework,DLT_LOG_INFO,
                    DLT_STRING("TunerControl : - Object created & exported"));
        }



        /* signal connection for TunerControl*/
        g_signal_connect (poTunerControlProxyObj,
                          "handle-seek",
                          G_CALLBACK (handle_seek), NULL);

        g_signal_connect (poTunerControlProxyObj,
                          "handle-cancel-seek",
                          G_CALLBACK (handle_cancel_seek), NULL);

        g_signal_connect (poTunerControlProxyObj,
                          "handle-search-available-stations",
                          G_CALLBACK (handle_search_available_stations), NULL);
        g_signal_connect (poTunerControlProxyObj,
                          "handle-cancel-search-stations",
                          G_CALLBACK (handle_cancel_search_stations), NULL);

        g_signal_connect (poTunerControlProxyObj,
                          "handle-set-frequency",
                          G_CALLBACK (handle_set_frequency), NULL);

        g_signal_connect (poTunerControlProxyObj,
                          "handle-save-preset",
                          G_CALLBACK (handle_save_preset), NULL);

        g_signal_connect (poTunerControlProxyObj,
                          "handle-tune",
                          G_CALLBACK (handle_tune), NULL);

        g_signal_connect (poTunerControlProxyObj,
                          "handle-tune-to-program-id",
                          G_CALLBACK (handle_tune_to_program_id), NULL);

        g_signal_connect (poTunerControlProxyObj,
                          "handle-tune-to-preset",
                          G_CALLBACK (handle_tune_to_preset), NULL);
        g_signal_connect (poTunerControlProxyObj,
                          "handle-set-band",
                          G_CALLBACK (handle_set_band), NULL);
        g_signal_connect (poTunerControlProxyObj,
                          "handle-set-configuration-settings",
                          G_CALLBACK (handle_set_configuration_settings), NULL);
    }
    else{
        DLT_LOG(AGW_framework,DLT_LOG_INFO,DLT_STRING(
                    "TunerControl : - proxy Object is NULL"));
        return FALSE;
    }

    //Tuner Info
    poTunerInfoProxyObj = poGetTunerInfoProxyObj();
    if(NULL != poTunerInfoProxyObj)
    {
        if(!g_dbus_interface_skeleton_export (
                    G_DBUS_INTERFACE_SKELETON (poTunerInfoProxyObj),
                    pConnection,
                    "/com/bosch/AutomotiveProxy/Tuner/Info",
                    &error))
        {
            if(error != NULL)
            {
                DLT_LOG(AGW_framework,DLT_LOG_ERROR,
                        DLT_STRING("TunerInfo :"
                                   "ERROR- Interface skeleton export failed -"),
                        DLT_STRING(error->message));
                g_clear_error (&error);
            }
            return FALSE;
        }
        else
        {
            DLT_LOG(AGW_framework,DLT_LOG_INFO,
                    DLT_STRING("TunerInfo- Object created & exported"));
        }
        /* signal connection for TunerInfo*/
        g_signal_connect (poTunerInfoProxyObj,
                          "handle-get-station-list-for-band",
                          G_CALLBACK (handle_get_station_list_for_band), NULL);
    }
    else{
        DLT_LOG(AGW_framework,DLT_LOG_INFO,DLT_STRING(
                    "TunerInfo : - proxy Object is NULL"));
        return FALSE;
    }

    return TRUE;
}
#endif

#ifdef VARIANT_S_FTR_ENABLE_FEAT_GW_RNAIVI

/******************************************************************************
 * Function:    	poGetSettingsProviderProxy
 * Description:	Gets the settings provider proxy pointer.
 * Parameters:		void
 * Return:			SettingsProvider
 *****************************************************************************/
SettingsProvider*  poGetSettingsProviderProxy()
{
    if(poSettingsProvider == NULL)
    {
        poSettingsProvider = settings_provider_skeleton_new ();
    }

    return poSettingsProvider;
}

/******************************************************************************
 * Function:    	poGetSettingsProviderMultimediaStateProxy
 * Description:	Gets the settings provider proxy pointer.
 * Parameters:		void
 * Return:			SettingsProvider
 *****************************************************************************/
SettingsProviderMultimediaState*  poGetSettingsProviderMultimediaStateProxy()
{
    if(poMultimediaStateProxyObj == NULL)
    {
        poMultimediaStateProxyObj =
                settings_provider_multimedia_state_skeleton_new ();
    }

    return poMultimediaStateProxyObj;
}

/******************************************************************************
 * Function:    	bDoSettingsProviderDBusSetup
 * Description:	    Performs the settings provider skeleton export and
                    registrations.
 * Parameters:		GDBusConnection
 * Return:			bool
 *****************************************************************************/
bool bDoSettingsProviderDBusSetup(GDBusConnection *pConnection)
{
    GError *error = NULL;
    if (!g_dbus_interface_skeleton_export (
                G_DBUS_INTERFACE_SKELETON (
                    (SettingsProvider*)poGetSettingsProviderProxy()),
                pConnection,
                "/com/bosch/AutomotiveProxy/SettingsProvider",
                &error))
    {
        if(error != NULL)
        {
            DLT_LOG(AGW_framework,DLT_LOG_ERROR,
                    DLT_STRING("Settings Interface "
                               "skeleton export error:"),
                    DLT_STRING(error->message));

            g_clear_error (&error);
        }
        return FALSE;
    }

    /* method signal connection for Settings Provider */

    g_signal_connect ((SettingsProvider*)poGetSettingsProviderProxy(),
                      "handle-read",
                      G_CALLBACK (handle_read), NULL);

    g_signal_connect ((SettingsProvider*)poGetSettingsProviderProxy(),
                      "handle-update-subscriptions",
                      G_CALLBACK (handle_settings_update_subscriptions), NULL);

    g_signal_connect ((SettingsProvider*)poGetSettingsProviderProxy(),
                      "handle-write",
                      G_CALLBACK (handle_write), NULL);
                      
    g_signal_connect ((SettingsProvider*)poGetSettingsProviderProxy(),
                      "handle-get-available-audio-settings",
                      G_CALLBACK (handle_get_available_audio_settings), NULL);
    return TRUE;
}

/******************************************************************************
 * Function:    	vExportNaviInterface
 * Description:	    Performs the navi skeleton export and registrations.
 * Parameters:		void
 * Return:			void
 *****************************************************************************/

void vExportNaviInterface()
{
    DLT_LOG(AGW_framework,DLT_LOG_INFO,DLT_STRING("+vExportNaviInterface"));
    GDBusConnection* pConnection = poGetGDBusConnection();
    GError *error = NULL;

    if (!g_dbus_interface_skeleton_export (
                G_DBUS_INTERFACE_SKELETON (
                    (PositionProvider*)poNavGetPositionProxyObj()),
                pConnection,
                "/com/bosch/AutomotiveProxy/PositionProvider",
                &error))
    {
        if(error != NULL)
        {
            DLT_LOG(AGW_framework,DLT_LOG_ERROR,
                    DLT_STRING("NAVI :- ERROR-- "
                               "Interface skeleton export failed - "),
                    DLT_STRING(error->message));
            g_clear_error (&error);
        }
        return;
    }
    else
    {
        DLT_LOG(AGW_framework,DLT_LOG_INFO,
                DLT_STRING("NAVI POSITION PROVIDER :"
                           " - Object Created & Exported"));
    }

    //RouteList

    if (!g_dbus_interface_skeleton_export (
                G_DBUS_INTERFACE_SKELETON (
                    (NavigationRouteList*)poNavGetRouteListProxyObj()),
                pConnection,
                "/com/bosch/AutomotiveProxy/Navigation/RouteList",
                &error))
    {
        if(error != NULL)
        {
            DLT_LOG(AGW_framework, DLT_LOG_ERROR,
                    DLT_STRING("NAVIGATION - RouteList:"
                               "ERROR-- Interface skeleton export failed - "),
                    DLT_STRING(error->message));
            g_clear_error(&error);
        }
        return;
    }
    else
    {
        DLT_LOG(AGW_framework,DLT_LOG_INFO,
                DLT_STRING("NAVI  : - Object Created & Exported"));

        pNaviRouteObjManager =
                g_dbus_object_manager_server_new (
                    "/com/bosch/AutomotiveProxy/Navigation/Route");

        g_dbus_object_manager_server_set_connection (pNaviRouteObjManager,
                                                     pConnection);
    }

    //GeoCode
    if (!g_dbus_interface_skeleton_export (
                G_DBUS_INTERFACE_SKELETON (
                    (Geocode*)poNavGetGeocodeProxyObj()),
                pConnection,
                "/com/bosch/AutomotiveProxy/Geocode",
                &error))
    {
        DLT_LOG(AGW_framework,DLT_LOG_ERROR,
                DLT_STRING("NAVI -  GEOCODE :"
                           "ERROR-- Interface skeleton export failed - "),
                DLT_STRING(error->message));
        g_clear_error(&error);
        return;
    }
    else
    {
        DLT_LOG(AGW_framework,DLT_LOG_INFO,
                DLT_STRING("NAVI GEOCODE"
                           " : - Object Created & Exported"));

        pNaviGeoCodeObjManager =
                g_dbus_object_manager_server_new (
                    "/com/bosch/AutomotiveProxy/Geocode/Place");

        g_dbus_object_manager_server_set_connection (pNaviGeoCodeObjManager,
                                                     pConnection);

        pNaviGeoCodeLocationObjManager =
                g_dbus_object_manager_server_new (
                    "/com/bosch/AutomotiveProxy/Geocode/Place/Location");

        g_dbus_object_manager_server_set_connection (pNaviGeoCodeLocationObjManager,
                                                     pConnection);
    }


    //RouteGuidance
    if (!g_dbus_interface_skeleton_export (
                G_DBUS_INTERFACE_SKELETON (
                    (NavigationRouteGuidance*)poNavGetRouteGuidanceProxyObj()),
                pConnection,
                "/com/bosch/AutomotiveProxy/Navigation/RouteGuidance",
                &error))
    {
        DLT_LOG(AGW_framework,DLT_LOG_ERROR,
                DLT_STRING("NAVI - ROUTEGUIDANCE :- ERROR--  Interface "
                           "skeleton export failed:")
                ,DLT_STRING(error->message));

        g_clear_error(&error);
        return;
    }
    else
    {
        DLT_LOG(AGW_framework,DLT_LOG_INFO,
                DLT_STRING("NAVI ROUTE GUIDANCE : - Object "
                           "Created & Exported"));
    }

    //Navigation
    if (!g_dbus_interface_skeleton_export (
                G_DBUS_INTERFACE_SKELETON (
                    (Navigation*)poGetNavigationProxyObj()),
                pConnection,
                "/com/bosch/AutomotiveProxy/Navigation",
                &error))
    {
        if(error != NULL)
        {
            DLT_LOG(AGW_framework,DLT_LOG_ERROR,
                    DLT_STRING("NAVI :- ERROR-- "
                               "Interface skeleton export failed - "),
                    DLT_STRING(error->message));
            g_clear_error (&error);
        }
        return;
    }
    else
    {
        DLT_LOG(AGW_framework,DLT_LOG_INFO,
                DLT_STRING("NAVI RouteList"
                           " : - Object Created & Exported"));
    }

    /* signal connection for Navi*/
    g_signal_connect ((PositionProvider*)poNavGetPositionProxyObj(),
                      "handle-get-position",
                      G_CALLBACK (handle_get_position_method), NULL);
    g_signal_connect ((Geocode*)poNavGetGeocodeProxyObj(),
                      "handle-reverse-resolve",
                      G_CALLBACK (handle_reverse_resolve_method), NULL);
    g_signal_connect ((Geocode*)poNavGetGeocodeProxyObj(),
                      "handle-forward-resolve",
                      G_CALLBACK (handle_forward_resolve_method), NULL);

    g_signal_connect ((NavigationRouteGuidance*)poNavGetRouteGuidanceProxyObj(),
                      "handle-start-guidance",
                      G_CALLBACK (handle_start_guidance_method), NULL);

    g_signal_connect ((NavigationRouteGuidance*)poNavGetRouteGuidanceProxyObj(),
                      "handle-stop-guidance",
                      G_CALLBACK (handle_stop_guidance_method), NULL);

    g_signal_connect ((NavigationRouteGuidance*)poNavGetRouteGuidanceProxyObj(),
                      "handle-get-current-destination",
                      G_CALLBACK (handle_get_current_destination_method), NULL);

    g_signal_connect ((NavigationRouteList*)poNavGetRouteListProxyObj(),
                      "handle-insert-way-point",
                      G_CALLBACK (handle_insert_way_point_method), NULL);

    g_signal_connect ((NavigationRouteList*)poNavGetRouteListProxyObj(),
                      "handle-replace-way-point",
                      G_CALLBACK (handle_replace_way_point_method), NULL);

    g_signal_connect ((NavigationRouteList*)poNavGetRouteListProxyObj(),
                      "handle-delete-way-points",
                      G_CALLBACK (handle_delete_way_points_method), NULL);

    g_signal_connect ((Navigation*)poGetNavigationProxyObj(),
                      "handle-read-favorites-list",
                      G_CALLBACK (handle_read_favorites_list), NULL);

    g_signal_connect ((Navigation*)poGetNavigationProxyObj(),
                      "handle-add-to-favorites-list",
                      G_CALLBACK (handle_add_to_favorites_list), NULL);


    DLT_LOG(AGW_framework,DLT_LOG_INFO,DLT_STRING("-vExportNaviInterface"));
}

void vUnexportNaviInterface()
{
    DLT_LOG(AGW_framework,DLT_LOG_INFO,DLT_STRING("+vUnexportNaviInterface"));

    g_dbus_interface_skeleton_unexport(
                G_DBUS_INTERFACE_SKELETON (
                    (PositionProvider*)poNavGetPositionProxyObj()));

    g_dbus_interface_skeleton_unexport(
                G_DBUS_INTERFACE_SKELETON (
                    (NavigationRouteList*)poNavGetRouteListProxyObj()));

    g_dbus_interface_skeleton_unexport(
                G_DBUS_INTERFACE_SKELETON (
                    (Geocode*)poNavGetGeocodeProxyObj()));

    g_dbus_interface_skeleton_unexport(
                G_DBUS_INTERFACE_SKELETON (
                    (NavigationRouteGuidance*)poNavGetRouteGuidanceProxyObj()));

    g_dbus_interface_skeleton_unexport(
                G_DBUS_INTERFACE_SKELETON (
                    (Navigation*)poGetNavigationProxyObj()));

    DLT_LOG(AGW_framework,DLT_LOG_INFO,DLT_STRING("-vUnexportNaviInterface"));
}

/******************************************************************************
 * Function:    	vUnexportE2ENaviInterface
 * Description:	    Performs the AGW E2E navigation skeleton unexport.
 * Parameters:		void
 * Return:			void
 *****************************************************************************/
void vUnexportE2ENaviInterface()
{
    DLT_LOG(AGW_framework, DLT_LOG_INFO, DLT_STRING("+vUnexportE2ENaviInterface"));

    g_dbus_interface_skeleton_unexport(
                G_DBUS_INTERFACE_SKELETON(
                    (E2ENavigation*)poGetAGWE2ENavigationProxy()));

    DLT_LOG(AGW_framework,DLT_LOG_INFO,DLT_STRING("-vUnexportE2ENaviInterface"));
}

/******************************************************************************
 * Function:    	vExportE2ENaviInterface
 * Description:	    Performs the AGW E2E navigation skeleton export and
                    registrations.
 * Parameters:		void
 * Return:			bool
 *****************************************************************************/
bool vExportE2ENaviInterface()
{
    DLT_LOG(AGW_framework, DLT_LOG_INFO, DLT_STRING("E2E-NAVI: ")
            , DLT_STRING( __FUNCTION__));

    GDBusConnection* pConnection = poGetGDBusConnection();
    GError *error = NULL;
    if(pConnection)
    {
        if (!g_dbus_interface_skeleton_export (
                    G_DBUS_INTERFACE_SKELETON (
                        (E2ENavigation*)poGetAGWE2ENavigationProxy())
                    , pConnection
                    , "/com/bosch/AutomotiveProxy/E2ENavigation"
                    , &error))
        {
            if(error != NULL)
            {
                DLT_LOG(AGW_framework, DLT_LOG_ERROR
                        , DLT_STRING("E2ENavigation Interface skeleton export error:")
                        , DLT_STRING(error->message));
                g_clear_error (&error);
            }
            return FALSE;
        }
    }
    else
    {
        DLT_LOG(AGW_framework, DLT_LOG_ERROR
                , DLT_STRING("Error: poGetGDBusConnection returned NULL"));

        return FALSE;
    }

    /* method signal connection for E2E Navigation methods */
    g_signal_connect ((E2ENavigation*)poGetAGWE2ENavigationProxy(),
                      "handle-send-destination-to-car",
                      G_CALLBACK (handle_send_destination_to_car), NULL);

    DLT_LOG(AGW_framework, DLT_LOG_INFO
            , DLT_STRING("E2ENavigation g_signal_connect done"));
    return TRUE;
}

/******************************************************************************
 * Function:       poGetNaviRouteObjManager
 * Description:   Get GDBusObjectManagerServer pointer.
 * Parameters:
 * Return:         GDBusObjectManagerServer
 *****************************************************************************/
GDBusObjectManagerServer* poGetNaviRouteObjManager()
{
    DLT_LOG(AGW_framework,DLT_LOG_INFO,DLT_STRING(
                "NAVI : "),DLT_STRING( __FUNCTION__));
    return pNaviRouteObjManager;
}
/******************************************************************************
 * Function:       poGetNaviRouteObjManager
 * Description:   Get GDBusObjectManagerServer pointer.
 * Parameters:
 * Return:         GDBusObjectManagerServer
 *****************************************************************************/
GDBusObjectManagerServer* poGetGeoCodeObjManager()
{
    DLT_LOG(AGW_framework,DLT_LOG_INFO,DLT_STRING(
                "NAVI : "),DLT_STRING( __FUNCTION__));
    return pNaviGeoCodeObjManager;
}

/******************************************************************************
 * Function:      poGetGeoCodeLocationObjManager
 * Description:   Get GDBusObjectManagerServer pointer.
 * Parameters:
 * Return:        GDBusObjectManagerServer
 *****************************************************************************/
GDBusObjectManagerServer* poGetGeoCodeLocationObjManager()
{
    DLT_LOG(AGW_framework,DLT_LOG_INFO,DLT_STRING(
                "NAVI : "),DLT_STRING( __FUNCTION__));
    return pNaviGeoCodeLocationObjManager;
}

/******************************************************************************
 * Function:       connect_to_nav_bus_cb
 * Description:   call back function for onNaviServiceBusNameAppeared
 * Parameters:
 * Return:         void
 *****************************************************************************/
void
connect_to_nav_bus_cb (GObject    *source_object,
                       GAsyncResult *res,
                       gpointer user_data)
{
    DLT_LOG(AGW_framework,DLT_LOG_INFO,DLT_STRING(
                "NAVI: - "),DLT_STRING( __FUNCTION__));

    GError *l_poError;
    l_poError = NULL;
    GDBusProxy *poProxy = g_dbus_proxy_new_finish (res, &l_poError);
    if(poProxy)
    {
        vSetNaviProxy(poProxy);        
        bRegisterNaviProperties();
        vExportNaviInterface();
        vSetAGWNavigationAvailabilty();

        // Update guidance status in the start up explicitly.
        vUpdateRouteGuidanceStatus();

        DLT_LOG(AGW_framework,DLT_LOG_INFO,DLT_STRING(
                    "Valid proxy pointer "));
    }
    else
    {
        if(l_poError != NULL)
        {
            DLT_LOG(AGW_framework,DLT_LOG_ERROR,DLT_STRING(
                        "fatal:Invalid proxy pointer "));
            g_clear_error (&l_poError);
        }
    }
}

/******************************************************************************
 * Function:       connect_to_nav_bus_cb
 * Description:   call back function for positioninformation property
 * Parameters:
 * Return:         void
 *****************************************************************************/
void connect_to_e2e_nav_bus_cb (GObject *source_object, GAsyncResult *res,
                                gpointer user_data)
{
    DLT_LOG(AGW_framework,DLT_LOG_INFO,DLT_STRING(
                "NAVI: - "),DLT_STRING( __FUNCTION__));

    GError *l_poError;

    l_poError = NULL;
    GDBusProxy *poProxy = g_dbus_proxy_new_finish (res, &l_poError);
    if(poProxy)
    {
        vSetE2ENaviProxy(poProxy);
        bRegisterE2EConnectedServiceProperties();
        vExportE2ENaviInterface();
        DLT_LOG(AGW_framework ,DLT_LOG_INFO, DLT_STRING("Valid proxy pointer"));
    }
    else
    {
        if(l_poError != NULL)
        {
            DLT_LOG(AGW_framework, DLT_LOG_ERROR
                    , DLT_STRING("fatal:Invalid proxy pointer "));
            g_clear_error (&l_poError);
        }
    }
}

/******************************************************************************
 * Function:    	poGetCarConfigurationProxy
 * Description:	Gets the CarConfiguration provider proxy pointer.
 * Parameters:		void
 * Return:			CarConfiguration
 *****************************************************************************/
CarConfiguration*  poGetCarConfigurationProxy()
{
    if(poCarConfiguration == NULL)
    {
        poCarConfiguration = car_configuration_skeleton_new ();
    }

    return poCarConfiguration;
}

/******************************************************************************
 * Function:    	bDoCarConfigurationDBusSetup
 * Description:	    Performs the car configuration skeleton export and
                    registrations.
 * Parameters:		void
 * Return:			bool
 *****************************************************************************/
bool bDoCarConfigurationDBusSetup(GDBusConnection *pConnection)
{
    GError *error = NULL;
    poCarConfiguration = poGetCarConfigurationProxy();
    if(NULL != poCarConfiguration)
    {
        if (!g_dbus_interface_skeleton_export (
                    G_DBUS_INTERFACE_SKELETON (
                        poCarConfiguration),
                    pConnection,
                    "/com/bosch/AutomotiveProxy/CarConfiguration",
                    &error))
        {
            if(error != NULL)
            {
                DLT_LOG(AGW_framework,DLT_LOG_ERROR,DLT_STRING(
                            "CarConfiguration Interface skeleton export error"),
                        DLT_STRING(error->message));
                g_clear_error (&error);
            }
            return FALSE;
        }

        /* method signal connection for CarConfiguration Provider */
        g_signal_connect (poCarConfiguration,
                          "handle-get-configuration-value",
                          G_CALLBACK (handle_get_configuration_value), NULL);
    }
    else{
        DLT_LOG(AGW_framework,DLT_LOG_INFO,DLT_STRING(
                    "CarConfig : - proxy Object is NULL"));
        return FALSE;
    }
    return TRUE;
}

#endif //VARIANT_S_FTR_ENABLE_FEAT_GW_RNAIVI

#ifdef VARIANT_S_FTR_ENABLE_FEAT_GW_PSARCC


/*********************************************************************************************
 * Function:    poGetDataNotifierProxyObj
 * Description: Gets the Data notifier proxy object
 * Parameters:  void
 * Return:      DataNotifier*
 ******************************************************************************/
DataNotifier* poGetDataNotifierProxyObj()
{
    if(poDataNotifierProxyObj == NULL)
    {
        poDataNotifierProxyObj = data_notifier_skeleton_new ();
        DLT_REGISTER_CONTEXT(AGW_DataNotifier,"GWDN","Data Notifier "
                             "context for DLT Logging");
    }

    return poDataNotifierProxyObj;
}

/******************************************************************************
 * Function:    	bDoDataNotifierSetup
 * Description:	    Performs the Data Notifier skeleton export and registrations.
 * Parameters:		void
 * Return:			bool
 *****************************************************************************/

bool bDoDataNotifierSetup(GDBusConnection *pConnection)
{
   bool l_bConnectionStatus = connectToAudioComp();
    if(TRUE == l_bConnectionStatus)
    {
        DLT_LOG(AGW_framework,DLT_LOG_INFO,
                DLT_STRING("connection to Genivi "
                "audio component is successfull"));
    }

    GError *error = NULL;
    poDataNotifierProxyObj = poGetDataNotifierProxyObj();
    if(NULL != poDataNotifierProxyObj)
    {
        if(!g_dbus_interface_skeleton_export (
                    G_DBUS_INTERFACE_SKELETON (poDataNotifierProxyObj),
                    pConnection,
                    "/com/bosch/AutomotiveProxy/DataNotifier",
                    &error))
        {
            if(error != NULL)
            {
                DLT_LOG(AGW_framework,DLT_LOG_ERROR,
                        DLT_STRING("DataNotifier : ERROR--"
                                   "Interface skeleton export failed - "),
                        DLT_STRING(error->message));

                g_clear_error (&error);
            }
            return FALSE;
        }
        else
        {
            DLT_LOG(AGW_framework,DLT_LOG_INFO,
                    DLT_STRING("DataNotifier :"
                               " - Object created & exported"));
        }

        /* signal connection for DataNotifier*/

        g_signal_connect (poDataNotifierProxyObj,
                          "handle-get-data",
                          G_CALLBACK (handle_get_data), NULL);

        g_signal_connect (poDataNotifierProxyObj,
                          "handle-set-data",
                          G_CALLBACK (handle_set_data), NULL);

        g_signal_connect (poDataNotifierProxyObj,
                          "handle-subscribe-for-data",
                          G_CALLBACK (handle_subscribe_for_data), NULL);
        return TRUE;
    }
    else{
        return FALSE;
    }

}
/*******************************************************************************/
// Only for PSA projects
gint iGetSubscriptionWaitTime()
{
    return iSubscription_Wait_Time;
}
#endif //VARIANT_S_FTR_ENABLE_FEAT_GW_PSARCC

#if ENABLE_CSM_IN_GATEWAY


/*********************************************************************************************
 * Function:   poGetVehicleInfoManagerProxyObj
 * Description: Gets the VehicleInfoManager proxy object
 * Parameters:  void
 * Return:      void
 *****************************************************************************/

VehicleInfoManager* poGetVehicleInfoManagerProxyObj()
{
    if(pVehicleInfoManagerObj == NULL)
    {
        pVehicleInfoManagerObj = vehicle_info_manager_skeleton_new ();
    }

    return pVehicleInfoManagerObj;
}

/******************************************************************************
 * Function:    	bDoVIMSetup
 * Description:	    Performs the VIM skeleton export and registrations.
 * Parameters:		void
 * Return:			bool
 *****************************************************************************/

bool bDoVIMSetup(GDBusConnection *pConnection)
{
    //VIM
    DLT_REGISTER_CONTEXT(AGW_VehicleData,"GWVD","Vehicle Data"
                         "context for DLT Logging");

#ifdef VARIANT_S_FTR_ENABLE_FEAT_GW_RNAIVI
    bool bReadSuccess = bReadConfigFile();
    if(true == bReadSuccess)
    {
#endif
        GError *error = NULL;
        pVehicleInfoManagerObj = poGetVehicleInfoManagerProxyObj();
        if(NULL != pVehicleInfoManagerObj)
        {


            if(!g_dbus_interface_skeleton_export (
                        G_DBUS_INTERFACE_SKELETON (pVehicleInfoManagerObj),
                        pConnection,
                        "/com/bosch/AutomotiveProxy/VehicleInfoManager",
                        &error))
            {
                if(error != NULL)
                {
                    DLT_LOG(AGW_framework,DLT_LOG_ERROR,
                            DLT_STRING("VIM : - "
                                       "ERROR-- Interface skeleton export failed"),
                            DLT_STRING(error->message));
                    g_clear_error (&error);
                }
                return FALSE;
            }
            else
            {
                DLT_LOG(AGW_framework,DLT_LOG_INFO,
                        DLT_STRING("VIM : - Object created & exported"));
            }



            /* signal connection for VIM*/
            g_signal_connect (pVehicleInfoManagerObj,"handle-get-vehicle-info",
                              G_CALLBACK (handle_get_vehicle_info), NULL);

            g_signal_connect (pVehicleInfoManagerObj, "handle-update-subscriptions",
                              G_CALLBACK (handle_update_subscriptions), NULL);

            g_signal_connect (pVehicleInfoManagerObj,
                              "handle-get-vehicle-data-by-name",
                              G_CALLBACK (handle_get_vehicle_data_by_name), NULL);
        }
        else{
            DLT_LOG(AGW_VehicleData,DLT_LOG_INFO,DLT_STRING(
                        "VehicleInfoManager : - proxy Object is NULL"));
            return FALSE;
        }
#ifdef VARIANT_S_FTR_ENABLE_FEAT_GW_RNAIVI
    }
    else
    {
        DLT_LOG(AGW_VehicleData, DLT_LOG_ERROR, DLT_STRING(
                    "Error: Reading vehicle signals conf file failed"));
        return FALSE;
    }
#endif

    return TRUE;
}
#endif //ENABLE_CSM_IN_GATEWAY


/******************************************************************************
 * Function:    poGetGDBusConnection
 * Description: Gets the GDBus session connection pointer
 * Parameters:  void
 * Return:      GDBusConnection*
 *****************************************************************************/
GDBusConnection* poGetGDBusConnection()
{

    if(poBusConn)
        return poBusConn;

    return NULL;

}

/*****************************************************************************
 * Function:    poGetSystemBusConnection
 * Description: stores the connection pointer for future use.
 * Parameters:  void
 * Return:      void
 *****************************************************************************/
GDBusConnection* poGetSystemBusConnection()
{
    if(poSystemBusConn)
        return poSystemBusConn;

    return NULL;
}
/*****************************************************************************
 * Function:    poGetSessionBusConnection
 * Description: stores the session connection pointer for future use.
 * Parameters:  void
 * Return:      void
 *****************************************************************************/
GDBusConnection* poGetSessionBusConnection()
{
    if(poSessionBusConn)
        return poSessionBusConn;

    return NULL;
}

/******************************************************************************
 * Function:    	bDoComponentsSetup
 * Description:	    Performs all components setup.
 * Parameters:		void
 * Return:			bool
 *****************************************************************************/
bool bDoComponentsSetup(GDBusConnection *pConnection)
{
    bool bRet = FALSE;
    if(!pConnection){
        DLT_LOG(AGW_framework,DLT_LOG_ERROR,DLT_STRING("pConnection is NULL"));
        return bRet;
    }


    bRet = bDoLCMDBusServerSetup(pConnection);
    if(!bRet)
    {
        DLT_LOG(AGW_framework,DLT_LOG_ERROR,DLT_STRING(
                    "ERROR- LCM Interface skeleton export failed"));
        //return FALSE;
    }
#ifndef VARIANT_S_FTR_ENABLE_FEAT_GW_PSARCC
    bRet = bDoAudioSetup(pConnection);
    if(!bRet)
    {
        DLT_LOG(AGW_framework,DLT_LOG_ERROR,DLT_STRING(
                    "Audio Interface skeleton export failed"));
        //return FALSE;
    }


    bRet = bDoUriSetup(pConnection);
    if(!bRet)
    {
        DLT_LOG(AGW_framework,DLT_LOG_ERROR,DLT_STRING(
                    "URI Interface skeleton export failed"));
        //return FALSE;
    }

    bRet = bDoSrcMngrDBusSetup (pConnection);
    if(!bRet)
    {
        DLT_LOG(AGW_framework,DLT_LOG_ERROR,DLT_STRING(
                    "SourceManager : ERROR--"
                    "source manager Interface skeleton export failed"));
        //return FALSE;
    }

    initialise_bluetooth_manager(pConnection, NULL);

    bRet = bDoTunerSetup(pConnection);
    if(!bRet)
    {
        DLT_LOG(AGW_framework,DLT_LOG_ERROR,
                DLT_STRING("Tuner :"
                           " - ERROR- Tuner Interface skeleton export failed"));
        //return FALSE;
    }
#endif

#if defined(VARIANT_COMPILER_ENV_GEN3ARM) || defined(VARIANT_COMPILER_ENV_GEN4RCAR)
    bRet = bInitializeSDC();
    if(!bRet)
    {
        DLT_LOG(AGW_framework,DLT_LOG_ERROR,DLT_STRING(
                    "SDC Initialization failed export failed"));
        //return FALSE;
    }
#endif//VARIANT_COMPILER_ENV_GEN3ARM || VARIANT_COMPILER_ENV_GEN4RCAR

#ifdef VARIANT_S_FTR_ENABLE_FEAT_GW_PSARCC

    bRet = bDoDataNotifierSetup(pConnection);
    if(!bRet)
    {
        DLT_LOG(AGW_framework,DLT_LOG_ERROR,DLT_STRING(
                    "ERROR- DataNotifier Interface skeleton export failed"));
        //return FALSE;
    }


#endif //VARIANT_S_FTR_ENABLE_FEAT_GW_PSARCC

#if ENABLE_CSM_IN_GATEWAY
    bRet = bDoVIMSetup(pConnection);
    if(!bRet)
    {
        DLT_LOG(AGW_framework,DLT_LOG_ERROR,DLT_STRING(
                    "VIM Interface skeleton export failed"));
        //return FALSE;
    }

#ifdef VARIANT_S_FTR_ENABLE_FEAT_GW_PSARCC
    bStartVehicleFunctionClient();
#endif
#endif

#ifdef VARIANT_S_FTR_ENABLE_FEAT_GW_RNAIVI
    bRet = bDoCarConfigurationDBusSetup(pConnection);
    if(!bRet)
    {
        DLT_LOG(AGW_framework,DLT_LOG_ERROR,
                DLT_STRING("CarConfiguration :- ERROR"
                           "CarConfiguration Interface skeleton export failed"));
        //return FALSE;
    }

    bRet = bDoSettingsProviderDBusSetup(pConnection);
    if(!bRet)
    {
        DLT_LOG(AGW_framework,DLT_LOG_ERROR,
                DLT_STRING("Settings : - ERROR"
                           "Settings Interface skeleton export failed"));
        //return FALSE;
    }

    bRet = bInitializeNavi();
    if(!bRet)
    {
        DLT_LOG(AGW_framework,DLT_LOG_ERROR,
                DLT_STRING("Tuner :"
                           " - ERROR- Navi init failed"));
        //return FALSE;
    }

   bRet = bInitializeE2ENavi();
   if(!bRet)
   {
      DLT_LOG(AGW_framework, DLT_LOG_ERROR
              , DLT_STRING("E2ENavigation : - ERROR- E2ENavigation Interface skeleton export failed"));
      //return FALSE;
   }
    bool bResult = bStartProfileManagerClient();
    if(!bResult)//This is just an info. D-Bus will catch up once i/f is up.
    {
        DLT_LOG(AGW_framework,DLT_LOG_ERROR,DLT_STRING(
                    "failed to start personalization"));
    }

	
	bRet = bInitializePDE(pConnection);
    if(!bRet)
    {
        DLT_LOG(AGW_framework,DLT_LOG_ERROR,DLT_STRING(
                    "PDE Initialization failed export failed"));
        //return FALSE;
    }


    bRet = bExportWebMplayerAutomInterface(pConnection);
    if(!bRet)
    {
        DLT_LOG(AGW_framework,DLT_LOG_ERROR,DLT_STRING(
                    "MetaData Autom domain setup failed"));
        //return FALSE;
    }

	
#endif  //VARIANT_S_FTR_ENABLE_FEAT_GW_RNAIVI


#ifdef VARIANT_COMPILER_ENV_GEN4RCAR
    bRet = bStartFcOtaClient(pConnection);
    if(!bRet)
    {
        DLT_LOG(AGW_framework,DLT_LOG_ERROR,
                DLT_STRING("FOTA :"
                           "failed to start FcOTA"));
    }
#endif

#ifdef VARIANT_GEN3ARM_RNAIVI
    bRet = bStartFcOtaClient();
    if(!bRet)
    {
        DLT_LOG(AGW_framework,DLT_LOG_ERROR,
                DLT_STRING("FOTA :"
                           "failed to start FcOTA"));
    }
#endif


    bRet = bStartManAppService();
    if(!bRet)
    {
        DLT_LOG(AGW_framework,DLT_LOG_ERROR,DLT_STRING(__FUNCTION__),
                DLT_STRING("(): Failed starting ManApp service"));
        //return FALSE;
    }
#ifdef ENABLE_NON_CMD_IN_GATEWAY
#if defined(VARIANT_S_FTR_ENABLE_FEAT_GW_RNAIVI) || defined(VARIANT_S_FTR_ENABLE_FEAT_GW_PSARCC)
    bRet = bDoFactoryResetDBusSetup(pConnection);
    if(!bRet)
    {
        DLT_LOG(AGW_framework,DLT_LOG_INFO,
                DLT_STRING("FactoryReset :- ERROR- FactoryReset Interface "
                           "skeleton export failed"));

        //return FALSE;
    }
#endif
    bRet = bInitializeECM();
    if(!bRet)
    {
        DLT_LOG(AGW_framework,DLT_LOG_ERROR,DLT_STRING(
                    "ECM Initialization failed export failed"));
        //return FALSE;
    }
#endif //ENABLE_NON_CMD_IN_GATEWAY
  
    bRet = bInitializeSP();
    if(!bRet)
    {
        DLT_LOG(AGW_framework,DLT_LOG_ERROR,DLT_STRING(__FUNCTION__),
                 DLT_STRING("(): initializeSeamlessPairing failed"));
            //return FALSE;
    }

    bRet = initializeConfigMgmt();
    if(!bRet)
    {
        DLT_LOG(AGW_framework,DLT_LOG_ERROR,DLT_STRING(__FUNCTION__),
                DLT_STRING("():	initializeConfigMgmt failed"));	
    //return FALSE;
    }	 
    malloc_trim(0);
    return TRUE;
}

/*****************************************************************************
 * Function:    checkMsgOverflow
 * Description: Runs periodically for every second. Checks
                gateway incoming requests and blocks the
                gateway requests when there is overflow
 * Parameters:  user_data
 * Return:      gboolean
 *****************************************************************************/
gboolean checkMsgOverflow(gpointer user_data)
{
    //   gint limit = readIncomingLimit();
    if(igatewayMsgLimitPerSec <= 0)
    {
        //DLT_LOG(AGW_framework, DLT_LOG_ERROR, DLT_STRING(__FUNCTION__),
        //  DLT_STRING("():Failed reading config value on Limit"));
    }
    else if(igatewayMsgCounter >= (guint)igatewayMsgLimitPerSec)
    {
        DLT_LOG(AGW_framework, DLT_LOG_ERROR, DLT_STRING(__FUNCTION__),
                DLT_STRING("(): Crossed msg limit per second, msg count="),
                DLT_UINT(igatewayMsgCounter),
                DLT_STRING("Blocking all incoming requests"));

        g_blockGatewayIncoming = TRUE;
        return FALSE;
    }
    // Reset the Msg counter for next cycle
    igatewayMsgCounter = 0;
    return TRUE;
}

/*****************************************************************************
 * Function:    message_path_exception
 * Description: This function checks if the message is exception
                for absolute path
 * Parameters:  message, pathStr
 * Return:      TRUE -  if exception
                FALSE - if no exception
 *****************************************************************************/
gboolean message_path_exception(GDBusMessage* message,
                                gchar*        pathStr)
{
    if(message == NULL || pathStr==NULL)
    {
        DLT_LOG(AGW_framework,DLT_LOG_ERROR, DLT_STRING(__func__),
                DLT_STRING("(): Null input"));
        return FALSE;
    }
    gchar* interface = strdup(g_dbus_message_get_interface (message));

    if(interface == NULL)
    {
        DLT_LOG(AGW_framework,DLT_LOG_ERROR, DLT_STRING(__func__),
                DLT_STRING("Null interface"));
        return FALSE;
    }

    if((!strcmp(interface, "com.bosch.AutomotiveProxy.VehicleInfoManager"))
         && (!strcmp(pathStr, "/")) )
    {
        DLT_LOG(AGW_framework,DLT_LOG_ERROR, DLT_STRING(__func__),
                DLT_STRING("(): VehicleData relative path exception"));
        free(interface);
        return TRUE;
    }
    else if( (!strcmp(interface, "com.bosch.AutomotiveProxy.SeamlessPairingServer")) ||
             (!strcmp(interface, "com.bosch.AutomotiveProxy.WebMPlayer")) ||
             (!strcmp(interface, "com.bosch.CEProxy.WebMPlayer")) )
    {
        DLT_LOG(AGW_framework,DLT_LOG_ERROR, DLT_STRING(__func__),
                DLT_STRING("(): Absolute path exception"));
        free(interface);
        return TRUE;
    }
    // Add any exceptions found in future here
    free(interface);
    return FALSE;
}

/*****************************************************************************
 * Function:    message_path_detected
 * Description: This function loops through all string arguments in
                the msg_body and checks if it contains file paths
 * Parameters:  msg_body
 * Return:      TRUE -  if file path found
                FALSE - if not found any file path
 *****************************************************************************/
gboolean message_path_detected( GVariant*       msg_body,
                                GDBusMessage    *message)
{
    GVariantIter iter;
    GVariant *item = NULL;
    struct stat  statbuf;

    if(msg_body == NULL)
    {
        DLT_LOG(AGW_framework,DLT_LOG_INFO,DLT_STRING("Null msg receieved"));
        return FALSE;
    }

    const GVariantType *msg_type = g_variant_get_type(msg_body);

    if(g_variant_type_equal (msg_type, G_VARIANT_TYPE_STRING))
    {
        gchar *arg_value = NULL;
        gsize len;
        arg_value = strdup(g_variant_get_string (msg_body, &len));
        trim_leading_space(arg_value);

        if((stat(arg_value, &statbuf) != -1)
                && (!message_path_exception(message, arg_value)) )
        {
            DLT_LOG(AGW_framework,DLT_LOG_INFO,
                    DLT_STRING("Absolute path found:"),
                    DLT_STRING(arg_value));
            free(arg_value);
            return TRUE;
        }
        free(arg_value);
    }
    else if(g_variant_type_equal(msg_type, G_VARIANT_TYPE_BYTESTRING))
    {
        gchar* arg_value = NULL;
        arg_value = strdup(g_variant_get_bytestring(msg_body));

        trim_leading_space(arg_value);

        if((stat(arg_value, &statbuf) != -1)
                && (!message_path_exception(message, arg_value)) )
        {
            DLT_LOG(AGW_framework,DLT_LOG_INFO,
                    DLT_STRING("Absolute path found: "),
                    DLT_STRING(arg_value));
            free(arg_value);
            return TRUE;
        }
        free(arg_value);
    }

    else if(g_variant_type_is_array (msg_type))
    {
        if (g_variant_n_children (msg_body) != 0)
        {
            g_variant_iter_init (&iter, msg_body);
            while ((item = g_variant_iter_next_value (&iter)) != NULL)
            {
                if(message_path_detected(item, message) == TRUE)
                {
					g_variant_unref (item);
                    return TRUE;
                }
				g_variant_unref (item);
            }
        }
    }

    else if (g_variant_type_is_dict_entry (msg_type)
             || g_variant_type_is_tuple (msg_type))
    {
        g_variant_iter_init (&iter, msg_body);
        while ((item = g_variant_iter_next_value (&iter)) != NULL)
        {
            if(message_path_detected(item, message) == TRUE)
            {
				g_variant_unref (item);
                return TRUE;
            }
			g_variant_unref (item);
        }
    }

    return FALSE;
}
/****************************************************************************
 * Function:    gateway_message_filter
 * Description: Filter function that tracks all dbus messages
                through gateway connection
 * Parameters:  Connection , message, incoming/outgoing flag, user_data
 * Return:      GDBusMessage pointer
 *****************************************************************************/
static GDBusMessage *
gateway_message_filter ( GDBusConnection *connection,
                         GDBusMessage    *message,
                         gboolean         incoming,
                         gpointer         user_data)
{
    GDBusMessage *reply = NULL;
    if( (incoming) && (message != NULL) )
    {
        if(g_blockGatewayIncoming == TRUE)
        {
            reply = g_dbus_message_new_method_error (
                        message,
                        "com.bosch.AutomotiveProxy.Error",
                        "Gateway communication disabled due to overflow ",
                        g_dbus_message_get_path (message));

            g_dbus_connection_send_message (connection,
                                            reply,
                                            G_DBUS_SEND_MESSAGE_FLAGS_NONE,
                                            NULL, NULL);

            message = NULL;
			g_object_unref(reply);
        }
        else
        {
            GDBusMessage *msg_cpy = NULL;
            GVariant *msg_body = NULL;

            igatewayMsgCounter++;
            msg_cpy = g_dbus_message_copy(message, NULL);
            if(msg_cpy != NULL)
            {
                if((msg_body = g_dbus_message_get_body(msg_cpy))!= NULL)
                {
                    if(message_path_detected(msg_body, msg_cpy) == TRUE)
                    {

                        DLT_LOG(AGW_framework,DLT_LOG_ERROR,
                                DLT_STRING(__FUNCTION__),
                                DLT_STRING("Absolute paths not allowed,"\
                                           "dropping req"));

                        reply = g_dbus_message_new_method_error (
                                    message,
                                    "com.bosch.AutomotiveProxy.Error",
                                    "Absolute or relative filepaths not allowed",
                                    g_dbus_message_get_path (message) );

                        g_dbus_connection_send_message (
                                    connection,
                                    reply,
                                    G_DBUS_SEND_MESSAGE_FLAGS_NONE,
                                    NULL, NULL );
                        message = NULL;
						g_object_unref(reply);
                    }
                }
            }
            else
                DLT_LOG(AGW_framework, DLT_LOG_ERROR,
                        DLT_STRING(__FUNCTION__),
                        DLT_STRING("(): g_dbus_message_copy failed"));

            g_object_unref(msg_cpy);

        } //end of else (g_blockgatewayIncoming)
    } // end of if(incoming && message!=NULL)
    return message;
}

/*****************************************************************************
 * Function:    on_bus_acquired
 * Description: Callback on D-Bus acquired
 * Parameters:  Connection , Bus name, user data
 * Return:      void
 *****************************************************************************/
static void
on_bus_acquired (GDBusConnection *pConnection, const gchar *pName,
                 gpointer pUserData)
{

    DLT_LOG(AGW_framework,DLT_LOG_ERROR,DLT_STRING(
                "GatewayDBusServer : - Bus acquired - "),DLT_STRING( pName));

    poSessionBusConn = g_object_ref(pConnection);
    
    //Initialize config data, the only session bus interface in AGW
    if(!initializeConfigData())
    {
        DLT_LOG(AGW_framework,DLT_LOG_ERROR,DLT_STRING(__FUNCTION__),
                DLT_STRING("(): initializeConfigData failed"));
    }
#ifdef VARIANT_S_FTR_ENABLE_FEAT_GW_RNAIVI

    if(!bExportWebMPlayerCEInterface(pConnection))
    {
        DLT_LOG(AGW_framework,DLT_LOG_ERROR,DLT_STRING(
                    "MetaData setup failed"));
    }
#endif
    if(DBUS_TYPE_SESSION == uBusType)
    {
        poBusConn = poSessionBusConn;
        bool bRet = bDoComponentsSetup(pConnection);
        if(!bRet)
        {
            DLT_LOG(AGW_framework,DLT_LOG_ERROR,DLT_STRING(
                        "Unable to export a compoent interface"));
        }
        //Check gateway incoming requests overflow periodically
        if(!check_limit_timeout_id)
            check_limit_timeout_id = g_timeout_add(1000,
                                                   checkMsgOverflow,
                                                   NULL);

        //Add filter function for gateway message tracking
        msg_filter_id = g_dbus_connection_add_filter( pConnection,
                                                      gateway_message_filter,
                                                      NULL,
                                                      NULL );


#ifdef  ENABLE_CGROUP_IMPL
        read_memory_threshold();
#endif
    }

    DLT_LOG(AGW_framework,DLT_LOG_INFO,DLT_STRING(
                "GatewayDBusServer : - : Bus acquired "),DLT_STRING( pName));
}

/*******************************************************************************
 * Function:    on_name_acquired
 * Description: Callback on DBus name acquired
 * Parameters:  Connection , Bus name, user data
 * Return:      void
 *****************************************************************************/
static void
on_name_acquired (GDBusConnection *pConnection, const gchar *pName,
                  gpointer pUserData)
{
    UNUSED(pUserData);
    UNUSED(pConnection);

    DLT_LOG(AGW_framework,DLT_LOG_INFO,DLT_STRING(
                "GatewayDBusServer : -  name acquired"),DLT_STRING(pName));
}

/******************************************************************************
 * Function:    on_name_lost
 * Description: Callback on D-Bus name lost
 * Parameters:  Connection , Bus name, user data
 * Return:      void
 *****************************************************************************/
static void
on_name_lost (GDBusConnection *pConnection, const gchar *pName,
              gpointer pUserData)
{
    UNUSED(pConnection);
    UNUSED(pUserData);
    DLT_LOG(AGW_framework,DLT_LOG_INFO,DLT_STRING(
                "GatewayDBusServer : -   name lost"),DLT_STRING(pName));
    vCleanUpProxyOnShutDown();
    if(poBusConn)
    {
        if(msg_filter_id)
            g_dbus_connection_remove_filter(poBusConn, msg_filter_id);
        g_object_unref(poBusConn);
        poBusConn = NULL;
    }

    if(poSessionBusConn)
    {
        g_object_unref(poSessionBusConn);
        poSessionBusConn = NULL;
    }
}

//LCM needs interface to be exposed on systembus
static void
on_system_bus_acquired (GDBusConnection *pConnection, const gchar *pName,
                        gpointer pUserData)
{
    UNUSED(pConnection);
    UNUSED(pUserData);
    UNUSED(pName);
    DLT_LOG(AGW_framework,DLT_LOG_ERROR,
            DLT_STRING("+-on_system_bus_acquired"));

    poSystemBusConn = g_object_ref(pConnection);
}

static void
on_system_name_acquired (GDBusConnection *pConnection, const gchar *pName,
                         gpointer pUserData)
{
    UNUSED(pConnection);
    UNUSED(pUserData);
    UNUSED(pName);
    DLT_LOG(AGW_framework,DLT_LOG_ERROR,
            DLT_STRING("+-on_system_name_acquired"));
}


static void
on_system_name_lost (GDBusConnection *pConnection, const gchar *pName,
                     gpointer pUserData)
{
    UNUSED(pConnection);
    UNUSED(pUserData);
    UNUSED(pName);

    DLT_LOG(AGW_framework,DLT_LOG_ERROR,DLT_STRING("+-on_system_name_lost"));
    vDoLCMDBusServerTearDown();
    g_object_unref(poSystemBusConn);
    poSystemBusConn = NULL;
}

/*******************************************************************************
 * Function:    privateBus_name_acquired_handler
 * Description: Callback on DBus name acquired
 * Parameters:  Connection , Bus name, user data
 * Return:      void
 *****************************************************************************/
static void
privateBus_name_acquired_handler (GDBusConnection *pConnection,
                                  const gchar *pName,
                                  gpointer pUserData)
{
    DLT_LOG(AGW_framework,DLT_LOG_INFO,DLT_STRING(
                "Entered "),DLT_STRING( __FUNCTION__));
    bool bRet = bDoComponentsSetup(pConnection);
    if(!bRet)
    {
        DLT_LOG(AGW_framework,DLT_LOG_ERROR,DLT_STRING(
                    "Unable to export a compoent interface"));
    }
    //Check gateway incoming requests overflow periodically
    if(!check_limit_timeout_id)
        check_limit_timeout_id = g_timeout_add(1000, checkMsgOverflow, NULL);

    //Add filter function for gateway message tracking
    msg_filter_id = g_dbus_connection_add_filter(pConnection,
                                                 gateway_message_filter,
                                                 NULL, NULL);

#ifdef  ENABLE_CGROUP_IMPL
    read_memory_threshold();
#endif

    DLT_LOG(AGW_framework,DLT_LOG_INFO,DLT_STRING(
                "privateBus_name_acquired_handler : - : Bus acquired "),
            DLT_STRING( pName));

    DLT_LOG(AGW_framework,DLT_LOG_INFO,DLT_STRING(
                "Exited "),DLT_STRING( __FUNCTION__));
}

/******************************************************************************
 * Function:    privateBus_name_lost_handler
 * Description: Callback on D-Bus name lost
 * Parameters:  Connection , Bus name, user data
 * Return:      void
 *****************************************************************************/
static void
privateBus_name_lost_handler (GDBusConnection *pConnection, const gchar *pName,
                              gpointer pUserData)
{
    DLT_LOG(AGW_framework,DLT_LOG_INFO,DLT_STRING(
                "Entered "),DLT_STRING( __FUNCTION__));
    DLT_LOG(AGW_framework,DLT_LOG_INFO,DLT_STRING(
                "privateBus_name_lost_handler: -   name lost"),
            DLT_STRING(pName));

    vCleanUpProxyOnShutDown();
    if(poBusConn)
    {
        if(msg_filter_id)
            g_dbus_connection_remove_filter(poBusConn, msg_filter_id);
        g_object_unref(poBusConn);
        poBusConn = NULL;
    }
    DLT_LOG(AGW_framework,DLT_LOG_INFO,DLT_STRING(
                "Exited "),DLT_STRING( __FUNCTION__));
}


static void
iddbus_connection_callback (GObject *source_object,
                            GAsyncResult *res,
                            gpointer user_data)
{

    DLT_LOG(AGW_framework,DLT_LOG_INFO,DLT_STRING(
                "Entered "),DLT_STRING( __FUNCTION__));
    GError *error = NULL;
    poBusConn = g_dbus_connection_new_finish (res, &error);
    if(error != NULL)
    {
        DLT_LOG(AGW_framework,DLT_LOG_INFO,DLT_STRING(
                    "Error in iddbus_connection_callback msg: "),
                DLT_STRING(error->message));

        g_clear_error (&error);
    }
    if(poBusConn)
    {

        iIDDBusWatcherID = g_bus_own_name_on_connection (
                    poBusConn,
                    "com.bosch.AutomotiveProxy",
                    G_BUS_NAME_OWNER_FLAGS_NONE,
                    privateBus_name_acquired_handler,
                    privateBus_name_lost_handler,
                    NULL,
                    NULL);
    }
    else{

        DLT_LOG(AGW_framework,DLT_LOG_ERROR,
                DLT_STRING("***Error:GDbus connection pointer is NULL*** "));
    }

    DLT_LOG(AGW_framework,DLT_LOG_INFO,
            DLT_STRING("GatewayDBusServer :-iddbus_connection_callback"));
}

static void g_log_message_handler (const gchar    *log_domain
                                   , GLogLevelFlags  log_level
                                   , const gchar    *msg
                                   , gpointer        user_data)
{
    DLT_LOG(AGW_framework, DLT_LOG_INFO ,DLT_STRING("LOG LEVEL: ")
            , DLT_INT(log_level));
    DLT_LOG(AGW_framework, DLT_LOG_INFO ,DLT_STRING("LOG DOMAIN: ")
            , DLT_STRING(log_domain));

    if(G_LOG_LEVEL_CRITICAL == log_level)
        DLT_LOG(AGW_framework, DLT_LOG_ERROR,
                DLT_STRING("AGW GLIB LOG MESSAGE ***** CRITICAL:"),
                DLT_STRING(msg));

    if(G_LOG_LEVEL_WARNING == log_level)
        DLT_LOG(AGW_framework, DLT_LOG_ERROR,
                DLT_STRING("AGW GLIB LOG MESSAGE ***** WARNING:"),
                DLT_STRING(msg));

    if(G_LOG_LEVEL_MESSAGE == log_level)
        DLT_LOG(AGW_framework, DLT_LOG_INFO,
                DLT_STRING("AGW GLIB LOG MESSAGE ***** MESSAGE:"),
                DLT_STRING(msg));

    if(G_LOG_LEVEL_INFO == log_level)
        DLT_LOG(AGW_framework, DLT_LOG_INFO,
                DLT_STRING("AGW GLIB LOG MESSAGE ***** INFO:"),
                DLT_STRING(msg));
}

/*****************************************************************************
 * Function:    vStartAutomServer
 * Description: stats the automotive server
 * Parameters:  void
 * Return:      void
 *****************************************************************************/

void       vStartAutomServer()
{
    DLT_LOG(AGW_framework,DLT_LOG_INFO,DLT_STRING(
                "GatewayDBusServer : - +vStartAutomServer"));

    gchar *bus_address = NULL;
    //reset gateway incoming limit flags
    g_blockGatewayIncoming = FALSE;
    igatewayMsgCounter = 0;

    bool status = loadConfig();
    DLT_LOG(AGW_DataNotifier,DLT_LOG_INFO,DLT_STRING(
                "load config status = "),DLT_INT(status));
    if(status)
    {
        igatewayMsgLimitPerSec = readIncomingLimit();
        uBusType = readBusConfig();
	   iSubscription_Wait_Time = readWaitParam();

        if(DBUS_TYPE_IDDBUS == uBusType)
        {
            bus_address = readPrivateBusAddress();
            //On error in reading iddbus socket path, fall back to default

            if(NULL == bus_address)
            {
                DLT_LOG(AGW_framework,DLT_LOG_INFO,
                        DLT_STRING("Error in reading iddbus socket path"));

                DLT_LOG(AGW_framework,DLT_LOG_INFO,
                        DLT_STRING("socket path: "),
                        DLT_STRING(bus_address));

                uBusType = DBUS_TYPE_SESSION;
            }
        }

        // GLib logging for debugging purpose
        gboolean bEnable = readGLogSettings();
        if(bEnable)
            g_log_set_default_handler (g_log_message_handler, NULL);
    }
    DLT_LOG(AGW_DataNotifier,DLT_LOG_INFO,DLT_STRING(
                "uBusType = "),DLT_INT(uBusType));

    if(DBUS_TYPE_IDDBUS == uBusType)
    {
        DLT_LOG(AGW_framework,DLT_LOG_INFO,DLT_STRING("bus type is IDDBUS"));

        g_dbus_connection_new_for_address (
                    g_strdup(bus_address),
                    G_DBUS_CONNECTION_FLAGS_AUTHENTICATION_CLIENT |
                    G_DBUS_CONNECTION_FLAGS_MESSAGE_BUS_CONNECTION,
                    NULL,
                    NULL,
                    iddbus_connection_callback,
                    NULL);

    }
    else if(DBUS_TYPE_TCPIP == uBusType)
    {
        DLT_LOG(AGW_framework,DLT_LOG_INFO,DLT_STRING("bus type is TCP"));
        GError *error = NULL;
        guid = g_dbus_generate_guid ();
        server_flags = G_DBUS_SERVER_FLAGS_NONE;

        server_flags |= G_DBUS_SERVER_FLAGS_AUTHENTICATION_ALLOW_ANONYMOUS;

        server = g_dbus_server_new_sync (opt_address,
                                         server_flags,
                                         guid,
                                         NULL, /* GDBusAuthObserver */
                                         NULL, /* GCancellable */
                                         &error);

        if (server == NULL)
        {
            if(error != NULL)
            {
                DLT_LOG(AGW_framework,DLT_LOG_ERROR,
                        DLT_STRING("Error creating server at address:"),
                        DLT_STRING(opt_address));

                DLT_LOG(AGW_framework,DLT_LOG_ERROR,DLT_STRING(
                            "Error msg: "), DLT_STRING(error->message));
                g_clear_error (&error);
            }

            return;
        }

        g_dbus_server_start (server);
        g_free (guid);
        guid = NULL;


        DLT_LOG(AGW_framework,DLT_LOG_INFO,
                DLT_STRING("Server is listening at: "),
                DLT_STRING(g_dbus_server_get_client_address (server)));

        g_signal_connect (server,
                          "new-connection",
                          G_CALLBACK (on_new_connection),
                          NULL);
    }
    else
    {
        DLT_LOG(AGW_framework,DLT_LOG_INFO,DLT_STRING("bus type is session"));
    }
    iSessionWatcherID = g_bus_own_name( G_BUS_TYPE_SESSION,
                                        "com.bosch.AutomotiveProxy",
                                        G_BUS_NAME_OWNER_FLAGS_NONE,
                                        on_bus_acquired,
                                        on_name_acquired,
                                        on_name_lost,
                                        NULL,
                                        NULL);

    /* LCM wants to expose interfaces on system bus name */
    isystemWatcherID  = g_bus_own_name (
                G_BUS_TYPE_SYSTEM,
                "com.bosch.AutomotiveProxy.LifeCycleConsumer",
                G_BUS_NAME_OWNER_FLAGS_NONE,
                on_system_bus_acquired,
                on_system_name_acquired,
                on_system_name_lost,
                NULL,
                NULL);

    if(bus_address)
    {
        free(bus_address);
        bus_address = NULL;
    }
    unloadConfig();

}

/*******************************************************************************
 * Function:    on_new_connection
 * Description: Callback on DBus name acquired
 * Parameters:  pConnection , Bus name, user data
 * Return:      void
 *****************************************************************************/
static gboolean
on_new_connection (GDBusServer *server,
                   GDBusConnection *pConnection,
                   gpointer user_data)
{
    DLT_LOG(AGW_framework,DLT_LOG_INFO,DLT_STRING(
                "Entered "),DLT_STRING( __FUNCTION__));
    GCredentials *credentials;
    gchar *s;
    g_blockGatewayIncoming = FALSE;
    igatewayMsgCounter = 0;

    credentials = g_dbus_connection_get_peer_credentials (pConnection);
    if (credentials == NULL)
        s = g_strdup ("(no credentials received)");
    else
        s = g_credentials_to_string (credentials);

    DLT_LOG(AGW_framework,DLT_LOG_INFO,
            DLT_STRING("Client connected. Peer credentials:"),
            DLT_STRING(s));

    DLT_LOG(AGW_framework,DLT_LOG_INFO,
            DLT_STRING("Negotiated capabilities:unix-fd-passing= "),
            DLT_INT(g_dbus_connection_get_capabilities (pConnection)
                    & G_DBUS_CAPABILITY_FLAGS_UNIX_FD_PASSING));

    poBusConn = g_object_ref (pConnection);
    if(!check_limit_timeout_id)
        check_limit_timeout_id = g_timeout_add(1000, checkMsgOverflow, NULL);

    //Add filter function for gateway message tracking
    msg_filter_id =  g_dbus_connection_add_filter(poBusConn,
                                                  gateway_message_filter,
                                                  NULL, NULL);
    g_free(s);
    s = NULL;

    bool bRet = bDoComponentsSetup(pConnection);
    if(!bRet)
    {
        DLT_LOG(AGW_framework,DLT_LOG_ERROR,DLT_STRING(
                    "Unable to export a compoent interface"));
        return FALSE;
    }

    DLT_LOG(AGW_framework,DLT_LOG_INFO,DLT_STRING("-on_new_connection"));
    return TRUE;
}


static void
iddbus_connection_close_callback (GObject *source_object,
                                  GAsyncResult *res,
                                  gpointer user_data)
{
    DLT_LOG(AGW_framework,DLT_LOG_INFO,DLT_STRING(
                "Entered "),DLT_STRING( __FUNCTION__));
    GError *error = NULL;
	gboolean bRet = g_dbus_connection_close_finish (poBusConn,
                                    res,
                                    &error);
	if(!bRet)
	{
		DLT_LOG(AGW_framework,DLT_LOG_ERROR,DLT_STRING(
		        "Error in closing IDDBUS connection"));
                if(error != NULL)
                {

                    DLT_LOG(AGW_framework,DLT_LOG_ERROR,DLT_STRING(
                                "Error msg: "), DLT_STRING(error->message));
                    g_clear_error (&error);
                }
	}
        else{
            DLT_LOG(AGW_framework,DLT_LOG_ERROR,DLT_STRING(
                    "IDDBUS connection closed successfully"));
        }
	
	g_clear_object(&poBusConn);
        DLT_LOG(AGW_framework,DLT_LOG_INFO,DLT_STRING(
                    "Exit "),DLT_STRING( __FUNCTION__));
	
}

/*****************************************************************************
 * Function:    vStopAutomServer
 * Description: stops the automotive server
 * Parameters:  void
 * Return:      void
 *****************************************************************************/
void       vStopAutomServer()
{
    DLT_LOG(AGW_framework,DLT_LOG_INFO,DLT_STRING(
                "GatewayDBusServer : - +vStopAutomServer"));

    vCleanUpProxyOnShutDown();



    if(iSessionWatcherID)
    {
        g_bus_unown_name(iSessionWatcherID);
        iSessionWatcherID = 0;
    }

    if(DBUS_TYPE_TCPIP == uBusType)
    {
        DLT_LOG(AGW_framework,DLT_LOG_INFO,
                DLT_STRING("vStopAutomServer: bus type is TCP"));

        if(server)
        {
            g_dbus_server_stop(server);
            g_object_unref(server);
            server = NULL;
        }
        g_object_unref(poBusConn);
        poBusConn = NULL;

    }
	
	malloc_trim(0);
}


void vCloseDbusConnection()
{
    DLT_LOG(AGW_framework,DLT_LOG_INFO,DLT_STRING(
                "Entered "),DLT_STRING( __FUNCTION__));
    vDoLCMDBusServerTearDown();
    if(isystemWatcherID)
    {
            g_bus_unown_name(isystemWatcherID);
            isystemWatcherID = 0;
            DLT_LOG(AGW_framework,DLT_LOG_INFO,
                    DLT_STRING("unowned system bus"));
    }
    if(iIDDBusWatcherID)
    {
            g_bus_unown_name(iIDDBusWatcherID);
            iIDDBusWatcherID = 0;
			if(poBusConn)
         	{
            	DLT_LOG(AGW_framework,DLT_LOG_INFO,DLT_STRING(
                        "closing iddbus connection "));
            	g_dbus_connection_close (poBusConn,
                                     NULL,
                                     iddbus_connection_close_callback,
                                     NULL);
         	}
    }
    DLT_LOG(AGW_framework,DLT_LOG_INFO,DLT_STRING(
                "Left "),DLT_STRING( __FUNCTION__));
}

/******************************************************************************
 * Function:    vCleanUpProxyOnShutDown
 * Description: Cleans proxy object on shutdown
 * Parameters:  void
 * Return:      void
 *****************************************************************************/

void vCleanUpProxyOnShutDown()
{
    DLT_LOG(AGW_framework,DLT_LOG_INFO,DLT_STRING(
                "GatewayDBusServer :+vCleanUpProxyOnShutDown"));

#ifdef ENABLE_NON_CMD_IN_GATEWAY
#if defined(VARIANT_S_FTR_ENABLE_FEAT_GW_RNAIVI) || defined(VARIANT_S_FTR_ENABLE_FEAT_GW_PSARCC)
    if(poFactoryReset != NULL)
        g_clear_object (&poFactoryReset);
#endif
    uninitializeECM(); // do this for RNAIVI as well, but not CMD.
#endif //ENABLE_NON_CMD_IN_GATEWAY


#ifndef VARIANT_S_FTR_ENABLE_FEAT_GW_PSARCC
    if(poAudioManagerProxy != NULL)
        g_clear_object(&poAudioManagerProxy);
    
	vUninitializeTTS();
	
    un_initialise_bluetooth_manager();

    if(pCallManager!=NULL){
        g_clear_object (&pCallManager); //will unref be enough? -- rhk6kor
    }

    if(pContactManager != NULL)
    {
        g_clear_object(&pContactManager);
    }

    if(pSourceManager != NULL)
    {
        g_clear_object(&pSourceManager);
    }

    if(MediaPlayerProxyObj != NULL)
    {
        g_clear_object (&MediaPlayerProxyObj);
    }

    if(URIObj != NULL)
    {
        DLT_UNREGISTER_CONTEXT(AGW_URI);
        g_signal_handler_disconnect (URIObj, uri_handler_id);
        g_clear_object(&URIObj);
    }

    if(poTunerInfoProxyObj != NULL)
    {
        g_clear_object (&poTunerInfoProxyObj);
    }

    if(poTunerControlProxyObj != NULL)
    {
        g_clear_object (&poTunerControlProxyObj);
    }
#endif

#ifdef VARIANT_S_FTR_ENABLE_FEAT_GW_RNAIVI

    if(poSettingsProvider != NULL)
        g_clear_object (&poSettingsProvider);

    if(poMultimediaStateProxyObj != NULL){
        g_clear_object(&poMultimediaStateProxyObj);
    }

    if(pNaviRouteObjManager != NULL){
        g_clear_object(&pNaviRouteObjManager);
    }

    if(pNaviGeoCodeObjManager != NULL){
        g_clear_object(&pNaviGeoCodeObjManager);
    }

    if(pNaviGeoCodeLocationObjManager != NULL){
        g_clear_object(&pNaviGeoCodeLocationObjManager);
    }

    vUninitializeNavi();
#endif //VARIANT_S_FTR_ENABLE_FEAT_GW_RNAIVI

#ifdef VARIANT_S_FTR_ENABLE_FEAT_GW_PSARCC

   disconnectToAudioComp();
	  
   if(poDataNotifierProxyObj != NULL)
   {
       g_dbus_interface_skeleton_unexport(
                   G_DBUS_INTERFACE_SKELETON(poDataNotifierProxyObj));
       DLT_UNREGISTER_CONTEXT(AGW_DataNotifier);
       g_clear_object(&poDataNotifierProxyObj);
    }
#endif //VARIANT_S_FTR_ENABLE_FEAT_GW_PSARCC

#if defined(VARIANT_COMPILER_ENV_GEN3ARM) || defined(VARIANT_COMPILER_ENV_GEN4RCAR)
    uninitializeSDC();
#endif //VARIANT_COMPILER_ENV_GEN3ARM || VARIANT_COMPILER_ENV_GEN4RCAR
	
#if ENABLE_CSM_IN_GATEWAY
    if(pVehicleInfoManagerObj != NULL)
    {
        g_dbus_interface_skeleton_unexport(
          G_DBUS_INTERFACE_SKELETON(pVehicleInfoManagerObj));
        g_clear_object (&pVehicleInfoManagerObj);
    }
    DLT_UNREGISTER_CONTEXT(AGW_VehicleData);
#ifdef VARIANT_S_FTR_ENABLE_FEAT_GW_PSARCC
    vStopVehicleFunctionClient();
#endif
#endif //ENABLE_CSM_IN_GATEWAY

#ifdef VARIANT_S_FTR_ENABLE_FEAT_GW_RNAIVI
    if(poCarConfiguration != NULL)
    {
        g_clear_object (&poCarConfiguration);
    }
#endif //VARIANT_S_FTR_ENABLE_FEAT_GW_RNAIVI
    uninitializeConfigMgmt();
    uninitializeConfigData();
#ifdef VARIANT_S_FTR_ENABLE_FEAT_GW_RNAIVI
    vStopProfileManagerClient();

	vUninitializePDE();
    vUnInitializeAutomWebMPlayer();
    vUnInitializeCEWebMPlayer();
#endif

#if defined(VARIANT_GEN3ARM_RNAIVI) || defined(VARIANT_COMPILER_ENV_GEN4RCAR)
    vStopFcOtaClient();
#endif
	
    vStopManAppService();
	vUninitializeSP();
    g_blockGatewayIncoming = FALSE;
    igatewayMsgCounter = 0;
    if(check_limit_timeout_id > 0)
        g_source_remove(check_limit_timeout_id);
    DLT_LOG(AGW_framework,DLT_LOG_INFO,DLT_STRING(
                "GatewayDBusServer :-vCleanUpProxyOnShutDown"));
}

#ifdef ENABLE_CGROUP_IMPL
void unexportAllInterfaces()
{
    DLT_LOG(AGW_framework,DLT_LOG_INFO,DLT_STRING(
                "In unexportAllInterfaces"));

    DLT_LOG(AGW_framework,DLT_LOG_INFO,
            DLT_STRING("unExport AudioManager"));

    g_dbus_interface_skeleton_unexport(
                G_DBUS_INTERFACE_SKELETON (
                    (AudioManager*)poGetAudioProxyObj()));

#ifndef VARIANT_S_FTR_ENABLE_FEAT_GW_PSARCC
    DLT_LOG(AGW_framework,DLT_LOG_INFO,DLT_STRING("unExport TunerControl"));

    g_dbus_interface_skeleton_unexport(
                G_DBUS_INTERFACE_SKELETON (
                    (TunerControl*)poGetTunerControlProxyObj()));

    DLT_LOG(AGW_framework,DLT_LOG_INFO,DLT_STRING("unExport TunerInfo"));

    g_dbus_interface_skeleton_unexport(
                G_DBUS_INTERFACE_SKELETON (
                    (TunerInfo*)poGetTunerInfoProxyObj()));
    DLT_LOG(AGW_framework,DLT_LOG_INFO,DLT_STRING("unExport URI"));
    g_dbus_interface_skeleton_unexport(G_DBUS_INTERFACE_SKELETON (
                                           (URI*)poGetURIProxyObj()));
    vUnexportAGWTTSInterface();
    vUnexportSMSManagerInterface();
    vUnexportCallManagerInterface();
    vUnexportContactManagerInterface();

    DLT_LOG(AGW_framework,DLT_LOG_INFO,DLT_STRING("unExport SourceManager"));
    g_dbus_interface_skeleton_unexport(
                G_DBUS_INTERFACE_SKELETON (
                    (SourceManager*)poGetSourceManager()));

    vUnexportAGWMediaPlayerInterface();
    vUnexportAGWBluetoothInterface();
#endif
#ifdef ENABLE_NON_CMD_IN_GATEWAY
    DLT_LOG(AGW_framework,DLT_LOG_INFO,
            DLT_STRING("unExport SettingsProviderFactoryReset"));

    g_dbus_interface_skeleton_unexport(
                G_DBUS_INTERFACE_SKELETON (
                    (SettingsProviderFactoryReset*)poGetFactoryResetProxy()));
#endif

#ifdef VARIANT_S_FTR_ENABLE_FEAT_GW_RNAIVI
    vUnexportNaviInterface();
    vUnexportE2ENaviInterface();
    vUnexportAGWPersonalizationInterface();
    vUnexportAGWSPMInterface();

    DLT_LOG(AGW_framework,DLT_LOG_INFO,DLT_STRING("unExport CarConfiguration"));
    g_dbus_interface_skeleton_unexport(
                G_DBUS_INTERFACE_SKELETON (
                    (CarConfiguration*)poGetCarConfigurationProxy()));

    DLT_LOG(AGW_framework,DLT_LOG_INFO,
            DLT_STRING("unExport SettingProvider"));
    g_dbus_interface_skeleton_unexport(
                G_DBUS_INTERFACE_SKELETON (
                    (SettingsProvider*)poGetSettingsProviderProxy()));

#endif //VARIANT_S_FTR_ENABLE_FEAT_GW_RNAIVI

#if ENABLE_CSM_IN_GATEWAY
    DLT_LOG(AGW_framework,DLT_LOG_INFO,
            DLT_STRING("unExport VehicleInfoManager"));
    g_dbus_interface_skeleton_unexport(
                G_DBUS_INTERFACE_SKELETON (
                    (VehicleInfoManager*)poGetVehicleInfoManagerProxyObj()));
#endif

#ifdef VARIANT_S_FTR_ENABLE_FEAT_GW_PSARCC
    DLT_LOG(AGW_framework,DLT_LOG_INFO,DLT_STRING("unExport DataNotifier"));
    g_dbus_interface_skeleton_unexport(
                G_DBUS_INTERFACE_SKELETON (
                    (DataNotifier*)poGetDataNotifierProxyObj()));

#endif //VARIANT_S_FTR_ENABLE_FEAT_GW_PSARCC
}
#endif //ENABLE_CGROUP_IMPL
