/******************************************************************
*FILE: ap-navigation-enums.h
*SW-COMPONENT: Internet
*DESCRIPTION: This file contains the enums related to
*             com.bosch.AutomotiveProxy.Navigation.RouteGuidance interface.
*COPYRIGHT: © 2016 Robert Bosch GmbH
*
*The reproduction, distribution and utilization of this file as
*well as the communication of its contents to others without express
*authorization is prohibited. Offenders will be held liable for the
*payment of damages. All rights reserved in the event of the grant
*of a patent, utility model or design.
******************************************************************/

#ifndef __AP_NAVIGATION_ENUMS__
#define __AP_NAVIGATION_ENUMS__

#include <glib.h>

G_BEGIN_DECLS

/**
 * AP_NAVIGATION_ERROR:
 *
 * Error domain for #ApNavigationError.
 *
 * Since: 5.0
 */
#define AP_NAVIGATION_ERROR ap_navigation_error_quark ()
GQuark ap_navigation_error_quark (void);

/**
 * ApNavigationError:
 * @AP_NAVIGATION_ERROR_INVALID_LATITUDE: Invalid latitude value
 * @AP_NAVIGATION_ERROR_INVALID_LONGITUDE: Invalid longitude value.
 * @AP_NAVIGATION_ERROR_GUIDANCE_FAILED: Start of route guidance failed.
 *
 * Represents the error values for Navigation methods
 *
 * Since: 5.0
 */
typedef enum
{
    AP_NAVIGATION_ERROR_INVALID_LATITUDE,
    AP_NAVIGATION_ERROR_INVALID_LONGITUDE,
    AP_NAVIGATION_ERROR_GUIDANCE_FAILED,
    AP_NAVIGATION_ERROR_INTERNAL_ERR,
    AP_NAVIGATION_RESPONSE_ERROR,
    AP_NAVIGATION_ERROR_UNKNOWN,
    AP_NAVIGATION_INVALID_PARAMETER,
    AP_NAVIGATION_INVALID_WAYPOINT
} ApNavigationError;

/**
 * ApNavigationRouteCriteria:
 * @AP_NAVIGATION_ROUTE_CRITERIA_NONE: No criteria or default value.
 * @AP_NAVIGATION_ROUTE_CRITERIA_FAST: criteria for fast route.
 * @AP_NAVIGATION_ROUTE_CRITERIA_SHORT: criteria for short route.
 * @AP_NAVIGATION_ROUTE_CRITERIA_ECO: criteria for eco route.
 *
 * Represents the enum values for route criteria
 *
 * Since: 5.0
 */
typedef enum
{
    AP_NAVIGATION_ROUTE_CRITERIA_ECO,
    AP_NAVIGATION_ROUTE_CRITERIA_FAST,
    AP_NAVIGATION_ROUTE_CRITERIA_SHORT,
    AP_NAVIGATION_ROUTE_CRITERIA_NONE
} ApNavigationRouteCriteria;

/**
 * ApNavigationNotifyIcon:
 * @AP_NAVIGATION_NOTIFY_ICON_LEFT: Icon representing turn left.
 * @AP_NAVIGATION_NOTIFY_ICON_RIGHT: Icon representing  turn right.
 * @AP_NAVIGATION_NOTIFY_ICON_U_LEFT: Icon representing "U" turn towards left.
 * @AP_NAVIGATION_NOTIFY_ICON_U_RIGHT: Icon representing "U" turn towards right.
 * @AP_NAVIGATION_NOTIFY_ICON_HEAD_STRAIGHT: Icon representing to head straight.
 * Represents the enum values for notification icon type
 *
 * Since: 5.0
 */
typedef enum
{
    AP_NAVIGATION_NOTIFY_ICON_NO_SYMBOL,
    AP_NAVIGATION_NOTIFY_ICON_NO_INFO,
    AP_NAVIGATION_NOTIFY_ICON_FOLLOW,
    AP_NAVIGATION_NOTIFY_ICON_RECALC,
    AP_NAVIGATION_NOTIFY_ICON_DEST_REACHED,
    AP_NAVIGATION_NOTIFY_ICON_DEST_REACHED_LEFT,
    AP_NAVIGATION_NOTIFY_ICON_DEST_REACHED_RIGHT,
    AP_NAVIGATION_NOTIFY_ICON_INTERMEDIATEDEST_REACHED,
    AP_NAVIGATION_NOTIFY_ICON_INTERMEDIATEDEST_REACHED_LEFT,
    AP_NAVIGATION_NOTIFY_ICON_INTERMEDIATEDEST_REACHED_RIGHT,
    AP_NAVIGATION_NOTIFY_ICON_EXIT_LEFT,
    AP_NAVIGATION_NOTIFY_ICON_EXIT_RIGHT,
    AP_NAVIGATION_NOTIFY_ICON_ROUNDABOUT_TRAFFICSENSE_LEFT_STRAIGHT,
    AP_NAVIGATION_NOTIFY_ICON_ROUNDABOUT_TRAFFICSENSE_LEFT_HALFRIGHT,
    AP_NAVIGATION_NOTIFY_ICON_ROUNDABOUT_TRAFFICSENSE_LEFT_RIGHT,
    AP_NAVIGATION_NOTIFY_ICON_ROUNDABOUT_TRAFFICSENSE_LEFT_SHARPRIGHT,
    AP_NAVIGATION_NOTIFY_ICON_ROUNDABOUT_TRAFFICSENSE_LEFT_REVERSE,
    AP_NAVIGATION_NOTIFY_ICON_ROUNDABOUT_TRAFFICSENSE_LEFT_SHARPLEFT,
    AP_NAVIGATION_NOTIFY_ICON_ROUNDABOUT_TRAFFICSENSE_LEFT_LEFT,
    AP_NAVIGATION_NOTIFY_ICON_ROUNDABOUT_TRAFFICSENSE_LEFT_HALFLEFT,
    AP_NAVIGATION_NOTIFY_ICON_ROUNDABOUT_TRAFFICSENSE_RIGHT_STRAIGHT,
    AP_NAVIGATION_NOTIFY_ICON_ROUNDABOUT_TRAFFICSENSE_RIGHT_HALFRIGHT,
    AP_NAVIGATION_NOTIFY_ICON_ROUNDABOUT_TRAFFICSENSE_RIGHT_RIGHT,
    AP_NAVIGATION_NOTIFY_ICON_ROUNDABOUT_TRAFFICSENSE_RIGHT_SHARPRIGHT,
    AP_NAVIGATION_NOTIFY_ICON_ROUNDABOUT_TRAFFICSENSE_RIGHT_REVERSE,
    AP_NAVIGATION_NOTIFY_ICON_ROUNDABOUT_TRAFFICSENSE_RIGHT_SHARPLEFT,
    AP_NAVIGATION_NOTIFY_ICON_ROUNDABOUT_TRAFFICSENSE_RIGHT_LEFT,
    AP_NAVIGATION_NOTIFY_ICON_ROUNDABOUT_TRAFFICSENSE_RIGHT_HALFLEFT,
    AP_NAVIGATION_NOTIFY_ICON_STRAIGHT,
    AP_NAVIGATION_NOTIFY_ICON_TURN_HALFRIGHT,
    AP_NAVIGATION_NOTIFY_ICON_TURN_RIGHT,
    AP_NAVIGATION_NOTIFY_ICON_TURN_SHARPRIGHT,
    AP_NAVIGATION_NOTIFY_ICON_TURN_SHARPLEFT,
    AP_NAVIGATION_NOTIFY_ICON_TURN_LEFT,
    AP_NAVIGATION_NOTIFY_ICON_TURN_HALFLEFT,
    AP_NAVIGATION_NOTIFY_ICON_UTURN_LEFT,
    AP_NAVIGATION_NOTIFY_ICON_UTURN_RIGHT,
    AP_NAVIGATION_NOTIFY_ICON_KEEP_LEFT,
    AP_NAVIGATION_NOTIFY_ICON_KEEP_RIGHT,
    AP_NAVIGATION_NOTIFY_ICON_DEST_DIRECTION_001,
    AP_NAVIGATION_NOTIFY_ICON_DEST_DIRECTION_009,
    AP_NAVIGATION_NOTIFY_ICON_DEST_DIRECTION_025,
    AP_NAVIGATION_NOTIFY_ICON_DEST_DIRECTION_041,
    AP_NAVIGATION_NOTIFY_ICON_DEST_DIRECTION_057,
    AP_NAVIGATION_NOTIFY_ICON_DEST_DIRECTION_073,
    AP_NAVIGATION_NOTIFY_ICON_DEST_DIRECTION_089,
    AP_NAVIGATION_NOTIFY_ICON_DEST_DIRECTION_105,
    AP_NAVIGATION_NOTIFY_ICON_DEST_DIRECTION_121,
    AP_NAVIGATION_NOTIFY_ICON_DEST_DIRECTION_137,
    AP_NAVIGATION_NOTIFY_ICON_DEST_DIRECTION_153,
    AP_NAVIGATION_NOTIFY_ICON_DEST_DIRECTION_169,
    AP_NAVIGATION_NOTIFY_ICON_DEST_DIRECTION_185,
    AP_NAVIGATION_NOTIFY_ICON_DEST_DIRECTION_201,
    AP_NAVIGATION_NOTIFY_ICON_DEST_DIRECTION_217,
    AP_NAVIGATION_NOTIFY_ICON_DEST_DIRECTION_233,
    SIZE
} ApNavigationNotifyIcon;

/**
 * ApNavigationDirectionDescription:
 * @AP_NAVIGATION_DIRECTION_NORTH: Towards north
 * @AP_NAVIGATION_DIRECTION_NORTH_NORTH_EAST: Towards north north east
 * @AP_NAVIGATION_DIRECTION_NORTH_EAST: Towards north east
 * @AP_NAVIGATION_DIRECTION_EAST_NORTH_EAST: Towards east north east
 * @AP_NAVIGATION_DIRECTION_EAST: Towards east
 * @AP_NAVIGATION_DIRECTION_EAST_SOUTH_EAST: Towards east south east
 * @AP_NAVIGATION_DIRECTION_SOUTH_EAST: Towards south east
 * @AP_NAVIGATION_DIRECTION_SOUTH_SOUTH_EAST: Towards south south east
 * @AP_NAVIGATION_DIRECTION_SOUTH: Towards south
 * @AP_NAVIGATION_DIRECTION_SOUTH_SOUTH_WEST: Towards south south west
 * @AP_NAVIGATION_DIRECTION_SOUTH_WEST: Towards south west
 * @AP_NAVIGATION_DIRECTION_WEST_SOUTH_WEST: Towards west south west
 * @AP_NAVIGATION_DIRECTION_WEST: Towards west
 * @AP_NAVIGATION_DIRECTION_WEST_NORTH_WEST: Towards west north west
 * @AP_NAVIGATION_DIRECTION_NORTH_WEST: Towards north west
 * @AP_NAVIGATION_DIRECTION_NORTH_NORTH_WEST: Towards north north west
 * @AP_NAVIGATION_DIRECTION_INVALID: Invalid direction
 * Represents the enum values for direction relative to car heading
 *
 * Since: 5.1
 */
typedef enum
{
    AP_NAVIGATION_DIRECTION_NORTH,
    AP_NAVIGATION_DIRECTION_NORTH_NORTH_EAST,
    AP_NAVIGATION_DIRECTION_NORTH_EAST,
    AP_NAVIGATION_DIRECTION_EAST_NORTH_EAST,
    AP_NAVIGATION_DIRECTION_EAST,
    AP_NAVIGATION_DIRECTION_EAST_SOUTH_EAST,
    AP_NAVIGATION_DIRECTION_SOUTH_EAST,
    AP_NAVIGATION_DIRECTION_SOUTH_SOUTH_EAST,
    AP_NAVIGATION_DIRECTION_SOUTH,
    AP_NAVIGATION_DIRECTION_SOUTH_SOUTH_WEST,
    AP_NAVIGATION_DIRECTION_SOUTH_WEST,
    AP_NAVIGATION_DIRECTION_WEST_SOUTH_WEST,
    AP_NAVIGATION_DIRECTION_WEST,
    AP_NAVIGATION_DIRECTION_WEST_NORTH_WEST,
    AP_NAVIGATION_DIRECTION_NORTH_WEST,
    AP_NAVIGATION_DIRECTION_NORTH_NORTH_WEST,
    AP_NAVIGATION_DIRECTION_INVALID
} ApNavigationDirectionDescription;

/**
 * ApNavigationTimeType:
 * @AP_NAVIGATION_REMAININGTIME: Remaining time.
 * @AP_NAVIGATION_ETA: Estimated time of Arrival.
 * Represents the enum values for time type
 *
 * Since: 5.1
 */
typedef enum
{
    AP_NAVIGATION_REMAININGTIME,
    AP_NAVIGATION_ETA
}ApNavigationTimeType;

/**
 * ApNavigationTimeFormat:
 * @AP_NAVIGATION_UNSET: Not set any valid time format.
 * @AP_NAVIGATION_TWELVEHOURS: 12 hour format.
 * @AP_NAVIGATION_TWENTYFOURHOURS: 24 hours format.
 * Represents the enum values for time format
 *
 * Since: 5.1
 */
typedef enum
{
    AP_NAVIGATION_UNSET,
    AP_NAVIGATION_TWELVEHOURS,
    AP_NAVIGATION_TWENTYFOURHOURS
}ApNavigationTimeFormat;

/**
 * ApNavigationEndPoint:
 * @AP_NAVIGATION_ENDPOINT_UNKNOWN: End point is unknown.
 * @AP_NAVIGATION_ENDPOINT_WAYPOINT: End point is a Waypoint.
 * @AP_NAVIGATION_ENDPOINT_DESTINATION: End point is the final destination.
 * Represents the enum values for EndPoint type
 *
 * Since: 5.1
 */
typedef enum
{
    AP_NAVIGATION_ENDPOINT_UNKNOWN,
    AP_NAVIGATION_ENDPOINT_WAYPOINT,
    AP_NAVIGATION_ENDPOINT_DESTINATION
}ApNavigationEndPoint;

/**
 * ApNavigationRouteRecalculationReason:
 * @AP_NAVIGATION_ROUTE_REASON_UNKNOWN: Unknown reason
 * @AP_NAVIGATION_ROUTE_NO_ROUTE_AVAILABLE: Sent if alternative route
 *                                      calculation is not enabled.
 * @AP_NAVIGATION_ROUTE_NO_CALCULATION: Sent if alternative route is not
 *                                      calculated.
 * @AP_NAVIGATION_ROUTE_FIRST_CALCULATION: sent on First route calculation to
                                        the destination set.
 * @AP_NAVIGATION_ROUTE_OPTIMIZE_ROUTE: Sent on reCalculation is self triggered
 *                                      to optimize the route.
 * @AP_NAVIGATION_ROUTE_REGARDING_TRAFFIC_MESSAGE: sent on reCalculation is self
 *                                      triggered by traffic messages .
 * @AP_NAVIGATION_ROUTE_DISREGARD_ROUTE: reCalculation due to detour or route
 *                                      erased by the user.
 * @AP_NAVIGATION_ROUTE_REGARDING_USER_TRAFFIC_MESSAGE: sent if reCalculation is
 *                                      triggered by user considering traffic
 *                                      messages.
 * @AP_NAVIGATION_ROUTE_ROUTE_OPTION_CHANGED: Route re-calculation on route
 *                                       option change like avoid streets
 *                                      , alternative route.
 * @AP_NAVIGATION_ROUTE_ROUTE_CRITERION_CHANGED : Route re-calculation on route
 *                                      criterion change.
 * @AP_NAVIGATION_ROUTE_ROUTE_OPTION_CRITERION_CHANGED: Route re-calculation on
 *                                      route option and criterion change.
 * @AP_NAVIGATION_ROUTE_ENTRY_POINT_EVALUATION: Route re-calculation done due to
                                        change in entry point.

 * Represents the enum values for route re-calculation reason
 *
 * Since: 5.2
 */

typedef enum
{
    AP_NAVIGATION_ROUTE_REASON_UNKNOWN,
    AP_NAVIGATION_ROUTE_NO_ROUTE_AVAILABLE,
    AP_NAVIGATION_ROUTE_NO_CALCULATION,
    AP_NAVIGATION_ROUTE_FIRST_CALCULATION,
    AP_NAVIGATION_ROUTE_OPTIMIZE_ROUTE,
    AP_NAVIGATION_ROUTE_REGARDING_TRAFFIC_MESSAGE,
    AP_NAVIGATION_ROUTE_DISREGARD_ROUTE,
    AP_NAVIGATION_ROUTE_REGARDING_USER_TRAFFIC_MESSAGE,
    AP_NAVIGATION_ROUTE_ROUTE_OPTION_CHANGED,
    AP_NAVIGATION_ROUTE_ROUTE_CRITERION_CHANGED,
    AP_NAVIGATION_ROUTE_ROUTE_OPTION_CRITERION_CHANGED,
    AP_NAVIGATION_ROUTE_ENTRY_POINT_EVALUATION
} ApNavigationRouteRecalculationReason;

/**
 * ApNavigationWaypointOperationStatus:
 * @AP_NAVIGATION_WAYPOINT_INSERTED: Waypoint is inserted.
 * @AP_NAVIGATION_WAYPOINT_DELETED: Waypoint is deleted.
 * @AP_NAVIGATION_WAYPOINT_MOVED: Waypoint is moved.
 * @AP_NAVIGATION_WAYPOINT_REPLACED: Waypoint is replaced.
 * @AP_NAVIGATION_WAYPOINT_STATUS_UNKNOWN: Waypoint status is unknown.
 *
 * Represents the enum values for operation performed on waypoints
 *
 * Since: 5.2
 */
typedef enum
{
    AP_NAVIGATION_WAYPOINT_INSERTED,
    AP_NAVIGATION_WAYPOINT_DELETED,
    AP_NAVIGATION_WAYPOINT_MOVED,
    AP_NAVIGATION_WAYPOINT_REPLACED,
    AP_NAVIGATION_WAYPOINT_STATUS_UNKNOWN
} ApNavigationWaypointOperationStatus;


G_END_DECLS

#endif /* !__AP_NAVIGATION_ENUMS__ */
