#include "vd_adr3_INC.h"
#include "vd_adr3_thread.h"
#include "vd_adr3_If.h"
#include "vd_adr3Msg_If.h"

#ifdef __cplusplus
extern "C"
{
#include "inc.h"
}
#endif

/* C++ headers*/
#include <iostream>
#include <algorithm>
#include <fstream>
#include <sstream>

/* C headers */
#include <errno.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <time.h>

#include <sys/types.h>
#include <sys/poll.h>
#include <sys/socket.h>
#include <sys/ioctl.h>
#include <sys/time.h>
#include <netdb.h>
#include <netinet/in.h>
#include <netinet/tcp.h>

#include "inc_ports.h"

#define PORT_OFFSET 0xC700
#define LUNID 0x16

#define BUFFER_SIZE 1024

#define SCD_S_IMPORT_INTERFACE_GENERIC
#include "scd_if.h"

#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#define ET_TRACE_INFO_ON
#include "etrace_if.h"
#include "../../fc_audiomanager_trace.h"

#include "../../fc_audiomanager_main.h"
#include "aud_sinkmgr_main.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_VD_ADR3_IF
#include "trcGenProj/Header/vd_adr3_INC.cpp.trc.h"
#endif

/* audio sink device configuration file path */
#define AUD_SINK_CFG_FILE "/etc/audio/audsink.cfg"

/* key string values to be read from configuration */
static std::string   str_sinkdev_name   = "sinkdevice.name";
static std::string   str_sinkdev_port   = "sinkdevice.port";
static std::string   str_sinkdev_domain = "sinkdevice.domain";
static std::string   str_sinkdev_stream = "sinkdevice.stream";

/*----------------------------------------*/
/* Audio thread configuration in registry */
/*----------------------------------------*/

#define AUD_REGPATH_THREAD                       "APP_THREAD"
#define AUD_REGVALUE_ADR3THREAD_PRIO_NAME        "WORKER_PRIO"
#define AUD_REGVALUE_ADR3THREAD_STACK_SIZE_NAME  "WORKER_STCK"

int       vd_adr3_INC::m_nADRSocketFd = -1;
sk_dgram* vd_adr3_INC::m_pSkDgram     = NULL;

/******************************************************************************
 * FUNCTION     : vd_adr3_INC
 * DESCRIPTION  : Constructor creates the INC communication channel with the
 *                INC Socket which interacts with the ADR3 or V850.
 * PARAMETER    : poAdr3Main -> pointer to adr3 main object.
 * RETURN VALUE : void
 * HISTORY      :
 * Date         | Author                  | Modification
 * 16.04.2013   | RBEI/ECG - Vyankatesh VD| Initial Revision
 * 12.12.2018   | RBEI/ECO - Ranjit Susal | Added sink device configuration init
 *              |                         | and parse configuration in runtime.
 *****************************************************************************/
vd_adr3_INC::vd_adr3_INC(vd_adr3_main *poAdr3Main) :
            m_vd_adr3_main(poAdr3Main)
{
    _bTerminate  =  FALSE;
    _hRxThreadId = -1;


    /* Default Sink configuration */
    m_stSinkCfg.u16AudioPort = (tU16)(PORT_OFFSET | LUNID);

    memset(&m_stSinkCfg.c8HostRemote,0, sizeof(tC8)*SINK_DEVICE_NAME_SIZE);
    memset(&m_stSinkCfg.c8HostLocal ,0, sizeof(tC8)*SINK_DEVICE_NAME_SIZE);
    memset(&m_stSinkCfg.c8RemotePort,0, sizeof(tC8)*SINK_DEVICE_PORT_SIZE);
    memset(&m_stSinkCfg.c8LocalPort ,0, sizeof(tC8)*SINK_DEVICE_PORT_SIZE);

    sprintf(m_stSinkCfg.c8HostRemote, "adr3");
    sprintf(m_stSinkCfg.c8HostLocal,  "adr3-local");

    sprintf(m_stSinkCfg.c8RemotePort, "%d", m_stSinkCfg.u16AudioPort);
    sprintf(m_stSinkCfg.c8LocalPort, "%d", m_stSinkCfg.u16AudioPort);

    m_stSinkCfg.u8Domain      = (tU8)AF_BOSCH_INC_ADR;
    m_stSinkCfg.u8Stream      = (tU8)SOCK_STREAM;

    /* parse configuration file and configure sink configuration */
    tBool bStatus = bParseSinkConfiguration();

    if(bStatus){
        ETG_TRACE_USR1(("vd_adr3_INC::vd_adr3_INC():  parse configuration file succeed status(%d)",(tU8) bStatus));
    }else{
        ETG_TRACE_USR1(("vd_adr3_INC::vd_adr3_INC():  parse configuration file failed status(%d)",(tU8) bStatus));
    }

    /**Calls the function for creating the INC communication socket.*/
    for(int i=0; i<500;i++)
    {
        if(bOpenINCCommunication())
        {
            break;
        }
        usleep(100000);
    }
}

/*******************************************************************************
* FUNCTION : vd_adr3_INC
*
* DESCRIPTION : write data to ADR3 socket
*
* PARAMETER : int,void*,size_t
*
* RETURNVALUE : int
*
* HISTORY : 16.04.2013 vdu1kor
* Initial version
* 12.12.2014 prm4kor  Changed to dgram_send
*********************************************************************************/
int vd_adr3_INC::vSendINC(void* pData,size_t dataLen)
{
  if(pData && m_pSkDgram && (dataLen>0))
    return (int)(dgram_send(m_pSkDgram,pData,dataLen));
  else
  {
    ETG_TRACE_ERR(("vd_adr3_INC::vSendINC can not send, m_pSkDgram 0x%x, pData 0x%x, dataLen %d",m_pSkDgram,pData,dataLen));
    return -1;
  }
}
/*******************************************************************************
* FUNCTION : ~vd_adr3_INC
*
* DESCRIPTION : destructor, kills the thread created for receiving the messages from the
        ADR3 via INC Sopcket and close the SOCKET..
*
* PARAMETER :
*
* RETURNVALUE : tVoid
*
* HISTORY : 16.04.2013 vdu1kor
* Initial version
* 12.12.2014 prm4kor  Changes for dgram
*********************************************************************************/
vd_adr3_INC::~vd_adr3_INC()
{
  /**Stop ADR3 receiver thread.*/
  _bTerminate = TRUE;

  if(NULL != m_pSkDgram)
  {
    dgram_exit(m_pSkDgram);
    m_pSkDgram = NULL;
  }

  /**Closes the socket at the time of Shutdown.*/
  if(-1 != m_nADRSocketFd)
  {
    (void)close(m_nADRSocketFd);
    m_nADRSocketFd = -1;
  }

  m_vd_adr3_main = NULL;

  /**deleting the thread at the time of Shutdown.*/
  OSAL_s32ThreadDelete(_hRxThreadId);
}
/*******************************************************************************
* FUNCTION : vGetThreadPrioandStacksize
*
* DESCRIPTION : gets the Thread Prio and Stack size from the registry.
*
* PARAMETER : (tU32 &u32ThreadPrio,tU32 &u32StackSize)
*
* RETURNVALUE : tVoid
*
* HISTORY : 16.04.2013 vdu1kor
* Initial version
*********************************************************************************/
tVoid vd_adr3_INC::vGetThreadPrioandStacksize(tU32 &u32ThreadPrio,tU32 &u32StackSize) const
{

    // Read thread priority from registry
    if ( FALSE == scd_bGetAppConfigurationValue
                                      (
                                         // CCA App ID
                                         CCA_C_U16_APP_AUDIO,

                                         // Registry path
                                         AUD_REGPATH_THREAD,

                                         // Registry entry for thread priority
                                         AUD_REGVALUE_ADR3THREAD_PRIO_NAME,

                                         // Read priority value
                                         &u32ThreadPrio
                                       )
          )
    {
      // Assign default value.
      u32ThreadPrio = AUD_WAITINGTHREAD_DEFAULT_PRIO;

      // Indicate that the thread is running on default priority
      ETG_TRACE_ERR(( " vd_adr3_INC::vGetThreadPrioandStacksize ->  the thread is running on default priority=%d"
          ,u32ThreadPrio
          ));

    }
    else
    {
      ETG_TRACE_USR2(( "vd_adr3_INC::vGetThreadPrioandStacksize ->  the thread is running on Registry priority=%d"
            ,u32ThreadPrio
            ));
    }


    // Read thread stack size from registry
  if ( FALSE == scd_bGetAppConfigurationValue
                                    (
                                       // CCA App ID
                                       CCA_C_U16_APP_AUDIO,

                                       // Registry path
                                       AUD_REGPATH_THREAD,

                                       // Registry entry for stack size
                                       AUD_REGVALUE_ADR3THREAD_STACK_SIZE_NAME,

                                       // Read stack size value
                                       &u32StackSize
                                     )
     )
  {
    u32StackSize  = AUD_WAITINGTHREAD_DEFAULT_STACKSIZE;

    // Indicate that the thread is running on default priority
    ETG_TRACE_ERR(( "vd_adr3_INC::vGetThreadPrioandStacksize ->  the thread is running with default stack size (=%d)."
                    ,u32StackSize
                  ));
  }
  else
  {
    ETG_TRACE_USR2(( "vd_adr3_INC::vGetThreadPrioandStacksize ->  the thread is running with Registry stack size =%d."
                    ,u32StackSize));
  }
}

/*******************************************************************************
* FUNCTION : vRxThreadSetup
*
* DESCRIPTION : Creates the Receiving Thread called AUD_RX and waits for the Messages from ADR3.
*
* PARAMETER :
*
* RETURNVALUE : tVoid
*
* HISTORY : 16.04.2013 vdu1kor
* Initial version
*********************************************************************************/
tVoid vd_adr3_INC::vRxThreadSetup()
{
  OSAL_trThreadAttribute  rAttr;
    tC8                     szThreadName[OSAL_C_U32_MAX_NAMELENGTH] = "\0";

      // Note:Maximum size of thread name limited to 8
  tString tStrTemp = OSAL_szStringCopy( szThreadName, "AUD_RX");
  tString tStrDummy = tStrTemp;//Added to remove lint warning.
  tStrTemp = tStrDummy;

  // Local variables to store data read from registry
    tU32 u32ThreadPrio = 0;
    tU32 u32StackSize  = 0;

  vGetThreadPrioandStacksize(u32ThreadPrio, u32StackSize);

    ETG_TRACE_USR3(("vd_adr3_INC::vRxThreadSetup() ,u32ThreadPrio= %d,u32StackSize= %d",u32ThreadPrio,u32StackSize));

  // Initialize thread parameters
  rAttr.szName       = szThreadName;
  rAttr.s32StackSize = (tS32)u32StackSize;
  rAttr.u32Priority  = u32ThreadPrio;
  rAttr.pfEntry      = (OSAL_tpfThreadEntry)&vd_adr3_INC::vWaitForADRMessage;
  rAttr.pvArg        = (tPVoid)this;

  _hRxThreadId = OSAL_ThreadSpawn(&rAttr);

  if ( _hRxThreadId == OSAL_ERROR )
      {
          tS32 s32OsalError = (tS32)OSAL_u32ErrorCode(  );
          ETG_TRACE_USR2(( "vd_adr3_INC::vRxThreadSetup() -> Thread spawn failed errorcode=%x" ,s32OsalError ));

      }
    else
    {
        ETG_TRACE_USR1(("vd_adr3_INC::vRxThreadSetup() ,OSAL_ThreadSpawn()-> Thread creation successful."));

    }
}

/*************************************************************************************
* FUNCTION : vWaitForADRMessage
*
* DESCRIPTION : this is a blocking call which waits for a Message from ADR INC socket
*
* PARAMETER : pvArg
*
* RETURNVALUE : void
*
* HISTORY : 16.04.2013
* Initial version
* 12.12.2014 prm4kor  Changes for dgram
***************************************************************************************/
tVoid vd_adr3_INC::vWaitForADRMessage(tVoid* pvArg)
{

  vd_adr3_INC*        poINCComm;

  poINCComm = (vd_adr3_INC*) pvArg;

  ETG_TRACE_USR1(("vd_adr3_INC::vWaitForADRMessage \n"));

  /*ET_TRACE_INFO_BIN(
    TUN_TRACE_CLASS_DRVADRIF_LOW,
    ET_EN_T8 _ TUN_TRACE_ADRIF_LOW_THREAD_ENTRY _
    ET_EN_DONE);*/

  tU8 au8RecvBuffer[BUFFER_SIZE+1];
  int iNumberOfBytesReceived =0;

  /**Polling on the INC Socket for ever to receive the messages from ADR3.*/
  while (poINCComm->_bTerminate == FALSE)
  {
    iNumberOfBytesReceived = (int)dgram_recv(
      m_pSkDgram,\
      au8RecvBuffer,\
      sizeof(au8RecvBuffer)-1);

    ETG_TRACE_USR4(("vd_adr3_INC::vWaitForADRMessage,Number of Bytes received:= %d ", iNumberOfBytesReceived));

    if(0 == iNumberOfBytesReceived)
    {
      ETG_TRACE_FATAL(("vd_adr3_INC::vWaitForADRMessage, dgram_recv returns 0 indicating connection is closed by peer"));
      break;
    }
    else if (-1 == iNumberOfBytesReceived)
    {
      ETG_TRACE_USR1(("vd_adr3_INC::vWaitForADRMessage,Receiving from ADR failed..."));
    }
    else
    {
      vd_adr3_If::vDataInd(au8RecvBuffer,iNumberOfBytesReceived);
      au8RecvBuffer[iNumberOfBytesReceived] = 0;
    }
  }
}

/******************************************************************************
 * FUNCTION     : bOpenINCCommunication
 * DESCRIPTION  : open the socket and creates the receiving thread for POLLING
 *                on INC socket.
 * PARAMETER    : void
 * RETURN VALUE : true on sink device successful open or else false.
 * HISTORY      :
 * Date         | Author                  | Modification
 * 16.04.2013   | RBEI/ECG - Vyankatesh VD| Initial Revision
 * 12.12.2018   | RBEI/ECO - Ranjit Susal | added generic sink device open call
 *              |                         | & sink devices:(adr3/alsactrl etc)
 *****************************************************************************/
tBool vd_adr3_INC::bOpenINCCommunication()
{
  tBool bSuccess = false;

#if 0 /*  bOpenADRSocket is deprecated and call instead bOpenSinkDevice */
    /**Opens the INC Socket with family of ADR and with AUDIO LUNID.*/
     bSuccess = bOpenADRSocket(AF_BOSCH_INC_ADR,SOCK_STREAM,LUNID);
#endif

   /* Opens sink device based on the sink device configuration read on startup*/
   bSuccess = bOpenSinkDevice();

    if(true == bSuccess){
      ETG_TRACE_USR2(("vd_adr3_INC::vOpenINCCommunication():  OpenSinkDevice Socket succeed ,bSuccess= %d",(tU8) bSuccess));

      /**Creats the thread for receiving the messages from ADR3 via INC socket.*/
        vRxThreadSetup();
    }else{
      ETG_TRACE_USR2(("vd_adr3_INC::vOpenINCCommunication():  OpenSinkDevice Socket failed ,bSuccess= %d",(tU8) bSuccess));
    }

  return bSuccess;
}

#if 0 /*  bOpenADRSocket is deprecated and call instead bOpenSinkDevice */
/*******************************************************************************
* FUNCTION : bOpenADRSocket
*
* DESCRIPTION : Opens the INC socket and sets local variable to store socket fd
*
* PARAMETER : channel - channel number
*        port  - port
*             protocol - protocol number
*
* RETURNVALUE : bool
*
* HISTORY : 16.04.2013
* Initial version
* 12.12.2014 prm4kor  Changes for dgram
*********************************************************************************/
bool vd_adr3_INC::bOpenADRSocket(uint8_t u8Domain,\
                 uint16_t u16Type,\
                 int iChannel) const
{
  int iLunID = (uint16_t)PORT_OFFSET |(uint16_t)iChannel;

  bool bSetupfail = true;

  struct sockaddr_in serv_addr;
  struct hostent *server, *lHost;

  memset(serv_addr.sin_zero, 0 , sizeof (struct sockaddr) - __SOCKADDR_COMMON_SIZE - sizeof (in_port_t) - sizeof (struct in_addr));

  //Create a Socket for communication with ADR
  m_nADRSocketFd = socket(
    u8Domain,\
    u16Type,\
    0);

  if(0 > m_nADRSocketFd)
  {
    ETG_TRACE_USR1(("vd_adr3_INC::bOpenADRSocket(), Failed to create socket"));
    return false;
  }

  //Set socket send timeout
  int enabled = 1;
  if(setsockopt(m_nADRSocketFd,SOL_SOCKET,SO_SNDTIMEO,(char*)&enabled,sizeof(enabled)) < 0)
  {
    ETG_TRACE_USR1(("vd_adr3_INC::bOpenADRSocket(), Failed to send timeout of socket"));
  }

   ETG_TRACE_USR2(("vd_adr3_INC::bOpenADRSocket() with fd = %u ", m_nADRSocketFd));

   //Initialise the dgram
   m_pSkDgram = dgram_init(m_nADRSocketFd, DGRAM_MAX, NULL);

  if (m_pSkDgram == NULL)
  {
     ETG_TRACE_ERR(("vd_adr3_INC::bOpenADRSocket(), Failed to initialize dgram service"));
    bSetupfail = false;
   }

  /**Get the Local Host for Adr3 and Bind to that Host,
  which is dynamically created at runtime by kernal in the file /etc/hosts*/
  lHost = gethostbyname("adr3-local");

  if(NULL == lHost)
  {
    ETG_TRACE_USR1(( "vd_adr3_INC::bOpenADRSocket(), adr3-local Host Not Available"));
    bSetupfail = false;
  }

  /** Bind to a local port to recieve response*/

  serv_addr.sin_family = AF_INET;
  /**Address received from gethostbyname function*/
  if(NULL != lHost)
  {
    memcpy((char *)&serv_addr.sin_addr.s_addr, (char *)lHost->h_addr, lHost->h_length);
  }
  //serv_addr.sin_addr.s_addr = 0;
  serv_addr.sin_port = htons(iLunID);

  if (
    bind(
    m_nADRSocketFd,\
    (struct sockaddr *) &serv_addr,\
    sizeof(serv_addr)\
    )
    < 0)
  {
    ETG_TRACE_USR2(( " vd_adr3_INC::bOpenADRSocket(),Binding to local host port failed - Port No:= %d ", iLunID));
    bSetupfail = false;
  }

  /**Get the ADR3 hosts IP Address and try to connect to the ADR3 socket*/
  server = gethostbyname("adr3");//Change it to v850 if you want to talk to v850
  if (server == NULL)
  {
    ETG_TRACE_USR1(("vd_adr3_INC::bOpenADRSocket(),Failed to get host info, check if info is present in /etc/hosts"));
    bSetupfail = false;
  }
  else
  {
    /**
    * Connect to port of the ADR3/v850.
    * Note to Developer > Configure remote client and server by following
    * commands, or else connect call will fail
    */
    serv_addr.sin_family = AF_INET;//Protocol
    memcpy(
      (char *)&serv_addr.sin_addr.s_addr,\
      (char *)server->h_addr,\
      server->h_length);
    //Address received from gethostbyname function
    serv_addr.sin_port = htons(iLunID);//Portnumber to listen to

    if (
      connect(
      m_nADRSocketFd,\
      (struct sockaddr *) &serv_addr,\
      sizeof(serv_addr)
      )
      < 0)
    {
      ETG_TRACE_USR1(("vd_adr3_INC::bOpenADRSocket(),Failed to connect to the desired target port, is allowed_connections configured ?"));
      bSetupfail = false;
    }
  }

  /**If the binding or connecting to ADR3 via INC socket fails then returns false to exist the loop.*/
  if(false == bSetupfail)
  {
    if(NULL != m_pSkDgram)
    {
      dgram_exit(m_pSkDgram);
      m_pSkDgram = NULL;
    }

    if(-1 != m_nADRSocketFd)
    {
      close(m_nADRSocketFd);
      m_nADRSocketFd = -1;
    }
  }
  /**For time being till the ADR3 ctrl driver is available the
  status of the Communication state of ADR3 is made as RUNNING.*/
  else
  {
      /*TODO : This a workaround.
    During Testing it is found that no Start Up message is received from ADR sometimes.
    So as soon as the socket connection is established, we are sending 'power on' message from here and setting the communication state as run
    by assuming ADR startup is already received, this needs to be investigated*/

    ETG_TRACE_USR1(("ADR not Power UP after Socket Creation"));



    //char cDefaultData[11] = {0x00, 0x10, 0x00, 0x91, 0x01, 0x0f, 0x11, 0x02, 0x00, 0x01, 0x02};
    //AUD_SET_U16(&cDefaultData[enAdrMsgOffset_CLIENT_ID], AUD_CLIENT_ID);//Set client ID
      //vSendINC(cDefaultData,11);

    vd_adr3_If::vSetCommunicationState(enComState_Run);
  }

  return bSetupfail;
}
#endif

/******************************************************************************
 * FUNCTION     : bParseSinkConfiguration
 * DESCRIPTION  : -> parse the configuration file from provided path and read
 *              : the sink device name, port number, domain and stream and
 *              : store it in a sink device configuration structure.
 *              : -> the parser reads each line from file and removes
 *              : white spaces, empty lines and commented lines (start by "#")
 *              : and reads data in a (key):(value) pair and using delimiter
 *              : (:) as separator.
 *              : -> the parser can read both formatted and non-formatted files
 *              : -> stores the values to a sink device structure.
 * PARAMETER    : void
 * RETURN VALUE : true on successful configuration read or else false.
 * HISTORY      :
 * Date         | Author                  | Modification
 * 12.12.2018   | RBEI/ECO - Ranjit Susal | Initial Revision
 *****************************************************************************/
tBool vd_adr3_INC::bParseSinkConfiguration(void)
{
    tBool     bStatus = true;
    ETG_TRACE_USR1(("vd_adr3_INC::bParseSinkConfiguration:  ENTERED"));

    /* open file in an input file(read) stream mode */
    std::ifstream cFile (AUD_SINK_CFG_FILE);
    if (cFile.is_open())
    {
        std::string line;
        std::string key;
        std::string value;
        /* iterates line by line till the end of file*/
        while(getline(cFile, line))
        {
            /*
             * erase(): erases the sequence of characters in the range (first, last)
             * remove_if(): takes a sequence (first, last from the line) and transforms
             * it into a sequence without the undesired characters (i.e. white spaces).
             * The length of the sequence does not get altered, however the elements representing
             * the undesired characters are moved to the end of the sequence and remain in an
             * unspecified state. The function returns an iterator to the new end of the sequence.
             * isspace : checks whether an individual character is a whitespace character.
             */
            line.erase
            (
                  std::remove_if(line.begin(),line.end(),::isspace)
                 ,line.end()
            );

            /* skip the line if it is empty or contains a comment (indicated by "#") */
            if(line[0] == '#' || line.empty())
                continue;

            /* split the string "key:value" at the delimiter ":" */
            std::size_t delimiterPos = line.find(":");
            key   = line.substr(0, delimiterPos);
            value = line.substr(delimiterPos + 1);

            /* ignore comment (#) from value part */
            delimiterPos = value.find("#");
            value = value.substr(0, delimiterPos);

            if(key.compare(str_sinkdev_name) == 0)
            {
                memset(&m_stSinkCfg.c8HostRemote,'\0', sizeof(tC8)*SINK_DEVICE_NAME_SIZE);
                memset(&m_stSinkCfg.c8HostLocal ,'\0', sizeof(tC8)*SINK_DEVICE_NAME_SIZE);

                std::string temp = value + "-local";
                memcpy(m_stSinkCfg.c8HostRemote, value.c_str(), sizeof(m_stSinkCfg.c8HostRemote));
                memcpy(m_stSinkCfg.c8HostLocal, temp.c_str(), sizeof(m_stSinkCfg.c8HostLocal));
            }
            else if(key.compare(str_sinkdev_port) == 0)
            {
                memset(&m_stSinkCfg.c8RemotePort,0, sizeof(tC8)*SINK_DEVICE_PORT_SIZE);
                memset(&m_stSinkCfg.c8LocalPort ,0, sizeof(tC8)*SINK_DEVICE_PORT_SIZE);

                m_stSinkCfg.u16AudioPort = (tU16)strtoul(value.c_str(), NULL, 0);

                sprintf(m_stSinkCfg.c8RemotePort, "%d", m_stSinkCfg.u16AudioPort);
                sprintf(m_stSinkCfg.c8LocalPort, "%d", m_stSinkCfg.u16AudioPort);
            }
            else if(key.compare(str_sinkdev_domain) == 0)
            {
                m_stSinkCfg.u8Domain = (tU8)strtoul(value.c_str(), NULL, 0);
                switch(m_stSinkCfg.u8Domain)
                {
                case 1: m_stSinkCfg.u8Domain = AF_BOSCH_INC_ADR; break;
                case 2: m_stSinkCfg.u8Domain = AF_BOSCH_INC_LINUX; break;
                default: m_stSinkCfg.u8Domain = AF_BOSCH_INC_ADR; break;
                }
            }
            else if(key.compare(str_sinkdev_stream) == 0)
            {
                m_stSinkCfg.u8Stream = (tU8)strtoul(value.c_str(), NULL, 0);
                switch(m_stSinkCfg.u8Stream)
                {
                case 1: m_stSinkCfg.u8Stream = SOCK_STREAM; break;
                case 2: m_stSinkCfg.u8Stream = SOCK_DGRAM;  break;
                default: m_stSinkCfg.u8Stream = SOCK_STREAM; break;
                }
            }
            else
            {
                ETG_TRACE_FATAL(("vd_adr3_INC::bParseSinkConfiguration:  !! Invalid configuration line read !!"));
            }

            ETG_TRACE_USR1(("vd_adr3_INC::bParseSinkConfiguration:  %s",value.c_str()));
        }


        ETG_TRACE_USR1(("vd_adr3_INC::bParseSinkConfiguration:  Server name : %s",m_stSinkCfg.c8HostRemote));
        ETG_TRACE_USR1(("vd_adr3_INC::bParseSinkConfiguration:  Client name : %s",m_stSinkCfg.c8HostLocal));
        ETG_TRACE_USR1(("vd_adr3_INC::bParseSinkConfiguration:  Server port : %d",m_stSinkCfg.u16AudioPort));
        ETG_TRACE_USR1(("vd_adr3_INC::bParseSinkConfiguration:  Client port : %d",m_stSinkCfg.u16AudioPort));
        ETG_TRACE_USR1(("vd_adr3_INC::bParseSinkConfiguration:  Domain      : %d",m_stSinkCfg.u8Domain));
        ETG_TRACE_USR1(("vd_adr3_INC::bParseSinkConfiguration:  Stream      : %d",m_stSinkCfg.u8Stream));

        /* close the file */
        cFile.close();
    }
    else
    {
        bStatus = false;
        ETG_TRACE_FATAL(("vd_adr3_INC::bParseSinkConfiguration:  !! failed to open configuration file for reading !!"));
        ETG_TRACE_FATAL(("vd_adr3_INC::bParseSinkConfiguration:  !! LOAD DEFAULT CONFIGURATION !!"));
    }

    return bStatus;
}

/******************************************************************************
 * FUNCTION     : bOpenSinkDevice
 * DESCRIPTION  : Opens the socket and connects with server based on the sink
 *              : device configuration stored structure(read on startup).
 *              : getaddrinfo() is used to get server and client info from
 *              : /etc/host file.
 * PARAMETER    : void
 * RETURN VALUE : true on successful socket creation and connect or else false.
 * HISTORY      :
 * Date         | Author                  | Modification
 * 12.12.2018   | RBEI/ECO - Ranjit Susal | Initial Revision
 *****************************************************************************/
tBool vd_adr3_INC::bOpenSinkDevice() const
{
    tBool     bStatus = true;
    tInt      iReturn = 0;
    tInt      iEnable = 1;
    tInt      errsv   = 0;
    struct addrinfo hints;
    struct addrinfo *pHostAddrLocal  = NULL;
    struct addrinfo *pHostAddrRemote = NULL;

    ETG_TRACE_USR1(("vd_adr3_INC::bOpenSinkDevice:  ENTERED"));

    /*----------------------- socket creation and dgram init ------------------------*/

    /* create a socket for communication with sink device */
    m_nADRSocketFd = socket( m_stSinkCfg.u8Domain, m_stSinkCfg.u8Stream, 0);

    if(0 > m_nADRSocketFd)
    {
        bStatus = false;
        ETG_TRACE_USR1(("vd_adr3_INC::bOpenSinkDevice:  !! failed to create socket !!"));
        return bStatus;
    }

    /* set socket options and send timeout */
    errno = 0;  /* init error number */
    if(setsockopt(m_nADRSocketFd,SOL_SOCKET,SO_SNDTIMEO,(char*)&iEnable,sizeof(iEnable)) < 0)
    {
        errsv = errno; /* save the error number */
        ETG_TRACE_USR1(("vd_adr3_INC::bOpenSinkDevice:  !! failed to send timeout of socket !!"));
        ETG_TRACE_USR1(("vd_adr3_INC::bOpenSinkDevice:  !! failed with error(%d)->(%s) !!",errsv,strerror( errsv )));
    }

    ETG_TRACE_USR1(("vd_adr3_INC::bOpenSinkDevice:  sink device file descriptor: = %u ", m_nADRSocketFd));

    /* Initialize the dgram */
    m_pSkDgram = dgram_init(m_nADRSocketFd, DGRAM_MAX, NULL);
    if (NULL == m_pSkDgram)
    {
        bStatus = false;
        ETG_TRACE_ERR(("vd_adr3_INC::bOpenSinkDevice:  !! failed to initialize dgram service !!"));
    }

    /*----------------------- client/local host setup ------------------------*/

    memset( &hints, 0, sizeof(struct addrinfo));
    hints.ai_family = AF_INET;

    /* get the local host/client information ex: adr3-local/alsactrl-local
     * and BIND to that host which is dynamically created at runtime
     * by kernel in the file /etc/hosts
     */
    iReturn = getaddrinfo(m_stSinkCfg.c8HostLocal, m_stSinkCfg.c8LocalPort, &hints, &pHostAddrLocal);
    if ( 0 != iReturn )
    {
        ETG_TRACE_USR1(("vd_adr3_INC::bOpenSinkDevice:  !! failed to getaddrinfo for %s !!",m_stSinkCfg.c8HostLocal));
        ETG_TRACE_USR1(("vd_adr3_INC::bOpenSinkDevice:  !! failed with error(%d)->(%s) !!",iReturn,gai_strerror( iReturn )));
        bStatus = false;
    }

    if(NULL == pHostAddrLocal)
    {
        bStatus = false;
        ETG_TRACE_USR1(("vd_adr3_INC::bOpenSinkDevice:  !! failed no such client/local host %s !!",m_stSinkCfg.c8HostLocal));
    }
    else
    {
        /* bind to a local port to receive response */
        errno = 0; /* init error number */
        if ( bind( m_nADRSocketFd, pHostAddrLocal->ai_addr, pHostAddrLocal->ai_addrlen ) < 0)
        {
            errsv = errno; /* save the error number */
            bStatus = false;
            ETG_TRACE_USR1(("vd_adr3_INC::bOpenSinkDevice:  !! failed to BIND for client/local host Port No: %d !!", m_stSinkCfg.u16AudioPort));
            ETG_TRACE_USR1(("vd_adr3_INC::bOpenSinkDevice:  !! failed with error(%d)->(%s) !!",errsv,strerror( errsv )));
        }

        /* address info no longer needed for local host/client */
        freeaddrinfo(pHostAddrLocal);
    }

    /*----------------------- server/remote host setup ------------------------*/

    memset( &hints, 0, sizeof(struct addrinfo));
    hints.ai_family = AF_INET;

    /* get the server/remote host information for ex: adr3/alsactrl
     * and CONNECT to that host which is dynamically created at runtime
     * by kernel in the file /etc/hosts
     */
    iReturn = getaddrinfo(m_stSinkCfg.c8HostRemote, m_stSinkCfg.c8RemotePort, &hints, &pHostAddrRemote);
    if ( 0 != iReturn )
    {
        ETG_TRACE_USR1(("vd_adr3_INC::bOpenSinkDevice:  !! failed to getaddrinfo for %s !!",m_stSinkCfg.c8HostRemote));
        ETG_TRACE_USR1(("vd_adr3_INC::bOpenSinkDevice:  !! failed with error(%d)->(%s) !!",iReturn,gai_strerror( iReturn )));
        bStatus = false;
    }

    if(NULL == pHostAddrRemote)
    {
        bStatus = false;
        ETG_TRACE_USR1(("vd_adr3_INC::bOpenSinkDevice:  !! failed no such server/remote host %s !!",m_stSinkCfg.c8HostRemote));
    }
    else
    {
        /* Connect to port of the sink device ex: adr3/alsactrl.
         * Note to Developer > Configure remote client and server by following commands, or else connect call will fail
         */
        errno = 0; /* reset error number */
        if ( connect( m_nADRSocketFd, pHostAddrRemote->ai_addr, pHostAddrRemote->ai_addrlen ) < 0 )
        {
            errsv = errno; /* save the error number */
            bStatus = false;
            ETG_TRACE_USR1(("vd_adr3_INC::bOpenSinkDevice:  !! failed to CONNECT for server/remote host Port No: %d !!", m_stSinkCfg.u16AudioPort));
            ETG_TRACE_USR1(("vd_adr3_INC::bOpenSinkDevice:  !! failed with error(%d)->(%s) !!",errsv,strerror( errsv )));
        }

        /* address info no longer needed for remote host/server */
        freeaddrinfo( pHostAddrRemote );
    }


    /* if the binding or connecting to sink device via INC/IPC socket fails then returns false to exist the loop */
    if(false == bStatus)
    {
        if(NULL != m_pSkDgram)
        {
            dgram_exit(m_pSkDgram);
            m_pSkDgram = NULL;
        }

        if(-1 != m_nADRSocketFd)
        {
            (void)close(m_nADRSocketFd);
            m_nADRSocketFd = -1;
        }
    }
    /**For time being till the ADR3 ctrl driver is available the
      status of the Communication state of ADR3 is made as RUNNING.*/
    else
    {
        /*TODO : This a workaround.
        During Testing it is found that no Start Up message is received from ADR sometimes.
        So as soon as the socket connection is established, we are sending 'power on' message from here and setting the communication state as run
        by assuming ADR startup is already received, this needs to be investigated*/

        ETG_TRACE_USR1(("vd_adr3_INC::bOpenSinkDevice:  !! sink device is not power up after socket creation !!"));


        //char cDefaultData[11] = {0x00, 0x10, 0x00, 0x91, 0x01, 0x0f, 0x11, 0x02, 0x00, 0x01, 0x02};
        //AUD_SET_U16(&cDefaultData[enAdrMsgOffset_CLIENT_ID], AUD_CLIENT_ID);//Set client ID
        //vSendINC(cDefaultData,11);

        vd_adr3_If::vSetCommunicationState(enComState_Run);
    }

    return bStatus;
}
