
#ifndef ADR3CONFIGSTREAMERS_HEADER
#define ADR3CONFIGSTREAMERS_HEADER

// =============================================================================
//
//                        A D R 3 - F i l e  S t r e a m e r s
//
//                     Set of utility classes to handle streaming
//                         to and from ADR3-formatted buffers
//
// Concept partly taken from FI message framework (fi_msgfw) iocontext
//
// Author: Martin Koch, Fa. ESE, 2013-09-24
//


// =============================================================================
//
//         conversion functions

inline tS16 S16 (tU8 MSB, tU8 LSB)
{
   // combine MSB and LSB to 16-Bit value
   return (tS16)((MSB << 8) + LSB);
}

inline tU8 LSB(tU16 Value)
{
   // extract least significant byte (LSB) from 16-bit value
   return (tU8)Value;
}

inline tU8 MSB(tU16 Value)
{
   // extract most significant byte (MSB) from 16-bit value
   return (tU8)((Value & 0xff00) >> 8);
}

// =============================================================================

// mapping of data pool element names to data function IDs
tCString sGetDpElementName(adr3_tenDataFunctionID enDataFID);

// =============================================================================

#define MAP_ID  (tU16)0x0001
#define CONF_ID (tU16)0x0002

// =============================================================================
//
//                          adr3 tcl  S t r e a m e r  B a s e
//
//                         Base Class for Read-Write Operations
//                           to / from ADR3-formatted buffer
//
// Note: Stand-alone usage is prohibited; no public constructor is available.
//       Use derived classes instead.

class adr3_tclStreamerBase
{
   // member variables
   // (must be in this order due to initialization dependencies)
protected:
   tCString sName;
   const tU32 u32BufSize;
private:
   tU8* pLocalBuffer;
protected:
   tU8* const pBuffer;
   tU8* pu8Position;


  // member functions
public:
   /* default constructor */ adr3_tclStreamerBase();  // intentionally not implemented
   virtual /* destructor */ ~adr3_tclStreamerBase();

   // Deep copy not supported: below functions are declared, but intentionally not implemented.
   // Use references or pointers instead.
   /* copy constructor */ adr3_tclStreamerBase(const adr3_tclStreamerBase& roRef);  // intentionally not implemented
   const adr3_tclStreamerBase& /* assignment */ operator=(const adr3_tclStreamerBase& roRef); // intentionally not implemented

   inline tCString sGetName() const;
   tVoid  vPrintFormatted (tCString sFilePath);
   tVoid  vDumpHex (FILE* poDumpFile) const;
   tVoid  vViewData(const tU8*& rpu8Data, tU32& ru32DataSize) const;
   tFileVersion oReadVersion() const;
   tCRCCheckSum oReadCRC() const;

   // send entire file to ADR3 co-processor
   tBool bSendToADR(tU8 u8FileID);

   // reset read/write position back to beginning
   inline tVoid   vRewind();

   // helper functions to manipulate data-pool elements
   static tCString sGetDpElementName(adr3_tenDataFunctionID enDataFID);
   static tU32 u32GetDatapoolElementSize (tCString sDpElementName);
   static tVoid vCopyFromDatapool(tCString sDpElementName, tU8 pau8DestinationBuffer[], tU32 u32DestBufSize, tU32& ru32BytesCopied);
   tVoid vUpdateDatapoolElement();

protected:
   /* constructor */ adr3_tclStreamerBase(tCString sName, tU8* pBuffer, tU32 u32BufSize);  // uses existing buffer in read-context
   /* constructor */ adr3_tclStreamerBase(tCString sName, tU32 u32BufSize);    // allocates new buffer for write context

   // print a 2-byte hex-representation of given value into file
   static tVoid  fprintValue (FILE* f, tS16 s16Value);

   // print buffer content as C++ initializer list in human readable form,
   // arranged according to internal structure
   tVoid    vPrintFormatted (FILE* f);


   inline tBool    bCheckSpace (tU32 u32CheckSize) const;
   inline tU32     u32GetFreeSize () const;
   inline tU32     u32GetFilledSize () const;

};

// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

inline tCString adr3_tclStreamerBase :: sGetName() const
   {  return sName;   }

inline tBool adr3_tclStreamerBase :: bCheckSpace (tU32 u32CheckSize) const
   {  return ((pu8Position + u32CheckSize) < pBuffer + u32BufSize) ? (tBool)TRUE : (tBool)FALSE;  }

inline tU32 adr3_tclStreamerBase :: u32GetFreeSize () const
   {  return (pBuffer + u32BufSize - pu8Position) > 0 ? (tU32)(pBuffer + u32BufSize - pu8Position) : 0;  }

inline tU32 adr3_tclStreamerBase :: u32GetFilledSize () const
   {  return (pu8Position - pBuffer) > 0 ? (tU32)(pu8Position - pBuffer) : 0;  }

inline tVoid adr3_tclStreamerBase :: vRewind()
   {  pu8Position = pBuffer + 4;  }

// =============================================================================
//
//                                 S e r i a l i z e r
//
//                     Stream for Writing into ADR3-formatted Buffer
//
// extend StreamerBase class with writing functionality


class adr3_tclSerializer : public adr3_tclStreamerBase
{

public:
   /* default constructor */ adr3_tclSerializer (); // intentionally not implemented
   /* constructor */ adr3_tclSerializer (const adr3_tclFile& roAudioFile);  // serialize structured file
   /* constructor */ adr3_tclSerializer (adr3_tenDataFunctionID enDataFID, tU8* pu8Data, tU32 u32Size);  // use existing buffer (e. g. from SoundTool)

   // methods for writing of single values
   tVoid vUpdateVersion(const tFileVersion& roVersion);
   tVoid vUpdateCRC();
   tVoid vWriteString (tCString sString);
   inline tVoid vWriteU8  (const tU8 u8Value);
   tVoid vWriteS16 (const tS16 s16Value);
   inline tVoid vWriteU16 (const tU16 u16Value);
};

// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

inline  tVoid  adr3_tclSerializer :: vWriteU8  (const tU8 u8Value)
   { vWriteS16((tS16) u8Value);  }

inline  tVoid  adr3_tclSerializer :: vWriteU16 (const tU16 u16Value)
   { vWriteS16((tS16) u16Value);  }


// =============================================================================
//
//                                     P a r s e r
//
//                     Stream for Reading from ADR3-formatted Buffer
//
// extend StreamerBase class with reading functionality


class adr3_tclParser : public adr3_tclStreamerBase
{

public:
   /* default constructor */ adr3_tclParser ();  // intentionally not implemented
   /* constructor */ adr3_tclParser (tCString sName, const tU8* pu8Start, size_t u32Size);  // provide buffer
   /* constructor */ adr3_tclParser (adr3_tenDataFunctionID enDataFID);  // use Datapool element

   // read single values
   tString sRead();
   tU8  u8Read();
   inline tU16 u16Read();
   tS16 s16Read();
};

// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

inline tU16 adr3_tclParser :: u16Read ()
   { return (tU16) s16Read();  }


// =============================================================================

#endif // ADR3CONFIGSTREAMERS_HEADER
