//
// VolumeManager/Configuration/Stream.cpp
//
//  Created on: Jun 26, 2014
//      Author: Martin Koch, Fa. ESE.de
//


// framework
#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#include <etrace_if.h>

#include "AudioProxy/aud_gio_dbus_handler.h"
#include "AudioProxy/aud_gio_dbus_audproc_proxy.h"

#include "Volume/Types.h"
#include "Volume/Utilities/Uncopyable.h"
#include "./Stream.h"
// - - - - - - - - - - - - -

#include "Volume/Utilities/StringCopy.hpp"
#include "Volume/VolumeManager.h"
#include "Volume/Configuration/ConfigDetails.hpp"
#include "Volume/Configuration/dBCalculator.h"
#include "InternalComponentCommunication/DataTypes/MessageDataTypes/AmpVolumeData.h"

#include "fc_audiomanager_trace.h"
#define ETG_DEFAULT_TRACE_CLASS  TR_CLASS_FC_AUDIOMANAGER_VOLUME
#include "trcGenProj/Header/Stream.cpp.trc.h"

namespace VolumeManager
{

   // ==========================================================================
   //
   //                               S t r e a m
   //

   /* constructor */ Stream:: Stream (const StreamConfig& stream)
    : _pConfig(&stream)
    , _enCurrentSource(AUD_INT_SRC_NOSOURCE)
    , _enCurrentMode(EN_INT_VOLMODE_NORMAL)
    , _enRecentEntertainmentSource(AUD_INT_SRC_NOSOURCE)
    , _s16AttenuationdBLevel(MUTE)
    , _bCANAttenuationActive(false)
   {

   }

   // -----------------------------------------------------------------------------

   /* default constructor */ Stream:: Stream ()
     : _pConfig(NULL)
     , _enCurrentSource(AUD_INT_SRC_NOSOURCE)
     , _enCurrentMode(EN_INT_VOLMODE_NOT_DEF)
     , _enRecentEntertainmentSource(AUD_INT_SRC_NOSOURCE)
     , _s16AttenuationdBLevel(MUTE)
     , _bCANAttenuationActive(false)
    {

    }

   // -----------------------------------------------------------------------------

   void Stream:: swap (Stream& rhs)
   {
      // exchanges (swaps) the content of rhs and this

      Stream temp;
      temp._pConfig = _pConfig;
      temp._enCurrentSource = _enCurrentSource;
      temp._enCurrentMode = _enCurrentMode;
      temp._enRecentEntertainmentSource = _enRecentEntertainmentSource;
      temp._s16AttenuationdBLevel = _s16AttenuationdBLevel;
      temp._bCANAttenuationActive = _bCANAttenuationActive;
      _pConfig = rhs._pConfig;
      _enCurrentSource = rhs._enCurrentSource;
      _enCurrentMode = rhs._enCurrentMode;
      _enRecentEntertainmentSource = rhs._enRecentEntertainmentSource;
      rhs._pConfig = temp._pConfig;
      rhs._enCurrentSource = temp._enCurrentSource;
      rhs._enCurrentMode = temp._enCurrentMode;
      rhs._enRecentEntertainmentSource = temp._enRecentEntertainmentSource;
      rhs._s16AttenuationdBLevel = temp._s16AttenuationdBLevel;
      rhs._bCANAttenuationActive = temp._bCANAttenuationActive;
   }

   // -----------------------------------------------------------------------------

   /* virtual destructor */ Stream:: ~Stream ()
   {
      _pConfig = NULL;
   }

   // -----------------------------------------------------------------------------

   /* virtual */ void Stream:: setVolume (const dBCalculator& calc, tU8 userStep, Ramp ramp)
   {
      if (NULL == _pConfig)
         return;

      AmpVolumeData AmpData;
      AmpData.m_enStream = _pConfig->enStreamID;
      AmpData.m_VolStep = userStep;
      AmpData.m_VoldB   = static_cast<tS16>(calc.s16GetdBValue(userStep, _pConfig->enStreamID) * dBCalculator::dBStepMultiplicator);
      AmpData.m_RampLin = ramp.lin;
      AmpData.m_RampdB = ramp.dB;
      AmpData.m_Sink = (tU16)enGetResource();
      VolumeManager::vLaunchAmplifierCommand(AmpData);

   }

   // -----------------------------------------------------------------------------

   tenStream Stream:: enGetStreamID ()
   {
      if (_pConfig)
         return _pConfig->enStreamID;
      else
         return EN_AUDIO_SOURCE_STREAM_DEFAULT;
   }

   // -----------------------------------------------------------------------------

   tenResource Stream:: enGetResource ()
   {
      if (_pConfig && (_pConfig->pStreamSet))
         return _pConfig->pStreamSet->enResource;
      else
         return UndefinedResource;
   }

   // -----------------------------------------------------------------------------

   void Stream:: vSetCurrentSource (tenInternalSource currentSource)
   {
     ETG_TRACE_USR2(("Stream:: vSetCurrentSource this %p, _enCurrentSource %d to currentSource %d",this,_enCurrentSource, currentSource));
      _enCurrentSource = currentSource;

      // remember latest entertainment source
      if ((EN_INT_VOLMODE_NORMAL == _enCurrentMode) && (isEntertainmentSource(currentSource)))
         _enRecentEntertainmentSource = currentSource;
   }

   // -----------------------------------------------------------------------------

   const char* Stream:: getName ()
   {
      if (_pConfig)
         return _pConfig->name;
      else
         return NULL;
   }

   // -----------------------------------------------------------------------------

   /* static */ bool Stream:: isEntertainmentSource (tenInternalSource enSource)
   {
      switch (enSource)
      {
      case AUD_INT_SRC_FM :
      case AUD_INT_SRC_AM :
      case AUD_INT_SRC_DAB :
      case AUD_INT_SRC_XM :
      case AUD_INT_SRC_CPRA :
      case AUD_INT_SRC_CDDA :
      case AUD_INT_SRC_AUX :
      case AUD_INT_SRC_BT_AUDIO :
      case AUD_INT_SRC_AUX2 :
      case AUD_INT_SRC_IPOD :
      case AUD_INT_SRC_PANDORA :
      case AUD_INT_SRC_CPRA2 :
      case AUD_INT_SRC_IAPPS :
      case AUD_INT_SRC_SPI_MAIN :
      case AUD_INT_SRC_SPI_ALERT :
      case AUD_INT_SRC_DRM :
      case AUD_INT_SRC_WEB_ENT :
      //case AUD_INT_SRC_ECORE_MAIN2 :
      case AUD_INT_SRC_BLU_RAY :
      case AUD_INT_SRC_EXTERNAL_CD :
      case AUD_INT_SRC_AUX3 :
      case AUD_INT_SRC_DTV:
      case  AUD_INT_SRC_VPA_MAIN:
      case  AUD_INT_SRC_RSE_IN:
            return true;

         default:
            return false;
      }
   }

   // -----------------------------------------------------------------------------
   tS16 Stream::s16GetStreamAttenuationLevel()
   {
      return _s16AttenuationdBLevel;
   }
   // -----------------------------------------------------------------------------
   void Stream::vSetStreamAttenuationlevel(tS16 Level)
   {
     _s16AttenuationdBLevel = Level;
   }
   // -----------------------------------------------------------------------------
   void Stream::vSetCANAttenuationActive()
   {
     _bCANAttenuationActive = true;
   }
   // -----------------------------------------------------------------------------
   void Stream::vClearCANAttenuationActive()
   {
     _bCANAttenuationActive = false;
   }
   // -----------------------------------------------------------------------------
   bool Stream::bIsCANAttenuationActiveOnStream()
   {
      return _bCANAttenuationActive;
   }

   // -----------------------------------------------------------------------------

}  // namespace VolumeManager

