//
// SPIDuckingFilter.h
//
//  AUDIO_PROP_IPOD_DUCKING:
//    This property specially designed for Ipod-Out also known as CarPlay.
//    It is used to attenuate the main stream while an announcement on the mix stream is played.
//    This feature is called "ducking" by Apple Inc.
//
//  Created on: Nov 27, 2014
//      Author: Martin Koch, Fa. ESE
//



// framework
#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#include <etrace_if.h>

#include "Volume/Types.h"  // implicitly links <midw_fi_if.h>
#include "Volume/Utilities/Uncopyable.h"
#include "Volume/Utilities/Array.hpp"
#include "./IFunction.h"
#include "./FunctionBase.h"
#include "InternalComponentCommunication/InternalCommunicationAdapter.h"
#include "SPIDuckingFilter.h"
// - - - - - - - - - - - - -

#include "Volume/Configuration/Configuration.h"
#include "Volume/Engine/Stream.h"
#include "Volume/Engine/StreamSet.h"
#include "Volume/Configuration/dBCalculator.h"
#include "Volume/Configuration/ConfigDetails.hpp"
#include "Volume/VolumeManager.h"
#include "Volume/PropertyStore.h"
#include "Volume/Engine/Actions/IAction.h"
#include "Volume/Engine/Actions/ActionBase.h"
#include "Volume/Engine/Actions/VolumeSetAction.h"
#include "Volume/TypeConversions/Functions.h"

#include "InternalComponentCommunication/DataTypes/MessageDataTypes/VolumeData.h"
#include "InternalComponentCommunication/DataTypes/MessageDataTypes/AmpVolumeData.h"
#include "InternalComponentCommunication/DataTypes/MessageDataTypes/VolumeSPIMixData.h"
#include "InternalComponentCommunication/Messages/Volume/VolumeManager/ID_SPIMixVolume.h"
#include "InternalComponentCommunication/Messages/mute/IDNotifyMute.h"
#include "InternalComponentCommunication/Messages/Source/IDIntSource.h"


#include "fc_audiomanager_trace.h"
#define ETG_DEFAULT_TRACE_CLASS  TR_CLASS_FC_AUDIOMANAGER_VOLUME
#include "trcGenProj/Header/SPIDuckingFilter.cpp.trc.h"


namespace VolumeManager
{


   // -----------------------------------------------------------------------------

   /* constructor */ SPIDuckingFilter:: SPIDuckingFilter(PropertyStore& properties, const Configuration& config
         , const FunctionConfig& functionCfg, StreamSet& streamSet)
      : FunctionBase(properties, config, functionCfg, streamSet)
      , IF_MessageObserver<PO_MessageConfig::enID>("SPIDuckingFilter")
      , _enDuckingActive(EN_INT_SPI_UNDUCK)
      , _s16VolumeDbLevel(0)
      , _activeSourceIsSPIInfo(false)
      , _enLastSPIMuteState(EN_AUDIO_MUTE_MUTERESULT_MUTE)
   {
      PostOffice<PO_MessageConfig::enID>* pPO = InternalCommunicationAdapter::POMessages;
      pPO->AddObserver(this, PO_MessageConfig::ID_SPIMixVolume);

      // register for source change notifications
      pPO->AddObserver(this, PO_MessageConfig::IDIntSource);

      // register for mute changes
      pPO->AddObserver(this, PO_MessageConfig::ID_NotifyMute);
   }


   // -----------------------------------------------------------------------------

   /* virtual destructor */ SPIDuckingFilter:: ~SPIDuckingFilter()
   {
      PostOffice<PO_MessageConfig::enID>* pPO = InternalCommunicationAdapter::POMessages;
      pPO->RemoveObserver(this, PO_MessageConfig::ID_NotifyMute);
      pPO->RemoveObserver(this, PO_MessageConfig::IDIntSource);
      pPO->RemoveObserver(this, PO_MessageConfig::ID_SPIMixVolume);
   }

   // -----------------------------------------------------------------------------

   /* virtual */ tenFunctionType SPIDuckingFilter:: enGetType ()
   {
      return enDuckingFilter;
   }


   // -----------------------------------------------------------------------------

   /* virtual */ void SPIDuckingFilter:: vAdjustAmplifierVolume (tenFadingReason /* enReason */, const VolumeData& volumeData, bool& useStandardAmplifierCommand)
   {
      // Limit volume on main channel while SPI ducking is active
      //
      // Note: this filter is effective if and only if the following conditions are met:
      //       - AUD_INT_SRC_SPI_MIX (35) is the active source
      //       - Ducking mode is enabled
      //       - active source is demuted
      //       - the requested volume change refers to MAIN stream (which is not the usual case)

      // check applicability
      if ( ! _activeSourceIsSPIInfo)
         return;
      else if (EN_INT_SPI_DUCK != _enDuckingActive)
         return;
      else if (EN_AUDIO_MUTE_MUTERESULT_DEMUTE != _enLastSPIMuteState)
         return;

      tenStream enStreamID = fiAudioChannel_StreamID_Map::getSecond(volumeData.m_Channel);
      if (enStreamID != EN_AUDIO_SOURCE_STREAM_MAIN)
         return;

      // - - - - - - - - - -

      vSetAmplifierVolume(volumeData.m_Volume);

      useStandardAmplifierCommand = false;
   }

   // -----------------------------------------------------------------------------

   /* virtual */ void SPIDuckingFilter:: MessageNotification (PO_MessageConfig::enID MsgId)
   {
      // PostOffice message observer implementation

      bool bConditionsWerePreviouslyMet = bConditionsAreMet();
      PostOffice<PO_MessageConfig::enID>* pPO = InternalCommunicationAdapter::POMessages;
      switch (MsgId)
      {
         case PO_MessageConfig::ID_SPIMixVolume:
            {
               const ID_SPIMixVolume* poMsg = pPO->QueryMessage<ID_SPIMixVolume>(MsgId);
               if (_enDuckingActive != poMsg->value.m_SPIMixMode)
               {
                  ETG_TRACE_USR2(("SPIDuckingFilter::MessageNotification() : changing SPIMixVolume: %u to %u with volume limit: %d "
                        , ETG_CENUM(tenSPIMixMode, _enDuckingActive)
                        , ETG_CENUM(tenSPIMixMode, poMsg->value.m_SPIMixMode)
                        , poMsg->value.m_VolumeDbLevel))
                  _enDuckingActive = poMsg->value.m_SPIMixMode;
                  _s16VolumeDbLevel = poMsg->value.m_VolumeDbLevel;
               }
               else
                  ETG_TRACE_USR4(("SPIDuckingFilter::MessageNotification() : leaving SPIMixVolume: %u unchanged"
                        , ETG_CENUM(tenSPIMixMode, _enDuckingActive)  ))

              if(_enDuckingActive == EN_INT_SPI_DUCK)
              {
                  Stream*    pMix2Stream = _streamSet.pGetStream(EN_AUDIO_SOURCE_STREAM_MIX2);
                  _streamSet.vSetActiveStream(EN_AUDIO_SOURCE_STREAM_MIX2);
                  tenVolumeMode enMode = pMix2Stream->enGetCurrentMode();
                  tenStream enStreamID = pMix2Stream->enGetStreamID();

                 _properties.vUpdateMode(enMode, enStreamID, _streamSet.enGetResource());
                 const SourceConfig* pSource = _config.pGetSourceConfig(AUD_INT_SRC_SPI_MIX);
                 if(NULL != pSource)
                 {
                 _properties.vSetCurrentSource(*pSource,pMix2Stream->enGetResource());
                 }
                 _properties.NotifyAllUpdates();
               }


              if(EN_INT_SPI_UNDUCK == _enDuckingActive)
              {
                 Stream*    pMainStream = _streamSet.pGetStream(EN_AUDIO_SOURCE_STREAM_MAIN);
                  _streamSet.vSetActiveStream(EN_AUDIO_SOURCE_STREAM_MAIN);
                  const SourceConfig* pSecondarySource = _config.pGetSourceConfig(pMainStream->enGetCurrentSource());
                  if(NULL != pSecondarySource)
                  {
                  _properties.vSetCurrentSource(*pSecondarySource,pMainStream->enGetResource());
                  }
                  _properties.NotifyAllUpdates();
              }
            }
            break;

         case PO_MessageConfig::IDIntSource:
            {
               const VolumeSourceData& data = pPO->QueryMessage<IDIntSource>(MsgId)->value;
               if (    (_streamSet.enGetResource() == fiResource_Map::getSecond(data.m_Resource.getResource()))
                    && (midw_fi_tcl_e8_AudioChannel::FI_EN_AUDIO_CHANNEL_MIX2 == data.m_Channel)  )
               {
                  bool bNextSourceIsSIPInfo = (AUD_INT_SRC_SPI_MIX == data.m_Source);
                  if (_activeSourceIsSPIInfo != bNextSourceIsSIPInfo)
                  {
                     ETG_TRACE_USR2(("SPIDuckingFilter::MessageNotification() : changing SPI-Info source to : %s"
                           , (bNextSourceIsSIPInfo ? "true" : "false")  ))
                     _activeSourceIsSPIInfo = bNextSourceIsSIPInfo;
                  }
                  else
                  ETG_TRACE_USR4(("SPIDuckingFilter::MessageNotification() : leaving SPI-Info source untouched as : %s"
                        , (_activeSourceIsSPIInfo ? "true" : "false")  ))
               }
            }
            break;

         case PO_MessageConfig::ID_NotifyMute:
            {
               const ID_NotifyMute* pMsg = pPO->QueryMessage<ID_NotifyMute>(MsgId);
               if (EN_AUDIO_SOURCE_STREAM_MIX2 != pMsg->enStream)
                  break;  // we are not interested in other streams than the one dedicated

               tenMuteResult enResult = pMsg->enMuteResult;
               if (enResult != _enLastSPIMuteState)
               {
                  ETG_TRACE_USR2(("SPIDuckingFilter::MessageNotification() : changing SPI mute state from %u to %u"
                        , ETG_CENUM(tenMuteResult, _enLastSPIMuteState)
                        , ETG_CENUM(tenMuteResult, enResult)  ))
                  _enLastSPIMuteState = enResult;
               }
               else
                  ETG_TRACE_USR4(("SPIDuckingFilter::MessageNotification() : leaving SPI mute state untouched as %u"
                        , ETG_CENUM(tenMuteResult, _enLastSPIMuteState)  ))

              if( EN_AUDIO_MUTE_MUTERESULT_MUTE == _enLastSPIMuteState && AUD_INT_SRC_NAVI != _properties.enGetCurrentSource() && AUD_INT_SRC_NAVI_BY_SDS != _properties.enGetCurrentSource())
              {
                  Stream*    pMainStream = _streamSet.pGetStream(EN_AUDIO_SOURCE_STREAM_MAIN);
                 _streamSet.vSetActiveStream(EN_AUDIO_SOURCE_STREAM_MAIN);
                 const SourceConfig* pSecondarySource = _config.pGetSourceConfig(pMainStream->enGetCurrentSource());
                 if(NULL != pSecondarySource)
                 {
                 _properties.vSetCurrentSource(*pSecondarySource,pMainStream->enGetResource());
                 }
                 _properties.NotifyAllUpdates();

              }
            }
            break;

         default:
            ETG_TRACE_COMP(("SPIDuckingFilter::MessageNotification() : unhandled message received: ID = %u", MsgId ))
            break;
      }

      if (bConditionsAreMet() != bConditionsWerePreviouslyMet)
         vApplyChanges();
   }

   // -----------------------------------------------------------------------------

   bool SPIDuckingFilter:: bConditionsAreMet () const
   {

      if (    _activeSourceIsSPIInfo
           && (EN_INT_SPI_DUCK == _enDuckingActive)
           && (EN_AUDIO_MUTE_MUTERESULT_DEMUTE == _enLastSPIMuteState) )
         return true;
      else
         return false;
   }

   // -----------------------------------------------------------------------------

   void SPIDuckingFilter:: vApplyChanges () const
   {
      Stream*    pMainStream = _streamSet.pGetStream(EN_AUDIO_SOURCE_STREAM_MAIN);
      if (NULL == pMainStream)
      {
         ETG_TRACE_FATAL(("SPIDuckingFilter::vApplyChanges() - E R R O R : could not locate MAIN stream"))
         return;
      }
      tenInternalSource enActiveSourceOnMain = pMainStream->enGetCurrentSource();
      tU8 VolStep = 0;
      const SourceConfig* pSourceCfg = _config.pGetSourceConfig(enActiveSourceOnMain);
      if (NULL == pSourceCfg)
      {
         ETG_TRACE_ERR(("SPIDuckingFilter::vApplyChanges() : Source configuration not found for source %u"
               , ETG_CENUM(tenInternalSource, enActiveSourceOnMain)))
         return;
      }

    const tU8*  pMainVolume = _properties.pGetVolume(_streamSet.enGetResource(), pSourceCfg->u8GetGroupType());

      if(enActiveSourceOnMain == AUD_INT_SRC_FM_TA || enActiveSourceOnMain == AUD_INT_SRC_DAB_TA)
      {
         #ifdef VARIANT_S_FTR_ENABLE_FEAT_AUDIO_TA_DAN
        const tU8* pEntVolume  = _properties.pGetVolume(pMainStream->enGetResource(), midw_fi_tcl_e8_Aud_VolumeType::FI_EN_VOLUME_ENTERTAINMENT);

          if(pEntVolume)
            VolStep = (tU8)(*pEntVolume + _properties.s8GetDanValue());   //gen4 prio3 warning fix

         #else
           if(pMainVolume)
              VolStep = *pMainVolume;
         #endif
          ETG_TRACE_USR4(("SetDependentVolume() -  re-setting dependent volume to %u for source %u"
                           , VolStep, ETG_CENUM(tenInternalSource, pMainStream->enGetCurrentSource())))
      }
      else
      {
      if (NULL == pMainVolume)
      {
       ETG_TRACE_ERR(("SPIDuckingFilter::vApplyChanges() : MAIN volume pointer not found for source %u"
           , ETG_CENUM(tenInternalSource, enActiveSourceOnMain)))
       return;
      }

      VolStep = *pMainVolume;
      }
      if (bConditionsAreMet())
         ETG_TRACE_USR3(("SPIDuckingFilter::vApplyChanges() : applying ducking filter on MAIN volume for source %u and volume %u"
               , ETG_CENUM(tenInternalSource, enActiveSourceOnMain), VolStep))
      else
         ETG_TRACE_USR3(("SPIDuckingFilter::vApplyChanges() : releasing ducking filter on MAIN volume for source %u and volume %u"
               , ETG_CENUM(tenInternalSource, enActiveSourceOnMain), VolStep))

      vSetAmplifierVolume(VolStep);
   }

   // -----------------------------------------------------------------------------

   void SPIDuckingFilter:: vSetAmplifierVolume (tU8 u8MainVolumeUserStep) const
   {
      // establish preconditions
      Stream*    pMainStream = _streamSet.pGetStream(EN_AUDIO_SOURCE_STREAM_MAIN);
      if (NULL == pMainStream)
      {
         ETG_TRACE_FATAL(("SPIDuckingFilter::vSetAmplifierVolume() - E R R O R : could not locate MAIN stream"))
         return;
      }
      const dBCalculator* pCalc = _config.pGetDBCalculator(pMainStream->enGetCurrentSource());
      if (NULL == pCalc)
      {
         ETG_TRACE_ERR(("SPIDuckingFilter::vSetAmplifierVolume() - E R R O R : could not locate dB-calculator for active source %u on MAIN stream"
               , ETG_CENUM(tenInternalSource, pMainStream->enGetCurrentSource())))
         return;
      }

      // calculate limited dB-value for amplifier
      tS16 dB  = pCalc->s16GetdBValue(u8MainVolumeUserStep, EN_AUDIO_SOURCE_STREAM_MAIN);
      if (bConditionsAreMet() && (dB > _s16VolumeDbLevel))
      {
         ETG_TRACE_USR2(("SPIDuckingFilter::vLimitAmplifierVolume() : reducing MAIN volume from %d to %d"
               , dB, _s16VolumeDbLevel))
         dB = _s16VolumeDbLevel;
      }

      // apply filtering
      AmpVolumeData AmpData;
      AmpData.m_enStream = EN_AUDIO_SOURCE_STREAM_MAIN;
      AmpData.m_VolStep  = u8MainVolumeUserStep;
      AmpData.m_VoldB    = (tS16)(dB * dBCalculator::dBStepMultiplicator);
      Ramp ramp          = _streamSet.queryRamp(EN_AUDIO_FADING_NOTIFY_CHG_VOL, EN_AUDIO_FADING_RES_USR, AmpData.m_enStream);
      AmpData.m_RampLin = ramp.lin;
      AmpData.m_RampdB = ramp.dB;
      VolumeManager::vLaunchAmplifierCommand(AmpData);
   }

   // -----------------------------------------------------------------------------

}  // namespace VolumeManager




