//
//  VolumeManager/Engine/Functions/PDCAttenuator.cpp
//
//  Created on: Sep 26, 2014
//      Author: Martin Koch, Fa. ESE
//


// framework
#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#include <etrace_if.h>  // implicitly links generic <osal_if.h>

// project includes
#define MIDW_FI_S_IMPORT_INTERFACE_MIDW_MASCFFI_ERRORCODES  // add loading of error codes
#include "Volume/Types.h"  // implicitly links types from <midw_fi_if.h>
#include "Volume/Utilities/Uncopyable.h"
#include "Volume/Utilities/Array.hpp"
#include "./IFunction.h"
#include "./FunctionBase.h"
#include "InternalComponentCommunication/MessageConfig.h"

#include "./PDCAttenuator.h"
// - - - - - - - - - - - - -

#include "Volume/VolumeManager.h"
#include "Volume/PropertyStore.h"
#include "Volume/Engine/StreamSet.h"
#include "Volume/Engine/Actions/IAction.h"
#include "Volume/Engine/Actions/ActionBase.h"
#include "Volume/Engine/Actions/VolumeSetAction.h"
#include "Volume/Configuration/Configuration.h"
#include "Volume/Configuration/ConfigDetails.hpp"
#include "Volume/Configuration/dBCalculator.h"
#include "Volume/TypeConversions/Functions.h"

#include "InternalComponentCommunication/DataTypes/MessageDataTypes/VolumeData.h"
#include "InternalComponentCommunication/DataTypes/MessageDataTypes/AmpVolumeData.h"  // for amplifier command
//#include "InternalComponentCommunication/DataTypes/MessageDataTypes/VolumeModeData.h"
#include "fc_audiomanager_service_Audio_Function.h"  // for fading module

#include "fc_audiomanager_trace.h"
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_FC_AUDIOMANAGER_VOLUME
#include "trcGenProj/Header/PDCAttenuator.cpp.trc.h"



namespace VolumeManager
{

   // -----------------------------------------------------------------------------
   //
   // class   P a r k - Di s t a n c e - C o n t r o l - V o l u m e - F i l t e r
   //
   //


   /* constructor */ PDCAttenuator:: PDCAttenuator (PropertyStore& properties, const Configuration& config
         , const FunctionConfig& functionCfg, StreamSet& streamSet)
      : FunctionBase(properties, config, functionCfg, streamSet)
      , _enAttenuationStatus(Inactive)
      , _attenuationValues(7)
      , _enCurrentMode(EN_INT_VOLMODE_NOT_DEF)
   {
      // in Configuration.xml a parameter values='list of 7 unsigned byte values' is required
      // so provide maximum allowed volume for OFF, LIGHTER, LIGHT, MEDIUM, STRONG, STRONGER, MUTE

      vPopulateAttenuationValues (pGetParameterValue(functionCfg.params, "values"));
   }

   // -----------------------------------------------------------------------------

   /* virtual destructor */ PDCAttenuator:: ~PDCAttenuator ()
   {

   }

   // -----------------------------------------------------------------------------

   /* virtual */ tenFunctionType PDCAttenuator:: enGetType ()
   {
      return enPDCAttenuator;
   }

   // -----------------------------------------------------------------------------

   // CCA support
   bool PDCAttenuator:: bProcessPDCCommand (midw_fi_tcl_e8_PDCAttenuation::tenType enCommand, tU16& u16Error)
   {
      tU8 newVolume = _properties.u8GetVolume();

      switch (enCommand)
      {
         case midw_fi_tcl_e8_PDCAttenuation::FI_EN_AUDIO_PDC_COMMAND_START:
            _enAttenuationStatus = Active;
            break;

         case midw_fi_tcl_e8_PDCAttenuation::FI_EN_AUDIO_PDC_COMMAND_STOP:
            _enAttenuationStatus = Inactive;
            break;

         case midw_fi_tcl_e8_PDCAttenuation::FI_EN_AUDIO_PDC_COMMAND_BREAK:
            // disable volume attenuation and use attenuated volume as new volume
            _enAttenuationStatus = Inactive;
            newVolume = newVolumeAfterBreak();
            break;

         default:
            u16Error = MIDW_MASCFFI_C_U16_ERROR_PARAMETEROUTOFRANGE;
            return false;
      }


      _properties.vUpdatePDCAttenuationCommand(enCommand);

      // delegate to embedded VolumeSet action
      VolumeData volumeData;
      volumeData.m_Channel = fiAudioChannel_StreamID_Map::getFirst(_streamSet.enGetActiveStreamID());
      volumeData.m_Resource.setResource(fiResource_Map::getFirst(_streamSet.enGetResource()));
      volumeData.m_VolumeType.setVolumeType(midw_fi_tcl_e8_AudioVolumeType::FI_EN_AUDIO_VAL_TYPE_ABS);
      volumeData.m_Volume = newVolume;
      VolumeSetAction(_properties, _config, volumeData, EN_AUDIO_FADING_RES_USR).vExecute (_streamSet);

      _properties.NotifyAllUpdates();
      return true;
   }

   // -----------------------------------------------------------------------------


   bool PDCAttenuator:: bProcessPDCLevel (tU8 PDCAttenutionLevel, tU16& /* u16Error */)
   {
      _properties.vUpdatePDCAttenuationLevel(PDCAttenutionLevel);

      _properties.NotifyAllUpdates();
      return true;
   }

   // -----------------------------------------------------------------------------

   /* virtual */ void PDCAttenuator:: vAdjustVolume (tenFadingReason enReason, VolumeData& volumeData)
   {
      if (_enAttenuationStatus != WaitForBreakingVolumeCommand)
         return;
      else if (EN_AUDIO_FADING_RES_USR != enReason)
         return;

      const SourceConfig* pSourceCfg = _config.pGetSourceConfig(_properties.enGetCurrentSource());

      if (pSourceCfg && affectsVolumeGroup(pSourceCfg->u8GetGroupType()))
      {
         // disable volume attenuation and use attenuated volume as new volume
         _enAttenuationStatus = Inactive;
         _properties.vUpdatePDCAttenuationCommand(midw_fi_tcl_e8_PDCAttenuation::FI_EN_AUDIO_PDC_COMMAND_NOT_DEF);
         volumeData.m_Volume = newVolumeAfterBreak();
      }

      /*ToDo : Check how to handle the SETUP_PDC_ATTENUATION mode. As of now no requirements for this?!!!*/
   }

   // -----------------------------------------------------------------------------


   /* virtual */ void PDCAttenuator:: vAdjustAmplifierVolume (tenFadingReason /* enReason */, const VolumeData& volumeData, bool& useStandardAmplifierCommand)
   {
      // check applicability
      if ( ! bIsApplicable())
         return;

      const dBCalculator* pCalc = _config.pGetDBCalculator(_properties.enGetCurrentSource());
      if (NULL == pCalc)
         return;

      tenStream enActiveStream = _streamSet.enGetActiveStreamID();

      tU8 userStep = volumeData.m_Volume;
      userStep = attenuatedVolume (userStep);

      AmpVolumeData AmpData;
      AmpData.m_enStream = enActiveStream;
      AmpData.m_VolStep = userStep;
      AmpData.m_VoldB   = static_cast<tS16>(pCalc->s16GetdBValue(userStep, enActiveStream) * dBCalculator::dBStepMultiplicator);
      Ramp ramp = _streamSet.queryRamp(EN_AUDIO_FADING_NOTIFY_SET_PDC_ATTENUATION, EN_AUDIO_FADING_RES_OTHER, AmpData.m_enStream);
//      Ramp ramp = queryRamp(VolumeSetAction::enGetFadingAction (enReason), enReason, AmpData.m_enStream);
      AmpData.m_RampLin = ramp.lin;
      AmpData.m_RampdB = ramp.dB;
      VolumeManager::vLaunchAmplifierCommand(AmpData);

      useStandardAmplifierCommand = false;
      if (Active == _enAttenuationStatus)
         _enAttenuationStatus = WaitForBreakingVolumeCommand;
   }

   // -----------------------------------------------------------------------------
   bool PDCAttenuator:: bIsApplicable ()
   {
      tenVolumeMode enMode = _properties.enGetMode();

      // check applicability of our volume filtering
      if ((Inactive != _enAttenuationStatus) && (EN_INT_VOLMODE_NORMAL == enMode))
      {
         const SourceConfig* pSourceCfg = _config.pGetSourceConfig(_properties.enGetCurrentSource());

         if (pSourceCfg && affectsVolumeGroup(pSourceCfg->u8GetGroupType()))
             return (_properties.enGetPDCAttenuationStatus() == midw_fi_tcl_e8_PDCAttenuation::FI_EN_AUDIO_PDC_COMMAND_START);
      }

      else if (EN_INT_VOLMODE_SETUP_PDC_ATTENUATION == enMode) /*ToDo : Check if we need the SETUP_PDC_ATTENUATION mode*/
         return true;

      return false;
   }
   // -----------------------------------------------------------------------------

   tU8 PDCAttenuator:: newVolumeAfterBreak()
   {
      tU8 userStep = _properties.u8GetVolume();
      const SourceConfig* pSourceCfg = _config.pGetSourceConfig(_properties.enGetCurrentSource());
      if (pSourceCfg && affectsVolumeGroup(pSourceCfg->u8GetGroupType()))
         userStep = attenuatedVolume (userStep);

      return userStep;
   }

   // -----------------------------------------------------------------------------

   tU8 PDCAttenuator:: attenuatedVolume (tU8 intendedVolume)
   {
      tU8 retVal = intendedVolume;

      tU8 attenuationValue = intendedVolume;
      tU8 attenuationLevel = _properties.u8GetPDCAttenuationLevel();

      if (attenuationLevel < _attenuationValues.size())
         attenuationValue = _attenuationValues[attenuationLevel];
      else
         ETG_TRACE_ERR(("PDCAttenuator::attenuatedVolume() - E R R O R :  attenuationLevel (%u) out of range, must be < %u"
               , attenuationLevel, _attenuationValues.size()))

      if (intendedVolume > attenuationValue)
         retVal = attenuationValue;

      return retVal;
   }

   // -----------------------------------------------------------------------------

   void PDCAttenuator:: vPopulateAttenuationValues (const char* pParams)
   {
      _attenuationValues.setSize(0);

      if ((NULL == pParams) || (0 == strlen(pParams)))
      {
         ETG_TRACE_ERR(("PDCAttenuator::vPopulateAttenuationValues() - E R R O R :  required value-set not found. Turning attenuation OFF!"))
         return;
      }

      const char* pEntryPosition = pParams;
      char* pParseEnd = NULL;
      while (pEntryPosition && (pEntryPosition[0] != '\''))  // filter for trailing single quote
      {
         unsigned len = getEntryLength (pEntryPosition);
         if (len)
         {
            unsigned foundValue = static_cast<unsigned>(strtoul(pEntryPosition, &pParseEnd, 0));
            if ((pParseEnd <= pEntryPosition) || (foundValue > 255))
            {
               ETG_TRACE_ERR(("PDCAttenuator::vPopulateAttenuationValues() - E R R O R :  failed reading attenuation value %s", pEntryPosition))

               break;   // further parsing meaningless after failure
            }

            // append found value to array
            unsigned i = _attenuationValues.size();
            _attenuationValues[i] = (tU8)foundValue;
            _attenuationValues.setSize(1 + i);

            pEntryPosition = pGetNextEntry(pParseEnd);
         }
      }

      if (_attenuationValues.size() < 7)  // array to small - ignore all values
         _attenuationValues.setSize(0);

      // log
      ETG_TRACE_USR1(("PDCAttenuator::vPopulateAttenuationValues() : found %u attenuation values"
            , _attenuationValues.size()))
      for (unsigned i = 0; i < _attenuationValues.size(); ++i)
         ETG_TRACE_USR3(("PDCAttenuator::vPopulateAttenuationValues() : \t\t %u: %u", i, _attenuationValues[i]))
   }

   // -----------------------------------------------------------------------------

}  // namespace VolumeManager


