//
// HeatProtector.cpp
//
//  Created on: Sep 26, 2014
//      Author: Martin Koch, Fa. ESE
//



// framework
#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#include <etrace_if.h>

#define FI_S_IMPORT_INTERFACE_FI_MESSAGE   // import fi_tclVisitorMessage
//#define MIDW_FI_S_IMPORT_INTERFACE_MIDW_MASCFFI_FUNCTIONIDS
#define MIDW_FI_S_IMPORT_INTERFACE_MIDW_MASCFFI_TYPES
#define MIDW_FI_S_IMPORT_INTERFACE_MIDW_MASCFFI_ERRORCODES
//#define MIDW_FI_S_IMPORT_INTERFACE_MIDW_MASCFFI_SERVICEINFO

#include "Volume/Types.h"  // implicitly links <midw_fi_if.h>
#include "Volume/Utilities/Uncopyable.h"
#include "Volume/Utilities/Array.hpp"
#include "./IFunction.h"
#include "./FunctionBase.h"
#include "InternalComponentCommunication/InternalCommunicationAdapter.h"
#include "./HeatProtector.h"
// - - - - - - - - - - - - -

#include "InternalComponentCommunication/DataTypes/MessageDataTypes/VolumeData.h"
#include "InternalComponentCommunication/Messages/Diag/IDGetDiagResult.h"
#include "InternalComponentCommunication/Messages/Diag/IDNotifyDiagResult.h"
#include "InternalComponentCommunication/Messages/Startup/IDNotifyStartup.h"
#include "InternalComponentCommunication/Messages/Startup/ID_NotifyRegular100msTicks.h"
#include "InternalComponentCommunication/Messages/vd_AmpResponses/ID_AmpdeviceId.h"
//#include "InternalComponentCommunication/DataTypes/TypeDefines/ADR3Startup.h"

#include "fc_audiomanager_trace.h"
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_FC_AUDIOMANAGER_VOLUME
#include "trcGenProj/Header/HeatProtector.cpp.trc.h"

//#include "fc_audiomanager_service_Audio_Function.h"
#include "Volume/PropertyStore.h"
#include "Volume/Engine/StreamSet.h"
#include "Volume/Engine/Actions/IAction.h"
#include "Volume/Engine/Actions/ActionBase.h"
#include "Volume/Engine/Actions/VolumeSetAction.h"
#include "Volume/Configuration/ConfigDetails.hpp"
#include "Volume/TypeConversions/Functions.h"

#ifdef VARIANT_S_FTR_ENABLE_FEAT_AUDIO_RNAIVI
//Based on assumption that only AmplifierDevice1 is active.Should be extended when multiple sinks are available.
#define DIGITALAMP1 1
#define DIGITALAMP2 2
#define ABAMP1 5
#define ABAMP2 6
#endif

namespace VolumeManager
{

   // -----------------------------------------------------------------------------
   //
   //                         class  H e a t - P r o t e c t o r
   //

   /* constructor */ HeatProtector:: HeatProtector (PropertyStore& properties, const Configuration& config
         , const FunctionConfig& functionCfg, StreamSet& streamSet)
      : FunctionBase(properties, config, functionCfg, streamSet)
      , IF_MessageObserver<PO_MessageConfig::enID>("HeatProtector")
      , _behavior(enGetBehaviorFromParams (functionCfg.params))
      , _trigger(enGetTriggerFromParams (functionCfg.params))
      , _interval(getIntervalFromParams(functionCfg.params))
      , _u8MinimumReductionStep(u8GetMinimumReductionFromParams (functionCfg.params))
      , _invocationTime(0)
      , _latestUpdateTime(0)
   {
      PostOffice<PO_MessageConfig::enID>* pPO = InternalCommunicationAdapter::POMessages;
      pPO->AddObserver(this, PO_MessageConfig::ID_NotifyDiagResult);
      pPO->AddObserver(this, PO_MessageConfig::ID_AmpdeviceId);
   }

   // -----------------------------------------------------------------------------

   /* virtual destructor */ HeatProtector:: ~HeatProtector ()
   {
      PostOffice<PO_MessageConfig::enID>* pPO = InternalCommunicationAdapter::POMessages;
      try
      {
    pPO->RemoveObserver(this, PO_MessageConfig::ID_NotifyDiagResult);
    pPO->RemoveObserver(this, PO_MessageConfig::ID_AmpdeviceId);
      }
      catch (...)   {  }
   }

   // -----------------------------------------------------------------------------

   /* virtual */ tenFunctionType HeatProtector:: enGetType ()
   {
      return enHeatProtector;
   }

   // -----------------------------------------------------------------------------

   /* virtual */ void HeatProtector:: vAdjustVolume (tenFadingReason enReason, VolumeData& volumeData)
   {
      // apply filtering on published volume

      // check applicability
      if ( ! bIsApplicable(enReason, volumeData))
         return;
   }

   // -----------------------------------------------------------------------------

   /* virtual */ void HeatProtector:: vAdjustAmplifierVolume (tenFadingReason enReason, const VolumeData& volumeData, bool& useStandardAmplifierCommand)
   {
      // apply filtering on amplifier command (invisible to CCA clients)

      // check applicability
      if ( ! bIsApplicable(enReason, volumeData) )
      {
       if( bIsActive() && ( enReason != EN_AUDIO_FADING_RES_SOURCESEL ) )
         useStandardAmplifierCommand = false;
         return;
      }

      tU8 currentVolume = _properties.u8GetVolume();
      if (volumeData.m_Volume == currentVolume && enReason != EN_AUDIO_FADING_RES_SOURCESEL)
         useStandardAmplifierCommand = false;
   }

   // -----------------------------------------------------------------------------

   /* virtual */ void HeatProtector:: MessageNotification (PO_MessageConfig::enID MsgId)
   {
//      ETG_TRACE_USR3(("MessageNotification entered. MsgId:0x%02x", MsgId));

      PostOffice<PO_MessageConfig::enID>* pPO = InternalCommunicationAdapter::POMessages;
      switch (MsgId)
      {
         case PO_MessageConfig::ID_NotifyDiagResult:
         {
            const ID_NotifyDiagResult*  pMsg = pPO->QueryMessage<ID_NotifyDiagResult>(MsgId);

            vHandleDiagResult(pMsg->enErrorSite, pMsg->enErrorCode, pMsg->enErrorValue);
         }
         break;

         case PO_MessageConfig::ID_NotifyRegular100msTicks:
         {
            const ID_NotifyRegular100msTicks* pMsg = pPO->QueryMessage<ID_NotifyRegular100msTicks>(MsgId);
            OSAL_tMSecond now = pMsg->TimeStamp;

            vHandleTimerEvent(now);
         }
         break;

         case PO_MessageConfig::ID_AmpdeviceId:
           {
#ifdef VARIANT_S_FTR_ENABLE_FEAT_AUDIO_RNAIVI
             const ID_AmpdeviceId* pMsg = pPO->QueryMessage<ID_AmpdeviceId>(MsgId);
             tU8 Amp = pMsg->m_u8AmpDeviceID;
             if(Amp == DIGITALAMP1 || Amp == DIGITALAMP2)
               _u8MinimumReductionStep = 24;
             else if(Amp == ABAMP1 || Amp == ABAMP2)
               _u8MinimumReductionStep = 18;
#endif
           }
           break;

         default:
            ETG_TRACE_ERR(("MessageNotification(): unknown Message %x", MsgId));
            break;
      }
   }

   // -----------------------------------------------------------------------------

   bool HeatProtector:: bProcessReductionCommand (bool downScaling_Active, tU8 reductionInterval, tU16& u16Error)
   {
      midw_mascffi_tclMsgOverTempReductionStatus fiStatus;
      if ( ! bPopulateReductionStatus(fiStatus))
      {
         u16Error = MIDW_MASCFFI_C_U16_ERROR_INTERNALFAILURE;
         return false;
      }

      PostOffice<PO_MessageConfig::enID>* pPO = InternalCommunicationAdapter::POMessages;
      if (NULL == pPO)
      {
         u16Error = MIDW_MASCFFI_C_U16_ERROR_INTERNALFAILURE;
         return false;
      }

      bool alreadyActive = (TRUE == fiStatus.ActiveByCommand) || (TRUE == fiStatus.ActiveByAmplifier);
      bool willBeActive  = downScaling_Active || (TRUE == fiStatus.ActiveByAmplifier);
      // apply changes
      if (willBeActive == alreadyActive)
         ;  // no change in timer listening
      else if (willBeActive)
      {
         ETG_TRACE_USR2(("HeatProtector - now activating"))
     _interval = reductionInterval * 1000; //converting to milliseconds
         pPO->AddObserver(this, PO_MessageConfig::ID_NotifyRegular100msTicks);  // register for periodical timer events
         _invocationTime = OSAL_ClockGetElapsedTime();
         _latestUpdateTime = _invocationTime;
         vExecuteDecrement();
      }
      else
      {
         ETG_TRACE_USR2(("HeatProtector - now deactivating"))
         pPO->RemoveObserver(this, PO_MessageConfig::ID_NotifyRegular100msTicks);  // unregister from periodical timer events
      }

      // transfer to propertyStore
      _properties.vUpdateOverTempReduction(downScaling_Active, fiStatus.ActiveByAmplifier, fiStatus.CurrentFactor, reductionInterval);

      // notify clients
      _properties.NotifyAllUpdates();
      return true;
   }

   // -----------------------------------------------------------------------------

   void HeatProtector:: vHandleTimerEvent (OSAL_tMSecond now)
   {
      OSAL_tMSecond nextUpdateTime = _latestUpdateTime + _interval;
      if (now < nextUpdateTime)
      {
         // skip intermediate time-ticks
//         ETG_TRACE_USR4(("vHandleTimerEvent() - intermediate ID_100mSecTimer received. timestamp = %u", now))
         return;
      }
      else if ( ! bIsActive())
      {
         ETG_TRACE_ERR(("vHandleTimerEvent() - ID_100mSecTimer received while inactive. timestamp = %u", now))
         return;
      }

      ETG_TRACE_USR1(("vHandleTimerEvent() - executing ID_100mSecTimer at timestamp = %u", now))
      _latestUpdateTime = now;
      vExecuteDecrement();
      _properties.NotifyAllUpdates();
   }

   // -----------------------------------------------------------------------------

   void HeatProtector:: vHandleDiagResult (tenErrorSite enErrorSite, tenErrorCode enErrorCode, tenErrorValue enErrorValue)
   {
      if ((EN_AUDIO_DIAG_AMPLIFIER != enErrorSite) || (EN_AUDIO_DIAG_ERR_THERMAL_WARNING != enErrorCode))
      {
         ETG_TRACE_USR4(("vHandleDiagResult() - unhandled message reseived: site = %u, code = %u, value = %u"
               , ETG_CENUM(tenErrorSite, enErrorSite), ETG_CENUM(tenErrorCode,  enErrorCode), ETG_CENUM(tenErrorValue, enErrorValue)))
      }
      else
      {
         ETG_TRACE_USR4(("vHandleDiagResult() - amplifier thermal warning received: value = %u"
               , ETG_CENUM(tenErrorValue, enErrorValue)))
         _properties.vUpdateOverTempAmplifierStatus(fiAmplifierWarningLevel_Map::getFirst(enErrorValue));
         _properties.NotifyAllUpdates();
      }

   }

   // -----------------------------------------------------------------------------

   bool HeatProtector:: bIsActive ()
   {
      midw_mascffi_tclMsgOverTempReductionStatus fiStatus;
      if ( ! bPopulateReductionStatus(fiStatus))
      {
         ETG_TRACE_ERR(("bIsActive() - E R R O R  : failed to obtain reduction status"))
         return false;
      }

      return (TRUE == fiStatus.ActiveByCommand) || (TRUE == fiStatus.ActiveByAmplifier);
   }

   // -----------------------------------------------------------------------------

   bool HeatProtector:: bIsApplicable (tenFadingReason enReason, const VolumeData& volumeData)
   {
      // check applicability of our volume filtering
      if (EN_AUDIO_FADING_RES_USR != enReason)
         return false;

      tenVolumeMode enMode = _properties.enGetMode();
      if ((EN_INT_VOLMODE_NORMAL != enMode))
         return false;

      if ( ! bIsActive())
         return false;

      tU8 currentVolume = _properties.u8GetVolume();

      // Increment should be possible only if volume is less than minReduction volume
      // Decrement always possible
      if (volumeData.m_Volume > currentVolume && volumeData.m_Volume > _u8MinimumReductionStep)
      {
         return false;
      }

      // finally we are applicable
      return true;
   }

   // -----------------------------------------------------------------------------

   void HeatProtector:: vExecuteDecrement ()
   {
      tU8 u8CurrentVolume = _properties.u8GetVolume();
      if (u8CurrentVolume <= _u8MinimumReductionStep)
      {
         ETG_TRACE_USR3(("vExecuteDecrement() - current volume (%u) not above minimum reduction (%u)"
               , u8CurrentVolume, _u8MinimumReductionStep))
         return;
      }

      // prepare data for a single-step decrement on current channel of parent StreamSet
      VolumeData data;
      data.m_Channel = fiAudioChannel_StreamID_Map::getFirst(_streamSet.enGetActiveStreamID());
      data.m_Resource.setResource(fiResource_Map::getFirst(_streamSet.enGetResource()));
      data.m_VolumeType.setVolumeType(midw_fi_tcl_e8_AudioVolumeType::FI_EN_AUDIO_VAL_TYPE_DEC);
      data.m_Volume = 1;
//      tS16 m_VolumeGain;
      tenFadingReason enReason = EN_AUDIO_FADING_RES_USR;

      // delegate to VolumeSetAction
      ETG_TRACE_USR2(("vExecuteDecrement() - decrementing volume to %u", u8CurrentVolume - 1))
      VolumeSetAction(_properties, _config, data, enReason).vExecute(_streamSet);
   }

   // -----------------------------------------------------------------------------

   bool HeatProtector:: bPopulateReductionStatus (midw_mascffi_tclMsgOverTempReductionStatus& fiStatus)
   {
      const fi_tclMessageBase& msg = _properties.rGetFIMessage(MIDW_MASCFFI_C_U16_OVERTEMPREDUCTION);
      const fi_tclTypeBase&   data = msg.corfoGetTypeBase();
      if (midw_mascffi_tclToken::EN_MSG_OVERTEMPREDUCTIONSTATUS != data.s32GetTypeId())
      {
         ETG_TRACE_ERR(("bPopulateReductionStatus() - E R R O R  : invalid OverTempReductionStatus message"
               "returned from PropertyStore, having TypeID = %u"
               , data.s32GetTypeId()))
         return false;
      }

      // copy elements using assignment operator from FI catalog
      fiStatus = static_cast<const midw_mascffi_tclMsgOverTempReductionStatus&>(data);
      return true;
   }

   // -----------------------------------------------------------------------------

   /* static */ HeatProtector::tenBehavior HeatProtector:: enGetBehaviorFromParams (const char* pParams)
   {
      const char* p   = pGetParameterValue (pParams, "behavior");
      unsigned    len = getParameterLength (p);
      if (len &&(0 == strncmp(p, "incremental decrease", len)))
         return IncrementalDecrease;

      ETG_TRACE_ERR(("enGetBehaviorFromParams() - E R R O R  : failed reading parameter 'behavior'"))
      return UndefinedBehavior;
   }

   // -----------------------------------------------------------------------------

   /* static */ HeatProtector::tenTrigger  HeatProtector:: enGetTriggerFromParams (const char* pParams)
   {
      const char* p   = pGetParameterValue (pParams, "trigger");
      unsigned    len = getParameterLength (p);
      if (len && (0 == strncmp(p, "CCA command", len)))
         return CCACommand;

      ETG_TRACE_ERR(("enGetTriggerFromParams() - E R R O R  : failed reading parameter 'trigger'"))
      return UndefinedTrigger;
   }

   // -----------------------------------------------------------------------------

   /* static */ OSAL_tMSecond HeatProtector:: getIntervalFromParams (const char* pParams)
   {
      const char* p   = pGetParameterValue (pParams, "interval");
      unsigned    len = getParameterLength (p);

      if (p && len)
         // read parameter, assume decimal representation in second and scale to milliseconds
         return static_cast<OSAL_tMSecond>(1000 * strtoul(p, NULL, 10));

      // provide a default value of 1 minute
      ETG_TRACE_ERR(("getIntervalFromParams() - E R R O R  : failed reading parameter 'interval'"))
      return 60000;
   }

   // -----------------------------------------------------------------------------

   /* static */ tU8 HeatProtector:: u8GetMinimumReductionFromParams (const char* pParams)
   {
      const char* p = pGetParameterValue (pParams, "minimumReduction");
      unsigned    len = getParameterLength (p);

      if (p && len)
      {
         // read parameter, assume decimal representation
         unsigned long val = strtoul(p, NULL, 10);
         if (val < 256)
            return (tU8)val;
      }

      // fallback to default
      ETG_TRACE_ERR(("u8GetMinimumReductionFromParams() - E R R O R  : failed reading parameter 'minimumReduction'"))
      return 0;
   }

   // -----------------------------------------------------------------------------


}  // namespace VolumeManager




