//
// Engine.cpp   for sub-component VolumeManager
//
//
//  Created on: Jun 17, 2014
//      Author: Martin Koch, Fa. ESE
//

#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#include <etrace_if.h> // implicitly links <osal_if.h>

#include "Volume/Types.h"
#include "Volume/Utilities/Uncopyable.h"
#include "Volume/Utilities/Array.hpp"
#include "InternalComponentCommunication/MessageConfig.h"

#define DP_S_IMPORT_INTERFACE_FI
#include "dp_audio_if.h"
#include "./Engine.h"
// - - - - - - - - - - -

//Include public FI interface of this service.
//#define FI_S_IMPORT_INTERFACE_FI_MESSAGE   // import fi_tclVisitorMessage
//#define MIDW_FI_S_IMPORT_INTERFACE_MIDW_MASCFFI_FUNCTIONIDS
#define MIDW_FI_S_IMPORT_INTERFACE_MIDW_MASCFFI_TYPES
#define MIDW_FI_S_IMPORT_INTERFACE_MIDW_MASCFFI_ERRORCODES
//#define MIDW_FI_S_IMPORT_INTERFACE_MIDW_MASCFFI_SERVICEINFO
#include <midw_fi_if.h>

//#define SYSTEM_S_IMPORT_INTERFACE_STRING
//#include <stl_pif.h>


#include <string>

#define OSAL_S_IMPORT_INTERFACE_GENERIC
#include "osal_if.h"

#include "Volume/PropertyStore.h"
#include "Volume/Engine/Actions/IAction.h"
#include "Volume/Engine/Functions/IFunction.h"
#include "Volume/Engine/Functions/FunctionBase.h"
#include "Volume/Engine/Functions/NavVolumeFilter.h"
#include "Volume/Engine/Functions/BeepVolumeFilter.h"
#include "Volume/Engine/Functions/PDCAttenuator.h"
#include "Volume/Engine/Functions/SPIDuckingFilter.h"
#include "Volume/Engine/Functions/TAVolumeFilter.h"
#include "Volume/Engine/Functions/HeatProtector.h"
#include "Volume/Engine/Functions/DiagVolume.h"
#include "Volume/Engine/Functions/AttenuationFilter.h"

#include "Volume/TypeConversions/Functions.h"
//#include "./Configuration/Source.h"
#include "Volume/Configuration/Configuration.h"
#include "./Stream.h"
#include "./StreamSet.h"
#include "Volume/Configuration/ConfigDetails.hpp"
#include "./ActionFactory.h"
#include "Volume/Configuration/dBCalculator.h"

#include "InternalComponentCommunication/InternalCommunicationAdapter.h"
#include "InternalComponentCommunication/DataTypes/MessageDataTypes/VolumeData.h"
#include "InternalComponentCommunication/Messages/Volume/VolumeManager/ID_CCA_Start_Volume.h"
#include "InternalComponentCommunication/Messages/Volume/VolumeManager/ID_CCA_Start_VolumeLock.h"
#include "InternalComponentCommunication/Messages/Volume/VolumeManager/ID_CCA_Start_VolumeMode.h"
#include "InternalComponentCommunication/Messages/Source/IDIntSource.h"
#include "InternalComponentCommunication/Messages/mute/IDNotifyMute.h"
#include "InternalComponentCommunication/Messages/Volume/VolumeManager/ID_SPIMixVolume.h"
#include "InternalComponentCommunication/Messages/Diag/IDDiagDefSet.h"
#include "InternalComponentCommunication/Messages/Diag/IDDiagAudioGain.h"
#include "InternalComponentCommunication/Messages/Diag/IDDiagRemoteControl.h"
#include "InternalComponentCommunication/Messages/power/ID_ApplicationStatus.h"
#include "InternalComponentCommunication/Messages/Volume/VolumeManager/ID_PDC_Attenuation.h"
#include "InternalComponentCommunication/Messages/Volume/ID_AmpVolumeStatus.h"
#include "InternalComponentCommunication/Messages/Volume/ID_AmpVolumeListStatus.h"
#include "InternalComponentCommunication/Messages/Source/IDIntSourceType.h"
#include "InternalComponentCommunication/DataTypes/TypeDefines/PhoneTypes.h"
#include "InternalComponentCommunication/DataTypes/TypeDefines/Ext_Amp_ConnectionState.h"
#include "InternalComponentCommunication/DataTypes/TypeDefines/VolumeModeDefines.h"
#include "InternalComponentCommunication/Messages/Volume/VolumeManager/ID_CANAttenuation.h"
#include "InternalComponentCommunication/Messages/Volume/VolumeManager/ID_PhoneDeviceList.h"
#include "InternalComponentCommunication/Messages/Beep/ID_ADASBeepActive.h"
#include "InternalComponentCommunication/Messages/settings/IDProfileDataChanged.h"
#include "InternalComponentCommunication/Messages/BOSE_Amp/ID_BOSE_Amplifier_ConnectionStatus.h"
#include "InternalComponentCommunication/Messages/Beep/ID_UpdateAmpValList.h"
#include "Volume/Utilities/Array.hpp"

#include "fc_audiomanager_trace.h"
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_FC_AUDIOMANAGER_VOLUME
#include "trcGenProj/Header/Engine.cpp.trc.h"


namespace VolumeManager
{

   // -----------------------------------------------------------------------------

   /* constructor */ Engine:: Engine (PropertyStore& store, const Configuration& config, Fading_IF* pFader)
      :  IF_MessageObserver<PO_MessageConfig::enID>("VolumeManager_Engine")
      , _properties(store)
      , _config(config)
      , _streamSets((unsigned)_config.getStreamSetCount())
      , _ppFunctions(NULL)
      , _pPDCAttenuator(NULL)
      , _pHeatProtector(NULL)
      , _pDuckingFilter(NULL)
      , _functionCount(0)
      , _pActionFactory(new ActionFactory(_properties, _config))
      , _isAppStateNormal(false)
    , _bSPIDuckingActive(false)
    , _bCANAttenuationActiveFlag(false)
    , _bADASAttenuationActiveFlag(false)
   {
      // make sure, InternalCommunicationAdapter::POMessages is properly initialized
      (void) InternalCommunicationAdapter::getInstance();

      // determine total number of functions and allocate accordingly
      for (unsigned i = 0; i < _config.getStreamSetCount(); ++i)
      {
         const StreamSetConfig* pSetCfg = config.pGetStreamSetConfig(i);
         if (pSetCfg)
            _functionCount += pSetCfg->numberOfFunctions;
      }
      if (_functionCount)
         _ppFunctions = new IFunction*[_functionCount];  // reserve space for pointers
      _functionCount = 0;   // but indicate all unassigned

      // initialize stream sets
      for (unsigned i = 0, j = 0; i < _config.getStreamSetCount(); ++i)
      {
         const StreamSetConfig* pSetCfg = config.pGetStreamSetConfig(i);
         if (NULL == pSetCfg)
            continue;

         _streamSets.setSize(j + 1);
         StreamSet(*pSetCfg, config, pFader) .swap(_streamSets[j]);

         // establish special functions according to streamSet configuration
         if (NULL == _ppFunctions)
            continue;
         for (unsigned k = 0; k < pSetCfg->numberOfFunctions; ++k)
         {
            const FunctionConfig& functionCfg = pSetCfg->pFunctions[k];
            IFunction* pFunction = createSpecialFunction(functionCfg, j);
            if (pFunction)
            {
               _ppFunctions[_functionCount] = pFunction;
               ++_functionCount;
               _streamSets[j].vAddFunction(pFunction);
            }
         }

         ++j;  // may be smaller than i due to some error conditions
      }

      PostOffice<PO_MessageConfig::enID>* pPO = InternalCommunicationAdapter::POMessages;
      if (pPO)
      {
         // register for forwarded CCA command messages
         pPO->AddObserver(this, PO_MessageConfig::ID_CCA_Start_Volume);
         pPO->AddObserver(this, PO_MessageConfig::ID_CCA_Start_VolumeLock);
         pPO->AddObserver(this, PO_MessageConfig::ID_CCA_Start_VolumeMode);

         // register for source change notifications
         pPO->AddObserver(this, PO_MessageConfig::IDIntSource);
         pPO->AddObserver(this, PO_MessageConfig::IDIntSourceType);

         // support diagnostic commands
         pPO->AddObserver(this, PO_MessageConfig::ID_DiagAudioGain);
         pPO->AddObserver(this, PO_MessageConfig::ID_DiagRemoteControl);

         // register for general information
         pPO->AddObserver(this, PO_MessageConfig::ID_ApplicationStatus);
         pPO->AddObserver(this, PO_MessageConfig::ID_ApplicationTrigger);

         // register for Factory-Reset trigger
         pPO->AddObserver(this, PO_MessageConfig::ID_DiagDefSet);
         pPO->AddObserver(this, PO_MessageConfig::ID_SPIMixVolume);

         // register for PDC Attenuation notifications from Vehicle Domain Handler
         pPO->AddObserver(this, PO_MessageConfig::ID_PDC_Attenuation);
         pPO->AddObserver(this, PO_MessageConfig::ID_NotifyMute);

         // register for CAN Attenuation Values
         pPO->AddObserver(this, PO_MessageConfig::ID_CANAttenuation);

         // register for ID_ADASBeepActive
         pPO->AddObserver(this, PO_MessageConfig::ID_ADASBeepActive);

         pPO->AddObserver(this, PO_MessageConfig::ID_PhoneDeviceList);
         pPO->AddObserver(this, PO_MessageConfig::ID_ProfileDataChanged);
         pPO->AddObserver(this, PO_MessageConfig::ID_BOSE_Amplifier_ConnectionStatus);
         pPO->AddObserver(this, PO_MessageConfig::ID_AmpVolumeStatus);
         pPO->AddObserver(this, PO_MessageConfig::ID_AmpVolumeListStatus);
         pPO->AddObserver(this, PO_MessageConfig::ID_UpdateAmpValList);
      }
//      else
//         ETG_TRACE_FATAL(("Engine():  E R R O R - could not initialize PostOffice"))

      // log
      ETG_TRACE_USR1(("Engine established successfully with %u Stream-Sets", _streamSets.size()))
   }

   // -----------------------------------------------------------------------------

   /* virtual destructor */ Engine:: ~Engine ()
   {
      PostOffice<PO_MessageConfig::enID>* pPO = InternalCommunicationAdapter::POMessages;
      if (pPO)
         try
         {
            pPO->RemoveObserver(this, PO_MessageConfig::ID_DiagDefSet);
            pPO->RemoveObserver(this, PO_MessageConfig::ID_CCA_Start_Volume);
            pPO->RemoveObserver(this, PO_MessageConfig::ID_CCA_Start_VolumeLock);
            pPO->RemoveObserver(this, PO_MessageConfig::ID_CCA_Start_VolumeMode);
            pPO->RemoveObserver(this, PO_MessageConfig::IDIntSource);
            pPO->RemoveObserver(this, PO_MessageConfig::IDIntSourceType);
            pPO->RemoveObserver(this, PO_MessageConfig::ID_DiagAudioGain);
            pPO->RemoveObserver(this, PO_MessageConfig::ID_DiagRemoteControl);
            pPO->RemoveObserver(this, PO_MessageConfig::ID_ApplicationStatus);
            pPO->RemoveObserver(this, PO_MessageConfig::ID_ApplicationTrigger);
            pPO->RemoveObserver(this, PO_MessageConfig::ID_SPIMixVolume);
            pPO->RemoveObserver(this, PO_MessageConfig::ID_NotifyMute);
            pPO->RemoveObserver(this, PO_MessageConfig::ID_CANAttenuation);
            pPO->RemoveObserver(this, PO_MessageConfig::ID_ADASBeepActive);
            pPO->RemoveObserver(this, PO_MessageConfig::ID_PhoneDeviceList);
            pPO->RemoveObserver(this, PO_MessageConfig::ID_ProfileDataChanged);
            pPO->RemoveObserver(this, PO_MessageConfig::ID_BOSE_Amplifier_ConnectionStatus);
            pPO->RemoveObserver(this, PO_MessageConfig::ID_AmpVolumeStatus);
            pPO->RemoveObserver(this, PO_MessageConfig::ID_AmpVolumeListStatus);
            pPO->RemoveObserver(this, PO_MessageConfig::ID_UpdateAmpValList);
         }
         catch(...)  {}

      delete _pActionFactory;

      _pHeatProtector = NULL;
      _pPDCAttenuator = NULL;
      _pDuckingFilter = NULL;

      if (_ppFunctions)
         for (unsigned i = 0; i < _functionCount; ++i)
            delete _ppFunctions[i];
      delete[] _ppFunctions;

      ETG_TRACE_USR1(("Engine terminating ..."))
   }

   // -----------------------------------------------------------------------------

   /* virtual */ void Engine:: MessageNotification (PO_MessageConfig::enID MsgId)
   {
      // implement IF_MessageObserver

      ETG_TRACE_USR4(("Engine::MessageNotification() received for ID %u", MsgId))
      if (NULL == _pActionFactory)   // won't happen
         return;

      tenResource enResource = Speaker;
      IAction* pAction = NULL;
      PostOffice<PO_MessageConfig::enID>* pPO = InternalCommunicationAdapter::POMessages;

      switch (MsgId)
      {
         case PO_MessageConfig::ID_CCA_Start_Volume:
            {
               const VolumeData& data = pPO->QueryMessage<ID_CCA_Start_Volume>(MsgId)->value;
               ETG_TRACE_USR4(("Current Source : %d", _properties.enGetCurrentSource()));
    if(_properties.bIsExtAmpConnected())
           {
             pAction = _pActionFactory->getVolumeSetActionExtAmp(data);
           }
               else
           {
               pAction = _pActionFactory->getVolumeSetAction(data);
               enResource = fiResource_Map::getSecond(data.m_Resource.getResource());
           }
            }
            break;

         case PO_MessageConfig::ID_PhoneDeviceList:
            {
              const ID_PhoneDeviceList* pMsg = pPO->QueryMessage<ID_PhoneDeviceList>(MsgId);

               map <tU8, tBool> DeviceList = pMsg->BTPhonedevicelistmap;
              _properties.vCheckAndResetPhoneDeviceVolume(DeviceList, _config);
              break;
            }

         case PO_MessageConfig::IDIntSource:
            {
                const VolumeSourceData& pMsg = pPO->QueryMessage<IDIntSource>(MsgId)->value;
    ETG_TRACE_USR4(("Incoming Source Sink : %d", pMsg.m_Resource.getResource()))
                enResource = fiResource_Map::getSecond(pMsg.m_Resource.getResource());

             if( bIsAttenuationActive() && _properties.enGetCurrentSource() != AUD_INT_SRC_NOSOURCE)
                {
                  switch(pMsg.m_Channel)
                  {
                  case (midw_fi_tcl_e8_AudioChannel::FI_EN_AUDIO_CHANNEL_EXC):
                        if(enResource == Headphone)
                        {
                            if(pMsg.m_Source == AUD_INT_SRC_PHONE)
                            {
#ifdef VARIANT_S_FTR_ENABLE_DEVICE_SPECIFIC_PHONE_VOL
                                VolumeSourceData InVolumeSourceData;
                                InVolumeSourceData.vSetToDefault();
                                InVolumeSourceData.m_Source = _properties.u8GetPhoneType();
                                InVolumeSourceData.m_Resource.setResource(pMsg.m_Resource.getResource());
                                InVolumeSourceData.m_Channel = pMsg.m_Channel;
                                ETG_TRACE_USR4(("For Sink 2 PHONE_VOL always create SourceSwitchAction with pMsg.m_Source %d",InVolumeSourceData.m_Source));
                                pAction = _pActionFactory->getSourceSwitchAction(InVolumeSourceData);
#else
                                pAction = _pActionFactory->getSourceSwitchAction(pMsg);
#endif
			    }
			    else 
			    {
                                ETG_TRACE_USR4(("For Sink 2 always create SourceSwitchAction"));
                                pAction = _pActionFactory->getSourceSwitchAction(pMsg);
			    }
                        }
                        else
                        {
                            //ETG_TRACE_USR4(("Create SourceSwitchActionWhenMixActive"));
                            pAction = _pActionFactory->getSourceSwitchActionWhenMixActive(pMsg);
                        }
                     break;
                  case (midw_fi_tcl_e8_AudioChannel::FI_EN_AUDIO_CHANNEL_MIX):
                       if(pMsg.m_Source == AUD_INT_SRC_NAVI|| pMsg.m_Source == AUD_INT_SRC_NAVI_BY_SDS)
                         _properties.vSetMix1ActiveFlag();

                     pAction = _pActionFactory->getSourceSwitchAction(pMsg);
                     break;
                  case (midw_fi_tcl_e8_AudioChannel::FI_EN_AUDIO_CHANNEL_MIX2):
                     pAction = _pActionFactory->getSourceSwitchAction(pMsg);
                    break;
                  default:
                    break;

                  }
                }
                else
                {
                  if( pMsg.m_Channel == (midw_fi_tcl_e8_AudioChannel::FI_EN_AUDIO_CHANNEL_MIX))
                  {
                      _properties.vSetMix1ActiveFlag();
                  }

                    if(pMsg.m_Source == AUD_INT_SRC_PHONE)
                    {
                    // Set Volume for Phone based on the type of phone connected.
#ifdef VARIANT_S_FTR_ENABLE_DEVICE_SPECIFIC_PHONE_VOL
                  VolumeSourceData InVolumeSourceData;
                  InVolumeSourceData.vSetToDefault();
                    InVolumeSourceData.m_Source = _properties.u8GetPhoneType();
                    InVolumeSourceData.m_Resource.setResource(pMsg.m_Resource.getResource());
                    InVolumeSourceData.m_Channel = pMsg.m_Channel;
                    pAction = _pActionFactory->getSourceSwitchAction(InVolumeSourceData);
#else
                    pAction = _pActionFactory->getSourceSwitchAction(pMsg);
#endif
                    }
                    else
                    {
                     pAction = _pActionFactory->getSourceSwitchAction(pMsg);
                    }
                }
            }
            break;

         case PO_MessageConfig::ID_CCA_Start_VolumeLock:
            {
               const ID_CCA_Start_VolumeLock* pMsg = pPO->QueryMessage<ID_CCA_Start_VolumeLock>(MsgId);
               pAction = _pActionFactory->getLockAction(pMsg->requester, pMsg->value);
            }
            break;

         case PO_MessageConfig::ID_CCA_Start_VolumeMode:
            {
               const ID_CCA_Start_VolumeMode* pMsg =  pPO->QueryMessage<ID_CCA_Start_VolumeMode>(MsgId);
               pAction = _pActionFactory->getModeChangeAction(pMsg->value);
               enResource = fiResource_Map::getSecond(pMsg->value.m_Resource);
            }
            break;

         case PO_MessageConfig::ID_DiagDefSet:
            {
               const ID_DiagDefSet* pMsg =  pPO->QueryMessage<ID_DiagDefSet>(MsgId);
               tenDiagDefSetPhase enPhase = pMsg->enDiagDefSetPhase;
               tenDiagDefSetType enType = pMsg->enDiagDefSetType;
               if (EN_DEFSET_FINISH == enPhase && ( enType == EN_DEFSET_TEF || enType == EN_DEFSET_HMI ))
                  vResetToFactoryDefaults ();
            }
            break;

         case PO_MessageConfig::ID_ApplicationStatus:
            {
               const ID_ApplicationStatus* pMsg = pPO->QueryMessage<ID_ApplicationStatus>(MsgId);
               switch (pMsg->value.m_State)
               {
                  case AUD_APP_PWR_EVENT_APP_NORMAL:
                     _isAppStateNormal = true;
                     vApplyOnVolumeLimits();
                     break;

                  case AUD_APP_PWR_EVENT_APP_PAUSE:
                  case AUD_APP_PWR_EVENT_APP_OFF:
                  case AUD_APP_PWR_EVENT_APP_DIAGNOSIS:
                     _isAppStateNormal = false;
                     break;

                  default:
                     _isAppStateNormal = false;  // t.b.d. consider ignoring
                     break;
               }
            }
            break;
         case PO_MessageConfig::ID_SPIMixVolume:
         {
             const ID_SPIMixVolume* poMsg = pPO->QueryMessage<ID_SPIMixVolume>(MsgId);
             if (EN_INT_SPI_DUCK == poMsg->value.m_SPIMixMode || EN_INT_SPI_NATIVE_DUCK == poMsg->value.m_SPIMixMode)
             {
                   _bSPIDuckingActive = true;
             }
             else
               _bSPIDuckingActive = false;
             break;
         }

         case PO_MessageConfig::ID_PDC_Attenuation:
            {
               tU16 u16Error = 0;

               const ID_PDC_Attenuation* pMsg = pPO->QueryMessage <ID_PDC_Attenuation>(MsgId);
               bProcessPDCCommand((midw_fi_tcl_e8_PDCAttenuation::tenType)(pMsg->m_u8PDCAttenuationCommand), u16Error);
            }
            break;

         case PO_MessageConfig::IDIntSourceType:
         {
             const IDIntSourceType* pIntSourceTypeMsg = pPO->QueryMessage<IDIntSourceType>(MsgId);
              if(NULL != pIntSourceTypeMsg)
            _properties.vUpdatePhonetype(pIntSourceTypeMsg->value.m_Source);
           break;
         }

         case PO_MessageConfig::ID_DiagAudioGain:
         {
          const ID_DiagAudioGain* pMsg = pPO->QueryMessage<ID_DiagAudioGain>(MsgId);
            VolumeData voldata;
            voldata.m_Channel = midw_fi_tcl_e8_AudioChannel::FI_EN_AUDIO_CHANNEL_NOT_DEF;
            voldata.m_VolumeGain = pMsg->sAudioGainValue;
            pAction = _pActionFactory->getVolumeSetAction(voldata);
         }
         break;
         case PO_MessageConfig::ID_DiagRemoteControl:
         {
           const ID_DiagRemoteControl* pMsg = pPO->QueryMessage<ID_DiagRemoteControl>(MsgId);

           VolumeModeData data;
           data.m_Channel = midw_fi_tcl_e8_AudioChannel::FI_EN_AUDIO_CHANNEL_EXC;
           data.m_Resource = midw_fi_tcl_e8_ResourceNo::FI_EN_AUDIO_RESOURCE_LS;

             switch (pMsg->enDiagRemoteControlPhase)
             {
            case EN_REMOTECONTROL_SET:
             {
               ETG_TRACE_USR4(("RemoteControlNotification: SessionStart"));
               data.m_VolumeModeType = EN_INT_VOLMODE_REMOTE_CONTROL_START;
               pAction = _pActionFactory->getModeChangeAction(data);
               enResource = fiResource_Map::getSecond(data.m_Resource);
             }
             break;

             case EN_REMOTECONTROL_UNFREEZE:
              {
               ETG_TRACE_USR4(("RemoteControlNotification: SessionEnd: enter"));
               data.m_VolumeModeType = EN_INT_VOLMODE_REMOTE_CONTROL_END;
               pAction = _pActionFactory->getModeChangeAction(data);
               enResource = fiResource_Map::getSecond(data.m_Resource);

              }
              break;

             default:
              break;
             }
         }
         break;
         case PO_MessageConfig::ID_NotifyMute:
         {
             const ID_NotifyMute* pMsg = pPO->QueryMessage<ID_NotifyMute>(MsgId);
             tenStream enStream = pMsg->enStream;
             tenMuteResult enResult = pMsg->enMuteResult;

             ETG_TRACE_USR4(("NavVolumeFilter::MessageNotification() : ID_NotifyMute received for Stream %u with result %u"
                   , ETG_CENUM(tenStream, enStream), ETG_CENUM(tenMuteResult, enResult)))

             if ((EN_AUDIO_MUTE_MUTERESULT_MUTE == enResult) && (EN_AUDIO_SOURCE_STREAM_MIX1 == enStream))
             {
                _properties.vClearMix1ActiveFlag();
             }
           break;
         }
         case PO_MessageConfig::ID_CANAttenuation:
            {
                const ID_CANAttenuation* pMsg = pPO->QueryMessage<ID_CANAttenuation>(MsgId);

                if(pMsg)
                   _bCANAttenuationActiveFlag = pMsg->m_bAttenuationCommand;
            }
            break;

         case PO_MessageConfig::ID_ADASBeepActive:
            {
                const ID_ADASBeepActive* pMsg = pPO->QueryMessage<ID_ADASBeepActive>(MsgId);
                if(pMsg->m_bAdasBeepActive)
                   _bADASAttenuationActiveFlag = true;
                else
                   _bADASAttenuationActiveFlag = false;
            }
            break;

         case PO_MessageConfig::ID_ProfileDataChanged:
           {
             const MSG_ProfileDataChanged* pMsgProfileData = pPO->QueryMessage<MSG_ProfileDataChanged>(MsgId);
             if(pMsgProfileData)
               {
           if(MSG_ProfileDataChanged::DATA_CHANGE_PROFILE_CHANGED == pMsgProfileData->m_enDataChangedType)
             {
               _properties.vReloadData();
      if(_properties.bIsExtAmpConnected())
           _properties.vSendVolumeListToPlugin();
               for (unsigned i = 0; i < _streamSets.size(); ++i)
                 if (Speaker == _streamSets[i].enGetResource())
             {
               VolumeData volData;
               volData.m_Resource.setResource(midw_fi_tcl_e8_ResourceNo::FI_EN_AUDIO_RESOURCE_LS);
               volData.m_Channel = fiAudioChannel_StreamID_Map::getFirst(_streamSets[i].enGetActiveStreamID());
               const tU8* vol = _properties.pGetVolume ( Speaker,  midw_fi_tcl_e8_Aud_VolumeType::FI_EN_VOLUME_ENTERTAINMENT);
               if(vol)
                   volData.m_Volume = *vol;
               else
                   return;
               volData.m_VolumeType.setVolumeType(midw_fi_tcl_e8_AudioVolumeType::FI_EN_AUDIO_VAL_TYPE_ABS);
               // volData.m_VolumeGain = XXX;

               IAction* pAction = _pActionFactory->getVolumeSetAction(volData);
               if (pAction)
                 pAction->vExecute(_streamSets[i]);
               delete pAction;
             }
               _properties.NotifyAllUpdates();
             }
               }
            }
            break;
         case PO_MessageConfig::ID_UpdateAmpValList:
         {
             tS32 s32Res = 0;
             const ID_UpdateAmpValList* pMsg = pPO->QueryMessage<ID_UpdateAmpValList>(MsgId);
             //dp_tclAudioManagerUsrDPAdas_ParkAssitVol_Multi odpParkAssistVol;
             ETG_TRACE_USR4(("ID_UpdateAmpValList value Received : %d", pMsg->m_volume))
             //s32Res = odpParkAssistVol.s32SetData(pMsg->m_volume);
             if (AudioUserDpIfSelect::pGetAudUsrDpIf())
                {
                    s32Res = AudioUserDpIfSelect::pGetAudUsrDpIf()->s32SetParkAssistVolume(pMsg->m_volume);
                }
             ETG_TRACE_USR4(("Persistent Memory storage Result for park assist volume = %x", s32Res));
            _properties.vSendVolumeListToPlugin();
         }
         break;
         case PO_MessageConfig::ID_BOSE_Amplifier_ConnectionStatus:
           {
             const ID_BOSE_Amplifier_ConnectionStatus* pMsg = pPO->QueryMessage<ID_BOSE_Amplifier_ConnectionStatus>(MsgId);

    ETG_TRACE_USR4(("Amplifier status Received : %d", pMsg->connectionstate))
    if(pMsg->connectionstate == (tU8)EXTAMP_CONNECTED)
      {
        _properties.vSetExtAmpFlag();
        _properties.vSendVolumeListToPlugin();
      }
    else
      {
        _properties.vClearExtAmpFlag();
      }
  }
  break;

         case PO_MessageConfig::ID_AmpVolumeStatus:
           {
             const ID_AmpVolumeStatus* pMsg = pPO->QueryMessage<ID_AmpVolumeStatus>(MsgId);
             const SourceConfig* pSourceCfg = _config.pGetSourceConfig(_properties.enGetCurrentSource());

             if(pSourceCfg != NULL)
               {
           tenStream enStream = pSourceCfg->enGetStreamID(enResource);
           tenInternalSource Source = pSourceCfg->enGetSourceID();
                 _properties.vUpdateVolume(Source, enStream, pMsg->volume, pMsg->volgroup);
               }
            }
            break;
      case PO_MessageConfig::ID_AmpVolumeListStatus:
  {
    const ID_AmpVolumeListStatus* pMsg = pPO->QueryMessage<ID_AmpVolumeListStatus>(MsgId);
    _properties.vUpdateVolumeInList(pMsg->VolumeList);
  }
  break;

      case PO_MessageConfig::ID_ApplicationTrigger:
         default:
            ETG_TRACE_ERR(("Engine::MessageNotification() - unhandled message %u received", MsgId))
            break;
      }

      if (pAction)
      {
         (void)bExecuteAction(*pAction, enResource);
         delete pAction;
      }
   }

   // -----------------------------------------------------------------------------

   bool Engine:: bProcessListChange (const midw_fi_tcl_Aud_VolumeStatusList& lst, tU8 Resource, tU16& u16Error)
      {
      IAction* pAction = NULL;
      if (_pActionFactory)  // shall always be valid
         pAction = _pActionFactory->getListChangeAction(lst);
      bool retVal = false;
      if (pAction)
      {
         ETG_TRACE_USR4(("Engine::bProcessListChange(midw_fi_tcl_Aud_VolumeStatusList) for Resource %d",Resource));
         retVal = bExecuteAction(*pAction,(tenResource)Resource);
         delete pAction;
      }

      // something got wrong
      if ( ! retVal)
         u16Error = MIDW_MASCFFI_C_U16_ERROR_INTERNALFAILURE;

      return retVal;
   }

   // -----------------------------------------------------------------------------

   bool Engine:: bProcessPDCCommand (midw_fi_tcl_e8_PDCAttenuation::tenType PDCAttenuationCommand, tU16& u16Error)
   {
      if (_pPDCAttenuator)
         return _pPDCAttenuator->bProcessPDCCommand(PDCAttenuationCommand, u16Error);

     return true;
   }

   // -----------------------------------------------------------------------------

   bool Engine:: bProcessPDCLevel (tU8 PDCAttenutionLevel, tU16& u16Error)
   {
      if (_pPDCAttenuator)
         return _pPDCAttenuator->bProcessPDCLevel(PDCAttenutionLevel, u16Error);

     return true;
   }

   // -----------------------------------------------------------------------------

   bool Engine:: bProcessOverTempReduction (bool downScaling_Active, tU8 reductionInterval, tU16& u16Error)
   {
      if (_pHeatProtector)
         return _pHeatProtector->bProcessReductionCommand(downScaling_Active, reductionInterval, u16Error);

      return true;
   }

   // -----------------------------------------------------------------------------

   void Engine:: vApplyOnVolumeLimits ()
   {
      if (NULL == _pActionFactory)
         return;

      IAction* pAction = _pActionFactory->getOnVolumeLimitAction();
      if (pAction)
         (void) bExecuteAction(*pAction, Speaker);
      delete pAction;
   }

   // -----------------------------------------------------------------------------

   void Engine:: vResetToFactoryDefaults ()
   {
     ETG_TRACE_USR4(("Engine:: vResetToFactoryDefaults ()"));
      _properties.vSetAllToDefaults(_config);
      _properties.vResetDANValue();

      if(_properties.bIsExtAmpConnected())
      _properties.vSendVolumeListToPlugin();

      // apply changes to stream-set affecting Speakers
      tU8 volume = _properties.u8GetVolume();
      tenInternalSource enSource = _properties.enGetCurrentSource();
      const SourceConfig* pSourceCfg = _config.pGetSourceConfig(enSource);
      if (pSourceCfg && _pActionFactory)
      {
         tenStream enStream = pSourceCfg->enGetStreamID(Speaker);
         for (unsigned i = 0; i < _streamSets.size(); ++i)
            if (Speaker == _streamSets[i].enGetResource())
            {
               _streamSets[i].vSetActiveStream(enStream);
               Stream* pStream = _streamSets[i].pGetActiveStream();
               if (pStream)
               {
                  pStream->vSetCurrentMode(_properties.enGetMode());
                  pStream->vSetCurrentSource(enSource);
               }

               {
                  VolumeData volData;
                  volData.m_Resource.setResource(midw_fi_tcl_e8_ResourceNo::FI_EN_AUDIO_RESOURCE_LS);
                  volData.m_Channel = fiAudioChannel_StreamID_Map::getFirst(enStream);
                  volData.m_Volume = volume;
                  volData.m_VolumeType.setVolumeType(midw_fi_tcl_e8_AudioVolumeType::FI_EN_AUDIO_VAL_TYPE_ABS);
                  // volData.m_VolumeGain = XXX;
                  IAction* pAction = _pActionFactory->getVolumeSetAction(volData);
                  if (pAction)
                     pAction->vExecute(_streamSets[i]);
                  delete pAction;
               }
               break;
            }
      }

      _properties.NotifyAllUpdates();
   }

   // -----------------------------------------------------------------------------

   IFunction* Engine:: createSpecialFunction (const FunctionConfig& functionCfg, unsigned streamSetIndex)
   {
      IFunction* pFunction = NULL;
      if (NULL == functionCfg.name)
         ;  // do nothing
      else if (0 == strcmp(functionCfg.name, "NavVolumeFilter"))
         pFunction = new NavVolumeFilter(_properties, _config, functionCfg, _streamSets[streamSetIndex]);
      else if (0 == strcmp(functionCfg.name, "BeepVolumeFilter"))
         pFunction = new BeepVolumeFilter(_properties, _config, functionCfg, _streamSets[streamSetIndex]);
      else if (0 == strcmp(functionCfg.name, "TAVolumeFilter"))
         pFunction = new TAVolumeFilter(_properties, _config, functionCfg, _streamSets[streamSetIndex]);
      else if (0 == strcmp(functionCfg.name, "HeatProtector"))
      {
         _pHeatProtector = new HeatProtector(_properties, _config, functionCfg, _streamSets[streamSetIndex]);
         pFunction = _pHeatProtector;
      }
      else if (0 == strcmp(functionCfg.name, "PDCAttenuator"))
      {
         _pPDCAttenuator = new PDCAttenuator(_properties, _config, functionCfg, _streamSets[streamSetIndex]);
         pFunction = _pPDCAttenuator;
      }
      else if (0 == strcmp(functionCfg.name, "SPI-DuckingFilter"))
      {
         _pDuckingFilter = new SPIDuckingFilter(_properties, _config, functionCfg, _streamSets[streamSetIndex]);
         pFunction = _pDuckingFilter;
      }
      else if (0 == strcmp(functionCfg.name, "DiagVolume"))
      {
        pFunction = new DiagVolume(_properties, _config, functionCfg, _streamSets[streamSetIndex]);
      }
      else if (0 == strcmp(functionCfg.name, "AttenuationFilter"))
      {
        pFunction = new AttenuationFilter(_properties, _config, functionCfg, _streamSets[streamSetIndex]);
      }
      else
         ETG_TRACE_FATAL(("createSpecialFunction() - E R R O R : no special function created for '%s'", functionCfg.name))

      return pFunction;
   }

   // -----------------------------------------------------------------------------

   bool Engine:: bExecuteAction (IAction& action, tenResource enResource)
   {
      for (unsigned i = 0; i < _streamSets.size(); ++i)
      {
         if (enResource == _streamSets[i].enGetResource())
         {
      ETG_TRACE_USR4(("Sink : %d",enResource))
            action.vExecute(_streamSets[i]);

            _properties.NotifyAllUpdates();
            return true;
         }
      }
      ETG_TRACE_ERR(("Engine:: bExecuteAction no Stream-Sets is bound to requested resource"));
      // none of the Stream-Sets is bound to requested resource
      return false;
   }

   // --------------------------------------------------------------------------
   bool Engine::bIsAttenuationActive()
   {
      if( _properties.bIsMix1Active () || _bSPIDuckingActive || _bADASAttenuationActiveFlag || _bCANAttenuationActiveFlag)
          return true;
      else
        return false;
   }

   // --------------------------------------------------------------------------

   void Engine:: vHandleTTFisCommand (tU32 size, const tU8* pcu8Data)
   {
      // TTFis support

      if (NULL == pcu8Data)
      {
         ETG_TRACE_FATAL(("vHandleTTFisCommand() - E R R O R  : invalid TTFis message"))
         return;
      }
      else if (size <= 1)
      {
         ETG_TRACE_FATAL(("vHandleTTFisCommand() - E R R O R  : TTFis message too short"))
         return;
      }

      ETG_TRACE_USR2(("Engine::vHandleTTFisCommand() - handling %u with %u bytes"
            , ETG_CENUM(TRC::tenTrcFuncIn_AudSrvFun, pcu8Data[1]), size))

      if (NULL == _pActionFactory)
         return;  // shall never happen

      IAction* pAction = NULL;
      switch (pcu8Data[1])
      {
         case TRC::OverTempReduction:
            {
               VolumeData data;
               data.m_Resource.setResource(midw_fi_tcl_e8_ResourceNo::FI_EN_AUDIO_RESOURCE_LS);
               data.m_Channel = midw_fi_tcl_e8_AudioChannel::FI_EN_AUDIO_CHANNEL_EXC;
               data.m_VolumeType.setVolumeType(midw_fi_tcl_e8_AudioVolumeType::FI_EN_AUDIO_VAL_TYPE_DEC);
               data.m_Volume = 1;
               pAction = _pActionFactory->getVolumeSetAction(data);
            }
            break;

         case TRC::enSetVolume:
            if (size >= 4)
            {
               VolumeData data;
               data.setVolumeType((midw_fi_tcl_e8_AudioVolumeType::tenType) pcu8Data[2]);
               data.setResource((midw_fi_tcl_e8_ResourceNo::tenType) pcu8Data[3]);
               data.setVolume(pcu8Data[4]);
             if(_properties.bIsExtAmpConnected())
                 pAction = _pActionFactory->getVolumeSetActionExtAmp(data);
             else
               pAction = _pActionFactory->getVolumeSetAction(data);
            }
            else
            {
               ETG_TRACE_FATAL(("TTFis message for enSetVolume too short -  only %u bytes"
                  , size))
            }
            break;

         case TRC::enSetVolumeMode:
            if (size >= 4)
            {
               VolumeModeData data;
               data.m_Resource = midw_fi_tcl_e8_ResourceNo::FI_EN_AUDIO_RESOURCE_LS;
               data.m_VolumeModeType = (tenVolumeMode) pcu8Data[3];
               data.m_Channel = (midw_fi_tcl_e8_AudioChannel::tenType) pcu8Data[4];
               pAction = _pActionFactory->getModeChangeAction(data);
            }
            else
            {
               ETG_TRACE_FATAL(("TTFis message for enSetVolumeMode too short -  only %u bytes"
                  , size))
            }
            break;

         case TRC::VolumeLock:
            if (size >= 3)
            {
               tenVolumeLock lock = (tenVolumeLock)pcu8Data[2];
               tenVolumeLockRequester requester = (tenVolumeLockRequester)pcu8Data[3];
               pAction = _pActionFactory->getLockAction (requester, lock);
            }
            else
            {
               ETG_TRACE_FATAL(("TTFis message for VolumeLock too short -  only %u bytes"
                  , size))
            }
            break;

         case TRC::PDCAttenuation:
            if (size >= 2)
            {
               if (_pPDCAttenuator)
               {
                  tU16 u16Error = 0;
                  midw_fi_tcl_e8_PDCAttenuation::tenType PDCAttenuationCommand
                   = static_cast<midw_fi_tcl_e8_PDCAttenuation::tenType>(pcu8Data[2]);
                  (void) _pPDCAttenuator->bProcessPDCCommand(PDCAttenuationCommand, u16Error);
               }
            }
            else
            {
               ETG_TRACE_FATAL(("TTFis message for PDCAttenuation too short -  only %u bytes"
                  , size))
            }
            break;

         case TRC::PDCAttenuationLevel:
            if (size >= 2)
            {
               {
                  _properties.vUpdatePDCAttenuationLevel(pcu8Data[2]);
                  _properties.NotifyAllUpdates();
               }
            }
            else
            {
               ETG_TRACE_FATAL(("TTFis message for PDCAttenuationLevel too short -  only %u bytes"
                  , size))
            }
            break;

         case TRC::PerformOnVolumeLimit:
            {
               pAction = _pActionFactory->getOnVolumeLimitAction();
            }
            break;

         default:
            break;
      }

      if (pAction)
      {
         (void) bExecuteAction(*pAction, Speaker);
         delete pAction;
     }
   }

   // -----------------------------------------------------------------------------

}  //  namespace VolumeManager
