
//
// VolumeManager/Engine/Actions/VolumeSetAction.h
//
//  Created on: Jul 3, 2014
//      Author: Martin Koch, Fa. ESE
//



// framework
#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#include <etrace_if.h>

#include "Volume/Types.h"  // implicitly links midw_fi alltypes.h and stl_pif.h vector
#include "Volume/Utilities/Uncopyable.h"
#include "Volume/Utilities/Array.hpp"
#include "./IAction.h"
#include "./ActionBase.h"
#include "./VolumeSetAction.h"
// - - - - - - - - - - - - -

#include "Volume/Engine/Stream.h"
#include "Volume/Engine/StreamSet.h"
#include "Volume/Engine/Functions/IFunction.h"
#include "Volume/Configuration/ConfigDetails.hpp"
#include "Volume/Configuration/dBCalculator.h"
#include "Volume/Configuration/Configuration.h"
#include "Volume/PropertyStore.h"
#include "Volume/TypeConversions/Functions.h"

#include "InternalComponentCommunication/DataTypes/MessageDataTypes/VolumeData.h"
#include "InternalComponentCommunication/DataTypes/TypeDefines/SourceDefines.h"

#include "fc_audiomanager_trace.h"
#define ETG_DEFAULT_TRACE_CLASS  TR_CLASS_FC_AUDIOMANAGER_VOLUME
#include "trcGenProj/Header/VolumeSetAction.cpp.trc.h"



namespace VolumeManager
{

   // -----------------------------------------------------------------------------

   /* constructor */ VolumeSetAction:: VolumeSetAction (PropertyStore& properties, const Configuration& config
         , const VolumeData& data, tenFadingReason enReason)
      : ActionBase(properties, config)
      , _data(data)
      , _enMode(_properties.enGetMode())
      , _enReason(enReason)
      , s8Dan(0)
   {
    ETG_TRACE_USR3(("VolumeSetAction() invoked for reason %u while in mode %u with active source %u,(Resource %d, Channel %d)"
            , ETG_CENUM(tenFadingReason, _enReason), ETG_CENUM(tenVolumeMode, _enMode)
      , ETG_CENUM(tenInternalSource, _properties.enGetCurrentSource())
      , data.getResource(),data.m_Channel));
      _data.printData();
   }

   // -----------------------------------------------------------------------------

   /* virtual destructor */ VolumeSetAction:: ~VolumeSetAction ()
   {
      ETG_TRACE_USR4(("VolumeSetAction() completed."))
   }

   // -----------------------------------------------------------------------------

   /* virtual */ void VolumeSetAction:: vExecute (StreamSet& streamSet)
   {
      tenResource enResource = streamSet.enGetResource();
      tU8 currentVol = u8GetCurrentVolume(streamSet);
      ETG_TRACE_USR4(("VolumeSetAction()::vExecute enResource %d, (_data.m_Channel %d)",enResource,_data.m_Channel));

      VolumeData nextVolume;
      if (midw_fi_tcl_e8_AudioChannel::FI_EN_AUDIO_CHANNEL_NOT_DEF == _data.m_Channel)
         nextVolume.m_Channel = fiAudioChannel_StreamID_Map::getFirst(streamSet.enGetActiveStreamID());
      else
         nextVolume.m_Channel = _data.m_Channel;
      nextVolume.m_Resource.setResource(fiResource_Map::getFirst(enResource));
      nextVolume.m_VolumeType.setVolumeType(midw_fi_tcl_e8_AudioVolumeType::FI_EN_AUDIO_VAL_TYPE_ABS);
      nextVolume.m_Volume = u8GetCalculatedTargetStep (_data, currentVol);
      nextVolume.m_VolumeGain = _data.m_VolumeGain;
      // apply external filtering on visible volume (published by Volume.Status message)
      streamSet.vApplyVolumeFilters(_enReason, nextVolume);

      // find configuration of active source on target stream
      const SourceConfig* pSourceCfg =  pGetTargetSourceConfig(streamSet);

      // determine intended volume according to change direction
      vApplyLimits (nextVolume, pSourceCfg, currentVol);

      // apply external filtering on intended amplifier command (invisible to the user)
      bool useStandardAmplifierCommand = true;
      streamSet.vApplyAmplifierFilters(_enReason, nextVolume, useStandardAmplifierCommand);

      if (useStandardAmplifierCommand)
      {
         vApply(streamSet, nextVolume);
         vUpdate(nextVolume, pSourceCfg);
      }
   }

   // -----------------------------------------------------------------------------

   void VolumeSetAction:: vApply (StreamSet& streamSet, VolumeData& nextVolume)
   {

     ETG_TRACE_USR4(("VolumeSetAction()::vApply VolumeData.m_Resource %d, channel %d",
         nextVolume.m_Resource.getResource(),nextVolume.m_Channel));
      // correct streamID (may be required, as CCA message dosn't have channel information)
      // workaround, as Channel information is invalid when initialized from FI-object
      tenStream enStreamID = fiAudioChannel_StreamID_Map::getSecond(_data.m_Channel);
      if (EN_AUDIO_SOURCE_STREAM_DEFAULT == enStreamID)
         enStreamID = streamSet.enGetActiveStreamID();

      // send requested volume to requested audio channel
      tenActions enAction = enGetFadingAction (_enReason);
      Ramp ramp = streamSet.queryRamp(enAction, _enReason, enStreamID);
      Stream* pStream = streamSet.pGetStream(enStreamID);
      tenInternalSource enTargetSource = _properties.enGetCurrentSource();
    const SourceConfig* pSourceCfg = NULL;
    if(pStream!= NULL)
        pSourceCfg = _config.pGetSourceConfig(pStream->enGetCurrentSource());
      if (pSourceCfg)
         enTargetSource = pSourceCfg->enGetSourceID();

      const dBCalculator* pCalcSink = streamSet.pGetCalculator();
      const dBCalculator* pCalcSource = _config.pGetDBCalculator(enTargetSource);
      if(pCalcSink && pCalcSource && ( 0 == strcmp(pCalcSink->sGetName(), pCalcSource->sGetName()) ) )
  {
     if (pStream)
         pStream->setVolume(*pCalcSource, nextVolume.m_Volume, ramp);
  }
      else
  {
     if (pStream && pCalcSink)
         pStream->setVolume(*pCalcSink, nextVolume.m_Volume, ramp);
  }
   }

   // -----------------------------------------------------------------------------

   void VolumeSetAction:: vUpdate (const VolumeData& nextVolume, const SourceConfig* pSourceConfig)
   {
      tU8 volume = nextVolume.getVolume();
      if (pSourceConfig)
      {
         tenResource enResource = fiResource_Map::getSecond(_data.m_Resource.getResource());
         if (pSourceConfig->enGetSourceID() == _properties.enGetCurrentSource())
            _properties.vUpdateVolume(enResource, _enMode, *pSourceConfig, volume);

         // update VolumeList also / special handling for TA-Setup
         if ((EN_INT_VOLMODE_SETUP_TA == _enMode) && _properties.bIsVolumeInList(enResource, midw_fi_tcl_e8_Aud_VolumeType::FI_EN_VOLUME_TASETUP))
            _properties.vUpdateVolumeInList(enResource, midw_fi_tcl_e8_Aud_VolumeType::FI_EN_VOLUME_TASETUP, volume);
         else
            _properties.vUpdateVolumeInList(enResource, pSourceConfig->u8GetGroupType(), volume);
      }
   }

   // -----------------------------------------------------------------------------

   const SourceConfig* VolumeSetAction:: pGetTargetSourceConfig (const StreamSet& streamSet)
   {
      // determine configuration of source currently active on mentioned stream

      tenStream enTargetStream = fiAudioChannel_StreamID_Map::getSecond(_data.m_Channel);

      // correct streamID (may be required, as CCA message dosn't have channel information)
      // workaround, as Channel information is invalid when initialized from FI-object
      if (enTargetStream == EN_AUDIO_SOURCE_STREAM_DEFAULT)
      {
         enTargetStream = streamSet.enGetActiveStreamID();
      }

      Stream* pTargetStream = streamSet.pGetStream(enTargetStream);
      if (pTargetStream)
      {
         return _config.pGetSourceConfig(pTargetStream->enGetCurrentSource());
      }
      else
      {
        ETG_TRACE_USR4(("VolumeSetAction:: pGetTargetSourceConfig pTargetStream is NULL returning NULL"));
         return NULL;
      }
   }

   // -----------------------------------------------------------------------------

   tU8 VolumeSetAction:: u8GetCurrentVolume (const StreamSet& streamSet)
   {
      // determine current volume from PropertyStore according to mentioned stream
      // Note: defaults to current active volume

      tenResource enResource = streamSet.enGetResource();
      if ((EN_INT_VOLMODE_SETUP_TA == _enMode) && _properties.bIsVolumeInList(enResource, midw_fi_tcl_e8_Aud_VolumeType::FI_EN_VOLUME_TASETUP))
      {
         const tU8* pTASetupVolume = _properties.pGetVolume(enResource, midw_fi_tcl_e8_Aud_VolumeType::FI_EN_VOLUME_TASETUP);
         if (pTASetupVolume)
            return *pTASetupVolume;
      }

      const SourceConfig* pSourceCfg = pGetTargetSourceConfig (streamSet);
      if (NULL == pSourceCfg)
         return _properties.u8GetVolume();

      const tU8* pCurrentVolume = _properties.pGetVolume(streamSet.enGetResource(), pSourceCfg->u8GetGroupType());
      if (NULL == pCurrentVolume)
         return _properties.u8GetVolume();

      // that's what we wanted
      return *pCurrentVolume;
   }

   // -----------------------------------------------------------------------------

   tU8 VolumeSetAction:: u8GetCalculatedTargetStep (const VolumeData& data, tU8 currentVol)
   {
      // determine intended volume according to change direction
      switch (data.m_VolumeType.getVolumeType())
      {
         case midw_fi_tcl_e8_AudioVolumeType::FI_EN_AUDIO_VAL_TYPE_ABS:
            return data.m_Volume;

         case midw_fi_tcl_e8_AudioVolumeType::FI_EN_AUDIO_VAL_TYPE_INC:
            return (tS16)(currentVol + data.m_Volume) <= 255 ? static_cast<tU8>(currentVol + data.m_Volume) : 255;

         case midw_fi_tcl_e8_AudioVolumeType::FI_EN_AUDIO_VAL_TYPE_DEC:
            return (tS16)(currentVol - data.m_Volume) >= 0 ? static_cast<tU8>(currentVol - data.m_Volume) : 0;

         default:
            ETG_TRACE_ERR(("VolumeSetAction undefined change direction: leaving current volume %u untouched ..."
                  , currentVol))
            return currentVol;  // skip further processing
      }
   }

   // -----------------------------------------------------------------------------

   void VolumeSetAction:: vApplyLimits (VolumeData& nextVolume, const SourceConfig* pSourceConfig, tU8 currentVol)
   {
      // first check if volume is not absolutely locked
      tenVolumeLock enLockState = _properties.enGetLockState();
      const tU8 u8TargetVol = nextVolume.getVolume();
      if (EN_INT_VOLUME_LOCK_MODE_TOTAL == enLockState)
      {
         ETG_TRACE_USR3(("VolumeSetAction() - Volume totally locked - ignoring request ..."))
         nextVolume.setVolume(currentVol);
         return;
      }
      if ((EN_INT_VOLUME_LOCK_MODE_INCREASING == enLockState) && (u8TargetVol > currentVol))
      {
         ETG_TRACE_USR3(("VolumeSetAction() - Volume locked for increase - keeping %u", currentVol))
         nextVolume.setVolume(currentVol);
         return;
      }
      // validate source configuration

      VolumeConstraints limits;
      if (pSourceConfig)
         limits = pSourceConfig->getVolumeLimits();
      else
         ETG_TRACE_ERR(("vApplyLimits() - E R R O R :  no source"))
      tenResource enResource = fiResource_Map::getSecond(_data.m_Resource.getResource());
      if ((EN_INT_VOLMODE_SETUP_TA == _enMode) && _properties.bIsVolumeInList(enResource, midw_fi_tcl_e8_Aud_VolumeType::FI_EN_VOLUME_TASETUP))
      {
         const VolumeConstraints* pLimits = _config.pGetVolumeLimitsByGroup(midw_fi_tcl_e8_Aud_VolumeType::FI_EN_VOLUME_TASETUP);
         if (pLimits)
            limits = *pLimits;
      }

#if defined VARIANT_S_FTR_ENABLE_FEAT_AUDIO_TA_DAN
       vGetCalculatedDANValue(limits,u8TargetVol);
#endif

       //If no persistance tag, set to default volume in case of source selection.
       const GroupConfig* pConfig;
       size_t groupCount = 0;
      _config.vGetSourceGroupConfigs(pConfig, groupCount);

if(pConfig != NULL && pSourceConfig != NULL)
 {
  for (unsigned i = 0; i < groupCount; ++i)
   if (!pConfig[i].persistent)
     {
          if(pConfig[i].typeId == pSourceConfig->u8GetGroupType() && _enReason == EN_AUDIO_FADING_RES_SOURCESEL)
            {
#if defined VARIANT_S_FTR_ENABLE_FEAT_AUDIO_TA_DAN
            if(midw_fi_tcl_e8_Aud_VolumeType :: FI_EN_VOLUME_TASETUP != pSourceConfig->u8GetGroupType())
            { nextVolume.setVolume(pConfig[i].constraints.defaultStep);
            ETG_TRACE_ERR(("VolumeSetAction leads to default volume : current = %u, intended = %i, setting to %u (default)"
                           , currentVol, u8TargetVol, pConfig[i].constraints.defaultStep));
        return;
            }
            #else
            nextVolume.setVolume(pConfig[i].constraints.defaultStep);
            ETG_TRACE_ERR(("VolumeSetAction leads to default volume : current = %u, intended = %i, setting to %u (default)"
                           , currentVol, u8TargetVol, pConfig[i].constraints.defaultStep));
        return;
#endif
               }
     }
}

      if (u8TargetVol < limits.minStep)
      {
         ETG_TRACE_ERR(("VolumeSetAction leads to underrange: current = %u, intended = %i, setting to %u (min)"
               , currentVol, u8TargetVol, limits.minStep))
         nextVolume.setVolume(limits.minStep);
      }
      else if (u8TargetVol > limits.maxStep)
      {
         ETG_TRACE_ERR(("VolumeSetAction leads to overrange: current = %u, intended = %i, setting to %u (max)"
               , currentVol, u8TargetVol, limits.maxStep))
         nextVolume.setVolume(limits.maxStep);
      }
      else
      {
         ETG_TRACE_USR3(("VolumeSetAction changing volume %u to %i ..."
               , currentVol,  u8TargetVol))
         nextVolume.setVolume(u8TargetVol);
      }

   }

/*****************************************************************************************
* void VolumeSetAction:: vGetCalculatedDANValue(VolumeConstraints limits,tU8 u8TargetVol)

 * DESCRIPTION:  This function is used to calculate the DAN Value on changing TA Volumes.
 *
 * PARAMETER:    VolumeConstraints limits,tU8 u8TargetVol
 *
 * RETURNVALUE: None
********************************************************************************************/
   void VolumeSetAction:: vGetCalculatedDANValue(VolumeConstraints limits,tU8 u8TargetVol)
   {
     tS8 danValue = 0;
     tenResource enResource = fiResource_Map::getSecond(_data.m_Resource.getResource());
     if(_properties.enGetCurrentSource() == AUD_INT_SRC_FM_TA || _properties.enGetCurrentSource() == AUD_INT_SRC_DAB_TA)
     {
       const tU8* pEntVolume  = _properties.pGetVolume(enResource, midw_fi_tcl_e8_Aud_VolumeType::FI_EN_VOLUME_ENTERTAINMENT);

       if(pEntVolume == NULL)
       {
         ETG_TRACE_FATAL(("vGetCalculatedDANValue(),pEntVolume is NULL"));
         return;
       }

       if(u8TargetVol != *pEntVolume)
       {
         ETG_TRACE_USR3(("pEntVolume :%d u8TargetVol : %d ", *pEntVolume, u8TargetVol))
           if(u8TargetVol > limits.maxStep)
             danValue = static_cast<tS8>(limits.maxStep - *pEntVolume);
           else if (u8TargetVol < limits.minStep)
             danValue = static_cast<tS8>(limits.minStep - *pEntVolume);
           else
             danValue = static_cast<tS8>(u8TargetVol - *pEntVolume);
       }
     _properties.vSetDanValue(danValue);

       ETG_TRACE_USR3(("DAN :%d", _properties.s8GetDanValue()))
     }
   }
 // -----------------------------------------------------------------------------

   /* static */ tenActions VolumeSetAction:: enGetFadingAction (tenFadingReason enReason)
   {
      switch (enReason)
      {
         case EN_AUDIO_FADING_RES_SOURCESEL:
            return EN_AUDIO_FADING_ACTION_SEND_VOL;

         default:
            return EN_AUDIO_FADING_NOTIFY_CHG_VOL;
      }
   }

   // -----------------------------------------------------------------------------

}  // namespace VolumeManager
