//
// VolumeManager/Actions/LockAction.cpp
//
//  Created on: Aug 25, 2014
//      Author: Martin Koch, Fa. ESE
//


// framework
#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#include <etrace_if.h>


#include "Volume/Types.h"  // implicitly links <midw_fi_if.h> for alltypes.h and <stl_pif.h> for vector
#include "Volume/Utilities/Uncopyable.h"
#include "Volume/Utilities/Array.hpp"
#include "./IAction.h"
#include "./ActionBase.h"
#include "./LockAction.h"
// - - - - - - - - - - - - -

#include "Volume/Configuration/Configuration.h"
#include "Volume/Engine/Stream.h"
#include "Volume/Engine/StreamSet.h"
#include "Volume/Configuration/ConfigDetails.hpp"
#include "Volume/PropertyStore.h"

//#include "InternalComponentCommunication/DataTypes/MessageDataTypes/VolumeLockData.h"


#include "fc_audiomanager_trace.h"
#define ETG_DEFAULT_TRACE_CLASS  TR_CLASS_FC_AUDIOMANAGER_VOLUME
#include "trcGenProj/Header/LockAction.cpp.trc.h"



namespace VolumeManager
{

   // -----------------------------------------------------------------------------

   /* static */ LockAction::LockStateTable  LockAction:: _lockStateTable;

   // -----------------------------------------------------------------------------

   /* constructor */  LockAction:: LockAction (PropertyStore& properties, const Configuration& config, tenVolumeLockRequester enRequester, tenVolumeLock enNewLockState)
      : ActionBase(properties, config)
   {
      if (EN_VOL_LOCK_REQUESTER_UNKNOWN == enRequester)
      {
         ETG_TRACE_ERR(("LockAction() - unknown requester - ignoring ..."))
         return;
      }

      ETG_TRACE_USR3(("LockAction() - invoked with lock state %u for requester %u"
         , ETG_CENUM(tenVolumeLock, enNewLockState)
         , ETG_CENUM(tenVolumeLockRequester, enRequester)))

      // update table with new values
      switch (enNewLockState)
      {
         case EN_INT_VOLUME_LOCK_MODE_UNDEF:
            ETG_TRACE_ERR(("LockAction() - undefined lock state - ignoring ..."))
            break;

         case EN_INT_VOLUME_LOCK_MODE_OFF:
            // remove if exists
            _lockStateTable.erase(enRequester);
            break;

         default:
            // insert if not existing
            _lockStateTable[enRequester] = enNewLockState;
            break;
      }
   }

   // -----------------------------------------------------------------------------

   /* virtual destructor */ LockAction:: ~LockAction ()
   {

   }

   // -----------------------------------------------------------------------------

   /* virtual */ void LockAction:: vExecute (StreamSet& /* streamSet */)
   {
      vUpdate ();
   }

   // -----------------------------------------------------------------------------

   void LockAction:: vUpdate ()
   {
      // scan list for entry with highest priority
      tenVolumeLockRequester enRequester = EN_VOL_LOCK_REQUESTER_UNKNOWN;
      tenVolumeLock enLockState = EN_INT_VOLUME_LOCK_MODE_OFF;
      tU8 u8Prio = 0;
      for (LockStateTable::iterator it = _lockStateTable.begin(); it != _lockStateTable.end(); it++)
      {
         ETG_TRACE_USR4(("Requester : %d, Lock State : %d ",it->first,it->second));
         //if priority of the listed lock state is greater than priority of old lock state
         if (u8GetPriority(it->second) > u8Prio)
         {
            enRequester = it->first;
            enLockState = it->second;
            u8Prio = u8GetPriority(it->second);
         }
      }

      // update
      _properties.vUpdateLock(enRequester, enLockState);
   }

   // -----------------------------------------------------------------------------

   /* static */ tU8 LockAction:: u8GetPriority (tenVolumeLock lockMode)
   {
      // returns the priority of the given lockMode, lowest valid is 1 for first table entry
      // 0 means lockMode is not in table

      const tenVolumeLock PriorityTable[]
       = {
            EN_INT_VOLUME_LOCK_MODE_OFF
          , EN_INT_VOLUME_LOCK_MODE_INCREASING
          , EN_INT_VOLUME_LOCK_MODE_TOTAL
         };
      const tU8 PriorityTableSize
       = static_cast<tU8>(sizeof(PriorityTable) / sizeof(PriorityTable[0]));

      // scan table for given entry and calculate its position
      for (int i = PriorityTableSize - 1; i >= 0; --i)
         if (PriorityTable[i] == lockMode)
            return static_cast<tU8>(1 + i);

      // not found
      return 0;
   }

   // -----------------------------------------------------------------------------

}
