//
// VolumeManager/Configuration/XMLParser.cpp
//
//  Created on: Jul 10, 2014
//      Author: Martin Koch, Fa. ESE
//


// framework
#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#include <etrace_if.h>

#include "Volume/Types.h"
#include "./XMLParser.h"
// - - - - - - - - - - - -

//#define SYSTEM_S_IMPORT_INTERFACE_VECTOR
//#include <stl_pif.h>
#define DP_S_IMPORT_INTERFACE_FI
#include "dp_audio_if.h"

#include <libxml/parser.h>
#include <libxml/xpath.h>
#include <stdlib.h>     /* strtoul */

#include "Volume/Utilities/Uncopyable.h"
#include "Volume/Utilities/StringCopy.hpp"
#include "./dBCalculator.h"
#include "./ConfigDetails.hpp"
//#include "./Stream.h"
//#include "./StreamSet.h"

#include "fc_audiomanager_trace.h"
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_FC_AUDIOMANAGER_VOLUME
#include "trcGenProj/Header/XMLParser.cpp.trc.h"

#if defined _LINUXX86MAKE_
#include <string>
using namespace std;
#endif

//#include <fstream>
#include "./ConfigDetails.hpp"
#define OEM_TYPE_INFINITI     0x16
#define OEM_TYPE_NISSAN       0x06


namespace VolumeManager
{

   // -----------------------------------------------------------------------------

   /* constructor */ XMLParser:: XMLParser ()
      : _document(NULL)
      , _context(NULL)
      , _streamNodes(NULL)
   {
     // -----------------------------------------------------------------------------

     // Location of Configuration.xml file
     //
     // Note: setting must match entry in
     //     ai_projects -> config -> packages -> pkg_procaudio.xml
     //
     #if defined _LINUXX86MAKE_

     std::string naviroot = getenv("_SWNAVIROOT");
     std::string path = "/../ai_audio/components/fc_audiomanager/Volume/Configuration.xml";
     std::string filename = (naviroot + path);
     /* extern */ const char* pConfigFile = filename.c_str();
     #else
     const char *filename;

     filename = "/var/opt/bosch/static/audio/audiomanager/volume_configuration.xml";

     tU8 u8VariantInfo = 0x00;
     if(DP_S32_NO_ERR == DP_s32GetConfigItem("CMVariantCoding","OEMType", &u8VariantInfo, 1))
       {
   if(OEM_TYPE_INFINITI == u8VariantInfo || OEM_TYPE_NISSAN == u8VariantInfo)
     {
       filename = "/var/opt/bosch/static/audio/audiomanager/volume_configuration_pivi.xml";
     }
       }

     const char* pConfigFile = filename;
     #endif

      _document = xmlParseFile(pConfigFile);
      if (_document == NULL )
      {
         ETG_TRACE_FATAL(("E R R O R - XML Configuration Document not parsed successfully - searched at '%s'.", pConfigFile))
         return;
      }

      _context = xmlXPathNewContext(_document);
      _streamNodes = xmlXPathEvalExpression((const xmlChar*)"//Stream", _context);

      ETG_TRACE_USR2(("successfully parsed XML Document: '%s'", pConfigFile))
   }

   // -----------------------------------------------------------------------------

   /* destructor */ XMLParser:: ~XMLParser ()
   {
      xmlXPathFreeObject(_streamNodes);
      _streamNodes = NULL;

      xmlXPathFreeContext(_context);
      _context = NULL;
      xmlFreeDoc(_document);
      _document = NULL;
      xmlCleanupParser();

      ETG_TRACE_USR2(("XML-Parser terminating ..."))
   }

   // -----------------------------------------------------------------------------

   void XMLParser:: vPopulateFunctions (/* output */ FunctionConfig*& pFunctions, size_t& functionCount)
   {
      if (NULL == _context)
         return;

      _xmlXPathObject* _functionNodes = xmlXPathEvalExpression((const xmlChar*)"//Function", _context);
      if (NULL == _functionNodes)
         return;

      functionCount = xmlXPathNodeSetGetLength(_functionNodes->nodesetval);
      pFunctions = new FunctionConfig[functionCount];
      for (size_t i = 0; pFunctions && (i < functionCount); ++i)
      {
         _xmlNode* functionNode = _functionNodes->nodesetval->nodeTab[i];
         pFunctions[i].name = stringCopy(getAttribute(functionNode, "name"));
         pFunctions[i].params = stringCopy(getAttribute(functionNode, "params"));
         pFunctions[i].groupRefs = stringCopy(getAttribute(functionNode, "groupRefs"));
      }

      xmlXPathFreeObject(_functionNodes);
   }

   // -----------------------------------------------------------------------------

   void XMLParser:: vPopulateCalculators (dBCalculator*& pCalculators, size_t& calculatorCount)
   {
      if (NULL == _context)
         return;

      _xmlXPathObject* dbTableNodes = xmlXPathEvalExpression((const xmlChar*)"//VolumeCurve", _context);

      calculatorCount = xmlXPathNodeSetGetLength(dbTableNodes->nodesetval);
      pCalculators = new dBCalculator[calculatorCount];
      if (NULL == pCalculators)
         calculatorCount = 0;
      ETG_TRACE_USR2(("XML-Parser: found %u dbCalculators", calculatorCount))

      for (size_t i = 0; pCalculators && (i < calculatorCount); ++i)
      {
         tDBTable* pNewTable = NULL;
         _xmlNode* pCurve = dbTableNodes->nodesetval->nodeTab[i];
         _xmlNode* pConstraintTarget = getChildElement(pCurve, "ConstraintTarget");
         const char* calculatorName = getAttribute(pCurve, "name");
         tenStream enSecondaryStream = EN_AUDIO_SOURCE_STREAM_DEFAULT;
         tenResource enSink = Speaker;
         const char* sName = NULL;
         if (pConstraintTarget && strlen(sName = getAttribute(pConstraintTarget, "stream")))
            enSecondaryStream = enGetStreamIDFromName(sName);

         const char* sinkName = NULL;
         if (pConstraintTarget && strlen(sinkName = getAttribute(pConstraintTarget, "sink")))
           enSink = enGetResourceIDFromName(sinkName);

         size_t valuesCount = getChildCount(pCurve, "Point");
        // ETG_TRACE_USR2(("XML-Parser: dbCalculator with %u points, named '%s' and Sink : %d", valuesCount, calculatorName, enSink))
         if (valuesCount)
         {
            pNewTable = new tDBTable[valuesCount];
            _xmlNode* pPoint = getChildElement (pCurve, "Point");
            size_t j = 0;
            while (pPoint && pNewTable)
            {
               pNewTable[j].u8UserStep = (tU8)strtoul(getAttribute(pPoint, "step"), NULL, 0);
               pNewTable[j].s16Primary = (tS16)strtol(getAttribute(pPoint, "dB"), NULL, 0);
               if (xmlNode* pConstraint = getChildElement(pPoint, "Constraint"))
               {
                  pNewTable[j].s16Secondary = (tS16)strtol(getAttribute(pConstraint, "value"), NULL, 0);
                  ETG_TRACE_USR3(("\t Point %u:\tstep = %u\tdB (primary) = %d\tdB (secondary) = %d"
                        , j, pNewTable[j].u8UserStep, pNewTable[j].s16Primary, pNewTable[j].s16Secondary))
               }
               else
               {
                  pNewTable[j].s16Secondary = pNewTable[j].s16Primary;
                  ETG_TRACE_USR3(("\t Point %u:\tstep = %u\tdB = %d", j, pNewTable[j].u8UserStep, pNewTable[j].s16Primary))
               }
               ++j;
               pPoint = getNextSibling(pPoint);
            }
            dBCalculator(pNewTable, (unsigned)valuesCount, calculatorName, enSecondaryStream, enSink).swap(pCalculators[i]);
         }
      }
      xmlXPathFreeObject(dbTableNodes);

   }

   // -----------------------------------------------------------------------------

   void XMLParser:: vPopulateStreamSets (FunctionConfig* pFunctions, size_t functionCount
         , /* output */ StreamSetConfig*& pSets, size_t& setCount
         , StreamConfig*& pStreams, size_t& streamCount, dBCalculator* pCalculators, size_t calculatorCount)
   {
      if (NULL == _context)
         return;

      _xmlXPathObject* nodes = xmlXPathEvalExpression((const xmlChar*)"//StreamSet", _context);
      setCount = xmlXPathNodeSetGetLength(nodes->nodesetval);
      pSets = new StreamSetConfig[setCount];
      ETG_TRACE_USR2(("XML-Parser: found %u StreamSets", setCount))

      size_t i = 0;
      streamCount = 0;
      for (i = 0; i < setCount; ++i)
         streamCount += getChildCount(nodes->nodesetval->nodeTab[i], "Stream");
      pStreams = new StreamConfig[streamCount];
      streamCount = 0;
      if ((NULL == pSets) || (NULL == pStreams))  // shall never happen
      {
         setCount = 0;
         return;
      }

      FunctionConfig* pFirstFunctionInSet = pFunctions;

      for (i = 0; i < setCount; ++i)
      {
         // extract attributes from nodes and, if id is valid, insert into StreamSet
         _xmlNode* n = nodes->nodesetval->nodeTab[i];
         const char* setName = getAttribute(n, "name");
         tenResource enResource = enGetResourceIDFromName(getAttribute(n, "resource"));
         ETG_TRACE_USR2(("XML-Parser: StreamSet for resource %u, named '%s'", ETG_CENUM(tenResource, enResource), setName))

         // query number of special functions for current set
         unsigned functionsInSet = (unsigned)getChildCount(n, "Function");
         if (pFirstFunctionInSet + functionsInSet > pFunctions + functionCount)
         {
            ETG_TRACE_FATAL(("vPopulateStreamSets() - E R R O R  : function count mismatch"))
            break;
         }

         const char* dBCalculatorName = getAttribute(n, "dBCalculation");
         dBCalculator* pCalculator = NULL;
         for (unsigned k = 0; k < calculatorCount; ++k)
            if (dBCalculatorName && (0 == strcmp(dBCalculatorName, pCalculators[k].sGetName())))
              pCalculator = &(pCalculators[k]);

             VolumeConstraints Startlimit(0, 0, 0);
             _xmlNode* setLimit = getChildElement(n, "StartUpLimit");
             if (setLimit)
             {
                 ETG_TRACE_USR2(("vPopulateStartUpLimit"))
                 vPopulateVolumeLimits(setLimit, pCalculator, Startlimit);
             }

         if(NULL != pCalculator)
            pSets[i] = StreamSetConfig(enResource, pFirstFunctionInSet, functionsInSet, stringCopy(setName), pCalculator,Startlimit);

         // advance pointer to first function for next set
         pFirstFunctionInSet += functionsInSet;

         // populate related streams
         for (_xmlNode* streamNode = getChildElement(n, "Stream");  streamNode; streamNode = getNextSibling(streamNode))
         {
            tU32 id = (tU32)strtoul(getAttribute(streamNode, "id"), NULL, 0);
            if ((EN_AUDIO_SOURCE_STREAM_DEFAULT < id) && (EN_AUDIO_SOURCE_STREAM_MAX > id))
            {
               tenStream enID = static_cast<tenStream>(id);
               const char* streamName = getAttribute(streamNode, "name");

               pStreams[streamCount] = StreamConfig(&pSets[i], enID, stringCopy(streamName));
               ETG_TRACE_USR2(("XML-Parser:   Stream %u, named '%s'", ETG_CENUM(tenStream, enID), streamName))
               streamCount++;
            }
         }
      }
      setCount = i;

      xmlXPathFreeObject(nodes);
   }

   // -----------------------------------------------------------------------------

   void XMLParser:: vPopulateSourceGroups (StreamConfig* pStreams, size_t streamCount
         , dBCalculator* pCalculators, size_t calculatorCount
         , /* output */ const StreamConfig**& ppStreamMap
         , GroupConfig*& pGroups, size_t& groupCount)
   {
      if (NULL == _context)
         return;
      else if ((NULL == pStreams) || (NULL == pCalculators))
      {
         ETG_TRACE_FATAL(("vPopulateSourceGroups() - E R R O R  : invalid pointer: pStreams = %08X, pCalculators = %08X"
               , pStreams, pCalculators))
         return;
      }

      _xmlXPathObject* groupNodes = xmlXPathEvalExpression((const xmlChar*)"//SourceGroup", _context);
      groupCount = xmlXPathNodeSetGetLength(groupNodes->nodesetval);
      ETG_TRACE_USR2(("XML-Parser: found %u Source-Groups", groupCount))

      // determine number of "Sink"-elements and allocate stream map accordingly
      size_t mapSize = 0;
      for (size_t i = 0; i < groupCount; ++i)
         mapSize += getChildCount(groupNodes->nodesetval->nodeTab[i], "Sink");
      ppStreamMap = new const StreamConfig*[mapSize];
      const StreamConfig** ppStreamsInGroup = ppStreamMap;

      pGroups = new GroupConfig[groupCount];
      const char** pStartupLimitGroupNames = new const char*[groupCount];  // buffer for names of linked OnVolLimit groups
      size_t groupIndex = 0;
      for (size_t i = 0; ppStreamsInGroup && pGroups && (i < groupCount); ++i)
      {
         // extract attributes from nodes
         _xmlNode* pGroupNode = groupNodes->nodesetval->nodeTab[i];
         const char*  groupName = getAttribute(pGroupNode, "name");
         if (NULL == groupName)
            continue;

         tU8  typeId
            = static_cast<tU8>(strtoul(getAttribute(pGroupNode, "typeId"), NULL, 0));

         // extract volume settings
         _xmlNode* pVolumeSettings = getChildElement(pGroupNode, "Volume");
         if (pVolumeSettings)
         {
            const char* dBCalculatorName = getAttribute(pVolumeSettings, "dBCalculation");
            dBCalculator* pCalculator = NULL;
            for (unsigned k = 0; k < calculatorCount; ++k)
               if (dBCalculatorName && (0 == strcmp(dBCalculatorName, pCalculators[k].sGetName())))
                 pCalculator = &(pCalculators[k]);

            VolumeConstraints constraints(0, 0, 0);
            vPopulateVolumeLimits(pVolumeSettings, pCalculator, constraints);

            _xmlNode* pPersistence = getChildElement(pVolumeSettings, "Persistence");
            bool persistent = (NULL != pPersistence);
            VolumeConstraints persistenceLimits = constraints;
            vPopulateVolumeLimits(pPersistence, pCalculator, persistenceLimits);

            const char* PersistanceType = getAttribute(pPersistence, "type");
            tU8 PersistanceTypeValue = 0;
            if(PersistanceType != NULL)
              {
                if ('\0' != PersistanceType[0])  // test against empty string
                  PersistanceTypeValue = static_cast<tU8>(strtoul(PersistanceType, NULL, 0));
              }
            size_t sinkCount = 0;
            vPopulateRelatedSinks (pGroupNode, pStreams, streamCount, ppStreamsInGroup, /* out */ sinkCount);

            // buffer name of linked OnVolLimit groups
            if (pStartupLimitGroupNames)
            {
               _xmlNode* pOnVolLimitGroup = getChildElement(pVolumeSettings, "StartupLimits");
               if (pOnVolLimitGroup)
                  pStartupLimitGroupNames[groupIndex] = getAttribute(pOnVolLimitGroup, "group");
               else
                  pStartupLimitGroupNames[groupIndex] = NULL;
            }

            // create volume group data from above variables
            GroupConfig  newGroup(ppStreamsInGroup, sinkCount, pCalculator, stringCopy(groupName), typeId, constraints, persistent, persistenceLimits, PersistanceTypeValue);
            ppStreamsInGroup += sinkCount;
            pGroups[groupIndex] = newGroup;
            ++groupIndex;
         }
      }
      groupCount = groupIndex;

      // link OnVolLimit groups
      if (pStartupLimitGroupNames && pGroups)
         for (size_t i = 0; i < groupCount; ++i)
            if (pStartupLimitGroupNames[i] && strcmp(pStartupLimitGroupNames[i], ""))
               for (size_t j = 0; j < groupCount; ++j)
               {
                  const char* pLimitGroup = pStartupLimitGroupNames[i];
                  if (0 == strcmp(pLimitGroup, pGroups[j].name))
                  {
                     pGroups[i].pOnVolLimits = &pGroups[j];
                     break;  // continue with next group
                  }
               }

      // log
      for (size_t i = 0; pGroups && (i < groupCount); ++i)
      {
         ETG_TRACE_USR3(("XML-Parser: SourceGroup(%d) '%s'", pGroups[i].typeId, pGroups[i].name))
         ETG_TRACE_USR4(("XML-Parser:\t with calculator '%s'", pGroups[i].pCalculator->sGetName()))
         ETG_TRACE_USR4(("XML-Parser:\t with constraints: min = %u, max = %u, default = %u"
               , pGroups[i].constraints.minStep, pGroups[i].constraints.maxStep, pGroups[i].constraints.defaultStep))
         if (pGroups[i].persistent)
            ETG_TRACE_USR4(("XML-Parser:\t persistent with constraints: min = %u, max = %u, Type=%u"
               , pGroups[i].persistenceConstraints.minStep, pGroups[i].persistenceConstraints.maxStep, pGroups[i].PersistanceType))
         if (pGroups[i].pOnVolLimits)
            ETG_TRACE_USR4(("XML-Parser:\t Startup Volume Limits taken from group '%s'"
                  , pGroups[i].pOnVolLimits->name))
      }

      // cleanup
      delete[] pStartupLimitGroupNames;
      xmlXPathFreeObject(groupNodes);
   }

   // -----------------------------------------------------------------------------

   void XMLParser:: vPopulateSources (GroupConfig* pGroups, size_t groupCount
         , /* output */ SourceConfig*& pSources, size_t& sourceIndex)
   {
      if (NULL == _context)
         return;

      _xmlXPathObject* sourceNodes = xmlXPathEvalExpression((const xmlChar*)"//Source", _context);
      if (NULL == sourceNodes)
         return;

      size_t sourceCount = xmlXPathNodeSetGetLength(sourceNodes->nodesetval);
      sourceIndex = 0;
      pSources = new SourceConfig[sourceCount];
      ETG_TRACE_USR2(("XML-Parser: found %u Sources", sourceCount))

      if (pGroups && pSources)
         for (int j = 0; j < xmlXPathNodeSetGetLength(sourceNodes->nodesetval); ++j)
         {
            xmlNode* srcNode = sourceNodes->nodesetval->nodeTab[j];
            const char* groupInSource = getAttribute(srcNode, "group");
            for (unsigned i = 0; i < groupCount; ++i)
            {
               if (groupInSource && (0 == strcmp(groupInSource, pGroups[i].name)))
               {
                  tenInternalSource enTypeID = static_cast<tenInternalSource>(strtoul(getAttribute(srcNode, "id"), NULL, 0));
                  const char* sourceName = getAttribute(srcNode, "name");
                  SourceConfig relatedSource(pGroups[i], enTypeID, stringCopy(sourceName));
                  pSources[sourceIndex] = relatedSource;
                  ETG_TRACE_USR2(("XML-Parser: \tSource[%u] with ID %u, named '%s'"
                        , sourceIndex, ETG_CENUM(tenInternalSource, enTypeID), pSources[sourceIndex].sGetName()))
                  ++sourceIndex;
               }
            }
         }
      xmlXPathFreeObject(sourceNodes);
   }

   // -----------------------------------------------------------------------------
   //
   //                         P r i v a t e   S e c t i o n
   //

   void XMLParser:: vPopulateRelatedSinks (_xmlNode* pGroupNode
         , const StreamConfig* pStreams, size_t streamCount
         , const StreamConfig** ppSinks, /* out */ size_t& sinkCount)
   {
      if ((NULL == pGroupNode) || (NULL == pStreams) || (NULL == ppSinks))
      {
         ETG_TRACE_FATAL(("vPopulateRelatedSinks() - E R R O R  : invalid pointer : pGroupNode = %08X, pStreams = %08X, ppSinks = %08X"
               , pGroupNode, pStreams, ppSinks))
         return;
      }

      _xmlNode* pSinkNode = getChildElement(pGroupNode, "Sink");
      sinkCount = 0;
      for (    ; pSinkNode;  pSinkNode = getNextSibling(pSinkNode))
      {
         const char* streamInGroup = getAttribute(pSinkNode, "stream");
         for (unsigned j = 0; j < streamCount; ++j)
            if (streamInGroup && (0 == strcmp(pStreams[j].name, streamInGroup)))
            {
               ppSinks[sinkCount] = &(pStreams[j]);
               ++sinkCount;
               break;  // proceed with next SinkNode
            }
      }
   }

   // -----------------------------------------------------------------------------

   void XMLParser:: vPopulateVolumeLimits (_xmlNode* pVolumeSettings, dBCalculator* pCalculator, /* out */ VolumeConstraints& volumeLimits)
   {
      {
         tU8 minVol = 0;
         const char* sMinStep = getAttribute(pVolumeSettings, "minStep");
         if ('\0' != sMinStep[0])  // test against empty string
            minVol = static_cast<tU8>(strtoul(sMinStep, NULL, 0));
         else if (pCalculator)
            minVol = pCalculator->u8GetMinimumUserStep();
         if (minVol > volumeLimits.minStep)
            volumeLimits.minStep = minVol;
      }

      {
         tU8 maxVol = 0;
         const char* sMaxStep = getAttribute(pVolumeSettings, "maxStep");
         if ('\0' != sMaxStep[0])
            maxVol = static_cast<tU8>(strtoul(sMaxStep, NULL, 0));
         else if (pCalculator)
            maxVol = pCalculator->u8GetMaximumUserStep();

         if ((0 == volumeLimits.maxStep) || (maxVol < volumeLimits.maxStep))
            volumeLimits.maxStep = maxVol;
      }

      {
         tU8 defaultVol = static_cast<tU8>(strtoul(getAttribute(pVolumeSettings, "defaultStep"), NULL, 0));
         if ((defaultVol >= volumeLimits.minStep) && (defaultVol <= volumeLimits.maxStep))
            volumeLimits.defaultStep = defaultVol;
      }
   }

   // -----------------------------------------------------------------------------

   /* static */ const char* XMLParser:: getAttribute (_xmlNode* node, const char* attributeName)
   {
      // if something goes wrong, provide at least an empty string
      static const char emptyAttribute[] = "";

      if (NULL == attributeName)
      {
         ETG_TRACE_FATAL(("XMLParser:: getAttribute() - E R R O R :  no attribute name given"))
         return emptyAttribute;
      }

      const char* retVal = (const char*)(xmlGetProp(node, (const xmlChar*)attributeName));
      if (NULL == retVal)
      {
//         ETG_TRACE_FATAL(("XMLParser:: getAttribute() - E R R O R :  attribute '%s' not found", attributeName))
         return emptyAttribute;
      }

      return retVal;
   }

   // -----------------------------------------------------------------------------

   /* static */ _xmlNode* XMLParser:: getChildElement (_xmlNode* node, const char* elementName)
   {
      // find the named child element in given node

      if ((NULL == node) || (XML_ELEMENT_NODE != node->type))
         return NULL;  // this is not an XML element, so cannot have requested child element

      for (xmlNode* child = node->children; child; child = child->next)
      {
         if (XML_ELEMENT_NODE != child->type)
            continue;

         if (0 == strcmp(reinterpret_cast<const char*>(child->name), elementName))
            return child;
      }

      // not found
      return NULL;
  }

   // -----------------------------------------------------------------------------

   /* static */ _xmlNode* XMLParser:: getNextSibling(_xmlNode* node)
   {
      // searches down for next child element having same name

      if ((NULL == node) || (XML_ELEMENT_NODE != node->type))
         return NULL;  // this is not an XML element, so cannot have requested child element

      for (xmlNode* sibling = node->next; sibling; sibling = sibling->next)
      {
         if (XML_ELEMENT_NODE != sibling->type)
            continue;

         if (0 == strcmp(reinterpret_cast<const char*>(sibling->name), reinterpret_cast<const char*>(node->name)))
            return sibling;
      }

      // not found
      return NULL;
  }

   // -----------------------------------------------------------------------------

   /* static */ size_t XMLParser:: getChildCount(_xmlNode* node, const char* elementName)
   {
      // count the child elements with given elementNamein given node

      if ((NULL == node) || (XML_ELEMENT_NODE != node->type))
         return 0;  // this is not an XML element, so cannot have requested child element
      else if (NULL == elementName)
         return 0;

      size_t retVal = 0;
      for (xmlNode* child = node->children; child; child = child->next)
         if (    (XML_ELEMENT_NODE == child->type)
              && (0 == strcmp(reinterpret_cast<const char*>(child->name), elementName)))
            ++retVal;

      return retVal;
  }

   // -----------------------------------------------------------------------------

   tenStream XMLParser:: enGetStreamIDFromName (const char* searchName)
   {
      if ((NULL == _streamNodes) || (NULL == searchName))
         return EN_AUDIO_SOURCE_STREAM_DEFAULT;

      // locate matching stream
      for (int i = 0; i < xmlXPathNodeSetGetLength(_streamNodes->nodesetval); ++i)
      {
         xmlNode* streamNode = _streamNodes->nodesetval->nodeTab[i];
         const char* name = getAttribute(streamNode, "name");
         if (strcmp(name, searchName))
            continue;

         tU32 id = (tU32)strtoul(getAttribute(streamNode, "id"), NULL, 0);
         if ((EN_AUDIO_SOURCE_STREAM_DEFAULT < id) && (EN_AUDIO_SOURCE_STREAM_MAX > id))
            return static_cast<tenStream>(id);
      }

      // no match found
      return EN_AUDIO_SOURCE_STREAM_DEFAULT;
   }

   // -----------------------------------------------------------------------------

   /* static */ tenResource XMLParser:: enGetResourceIDFromName (const char* resourceName)
   {
      if(NULL == resourceName)
  return Speaker;

      if (0 == strcmp(resourceName, "LS"))
         return Speaker;
      else if (0 == strcmp(resourceName, "HP"))
         return Headphone;
      else if (0 == strcmp(resourceName, "Sink17")) //daw2hi 24.05.2018
         return Sink17;
      // no match
      ETG_TRACE_ERR(("XMLParser:: enGetResourceIDFromName()  - no match for '%s'", resourceName))
      return UndefinedResource;
   }


}  // namespace VolumeManager
