//
// VolumeManager/Configuration/SourceConfigs.h
//
// Configuration details and parameters of audio sources
//
//  Created on: Jul 10, 2014
//      Author: Martin Koch, Fa. ESE
//

#if !defined(EA_855AC81D_EB73_430d_936C_CAB2E79FDE4B__INCLUDED_)
#define EA_855AC81D_EB73_430d_936C_CAB2E79FDE4B__INCLUDED_
//#ifndef SOURCE_H_
//#define SOURCE_H_

namespace VolumeManager
{

   // --------------------------------------------------------------------------

   class dBCalculator;

   // --------------------------------------------------------------------------

   struct FunctionConfig
   {
      const char* name;
      const char* params;
      const char* groupRefs;
   };

   // --------------------------------------------------------------------------

   struct VolumeConstraints
   {
      tU8   minStep;
      tU8   defaultStep;
      tU8   maxStep;

      inline /* constructor */  VolumeConstraints (tU8 min, tU8 def, tU8 max);
      inline /* default constructor */ VolumeConstraints ();
      // use compiler-generated assignment operator and destructor
   };

   struct StreamSetConfig
   {
      tenResource      enResource;   // Resource the context is responsible for.
      const char*     name;
      FunctionConfig*  pFunctions;
      unsigned        numberOfFunctions;
      const dBCalculator* pCalculator;
      VolumeConstraints         Startlimit;

      inline /* constructor */ StreamSetConfig (tenResource resource, FunctionConfig* pFunctions, unsigned numberOfFunctions_, const char* name_, dBCalculator* pCalc,VolumeConstraints StartUpLimit);
      inline /* default constructor */ StreamSetConfig ();
      // use compiler-generated assignment operator and destructor
   };

   // --------------------------------------------------------------------------

   struct StreamConfig
   {
      const StreamSetConfig*  pStreamSet;
      tenStream         enStreamID;
      const char*      name;

      inline /* constructor */ StreamConfig (StreamSetConfig* pSet, tenStream stream, char* name_);
      inline /* default constructor */ StreamConfig ();
      // use compiler-generated assignment operator and destructor
   };

   // --------------------------------------------------------------------------


   // --------------------------------------------------------------------------

   struct GroupConfig
   {
      const StreamConfig** ppSinks;
      size_t                sinkCount;
      const dBCalculator* pCalculator;
      const char*        name;
      tU8                typeId;
      VolumeConstraints  constraints;
      bool               persistent;
      VolumeConstraints  persistenceConstraints;
      const GroupConfig* pOnVolLimits;
      tU8 PersistanceType;

      inline /* constructor */ GroupConfig (const StreamConfig** ppSinks_, size_t sinkCount_
            , dBCalculator* pCalc, char* groupName, tU8 typeId, VolumeConstraints min_max
            , bool persistent, VolumeConstraints persistence_min_max, tU8 PersistanceType);
      inline /* default constructor */ GroupConfig ();
      // use compiler-generated assignment operator and destructor
   };

   // --------------------------------------------------------------------------

   class SourceConfig
   {
      friend class Configuration;  // allow final delete of name string

   private:
      const GroupConfig* pGroup;
      tenInternalSource  id;
      const char*       name;

   public:
      inline /* constructor */ SourceConfig (GroupConfig& group, tenInternalSource sourceID, char* sourceName);
      inline /* default constructor */ SourceConfig ();
      // use compiler-generated assignment operator and destructor

      inline tenInternalSource enGetSourceID() const;
      inline const char* sGetName() const;
      inline tenStream enGetStreamID (tenResource enResource) const;
      inline tU8 u8GetGroupType () const;
      inline VolumeConstraints getVolumeLimits (bool requestPersistenceLimits = false) const;
      inline const dBCalculator* pGetCalculator() const;

   };


   // ==========================================================================

   inline /* constructor */  VolumeConstraints:: VolumeConstraints (tU8 min, tU8 def, tU8 max)
      : minStep(min)
      , defaultStep(def)
      , maxStep(max)
   {

   }

   // --------------------------------------------------------------------------

   inline /* default constructor */  VolumeConstraints:: VolumeConstraints ()
      : minStep(0)
      , defaultStep(0)
      , maxStep(0)
   {

   }

   // ==========================================================================
   //
   //                        ---   inline implementations   ---
   //

   inline /* constructor */ StreamSetConfig:: StreamSetConfig (tenResource r, FunctionConfig* pFuncs, unsigned n, const char* nm, dBCalculator* pCalc,VolumeConstraints StartUpLimit)
      : enResource(r)
      , name(nm)
      , pFunctions(pFuncs)
      , numberOfFunctions(n)
      , pCalculator(pCalc)
      , Startlimit(StartUpLimit)
   {

   }

   // --------------------------------------------------------------------------

   inline /* default constructor */ StreamSetConfig:: StreamSetConfig ()
      : enResource(UndefinedResource)
      , name(NULL)
      , pFunctions(NULL)
      , numberOfFunctions(0)
      , pCalculator(NULL)
      , Startlimit()
   {

   }

   // ==========================================================================

   inline /* constructor */ StreamConfig:: StreamConfig (StreamSetConfig* pSet, tenStream stream, char* name_)
      : pStreamSet(pSet)
      , enStreamID(stream)
      , name(name_)
   {

   }

   // --------------------------------------------------------------------------

   inline /* default constructor */ StreamConfig:: StreamConfig ()
      : pStreamSet(NULL)
      , enStreamID(EN_AUDIO_SOURCE_STREAM_DEFAULT)
      , name(NULL)
   {

   }

   // ==========================================================================

   inline /* constructor */ GroupConfig:: GroupConfig (const StreamConfig** ppSinks_, size_t sinkCount_
         , dBCalculator* pCalc, char* groupName, tU8 type, VolumeConstraints min_max
         , bool persistent_, VolumeConstraints persistence_min_max_, tU8 persistancetype)
      : ppSinks(ppSinks_)
      , sinkCount(sinkCount_)
      , pCalculator(pCalc)
      , name(groupName)
      , typeId(type)
      , constraints(min_max)
      , persistent(persistent_)
      , persistenceConstraints(persistence_min_max_)
      , pOnVolLimits(NULL)
      , PersistanceType(persistancetype)
   {

   }

   // --------------------------------------------------------------------------

   inline /* default constructor */ GroupConfig:: GroupConfig ()
      : ppSinks(NULL)
      , sinkCount(0)
      , pCalculator(NULL)
      , name(NULL)
      , typeId(255)    // FI_EN_VOLUME_NOT_DEF
      , constraints()
      , persistent(false)
      , persistenceConstraints()
      , pOnVolLimits(NULL)
      , PersistanceType(255)
   {

   }

   // ==========================================================================

   inline /* constructor */ SourceConfig:: SourceConfig (GroupConfig& group, tenInternalSource sourceID, char* sourceName)
      : pGroup(&group)
      , id(sourceID)
      , name(sourceName)
   {

   }

   // --------------------------------------------------------------------------

   inline /* default constructor */ SourceConfig:: SourceConfig ()
      : pGroup(NULL)
      , id(AUD_INT_SRC_NOSOURCE)
      , name(NULL)
   {

   }

   // --------------------------------------------------------------------------

   inline tenInternalSource SourceConfig:: enGetSourceID() const
   {
      return id;
   }

   // --------------------------------------------------------------------------

   inline const char* SourceConfig:: sGetName() const
   {
      return name;
   }

   // --------------------------------------------------------------------------

   inline tenStream SourceConfig:: enGetStreamID (tenResource enResource) const
   {
      if (pGroup && pGroup->ppSinks)
         for (size_t i = 0; i < pGroup->sinkCount; i++)
            if (pGroup->ppSinks[i] && pGroup->ppSinks[i]->pStreamSet)// && (pGroup->ppSinks[i]->pStreamSet->enResource == enResource))
               return pGroup->ppSinks[i]->enStreamID;

      // not found
      (void)enResource;
      return EN_AUDIO_SOURCE_STREAM_DEFAULT;

   }

   // --------------------------------------------------------------------------

   inline tU8 SourceConfig:: u8GetGroupType () const
   {
      if (pGroup)
         return pGroup->typeId;
      else
         return 255;  // FI_EN_VOLUME_TYPE_NOT_DEF
   }

   // --------------------------------------------------------------------------

   inline VolumeConstraints SourceConfig:: getVolumeLimits (bool requestPersistenceLimits) const
   {
      if (pGroup)
         if (requestPersistenceLimits)
            return pGroup->persistenceConstraints;
         else
            return pGroup->constraints;
      else    // should not happen
         return VolumeConstraints(0, 0, 0);
   }

// --------------------------------------------------------------------------

   inline const dBCalculator* SourceConfig:: pGetCalculator() const
   {
      if (pGroup)
         return pGroup->pCalculator;
      else
         return NULL;
   }

   // --------------------------------------------------------------------------

}


//#endif /* SOURCE_H_ */
#endif // !defined(EA_855AC81D_EB73_430d_936C_CAB2E79FDE4B__INCLUDED_)
