/**
 * @file TuningDataService.cpp
 * @author CM-AI wie1hi
 * @copyright (c) 2014 Robert Bosch Car Multimedia GmbH
 * @addtogroup fc_audiomanager
 * @{
 */

// for TCP
#include <unistd.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <string.h>
#include <sys/stat.h>
#include <errno.h>
// end

#define ETRACE_S_IMPORT_INTERFACE_GENERIC
//#define ET_TRACE_INFO_ON
#include <etrace_if.h>

#include "fc_audiomanager_trace.h"
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_FC_AUDIOMANAGER_TUNING_DATA_SERVICE
#include "trcGenProj/Header/TuningDataService.cpp.trc.h"
#endif

#include "TuningDataServiceMessage.h"
#include "TuningDataService.h"

#ifdef VARIANT_S_FTR_ENABLE_FEAT_AUDIO_ARKAMYS

#ifdef VARIANT_S_FTR_ENABLE_FEAT_AUDIO_ARKAMYS_CLASSIC
#include "Sound/ArkamysTuningHandlerRNAIVI.h"
#include "Sound/ArkamysTuningHandlerRNAIVI2.h"
#endif //VARIANT_S_FTR_ENABLE_FEAT_AUDIO_ARKAMYS_CLASSIC

#ifdef VARIANT_S_FTR_ENABLE_FEAT_AUDIO_ARKAMYS_PSA
#include "Sound/ArkamysTuningHandler.h"
#endif // VARIANT_S_FTR_ENABLE_FEAT_AUDIO_ARKAMYS_PSA
#endif //VARIANT_S_FTR_ENABLE_FEAT_AUDIO_ARKAMYS

#ifdef VARIANT_S_FTR_ENABLE_FEAT_AUDIO_ENGINE_SOUND

#ifdef VARIANT_S_FTR_ENABLE_FEAT_AUDIO_ENGINE_SOUND_IVI
#include "Sound/ESETuningHandlerRNAIVI.h"
#endif //VARIANT_S_FTR_ENABLE_FEAT_AUDIO_ENGINE_SOUND_IVI

#endif //VARIANT_S_FTR_ENABLE_FEAT_AUDIO_ENGINE_SOUND

#define DP_S_IMPORT_INTERFACE_FI
#include "dp_generic_if.h"


// Number of elements
#define NELT(_array) (sizeof(_array) / sizeof(_array[0]))
#define DATA_OFFSET    2//Trc class+ Trc Func
#define SINKDEV_WITHOUT_ADR (tU8)42
#define SINKDEV_PASSTHRU_ADR (tU8)43

TuningDataService::TuningDataService()
{
   m_bkeepRunning = TRUE;
   m_iActiveConnectionHandles[0] = -1;
   m_responseBufferSize = 0;
   m_pResponseBuffer = OSAL_NULL;
   vSetError(ERR_NO_ERROR_YET);
   m_tenlastError = ERR_NO_ERROR_YET;
   m_hRxThreadId = -1;
   m_bSBRVariant = false;
   fc_audiomanager_tclApp::theServer()->vRegisterTraceInputs(TRC::enAudioTuning, this);
}

TuningDataService::~TuningDataService()
{
   if (m_pResponseBuffer != OSAL_NULL)
   {
      delete[] m_pResponseBuffer;
      m_responseBufferSize = 0;
   }
   m_pResponseBuffer = OSAL_NULL;
}

/**
 * @method TuningDataService* GetInstance()
 * @description Singleton pattern
 */

TuningDataService* TuningDataService::getInstance()
{
   static TuningDataService theInstance;
   return &theInstance;
}

/**
 * @method tBool TuningDataService::bRegisterUser()
 * @param None
 * @return Returns true if registration was successful, else false based on project feature switch
 */
tBool TuningDataService::bRegisterUser()
{
	tBool result = true;
#ifdef VARIANT_S_FTR_ENABLE_FEAT_AUDIO_ARKAMYS_PSA
      result = bRegisterUser(TuningDataService::ID_ARKAMYS_PSA, ArkamysTuningHandler::getInstance());
      if (result)
      {
         ETG_TRACE_USR3(("ArkamysTuningHandler->PSA registered with TuningDataService."));
      }
      else
      {
         ETG_TRACE_ERR(("Could not register ArkamysTuningHandler->PSA with TuningDataService."));
      }
#endif // VARIANT_S_FTR_ENABLE_FEAT_AUDIO_ARKAMYS_PSA
#ifdef VARIANT_S_FTR_ENABLE_FEAT_AUDIO_ARKAMYS_CLASSIC
      vCheckSBRConfig();
      if(m_bSBRVariant)
    	  result = bRegisterUser(TuningDataService::ID_ARKAMYS_RNAIVI, ArkamysTuningHandlerRNAIVI2::getInstance());
      else
    	  result = bRegisterUser(TuningDataService::ID_ARKAMYS_RNAIVI, ArkamysTuningHandlerRNAIVI::getInstance());
      if (result)
      {
         ETG_TRACE_USR3(("ArkamysTuningHandler->IVI registered with TuningDataService."));
      }
      else
      {
         ETG_TRACE_ERR(("Could not register ArkamysTuningHandler->IVI with TuningDataService."));
      }
#endif // VARIANT_S_FTR_ENABLE_FEAT_AUDIO_ARKAMYS
#ifdef VARIANT_S_FTR_ENABLE_FEAT_AUDIO_ENGINE_SOUND_IVI
      result = bRegisterUser(TuningDataService::ID_ESE_RNAIVI, ESETuningHandlerRNAIVI::getInstance());
      if (result)
      {
         ETG_TRACE_USR3(("ESETuningHandler->IVI registered with TuningDataService."));
      }
      else
      {
         ETG_TRACE_ERR(("Could not register ESETuningHandler->IVI with TuningDataService."));
      }
#endif//VARIANT_S_FTR_ENABLE_FEAT_AUDIO_ENGINE_SOUND_IVI
      return result;
}
/**
 * @method tBool TuningDataService::bRegisterUser(tenTuningDataClient id, ITuningDataClient *pReceiver)
 * @param id ID of the client that registers. See Header file for enum.
 * @param pReceiver Pointer to an instance of a class that implements the interface ITuningDataClient.
 *                   This object will get the received data via the respective callback method.
 * @return Returns true if registration was successful, else false.
 */
tBool TuningDataService::bRegisterUser(tenTuningDataServiceId id, ITuningDataClient *pReceiver)
{
   std::pair<tenTuningDataServiceId, ITuningDataClient*> newClient(id, pReceiver); // exception bad_alloc must not be caught by application
   m_vClients.push_back(newClient);
   ETG_TRACE_USR4(("TuningDataService: Service registered successfully."));
   return true;
}

/**
 * @method tVoid vDataTx()
 * @description This method sends data to the socket described by txHandle.
 *              It adds the appropriate header as defined for the protocol.
 *              The handle that has to be passed as txHandle is provided
 *              to functions that have registered by calling bRegisterUser()
 *              and have received a callback via the ITuningDataClient interface.
 * @param txHandle The socket handle that was provided by the callback function
 * @data Pointer to the data to be sent.
 * @param size The size of the data pointed to by parameter "data".
 *             Make sure not to violate any memory boundaries!
 */
tVoid TuningDataService::vDataTx(tU8 serviceID, tU8 txHandle, const tU8* data, tU32 size, tU8 msgType)
{
   ETG_TRACE_USR4(("TuningService:vDataTx(%i, %i, %p, %i, %i) entered.", serviceID, txHandle, data, size, msgType));
   NORMAL_M_ASSERT(txHandle < NELT(m_iActiveConnectionHandles));
   if (txHandle < NELT(m_iActiveConnectionHandles))
   {
      tU32 outSize = size + TUNING_DATA_SERVICE_RESPONSE_HEADER_SIZE;
      ASSERT_STATIC(TUNING_DATA_SERVICE_RESPONSE_HEADER_SIZE == 2); // As of version 1

      // check for inconsistency
      if ((m_responseBufferSize == 0 && m_pResponseBuffer != OSAL_NULL) ||
          (m_responseBufferSize != 0 && m_pResponseBuffer == OSAL_NULL))
      {
         ETG_TRACE_ERR(("TuningDataService: Inconsistent inner state. Bailing out."));
         NORMAL_M_ASSERT_ALWAYS();
         return;
      }

      // reallocate only if necessary
      if (m_responseBufferSize < outSize)
      {
         if (m_pResponseBuffer != OSAL_NULL)
         {
            delete[] m_pResponseBuffer;
         }
         if (outSize >= 2){ // always true, but lint needs to know this explicitly
            m_pResponseBuffer = OSAL_NEW tU8[outSize];
            m_responseBufferSize = outSize;
         }
      }

      NORMAL_M_ASSERT(m_responseBufferSize >= outSize);

      // Fill header
      if (m_pResponseBuffer != OSAL_NULL && m_responseBufferSize >= (TUNING_DATA_SERVICE_RESPONSE_HEADER_SIZE + size))
      {
         m_pResponseBuffer[0] = serviceID;
         m_pResponseBuffer[1] = msgType;
         if (m_responseBufferSize < (TUNING_DATA_SERVICE_RESPONSE_HEADER_SIZE + size))
         {
            ETG_TRACE_ERR(("m_responseBufferSize < (TUNING_DATA_SERVICE_RESPONSE_HEADER_SIZE + size): %i < ( %i + %i)", m_responseBufferSize, TUNING_DATA_SERVICE_RESPONSE_HEADER_SIZE, size));
         }
         NORMAL_M_ASSERT(m_responseBufferSize >= (TUNING_DATA_SERVICE_RESPONSE_HEADER_SIZE + size));
         if (size != 0 && data != OSAL_NULL && (size + 2 <= m_responseBufferSize))
         {
            //lint --e{826} Suspicious pointer-to-pointer conversion (area too small)
            // -> No, everything is fine here.
            // m_pResponseBuffer has at least 3 elements as is asserted multiple times above
            OSAL_pvMemoryCopy(&m_pResponseBuffer[2], data, size);
         }

         // txHandle is smaller than number of elements of m_iActiveConnectionHandles, as ensured by surrounding if-clause
         if (m_iActiveConnectionHandles[txHandle] != -1)
         {
            ETG_TRACE_USR4(("Sending data: (outSize:%i, buffer size:%i) %x", outSize, m_responseBufferSize, ETG_LIST_LEN(outSize), ETG_LIST_PTR_T8(m_pResponseBuffer)));
            (void)send(m_iActiveConnectionHandles[txHandle], m_pResponseBuffer, outSize, 0);
         }
         else
         {
            ETG_TRACE_USR4(("Not Sending data. Handle invalid."));
         }
      }
      else
      {
         ETG_TRACE_ERR(("m_pResponseBuffer=%p m_responseBufferSize=%i (should be >= %i)", m_pResponseBuffer, m_responseBufferSize, TUNING_DATA_SERVICE_RESPONSE_HEADER_SIZE + size));
      }
   }
}

/**
 * @method tVoid TuningDataService::vDataRxThread()
 * @description The actual server runs here. This method is called in an extra thread and will block until shutdown
 */
tVoid TuningDataService::vDataRxThread()
{
   ETG_TRACE_USR3(("TuningDataService: Receive thread started."));

   // Create socket
   int sock_desc = (int)socket(AF_INET, SOCK_STREAM, 0);
   if (sock_desc == -1)
   {
       ETG_TRACE_ERR(("TuningDataService: Create socket failed."));
       return;
   }

   // Bind on port TUNING_DATA_SERVICE_PORT
   struct sockaddr_in server;
   memset(&server, 0, sizeof(server));
   server.sin_family = AF_INET;
   server.sin_addr.s_addr = INADDR_ANY;
   server.sin_port = htons(TUNING_DATA_SERVICE_PORT);
   if (bind(sock_desc, reinterpret_cast<struct sockaddr*>(&server), sizeof(server)) != 0)
   {
       ETG_TRACE_ERR(("TuningDataService: bind socket failed."));
       close(sock_desc);
       return;
   }

   // Allow up to 5 clients to wait while connection is busy
   if (listen(sock_desc, 5) != 0)
   {
       ETG_TRACE_ERR(("TuningDataService: Listen on socket failed. Socket invalid."));
       close(sock_desc);
       return;
   }

   struct sockaddr_in client;
   memset(&client, 0, sizeof(client));
   socklen_t len = sizeof(client);

   // Prepare data structures

   // Receive buffer for raw TCP data
   tU8 recvBuf[TUNING_DATA_SERVICE_RECEIVE_BUFFER_SIZE];
   ASSERT_STATIC(sizeof(recvBuf) >= TUNING_DATA_SERVICE_MESSAGE_HEADER_SIZE);    // Must fit at least the size of the header
   ASSERT_STATIC(TUNING_DATA_SERVICE_MESSAGE_HEADER_SIZE == 8);                  // Adapt this assertion if header size changes

   ASSERT_STATIC(sizeof(recvBuf[0]) == 1); // Code relies on this assumption
   // Message object
   TuningDataServiceMessage oMsg; // Tuning data message

   // Pointers and counters to be used during receive
   tU32  bufWriteIndex = 0;         // write pointer (index) in buffer
   tU32  msgWriteIndex = 0;         // write pointer (index) in message
   tBool bReceivingHeader = true;   // Indicate if waiting for header
   tU32  totalReceivedBytes = 0;    // Counts total bytes received during connection

   // Handle connections loop
   while (m_bkeepRunning)
   {
      // Accept client (blocks until a client connects)
      m_iActiveConnectionHandles[0] = accept(sock_desc, reinterpret_cast<struct sockaddr*>(&client), &len);
      if (m_iActiveConnectionHandles[0] == -1)
      {
          ETG_TRACE_ERR(("TuningDataService: Accept client failed."));
          //close(sock_desc); solving coverity lint warning
          continue;
      }

      m_tenlastError = ERR_NO_ERROR_YET;
      ETG_TRACE_USR4(("TuningDataService: Client connected from %i.%i.%i.%i (Handle 0, Socket %i)."
            ,((client.sin_addr.s_addr >>  0) & 0xFF)
            ,((client.sin_addr.s_addr >>  8) & 0xFF)
            ,((client.sin_addr.s_addr >> 16) & 0xFF)
            ,((client.sin_addr.s_addr >> 24) & 0xFF)
            ,m_iActiveConnectionHandles[0]
      ));

      // Receive data from client
      while (true)
      {
//         ETG_TRACE_USR4(("bHeader:%d bufWritePtr:%d msgWritePtr:%d", bReceivingHeader, bufWriteIndex, msgWriteIndex));

         // Determine length of data to read from TCP stream
         int rcvLen = 0;
         if (bReceivingHeader)
         {
            NORMAL_M_ASSERT(bufWriteIndex <= TuningDataServiceMessage::getHeaderSize());
            rcvLen = (int)(TuningDataServiceMessage::getHeaderSize() - bufWriteIndex);
//            ETG_TRACE_USR4(("Waiting for header... %d bytes remaining.", rcvLen));
         }
         else
         {
            NORMAL_M_ASSERT(msgWriteIndex >= TuningDataServiceMessage::getHeaderSize());
            rcvLen = (int)(oMsg.dataLength - (msgWriteIndex - TuningDataServiceMessage::getHeaderSize()));
//            ETG_TRACE_USR4(("Waiting for data... %d bytes remaining.", rcvLen));

            // Ensure not to write beyond receive buffer boundary
            if ((rcvLen + bufWriteIndex) > sizeof(recvBuf))
            {
               rcvLen = (int)(sizeof(recvBuf) - bufWriteIndex);
            }
         }

         // Read from TCP stream (blocking)
         NORMAL_M_ASSERT(bufWriteIndex < sizeof(recvBuf)); // Never write beyond boundary!
         int iReceivedBytes = (int)recv(m_iActiveConnectionHandles[0], &recvBuf[bufWriteIndex], rcvLen, 0);
         bufWriteIndex += iReceivedBytes;
         totalReceivedBytes += iReceivedBytes;

         if (iReceivedBytes > 0)
         {
            ETG_TRACE_USR4(("TuningDataService: Received %i bytes.", iReceivedBytes));

            // Fill header data when header is received completely
            if (bReceivingHeader && bufWriteIndex >= TuningDataServiceMessage::getHeaderSize())
            {
               ETG_TRACE_USR4(("TuningDataService: Header (%i bytes) received.", TuningDataServiceMessage::getHeaderSize()));
               memcpy(&oMsg.preface, &recvBuf[msgWriteIndex], sizeof(oMsg.preface));
               msgWriteIndex += (tU32)sizeof(oMsg.preface);
               memcpy(&oMsg.version, &recvBuf[msgWriteIndex], sizeof(oMsg.version));
               msgWriteIndex += (tU32)sizeof(oMsg.version);
               memcpy(&oMsg.serviceID, &recvBuf[msgWriteIndex], sizeof(oMsg.serviceID));
               msgWriteIndex += (tU32)sizeof(oMsg.serviceID);
               memcpy(&oMsg.dataLength, &recvBuf[msgWriteIndex], sizeof(oMsg.dataLength));
               msgWriteIndex += (tU32)sizeof(oMsg.dataLength);
               NORMAL_M_ASSERT(msgWriteIndex == TuningDataServiceMessage::getHeaderSize());

               // convert bytes from network to host
               oMsg.preface = ntohs(oMsg.preface);
               oMsg.dataLength = ntohl(oMsg.dataLength);

               // Check Header
//               ETG_TRACE_USR4(("TuningDataService: Check header."));
               tBool bHeaderOk = true;
               if (oMsg.preface != 0)
               {
                  vSetError(ERR_PROTOCOL_ERROR);
                  bHeaderOk = false;
               }

               if (bHeaderOk && (oMsg.version < TUNING_DATA_SERVICE_MIN_VERSION || oMsg.version > TUNING_DATA_SERVICE_MAX_VERSION))
               {
                  vSetError(ERR_VERSION_NOT_SUPP);
                  bHeaderOk = false;
               }

               if (bHeaderOk && (oMsg.dataLength > TUNING_DATA_SERVICE_MAX_MSG_DATA_SIZE))
               {
                  vSetError(ERR_DATA_LENGTH_TOO_LARGE);
                  bHeaderOk = false;
               }

               if (bHeaderOk && oMsg.dataLength == 0)
               {
                  vSetError(ERR_NO_DATA);
                  bHeaderOk = false;
               }

               // Check service ID
               if (oMsg.serviceID != ID_TUNING_DATA_SRV)
               {
                  tBool bServiceFound = false;
                  std::vector<std::pair<tenTuningDataServiceId, ITuningDataClient*> >::iterator it;
                  for (it = m_vClients.begin(); it != m_vClients.end(); ++it)
                  {
                     if (it -> first == static_cast<tenTuningDataServiceId>(oMsg.serviceID))
                     {
                        bServiceFound = true;
                        break;
                     }
                  }

                  if (bHeaderOk && !bServiceFound)
                  {
                     vSetError(ERR_UNKNOWN_SERVICE);
                     bHeaderOk = false;
                  }
               }

               if (bHeaderOk)
               {
                  ETG_TRACE_USR4(("TuningDataService: Header ok. "
                        "  Preface: %i"
                        "  Version: %i"
                      "  Service ID: %i"
                       "  Data Length: %i", oMsg.preface, oMsg.version, ETG_CENUM(tenTuningDataServiceId, oMsg.serviceID), oMsg.dataLength));


                  // Prepare message data (allocate memory)
                  if (oMsg.dataLength > static_cast<tU32>(0)) // cast due to lint warning
                  {
                     oMsg.allocDataMemory(oMsg.dataLength); // exception bad_alloc must not be caught by application
                  }

                  // Copy remaining data to message
                  if (iReceivedBytes > static_cast<int>(msgWriteIndex)) // cast due to lint warning
                  {
//                     ETG_TRACE_USR4(("Copy remaining data to message."));
                     tU32 dataLength = bufWriteIndex - msgWriteIndex;
                     if (dataLength <= oMsg.dataLength && bufWriteIndex >= msgWriteIndex)
                     {
                        //tU8 *pData = oMsg.getData();
                        memcpy(oMsg.getData(), &recvBuf[msgWriteIndex], dataLength);
                        msgWriteIndex += dataLength;
                     }
                  }
                  bufWriteIndex = 0;

                  // Now wait for data if any
                  if (oMsg.dataLength > 0)
                  {
                     bReceivingHeader = false;
                  }
               }
               else //if (!bHeaderOk)
               {
                  ETG_TRACE_USR4(("TuningDataService: Header NOT OK! "
                        "  Error: %d"
                        "  Preface: %d"
                        "  Version: %d"
                        "  Service ID: %d"
                        "  Data Length: %i"
                        , m_tenlastError, oMsg.preface, oMsg.version, ETG_CENUM(tenTuningDataServiceId, oMsg.serviceID), oMsg.dataLength));

                  // Check if more data is available
                  fd_set readfds = { 0 };
                  FD_SET(m_iActiveConnectionHandles[0], &readfds);
                  struct timeval sTimeout = { 0, 0 }; // return immediately
                  bool bDataAvailable = 0 < select(m_iActiveConnectionHandles[0] + 1, &readfds, OSAL_NULL, OSAL_NULL, &sTimeout);

                  // Flush if needed
                  if (bDataAvailable)
                  {
                     int iReceived = (int)recv(m_iActiveConnectionHandles[0], &recvBuf[0], sizeof(recvBuf), MSG_DONTWAIT);
                     ETG_TRACE_USR4(("TuningDataService: Flushed %i bytes.", iReceived));
                  }

                  tU16 err = htonl(m_tenlastError);
                  vDataTx(ID_TUNING_DATA_SRV, 0, (tU8*)&err, sizeof(err), MSG_TYPE_ERROR); // 0: Index for m_iActiveConnectionHandles

                  bReceivingHeader = true;
                  bufWriteIndex = 0;
                  msgWriteIndex = 0;
               }

            }
            else if (!bReceivingHeader)
            {
               // Copy data to message
               int dataLength = MIN(static_cast<tU32>(iReceivedBytes), oMsg.dataLength);
               if (dataLength < 0)
               {
                  dataLength = 0;
               }
               tU32 tgtIndex = (tU32)(msgWriteIndex - TuningDataServiceMessage::getHeaderSize());

               //memcpy(&oMsg.data[tgtIndex], recvBuf, dataLength);
               memcpy(&(oMsg.getData()[tgtIndex]), recvBuf, dataLength);
               msgWriteIndex += dataLength;
               bufWriteIndex = 0;

               // Message complete!
               if (msgWriteIndex > oMsg.dataLength)
               {
                  ETG_TRACE_USR4(("Complete message received. Transferring to serviceID %d.", ETG_CENUM(tenTuningDataServiceId, oMsg.serviceID)));
                  ETG_TRACE_USR4(("%02x", ETG_LIST_LEN(oMsg.dataLength), ETG_LIST_PTR_T8(oMsg.getData())));

                  if (oMsg.serviceID == ID_TUNING_DATA_SRV)
                  {
                     vProcessRequest(oMsg.dataLength, oMsg.getData(), 0); // 0: Index for m_iActiveConnectionHandles
                  }
                  else
                  {
                     tU32 msgSize = (tU32)(TUNING_DATA_SERVICE_MESSAGE_HEADER_SIZE + DATA_OFFSET + oMsg.dataLength);
                     tPU8 pu8Message = OSAL_NEW tU8[msgSize];

                     if (pu8Message == OSAL_NULL) // Lint wants a check
                     {
                        ETG_TRACE_FATAL(("Could not allocate %i bytes.", msgSize));
                        return;
                     }
                     pu8Message[0] = (tU8)TRC::enAudioTuning;
                     pu8Message[1] = (tU8)TRC::enAudioTuningFunc;

                     OSAL_pvMemoryCopy(&pu8Message[2], &oMsg.preface, sizeof(tU16)); // copy the preface into the array

                     pu8Message[4] = oMsg.version;
                     pu8Message[5] = oMsg.serviceID;


                     OSAL_pvMemoryCopy(&pu8Message[6], &oMsg.dataLength, sizeof(tU32)); // copy the data length into the array
                     OSAL_pvMemoryCopy(&pu8Message[TUNING_DATA_SERVICE_MESSAGE_HEADER_SIZE + DATA_OFFSET], oMsg.getData(), oMsg.dataLength); // copy the data into the array

                     ETG_TRACE_USR4(("Message to 'other thread': Size=%i, Data=%02x", msgSize, ETG_LIST_LEN(msgSize), ETG_LIST_PTR_T8(pu8Message)));
                     // Can be called from different threads, so bring to Audio CCA thread
                     fc_audiomanager_tclApp::vRxFromOtherThreads(msgSize, pu8Message);
                     OSAL_DELETE []pu8Message;
                  }

                  bReceivingHeader = true;
                  msgWriteIndex = 0;
                  bufWriteIndex = 0;
               }

               // If more data available: move to beginning of buffer
               if (iReceivedBytes > dataLength){
                  memmove(recvBuf, &recvBuf[dataLength], iReceivedBytes - dataLength);
                  bufWriteIndex = iReceivedBytes - dataLength;
               }
            }
         }
         else
         {
            close(m_iActiveConnectionHandles[0]);
            m_iActiveConnectionHandles[0] = -1;
            ETG_TRACE_USR4(("TuningDataService: Client disconnected. Total received bytes: %d", totalReceivedBytes));
            totalReceivedBytes = 0;
            bReceivingHeader = true;
            bufWriteIndex = 0;
            msgWriteIndex = 0;
            break;
         }
      }
      oMsg.free();
   }

   ETG_TRACE_ERR(("Receive thread stopped."));

}

/**
 * @method tVoid vTraceRx()
 * @description Listens for loopback message.
 */

tVoid TuningDataService::vTraceRx(tU32 size, tPCUChar pcu8Data)
{
  ETG_TRACE_USR4(("TuningDataService::vTraceRx()"));

  // if only header data is available do nothing
  if((size == TUNING_DATA_SERVICE_MESSAGE_HEADER_SIZE + DATA_OFFSET) || (pcu8Data == OSAL_NULL))
  {
    ETG_TRACE_USR4(("TuningDataService::vTraceRx() No data available to send"));
    return;
  }

  TuningDataServiceMessage Msg; // Tuning data message

  Msg.serviceID = pcu8Data[5];

  OSAL_pvMemoryCopy(&Msg.dataLength, &pcu8Data[6], sizeof(tU32));

  Msg.allocDataMemory(Msg.dataLength); //allocate memory to copy the data

  OSAL_pvMemoryCopy(Msg.getData(), &pcu8Data[TUNING_DATA_SERVICE_MESSAGE_HEADER_SIZE + DATA_OFFSET], Msg.dataLength);

  //send the data to respective tuning handlers based on service ID
     std::vector<std::pair<tenTuningDataServiceId, ITuningDataClient*> >::iterator it;
     for (it = m_vClients.begin(); it != m_vClients.end(); ++it)
     {
    if (it -> first == (tenTuningDataServiceId)Msg.serviceID)
        {
           ETG_TRACE_USR4(("Transferring to serviceID %d.", ETG_CENUM(tenTuningDataServiceId, Msg.serviceID)));
       ETG_TRACE_USR4(("%02x", ETG_LIST_LEN(Msg.dataLength), ETG_LIST_PTR_T8(Msg.getData())));
       it -> second -> DataRx(Msg.dataLength, Msg.getData(), 0); // 0: Index for m_iActiveConnectionHandles
           break;
        }
     }

   Msg.free();
}
/**
 * @method tVoid TuningDataService::vStartThread(tVoid* pvArg)
 */
tVoid TuningDataService::vStartThread(tVoid* pvArg)
{
   if(pvArg != NULL)
   {
      TuningDataService* pTuningDataService = reinterpret_cast<TuningDataService*>(pvArg);
      // call the thread loop
      pTuningDataService->vDataRxThread();
   }
   else
   {
      NORMAL_M_ASSERT_ALWAYS();
   }
}

/**
 * @method tBool bStartupServer()
 * @description Starts the Tuning Data Service by starting up a thread
 *              that listens for incoming TCP connections.
 */
tBool TuningDataService::bStartupServer()
{
   // CMG3G-13417 IS3444_A-IVI Tuning Interface Arkamys/ESE@FC_AudioManager
   FILE* pTuningEnable = NULL;   /* pointer to file */
   const char       *TuningEnableFileName = TUNING_ENABLE_FILE;
   pTuningEnable = fopen(TuningEnableFileName, "r");
   if(NULL == pTuningEnable)
   {
      return false;
   }
   fclose(pTuningEnable);// end

   OSAL_trThreadAttribute  rAttr;
   tC8                     szThreadName[OSAL_C_U32_MAX_NAMELENGTH] = "";

   (void)OSAL_szStringCopy( szThreadName, "TuningDataService");

   // Local variables to store data read from registry
   tU32 u32ThreadPrio = 0;
   tU32 u32StackSize  = 0;

   vGetThreadPrioandStacksize(u32ThreadPrio, u32StackSize);

   //ETG_TRACE_USR3(("vRxThreadSetup() ,u32ThreadPrio= %d,u32StackSize= %d",u32ThreadPrio,u32StackSize));

   // Initialize thread parameters
   rAttr.szName       = szThreadName;
   rAttr.s32StackSize = (tS32)u32StackSize;
   rAttr.u32Priority  = u32ThreadPrio;
   rAttr.pfEntry      = (OSAL_tpfThreadEntry)&TuningDataService::vStartThread;
   rAttr.pvArg        = (tPVoid)this;

   m_hRxThreadId = OSAL_ThreadSpawn(&rAttr);

   if ( m_hRxThreadId == OSAL_ERROR )
   {
      tS32 s32OsalError = (tS32)OSAL_u32ErrorCode(  );
      ETG_TRACE_ERR(("setUpServer -> Thread spawn failed errorcode=%x" ,s32OsalError ));

   }
#if 1
   else
   {
      ETG_TRACE_USR3(("SocketReceive ,OSAL_ThreadSpawn()-> Thread creation successful."));
   }
#endif
   return true;
}

/**
 * @method tVoid vGetThreadPrioandStacksize(tU32 &u32ThreadPrio,tU32 &u32StackSize) const
 * @description returns thread priority and stack size to the passed parameters
 */
tVoid TuningDataService::vGetThreadPrioandStacksize(tU32 &u32ThreadPrio,tU32 &u32StackSize) const
{
   // check with vd_adr3_inc.cpp: it is also possible to have this config in registry
   // Assign default value.
   u32ThreadPrio = TUNING_DATA_SERVICE_DEFAULT_PRIO;
   u32StackSize  = TUNING_DATA_SERVICE_DEFAULT_STACKSIZE;

   // Indicate that the thread is running on default priority
   //ETG_TRACE_ERR(( "vGetThreadPrioandStacksize prio=%d, stack size=%d",u32ThreadPrio,u32StackSize));
}

/**
 * bCreateReply
 * @brief see header
 */
//tBool TuningDataService::bCreateReply(tU8 command, tU8 *pData, tU8 *replyBuf, tU16 bufSize)
//{
//   return true;
//}

/**
 * vProcessRequest()
 * Process a request to the "Meta Services" of this protocol.
 */
tVoid TuningDataService::vProcessRequest(tU32 size, tVoid* data, tU8 txHandle)
{
   ETG_TRACE_USR4(("TuningDataService: vProcessRequest(size:%d, *data, txHandle:%d) entered.", size, txHandle));
   if (size < 1)
   {
      ETG_TRACE_ERR(("TuningDataService: Received Meta Service message without data."));
      return;
   }

   tU8 *pData = (tU8*)data;
   tU8 command = pData[0];
   //lint --e{778} Constant expression evaluates to 0 in operation '-'
   // Yes, it does that deliberately
   tU8 reply[3 + MAX(
         (int)(TUNING_DATA_SERVICE_MAX_VERSION - TUNING_DATA_SERVICE_MIN_VERSION), // if replying to CMD_GET_SUPP_VERSIONS
         sizeof(tU32) // if replying to CMD_GET_MAX_DATA_LENGTH
         )]; // Adapt if longer replies become necessary

   ETG_TRACE_USR4(("TuningDataService: command: %d", ETG_CENUM(tenCommands, command)));

   switch (command)
   {
   case CMD_CMD_AVAIL:
      if (size == 2)
      {
         reply[0] = (tU8)CMD_CMD_AVAIL;
         reply[1] = (pData[1] < CMD_MAX) ? 1 : 0;
         vDataTx(ID_TUNING_DATA_SRV, txHandle, reply, 2);
      }
      else
      {
         vSetError(ERR_UNKNOWN_CMD);
      }
      break;
   case CMD_SERVICE_AVAIL:
      if (size == 2)
      {
         reply[0] = (tU8)CMD_SERVICE_AVAIL;
         reply[1] = 0;

         if (pData[1] == ID_TUNING_DATA_SRV)
         {
            reply[1] = 1;
         }
         else
         {
            std::vector<std::pair<tenTuningDataServiceId, ITuningDataClient*> >::iterator it;
            for (it = m_vClients.begin(); it != m_vClients.end(); ++it)
            {
               if (it -> first == (tenTuningDataServiceId)pData[1])
               {
                  reply[1] = 1;
                  break;
               }
            }
         }

         vDataTx(ID_TUNING_DATA_SRV, txHandle, reply, 2);
      }
      else
      {
         vSetError(ERR_UNKNOWN_CMD);
      }
      break;
   case CMD_GET_SUPP_VERSIONS:
      {
         reply[0] = (tU8)CMD_GET_SUPP_VERSIONS;

         tU8 ver = TUNING_DATA_SERVICE_MIN_VERSION;
         for (int n = 1; ver <= TUNING_DATA_SERVICE_MAX_VERSION; ++n)
         {
            reply[n] = ver;
            ++ver;
         }

         vDataTx(ID_TUNING_DATA_SRV, txHandle, reply, 2 + (int)(TUNING_DATA_SERVICE_MAX_VERSION - TUNING_DATA_SERVICE_MIN_VERSION));
      }
      break;
   case CMD_GET_MAX_DATA_LENGTH:
      {
         reply[0] = CMD_GET_MAX_DATA_LENGTH;
         tU32 maxDataLength = TUNING_DATA_SERVICE_MAX_MSG_DATA_SIZE;
         memcpy(&reply[1], &maxDataLength, sizeof(maxDataLength));
         ETG_TRACE_USR2(("Max Data Length: %d (%02x,%02x,%02x,%02x)", maxDataLength, reply[1], reply[2], reply[3], reply[4]));
         vDataTx(ID_TUNING_DATA_SRV, txHandle, reply, 1 + sizeof(maxDataLength));
      }
      break;
   case CMD_GET_LAST_ERROR:
      {
         reply[0] = CMD_GET_LAST_ERROR;
         tU16 errCode = m_tenlastError;
         memcpy(&reply[1], &errCode, sizeof(errCode));
         vDataTx(ID_TUNING_DATA_SRV, txHandle, reply, 1 + sizeof(errCode));

         // Reset last error if error was available
         if (m_tenlastError != ERR_NO_ERROR_YET)
         {
            m_tenlastError = ERR_NO_CUR_ERROR;
         }
      }
      break;
   default:
      reply[0] = command;
      tU16 errCode = ERR_UNKNOWN_CMD;
      memcpy(&reply[1], &errCode, sizeof(errCode));
      vDataTx(ID_TUNING_DATA_SRV, txHandle, reply, 1 + sizeof(errCode), MSG_TYPE_ERROR);
      vSetError(ERR_UNKNOWN_CMD);
      break;
   }
}

tVoid TuningDataService::vSetError(tenErrorCodes errCode)
{
   ETG_TRACE_USR2(("TuningDataService: Error occured: %d", ETG_CENUM(tenErrorCodes, errCode)));
   m_tenlastError = errCode;
}

/**
 * @method tBool TuningDataService::vCheckSBRConfig()
 * @description Function to determine ADR/SBR variant
 */
tVoid TuningDataService::vCheckSBRConfig()
{
	tU8 u8KDSParam = 0; //initializing with internal amplifier

	if((DP_S32_NO_ERR == DP_s32GetConfigItem("GenericTunerParameter","MuteOrAttenuationForAbsoluteStationChangeAMFM", &u8KDSParam, 1)))
	{
	   ETG_TRACE_USR4(("GenericTunerParameter value  : %d", u8KDSParam));
       if((u8KDSParam == SINKDEV_WITHOUT_ADR) || (u8KDSParam == SINKDEV_PASSTHRU_ADR))
         m_bSBRVariant = true;
	}
	else
	{
		ETG_TRACE_ERR(("Error in diagnosis read "));
	}

	if(m_bSBRVariant)
	{
		if (mkdir("/tmp/audiomanager/", 0770) == -1)
			ETG_TRACE_FATAL(("failed to create arkamys folder inside tmp %s",strerror(errno)));
	}
}

