/*******************************************************************************
 *
 * FILE:          SoundToolClient.cpp
 *
 * SW-COMPONENT:  SoundToolClient application
 *
 * PROJECT:
 *
 * DESCRIPTION:
 *
 * AUTHOR:
 *
 * COPYRIGHT:    (c) 2012 Robert Bosch GmbH, Hildesheim
 *
 *******************************************************************************/

/******************************************************************************/
/*                                                                            */
/* INCLUDES                                                                   */
/*                                                                            */
/******************************************************************************/

#include "../fc_audiomanager_main.h"
#include "../fc_audiomanager_service_Audio_Function.h"

#include "../fc_audiomanager_trace.h"
#include "../fc_audiomanager_trace_macros.h"


#include "../aud_sinkmanager/vd_adr3/vd_adr3Msg_If.h"
#include "fc_audiomanager_componentfactory_if.h"

#include "SoundTool_TcpRxTx.h"
#include "SoundToolClient.h"

#define DP_S_IMPORT_INTERFACE_FI
#include "dp_audio_if.h"

//#define GENERICMSGS_S_IMPORT_INTERFACE_GENERIC
//#include "generic_msgs_if.h"

// Include message framework interface (AMT, msgfw)
#define FI_S_IMPORT_INTERFACE_FI_MESSAGE

//Include FI interface of used service
#define MIDW_FI_S_IMPORT_INTERFACE_MIDW_MASCFFI_TYPES
#define MIDW_FI_S_IMPORT_INTERFACE_MIDW_MASCFFI_FUNCTIONIDS
#define MIDW_FI_S_IMPORT_INTERFACE_MIDW_MASCFFI_ERRORCODES
#define MIDW_FI_S_IMPORT_INTERFACE_MIDW_MASCFFI_SERVICEINFO

#include "midw_fi_if.h"


#define ETRACE_S_IMPORT_INTERFACE_GENERIC
//#define ET_TRACE_INFO_ON
#include "etrace_if.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_FC_AUDIOMANAGER_SOUND_TOOL_CLIENT
#include "trcGenProj/Header/SoundToolClient.cpp.trc.h"
#endif

#include "../../../PostOffice/PostOffice.hpp"
#include "../../InternalComponentCommunication/InternalCommunicationAdapter.h"
#include "../../InternalComponentCommunication/Messages/SoundConfig/IDNotifySendSoundConfigFile.h"
#include "../../InternalComponentCommunication/Messages/settings/IDSettings.h"
//#include "../../util/Macro.h"

//#include"../config/delay_default.h"
//#include"../config/equ_default.h"

#include "vd_adr3_Config_Types.h"
#include "vd_adr3_Config_File.h"
#include "vd_adr3_Config_Streamers.h"


/******************************************************************************/
/*                                                                            */
/* DEFINES                                                                    */
/*                                                                            */
/******************************************************************************/

/******************************************************************************/
/*                                                                            */
/* GLOBAL VARIABLES                                                           */
/*                                                                            */
/******************************************************************************/
// Static self reference to be used in function callbacks.
fc_audiomanager_tclApp*   SoundToolClient::poCcaMainApp  = NULL;
SoundToolClient*           SoundToolClient::poSoundToolClient = NULL;

SoundTool_TcpRxTx*         SoundToolClient::m_poSoundTool_TcpRxTx = NULL;

/******************************************************************************/
/*                                                                            */
/* METHODS                                                                    */
/*                                                                            */
/******************************************************************************/


SoundToolClient* SoundToolClient::getSoundToolClient()
{
   if(poSoundToolClient == NULL) {
      ETG_TRACE_USR4(("SoundToolClient::getSoundToolClient() return Null"));
   }

   return poSoundToolClient;
}
/*******************************************************************************
 *
 * FUNCTION: SoundToolClient::SoundToolClient()
 *
 * DESCRIPTION: Constructor.
 *
 *              Initialize static self reference to be used in function
 *              callbacks.
 *
 * PARAMETER: None.
 *
 * RETURNVALUE: None.
 *
 *******************************************************************************/
SoundToolClient::SoundToolClient(fc_audiomanager_tclApp* poMainAppl, fc_audiomanager_tclService_Audio_Function* poService) :
    fc_audiomanager_IClientHandler_Soundtool(poMainAppl,poService)
{
   ETG_TRACE_USR4(("SoundToolClient() constructor entered."));

   SoundToolClient::poCcaMainApp = poMainAppl;
   SoundToolClient::poAudioService = poService;

   poSoundToolClient = this;

   if(poMainAppl != NULL)
      poCcaMainApp->vRegisterTraceInputs(TRC::enSoundToolClient, this);

   ETG_TRACE_USR4(("SoundToolClient() constructor"));

   FILE            *pSoundtoolEnable = NULL;   /* pointer to file */
   const char       *SoundtoolEnableFileName = SOUNDTOOL_ENABLE_FILE;
   pSoundtoolEnable = fopen(SoundtoolEnableFileName, "r");
   if(NULL == pSoundtoolEnable)
   {
      ETG_TRACE_USR4(("enable file for soundtool tuning not found, communication will not be established"));
      return;
   }
   //we can close again
   fclose(pSoundtoolEnable);


   // now set up the TCP Socket
   m_poSoundTool_TcpRxTx = new SoundTool_TcpRxTx(this);
   if(m_poSoundTool_TcpRxTx != NULL)
   {
        tBool bRet;
        ETG_TRACE_USR4(("m_poSoundTool_TcpRxTx->bStart"));
        bRet = (tBool)m_poSoundTool_TcpRxTx->setUpServer();
        ETG_TRACE_USR4(("m_poSoundTool_TcpRxTx->setUpServer bRet = %u",bRet));
   }
   else
   {
        ETG_TRACE_USR4(("m_poSoundTool_TcpRxTx is NULL"));
   }
   return;

}

void SoundToolClient::vRxTcp(tPCUChar pcu8Data)
{
  ETG_TRACE_USR4(("vRxTcp() entered %x, %x, %x, %x",pcu8Data[0],pcu8Data[1],
      pcu8Data[2],pcu8Data[3]));

  tU16 len = (tU16)((pcu8Data[0]<<8)+pcu8Data[1]);
  // pcu8Data[2] left empty
    tU8 opCode = pcu8Data[3];

#ifdef DEBUG_PRINT
    tU16 idx=0;
    FILE* dbg_write_ptr = fopen("/var/opt/bosch/dynamic/ffs/Soundtool.log","a");
    fprintf (dbg_write_ptr, "vRxTcp() entered len = %d, %x, %x, %x, %x \n",len,pcu8Data[0],pcu8Data[1],pcu8Data[2],pcu8Data[3]);
    while(idx < len)
    {
      fprintf (dbg_write_ptr, "%02X, ",(unsigned char)*(pcu8Data+idx));
      idx++;
      if((idx>0) && (idx%20==0)) fprintf (dbg_write_ptr, "\n");
    }
    fclose(dbg_write_ptr);
#endif

    setSoundToolFlag(opCode);

    switch(opCode)
    {
    case TRC::GetPropertyListLength: //0x1D
    {
      handleGetPropertyListLength(len, const_cast<tU8*>(pcu8Data));
      break;
    }
    case TRC::SetConfigFile:   //0x02
    {
#ifdef DEBUG_PRINT
      dbg_write_ptr = fopen("/var/opt/bosch/dynamic/ffs/Soundtool.log","a");
      fprintf (dbg_write_ptr, "\nSetConfigFile \n");
      fclose(dbg_write_ptr);
#endif

      ETG_TRACE_USR4(("Received SetConfigFile 0x%x",*(pcu8Data+5)));
      tU16 u16FileID = *(pcu8Data+5);
    (void)saveConfigFile(u16FileID);

      sendSetConfigFileStatus(SOUNDTOOL_SUCCESS);

      ETG_TRACE_USR4(("Len: %d,",len-3));
      ETG_TRACE_USR4(("Data: %x, %x, %x, %x, %x",*(pcu8Data+3),*(pcu8Data+4),*(pcu8Data+5),*(pcu8Data+6),*(pcu8Data+7)));

      // daw2hi 30.11.2015 forgot to send it
        ID_NotifySendSoundConfigFile oSendSoundConfigFile((tU8)(u16FileID & 0x00FF), (tPCUChar)(pcu8Data + 9), len-9);
        InternalCommunicationAdapter::getInstance()->POMessages->DeliverMsg(&oSendSoundConfigFile);

      break;
    }
    case TRC::GetConfigFile:   //0x03
    {
      ETG_TRACE_USR4(("Received GetConfigFile"));
      handleGetConfigFile(len-2, pcu8Data+2);
      break;
    }
    default:
      break;
    }


    tChar u8MsgBuffer[5];

    u8MsgBuffer[0] = 0; //len_H
    u8MsgBuffer[1] = 5; //len_L

    u8MsgBuffer[2] = opCode; // mirror back the Msg Code
    u8MsgBuffer[3] = SOUNDTOOL_SUCCESS; //success
    u8MsgBuffer[4] = (tU8)0;

    //ET_TRACE_BIN(TR_CLASS_SOUND_TOOL_COMMUNICATION, TR_LEVEL_FATAL, et_enTraceType::ET_EN_T8LIST, 3, u8MsgBuffer);
    //et_vTraceBinary(TR_CLASS_SOUND_TOOL_COMMUNICATION, TR_LEVEL_FATAL, ET_EN_T8LIST, 3, u8MsgBuffer+2, ET_EN_DONE);

    if(m_poSoundTool_TcpRxTx != NULL)
    {
#ifdef DEBUG_PRINT
      dbg_write_ptr = fopen("/var/opt/bosch/dynamic/ffs/Soundtool.log","a");
      fprintf (dbg_write_ptr, "vRxTcp SendStatus opCode= 0x%x \n",opCode);
      fclose(dbg_write_ptr);
#endif
         //sleep(2);
         m_poSoundTool_TcpRxTx->SocketSend(5,(tU8*)u8MsgBuffer);
    }
  return;
}




/*******************************************************************************
 *
 * FUNCTION: SoundToolClient::~SoundToolClient()
 *
 * DESCRIPTION: Destructor.
 *
 *              Invalidate static self reference.
 *
 * PARAMETER: None.
 *
 * RETURNVALUE: None.
 *
 *******************************************************************************/
SoundToolClient::~SoundToolClient()
{
   ETG_TRACE_USR4(("SoundToolClient() destructor entered."));
   SoundToolClient::poCcaMainApp = NULL;
   SoundToolClient::poAudioService = NULL;

   SoundToolClient::poSoundToolClient=NULL;
   //SoundToolClient::m_pPropertyIF_BaseIter = NULL;

   return;
}

/********************************************************************************
 * vInit().
 *******************************************************************************/
tVoid SoundToolClient::vInit()const
{
   ETG_TRACE_USR4(("SoundToolClient vInit() entered."));
#if 0 //no one calls vInit move this to constructor
   if(poCcaMainApp != NULL) {
      poCcaMainApp->vRegisterTraceInputs(TRC::enSoundToolClient, this);
   }
#endif
   return;
}


/********************************************************************************
 * vOnApplicationClose().
 *******************************************************************************/
tVoid SoundToolClient::vOnApplicationClose()
{

   ETG_TRACE_USR4(("vOnApplicationClose() entered."));

   return;
}

/********************************************************************************
 * handleGetPropertyListLength().
 * This method will send all registered properties as a list back to Soundtool
 *
 * ToDO: Currently only hard coded for Soundtool adaptation (always 1 property available)
 *       Real implementation will follow
 *******************************************************************************/
tVoid SoundToolClient::handleGetPropertyListLength(tU32 size, tU8 const* const pu8Data)const
{
   (tVoid) size; // Lint
   tChar u8MsgBuffer[4];

   u8MsgBuffer[0] = pu8Data[1]; // mirror back the Msg Code
   u8MsgBuffer[1] = SOUNDTOOL_SUCCESS; //success
   //u8MsgBuffer[2] = (tU8)m_PropertyIF_BaseList.size();
   u8MsgBuffer[2] = (tU8)0;

   //ET_TRACE_BIN(TR_CLASS_SOUND_TOOL_COMMUNICATION, TR_LEVEL_FATAL, et_enTraceType::ET_EN_T8LIST, 3, u8MsgBuffer);
   et_vTraceBinary(TR_CLASS_SOUND_TOOL_COMMUNICATION, TR_LEVEL_FATAL, ET_EN_T8LIST, 3, u8MsgBuffer, ET_EN_DONE);


   if(m_poSoundTool_TcpRxTx != NULL)
   {
     tU8 u8SendBuf[6];
     memset(u8SendBuf,0,6);
     u8SendBuf[1] = 5; //the size of the message
     memcpy(u8SendBuf+2, u8MsgBuffer,sizeof(u8MsgBuffer));

     //???
     u8SendBuf[2] = 0x1d;

        m_poSoundTool_TcpRxTx->SocketSend(sizeof(u8SendBuf),(tU8*)u8SendBuf);
   }
}

#if 0
/********************************************************************************
 * handleSaveConfigFile().
 * This method will save a ConfigFile to DP
 *
 *******************************************************************************/
tVoid SoundToolClient::handleSaveConfigFile(tPCUChar pcu8Data)
{
   // ToDo
   tU16 u16FileID;
   tU8 u8MuteFlag;
   //check if size is big enough to avoid out of bounce
   if(pcu8Data[0] >= 0x04)
   {
      u8MuteFlag = pcu8Data[2];
      // LSB first as agreed with Christoph Montag
      u16FileID = pcu8Data[4];
      u16FileID = u16FileID <<8;
      u16FileID = u16FileID + pcu8Data[3];

      ETG_TRACE_USR4(("handleSaveConfigFile() for FileID = %x mute=%x",u16FileID,u8MuteFlag));
   }
   else //file too short
   {
      ETG_TRACE_USR4(("handleSaveConfigFile() Msg to short."));
   }
}
#endif
/********************************************************************************
 * saveConfigFile().
 *
 * Saves a config file into DP according to the given FileID.
 * The save is called within the SetConfigFile command
 *******************************************************************************/

tS32 SoundToolClient::saveConfigFile(tU16 u16FileID)const
{

   // return code for DP calls
   tS32 s32Res=1; //initial value 1 would be NOK. s32Res should be set to 0 if operation is successful

   tCString sDpElementName = adr3_tclParser::sGetDpElementName(adr3_tenDataFunctionID(u16FileID));
   tU32 u32DpElementSize = adr3_tclParser::u32GetDatapoolElementSize(sDpElementName);

   if (u32DpElementSize)
      s32Res = 0;  // data pool element exists and has a size, so vd_andr_Config will be able to update

   ETG_TRACE_USR4(("saveConfigFile() done with s32Res=%x for FileID %u (%s)",s32Res, u16FileID, sDpElementName))
   return s32Res;

}

/********************************************************************************
 * handleGetConfigFile().
 *
 * This method will read a config File according to the FileID given as parameter within the buffer.
 * See Soundtool documentation for the communication protocol.
 * The config file will be send to Soundtool with method sendGetConfigFile
 *******************************************************************************/
tVoid SoundToolClient::handleGetConfigFile(tU32 size, tPCUChar pcu8Data)const
{
   //check if size is big enough to avoid out of bounce access
   if (size < 0x04)
   {
      ETG_TRACE_ERR(("handleGetConfigFile() Error Msg to short."))
      return;
   }

   // provide error response as answer if something goes wrong;
   tU8 au8ADR3_ErrorBuf[6] = { pcu8Data[1], SOUNDTOOL_FAILURE, pcu8Data[2], pcu8Data[3], 0, 0 };

   // LSB first as agreed with Christoph Montag (implementer of Soundtool on PC)
   adr3_tenDataFunctionID enFileID  = adr3_tenDataFunctionID((pcu8Data[3] << 8) + pcu8Data[2]);

   tCString sDpElementName = adr3_tclParser::sGetDpElementName(enFileID);
   tU32 u32DpElementSize = adr3_tclParser::u32GetDatapoolElementSize(sDpElementName);
   if (0U == u32DpElementSize)  // validate
   {
      ETG_TRACE_ERR(("handleGetConfigFile() Error - No valid data-pool element for FileID %u. (%s)", enFileID, sDpElementName))
      sendGetConfigFile(au8ADR3_ErrorBuf, 6);

#ifdef DEBUG_PRINT
    FILE* dbg_write_ptr = fopen("/var/opt/bosch/dynamic/ffs/Soundtool.log","a");
    fprintf (dbg_write_ptr, "vRxTcp() handleGetConfigFile Error 0\n");
    fclose(dbg_write_ptr);
#endif

      return;
   }

   // Allocate buffer for config file data
   tU8* pu8ADR3_FileBuf = OSAL_NEW tU8[6 + u32DpElementSize];  // provide 6 bytes for message header
   if (NULL == pu8ADR3_FileBuf)
   {
      ETG_TRACE_ERR(("handleGetConfigFile() Error - message buffer allocation failed"))
      sendGetConfigFile(au8ADR3_ErrorBuf, 6);
#ifdef DEBUG_PRINT
    FILE* dbg_write_ptr = fopen("/var/opt/bosch/dynamic/ffs/Soundtool.log","a");
    fprintf (dbg_write_ptr, "vRxTcp() handleGetConfigFile Error 1\n");
    fclose(dbg_write_ptr);
#endif
      return;
   }


   tU32 u32BytesCopied;
   adr3_tclParser::vCopyFromDatapool(sDpElementName, pu8ADR3_FileBuf+6, u32DpElementSize, u32BytesCopied);
   if( (u32BytesCopied == 0) || (u32BytesCopied >= (6 + u32DpElementSize)) )
   {
      // available buffer should always be bigger to ensure we got it all
      ETG_TRACE_ERR(("ERROR handleGetConfigFile()  %u bytes copied to %x of size %u for FileID = %x (%s)"
         , u32BytesCopied, pu8ADR3_FileBuf+6, u32DpElementSize, enFileID, sDpElementName));

      sendGetConfigFile(au8ADR3_ErrorBuf, 6);

      // free the no more needed memory after it was send
      OSAL_DELETE [] pu8ADR3_FileBuf;
#ifdef DEBUG_PRINT
    FILE* dbg_write_ptr = fopen("/var/opt/bosch/dynamic/ffs/Soundtool.log","a");
    fprintf (dbg_write_ptr, "vRxTcp() handleGetConfigFile Error 2\n");
    fclose(dbg_write_ptr);
#endif

      return;
   }

   // copy received data into response
   pu8ADR3_FileBuf[0] = pcu8Data[1]; // reply with received MsgCode
   pu8ADR3_FileBuf[1] = SOUNDTOOL_SUCCESS; //status success
   pu8ADR3_FileBuf[2] = pcu8Data[2]; //reply received u16FileID LSB
   pu8ADR3_FileBuf[3] = pcu8Data[3]; //reply received u16FileID MSB
   // add the file size into message, LSB first, agreed with Christoph 14.01.2012
   pu8ADR3_FileBuf[4] = (tU8)(u32BytesCopied & 0x000000FF); //LSB
   pu8ADR3_FileBuf[5] = (tU8)((u32BytesCopied & 0x0000FF00) >> 8); //MSB

   ETG_TRACE_USR2(("sending %u bytes to Sound Tool for file %u (%s)", u32BytesCopied + 6, enFileID, sDpElementName))
   if (et_bIsTraceActive((tU16)TR_CLASS_FC_AUDIOMANAGER_SOUND_TOOL_CLIENT, (tU16)TR_LEVEL_USER_3))
   {
      adr3_tclParser oParser(sDpElementName, pu8ADR3_FileBuf + 6, u32DpElementSize);
      adr3_tclFile(enFileID, oParser).vPrintLog(NULL);
   }

   // consider here also the header offset 6Bytes pu8ADR3_FileBuf[0]...pu8ADR3_FileBuf[5]
   // and add it to the data red from DP
   sendGetConfigFile(pu8ADR3_FileBuf, (tU16)(u32BytesCopied + 6));

   if(m_poSoundTool_TcpRxTx != NULL)
   {
     tU8* pBuffer = new tU8[2048];
     if(pBuffer != NULL)
     {
       memset(pBuffer,0,2048);
       // here we put the overall size
       pBuffer[0] = (tU8)(((u32BytesCopied+8) & 0xFF00)>>8);  //High
       pBuffer[1] = (tU8)((u32BytesCopied+8) & 0x00FF);  //Low

       //now we leave 6 Bytes empty header
       // pBuffer[2] bis Buffer[7]

       pBuffer[2] = 0x03;
       pBuffer[3] = 0x55;
       pBuffer[4] = 0x07;
       pBuffer[5] = 0x00;

       //now 2 Bytes data size
       pBuffer[6] = (tU8)(u32BytesCopied & 0x000000FF); //LSB
       pBuffer[7] = (tU8)((u32BytesCopied & 0x0000FF00) >> 8); //MSB

       memcpy(pBuffer+8,pu8ADR3_FileBuf+6,u32BytesCopied);

  #ifdef DEBUG_PRINT
       FILE* dbg_write_ptr = fopen("/var/opt/bosch/dynamic/ffs/Soundtool.log","a");
       fprintf (dbg_write_ptr, "\nvRxTcp() handleGetConfigFile sending %d bytes \n",u32BytesCopied+8);
       fprintf (dbg_write_ptr, "---\n");
       for(int idx=0;idx<u32BytesCopied+8;idx++)
       {
         fprintf (dbg_write_ptr,"0x%02x, ",pBuffer[idx]);
         if((idx>0) && (idx%20==0)) fprintf (dbg_write_ptr,"\n");

       }
       fprintf (dbg_write_ptr, "\n---\n");
       fclose(dbg_write_ptr);
  #endif
       m_poSoundTool_TcpRxTx->SocketSend((tU16)u32BytesCopied+8,pBuffer);
       delete [] pBuffer;

  #if 0
       // don't need the full header here, we only need the size
       pu8ADR3_FileBuf[0] = 0;
       pu8ADR3_FileBuf[1] = 0;
       pu8ADR3_FileBuf[2] = 0;
       pu8ADR3_FileBuf[3] = 0;

       // this size is inclusive the first two bytes. They indicate the whole msg size (2 BYtes for size plus data)
       pu8ADR3_FileBuf[4] = (tU8)(((u32BytesCopied+2) & 0x0000FF00) >> 8); //MSB
       pu8ADR3_FileBuf[5] = (tU8)((u32BytesCopied+2) & 0x000000FF); //LSB

       FILE* dbg_write_ptr = fopen("/var/opt/bosch/dynamic/ffs/Soundtool.log","a");
       fprintf (dbg_write_ptr, "\nvRxTcp() handleGetConfigFile sending %d bytes \n",u32BytesCopied+2);
       fprintf (dbg_write_ptr, "\nvData 0x%x, 0x%x, 0x%x, 0x%x, 0x%x, ",
           *(pu8ADR3_FileBuf+4),*(pu8ADR3_FileBuf+5),*(pu8ADR3_FileBuf+6),*(pu8ADR3_FileBuf+7),*(pu8ADR3_FileBuf+8));
       fprintf (dbg_write_ptr, "0x%x, 0x%x, 0x%x, 0x%x, 0x%x \n",
           *(pu8ADR3_FileBuf+9),*(pu8ADR3_FileBuf+10),*(pu8ADR3_FileBuf+11),*(pu8ADR3_FileBuf+12),*(pu8ADR3_FileBuf+13));
       fclose(dbg_write_ptr);

       m_poSoundTool_TcpRxTx->SocketSend((tU16)u32BytesCopied+2,pu8ADR3_FileBuf+4);
       //sleep(2);
  #endif
     } //if(pBuffer != NULL)
   } //if(m_poSoundTool_TcpRxTx != NULL)

   // free the no more needed memory after it was send
   OSAL_DELETE [] pu8ADR3_FileBuf;
}


tVoid SoundToolClient::removeSizeInDelayAndEqu(tU8* pu8Data, tS32& s32Size) const
{
   tU16 u16Idx;
   tU16 u16SizeInFile = *pu8Data;
   u16SizeInFile = (tU16)(u16SizeInFile<<8);
   u16SizeInFile = (tU16)(u16SizeInFile + *(pu8Data+1));

   ETG_TRACE_USR4(("removeSizeInDelayAndEqu() s32Size=%d, u16SizeInFile=%d",
         s32Size,u16SizeInFile));

   for(u16Idx=0;u16Idx<s32Size;u16Idx++)
   {
      *(pu8Data+u16Idx) = *(pu8Data+u16Idx+2);
   }
   // clear the two bytes at the end
   *(pu8Data+u16Idx-1) = 0x00;
   *(pu8Data+u16Idx-2) = 0x00;
   s32Size = s32Size-2;
}
/********************************************************************************
 * sendGetConfigFile().
 *
 * This method will send a ConfigFile, given as parameters buffer and size, to Soundtool.
 * The sending is done with splitted messages because TTFis (or CSM) can not handle more than 239
 * bytes per message and the files are typically bigger. On the PC side the TTFisLib.dll will put
 * the pieces together and forward it as one single response to Soundtool.
 * See Soundtool documentation for the communication protocol.
 * The config file will be send to Soundtool with method sendGetConfigFile
 *
 * The splitted messages will use an own header that is removed by the TTFisLib.dll
 *
 * Header for first message (8bytes)
 * u8 MsgCode   // this is the same message code that was received. It is used to identify the message in TTFisLib.dll
 * u8 SplitFlag // this was just added to flag that the message is splitted and more pieces will come
 *              // finally it was not really needed
 * u16 MsgCounter // can be used to keep track for the messages, and is needed to identify the first message of a sequence
 * u32 Size       // Size of the overall data that is coming
 *
 * Header for all other message (4bytes)
 * same as above but without the overall u32 size, so just 4 bytes
 *******************************************************************************/
tVoid SoundToolClient::sendGetConfigFile(tU8 const* const pu8Data, tU16 u16Size)const
{
   ETG_TRACE_USR4(("sendGetConfigFile()"));
   // ToDo split it into pieces, we use the same format as for received splitted messages

   // byte array for one message
   tChar u8MsgBuffer[MSG_SIZE_REPLY+1];        // we might need a 0 termination????
   tU16 u16RemainingSize = u16Size;
   tU16 u16MsgCounter=MSG_FIRST_MSG;           // message counter will run from 1,2,..,n (0 is not used)
   tU8 u8HeaderOffset = MSG_HEADER_WITH_SIZE;  // header offset is 8 for 1st message and from 2nd message on it is 4
   tU16 u16ReadPos = 0;
   tU16 u16WritePtr=0;

   u8MsgBuffer[MSG_SIZE_REPLY] = 0; // do we need this 0 termination? (daw2hi: no, see et_vTraceBinary() in et_Trace.cpp)

   u8MsgBuffer[0] = pu8Data[0];        //Msg Code as defined in Soundtool interface
   u8MsgBuffer[1] = MSG_IS_SPLITTED;   //splitted

   // split loop. Here we split the file in pieces for sending
   // consider only 4 bytes header as remaining size.
   // ToDo: We have problems if a file is short enough to fit in one message. Split flag =0 for this case?
   while(u16RemainingSize >= (MSG_SIZE_REPLY-MSG_HEADER))
   {
      // add message counter
      u8MsgBuffer[2] = (tU8)((u16MsgCounter & 0xFF00)>>8); //MSB
      u8MsgBuffer[3] = (tU8)(u16MsgCounter & 0x00FF);      //LSB
      // First message counter contains an overall size information. The other side will use it to count
      // received and remaining bytes.
      if(u16MsgCounter == 1)
      {
         //add overall size
         u8MsgBuffer[4] = 0x00; // MSB only 16 Bit supported
         u8MsgBuffer[5] = 0x00; //     only 16 Bit supported
         u8MsgBuffer[6] = (tU8)((u16Size & 0xFF00)>>8);
         u8MsgBuffer[7] = (tU8)(u16Size & 0x00FF); //LSB
         u8HeaderOffset = MSG_HEADER_WITH_SIZE;
      }

      // loop to fill the data behind the header
      tU8 i;
      for(i=0; i<(MSG_SIZE_REPLY-u8HeaderOffset); i++)
      {
         u8MsgBuffer[u8HeaderOffset + i] = pu8Data[u16ReadPos+i];
         u16WritePtr++;
      }
      // keep track of the read position
      u16ReadPos = (tU16)(u16ReadPos + i);

      // Trace the message back to SoundTool via TTFis dll
      //ET_TRACE_BIN( TR_CLASS_SOUND_TOOL_COMMUNICATION, TR_LEVEL_FATAL, ET_EN_T8LIST _ MSG_SIZE_REPLY _ u8MsgBuffer _ ET_EN_DONE);
      et_vTraceBinary(TR_CLASS_SOUND_TOOL_COMMUNICATION, TR_LEVEL_FATAL, ET_EN_T8LIST, MSG_SIZE_REPLY, u8MsgBuffer, ET_EN_DONE);
      // this are the remaining bytes that we need to transfer
      u16RemainingSize = (tU16)(u16RemainingSize - (MSG_SIZE_REPLY-u8HeaderOffset));
      // we needed it only for first message, so set back to 4
      u8HeaderOffset = MSG_HEADER;
      u16MsgCounter++;


   }

   // last message if data left over
   if(u16RemainingSize > 0)
   {
      // add message counter
      u8MsgBuffer[2] = (tU8)((u16MsgCounter & 0xFF00)>>8); //MSB
      u8MsgBuffer[3] = (tU8)(u16MsgCounter & 0x00FF);      //LSB
      tU8 i;
      for(i=0; i<u16RemainingSize; i++)
      {
         u8MsgBuffer[u8HeaderOffset + i] = pu8Data[u16ReadPos+i];
         u16WritePtr++;
      }
      // Trace the message back to SoundTool
      //ET_TRACE_BIN(TR_CLASS_SOUND_TOOL_COMMUNICATION, TR_LEVEL_FATAL, ET_EN_T8LIST _ i + u8HeaderOffset _ u8MsgBuffer _ ET_EN_DONE);
      et_vTraceBinary(TR_CLASS_SOUND_TOOL_COMMUNICATION, TR_LEVEL_FATAL, ET_EN_T8LIST, i + u8HeaderOffset, u8MsgBuffer, ET_EN_DONE);
   }
   else
   {
      ETG_TRACE_USR4(("sendGetConfigFile() no more data"));
   }

   ETG_TRACE_USR4(("sendGetConfigFile() done with u16WritePtr=%d",u16WritePtr));

}



/********************************************************************************
 * getTotalFileLength().
 *
 * Short routine to combine a u32 out of byte array with MSB first
 *******************************************************************************/
tU32 SoundToolClient::getTotalFileLength(tPCUChar pcu8Data)const
{
    tU32 u32Length;
    u32Length = *pcu8Data;                //MSB
    u32Length = u32Length<<8;
    u32Length = u32Length+*(pcu8Data+1);
    u32Length = u32Length<<8;
    u32Length = u32Length+*(pcu8Data+2);
    u32Length = u32Length<<8;
    u32Length = u32Length+*(pcu8Data+3); //LSB

    return u32Length;
}


/********************************************************************************
 * handleSetConfigFile().
 *
 * Here we combine the splitted messages from Soundtool (TTFis.dll) to one file again
 * ToDo: What if transfer gets interrupted? Any timeout to reset receive mode??
 * ToDo: CHeck if we can make this cleaner??
 *******************************************************************************/
tVoid SoundToolClient::handleSetConfigFile(tU32 size, tPCUChar pcu8Data)const
{
   // Leave this as static or change them to members?

   static tU32 u32TotalLength = 0;
   static tU32 u32RemainingLength = 0;
   static tU32 u32WritePos=0;
   static tU8 * pcu8File = OSAL_NULL;

   static tU16 u16NextExpectedMsgIndex=1;
   static tU16 u16MsgCounter=1;

   tU16 u16MsgIndex = 0;
   ETG_TRACE_USR4(("handleSetConfigFile"));

   // we need to check if the file is splitted (this is usually the case)

   // if splitted
   if(pcu8Data[2] == MSG_IS_SPLITTED)
   {
      //get the MsgIndex
      u16MsgIndex = (tU16)((*(pcu8Data+3))<<8); //MSB
      //u16MsgIndex = u16MsgIndex << 8; //lint
      u16MsgIndex = (tU16)(u16MsgIndex + *(pcu8Data+4)); //LSB

      if(u16MsgIndex != u16NextExpectedMsgIndex)
      {
         ETG_TRACE_ERR(("handleSetConfigFile wrong Msg Index exp=%x, rec=%x",u16NextExpectedMsgIndex, u16MsgIndex));
      }
      u16NextExpectedMsgIndex++;

      // if first Msg, get the overall size (32Bit)
      if(u16MsgIndex == MSG_FIRST_MSG)
      {
       // reset all static variables also when first message arrives again (needed if a transfer was not terminated)
         u32TotalLength = 0;
         u32RemainingLength = 0;
         u32WritePos=0;
         u16MsgCounter=1;
         u16NextExpectedMsgIndex=1;


         u32TotalLength = getTotalFileLength(pcu8Data+5);

         u32RemainingLength = u32TotalLength;
         //Allocate for the data
         if(pcu8File == OSAL_NULL)
         {
            // if not yet allocated
            pcu8File = OSAL_NEW tU8[u32TotalLength];
            if(pcu8File == OSAL_NULL)
            {
              ETG_TRACE_FATAL(("Out of memory OSAL_NEW returned NULL"));
              return;
            }
         }
    (void)OSAL_pvMemoryCopy(pcu8File, pcu8Data+9, size-9);
    u32RemainingLength = u32RemainingLength - (size-9);
    u32WritePos = u32WritePos + (size-9);
    u16MsgCounter++;

      }
      else
      {
    if(u32RemainingLength >= size-5)
    {
       // no size inside if not first Msg
       (void)OSAL_pvMemoryCopy(pcu8File+u32WritePos, pcu8Data+5, size-5);
       u32WritePos = u32WritePos + (size-5);
       u32RemainingLength = u32RemainingLength - (size-5);
       u16MsgCounter++;
    }
    else
    {
       ETG_TRACE_USR4(("handleSetConfigFile size left = %x",u32RemainingLength));
    }
      }

      // If there was something to do and has been done then finish it
      if( (u32TotalLength != 0) && (u32RemainingLength == 0) )
      {
         //we got it all
         ETG_TRACE_USR4(("handleSetConfigFile receive all Data"));
         // handle the file
         // ToDo
         tU8 u8Status;
         tS32 s32Res;
         tU16 u16FileID = (tU16)(pcu8File[3]<<8);    //MSB
         //u16FileID = u16FileID<<8; // lint
         u16FileID = (tU16)(u16FileID + pcu8File[2]); //LSB

         // header is 6 Byte
         // u8MsgCode
         // u8MuteFlag
         // u16FileID
         // Length
     s32Res = saveConfigFile(u16FileID);

         ETG_TRACE_USR4(("handleSetConfigFile s32Res=%d u32TotalLength=%d",(tU32)s32Res, u32TotalLength));
         //if(s32Res == u32TotalLength-6)
         u8Status = SOUNDTOOL_SUCCESS; //success
         // we should send back a response message for the Set
         sendSetConfigFileStatus(u8Status);
         // end

         // We need to transmit the received Config File to ADR, but without save command
         // save will happen during next startup (different CRC in DP)
         //vd_adr3_config::vSendConfigFile(tU8 u8FileID, tU8* pu8FileBuf, tU32 u32Length)
         ID_NotifySendSoundConfigFile oSendSoundConfigFile((tU8)(u16FileID & 0x00FF), (tU8*)(pcu8File + 6), u32TotalLength-6);
         InternalCommunicationAdapter::getInstance()->POMessages->DeliverMsg(&oSendSoundConfigFile);

         // Reset all static variables
         u32TotalLength = 0;
         u32RemainingLength = 0;
         u32WritePos=0;
         u16MsgCounter=1;
         u16NextExpectedMsgIndex=1;

         // and free occupied memory
         OSAL_DELETE[] pcu8File;
         pcu8File = OSAL_NULL;
      }

      ETG_TRACE_USR4(("handleSetConfigFile file length = %x, (MsgCount=%d)", u32TotalLength,u16MsgCounter)); //to please LINT

   } //if(splitted)

}

/********************************************************************************
 * sendSetConfigFileStatus().
 *
 * Short routine to send a status back for received SetConfigFile
 *******************************************************************************/
tVoid SoundToolClient::sendSetConfigFileStatus(tU8 u8Status)const
{
   tChar u8MsgBuffer[3];
   u8MsgBuffer[0] = TRC::SetConfigFile; // MSg code
   u8MsgBuffer[1] = u8Status;
   u8MsgBuffer[2] = 0x00;
   // no need to check for trace is active, we use FATAL level and if this code is executed, we have for sure someone
   // who is listening to this (Soundtool)
   et_vTraceBinary(TR_CLASS_SOUND_TOOL_COMMUNICATION, TR_LEVEL_FATAL, ET_EN_T8LIST, 3, u8MsgBuffer, ET_EN_DONE);
   if(m_poSoundTool_TcpRxTx != NULL)
   {
#ifdef DEBUG_PRINT
       FILE* dbg_write_ptr = fopen("/var/opt/bosch/dynamic/ffs/Soundtool.log","a");
       fprintf (dbg_write_ptr, "supress sendSetConfigFileStatus SendStatus opCode= 0x%x \n",TRC::SetConfigFile);
       fclose(dbg_write_ptr);
#endif
     //m_poSoundTool_TcpRxTx->SocketSend(sizeof(u8MsgBuffer),(tU8*)u8MsgBuffer);

#if 0
     tU8 u8SendBuf[9];
     memset(u8SendBuf,0,6);
     memcpy(u8SendBuf+6, u8MsgBuffer,sizeof(u8MsgBuffer));
        m_poSoundTool_TcpRxTx->SocketSend(sizeof(u8SendBuf),(tU8*)u8SendBuf);
#endif
   }
}

#if 0
/********************************************************************************
 * sendSetPropertyStatus().
 *
 * Short routine to send a status back for received SetProperty
 *******************************************************************************/
tVoid SoundToolClient::sendSetPropertyStatus(tU8 u8Status)const
{
   tChar u8MsgBuffer[3];
   u8MsgBuffer[0] = TRC::SetProperty; // MSg code
   u8MsgBuffer[1] = u8Status;
   u8MsgBuffer[2] = 0x00;
   // no need to check for trace is active, we use FATAL level and if this code is executed, we have for sure someone
   // who is listening to this (Soundtool)
   et_vTraceBinary(TR_CLASS_SOUND_TOOL_COMMUNICATION, TR_LEVEL_FATAL, ET_EN_T8LIST, 3, u8MsgBuffer, ET_EN_DONE);
}


tVoid SoundToolClient::sendGetProperty(tU8 u8PropertyID, tU8* pcu8Data, trPropertyData const* const pProperty)const
{
   // ToDo: Adapt size to what we need, maybe better with Alloc and Free
   // [12] needs 13 plus 1 Byte length for u8PropertyName then we have 14
   // plus the two strings.
   // 14 + pProperty->u8LengthUnitName + pProperty->pu8PropertyName
   tChar u8MsgBuffer[200];
   tU8 u8Offset;
   (tVoid)pcu8Data;

   ETG_TRACE_USR4(("sendGetProperty ID=%d, min=%d, max=%d, val=%d",
         (tU16)u8PropertyID,pProperty->s16MinVal, pProperty->s16MaxVal, pProperty->s16Value));

   ETG_TRACE_USR4(("sendGetProperty Name=%s",pProperty->pu8PropertyName));
   ETG_TRACE_USR4(("sendGetProperty Unit=%s",pProperty->pu8UnitName));


   u8MsgBuffer[0] = TRC::GetProperty;
   u8MsgBuffer[1] = SOUNDTOOL_SUCCESS;
   u8MsgBuffer[2] = u8PropertyID;

   u8MsgBuffer[3] = (tU8)((pProperty->s16MinVal) & 0x00FF);
   u8MsgBuffer[4] = ((pProperty->s16MinVal) & 0xFF00)>>8;

   u8MsgBuffer[5] = (tU8)((pProperty->s16MaxVal) & 0x00FF);
   u8MsgBuffer[6] = ((pProperty->s16MaxVal) & 0xFF00)>>8;

   u8MsgBuffer[7] = (tU8)((pProperty->u16Resolution) & 0x00FF);
   u8MsgBuffer[8] = ((pProperty->u16Resolution) & 0xFF00)>>8;

   u8MsgBuffer[9] = (tU8)((pProperty->s16Value) & 0x00FF);
   u8MsgBuffer[10] = ((pProperty->s16Value) & 0xFF00)>>8;

   u8MsgBuffer[11] = pProperty->u8LengthUnitName;
   (void)OSAL_pvMemoryCopy(&u8MsgBuffer[12], pProperty->pu8UnitName, pProperty->u8LengthUnitName);

   u8Offset = 12+pProperty->u8LengthUnitName;
   u8MsgBuffer[u8Offset] = pProperty->u8LengthPropertyName;
   u8Offset = u8Offset + 1;
   (void)OSAL_pvMemoryCopy(&u8MsgBuffer[u8Offset], pProperty->pu8PropertyName, pProperty->u8LengthPropertyName);

   u8Offset = u8Offset + pProperty->u8LengthPropertyName;

   et_vTraceBinary(TR_CLASS_SOUND_TOOL_COMMUNICATION, TR_LEVEL_FATAL, ET_EN_T8LIST, u8Offset, u8MsgBuffer, ET_EN_DONE);
}
#endif


tVoid SoundToolClient::setSoundToolFlag(tU8 u8Command) const
{
   dp_tclAudioManagerDPSoundToolFlag oSoundToolFlag;
   tU8 u8SoundToolFlag;
   //tS32 s32Res; //lint

   //s32Res = oSoundToolFlag.s32GetData(u8SoundToolFlag);
   (void)oSoundToolFlag.s32GetData(u8SoundToolFlag); //lint
   ETG_TRACE_USR4(("setSoundToolFlag() Flag is %x",u8SoundToolFlag));
   // Only if it was not yet set to avoid unnecessary write cycles
   if(u8SoundToolFlag == 0)
   {
      switch(u8Command)
      {
         // For all Set or Save commands we set the SoundToolFlag if
         case TRC::SetConfigFile:
         case TRC::SaveConfigFile:
         case TRC::SetProperty:
         {
            u8SoundToolFlag = 1;
            //s32Res = oSoundToolFlag.s32SetData(u8SoundToolFlag);
            (void)oSoundToolFlag.s32SetData(u8SoundToolFlag); //lint
            ETG_TRACE_USR4(("setSoundToolFlag() Flag set to %x",u8SoundToolFlag));
            break;
         }

         default:
            break;
      }
   }
}
/********************************************************************************
 * vTraceRx(tPCUChar pcu8Data)
 *******************************************************************************/
tVoid SoundToolClient::vTraceRx(tU32 size, tPCUChar pcu8Data)
{
   ETG_TRACE_USR4(("vTraceRx() entered."));

   if(pcu8Data == NULL) return;
   // pu8Data[0] == tenTrcTrcClassName
   // pu8Data[1] == tenTrcFuncInput  SetAdrData
   // pu8Data[2] == the 1. parameter
   // pu8Data[3] == the 2. parameter
   // rest is defined here

   //tU8 u8PropertyID;
   //PropertyIF_Base* pPropertyOwner;

   if (size > 2)
   {
      setSoundToolFlag(pcu8Data[1]);

      switch ( pcu8Data[1] )
      {
         case TRC::SetConfigFile:   //0x02
         {
            ETG_TRACE_USR4(("Received SetConfigFile"));
            handleSetConfigFile(size, pcu8Data);
            break;
         }
         case TRC::GetConfigFile:   //0x03
         {
            ETG_TRACE_USR4(("Received GetConfigFile"));
            handleGetConfigFile(size, pcu8Data);
            break;
         }
         case TRC::SaveConfigFile:  //0x04
         {
            ETG_TRACE_USR4(("Received SaveConfigFile (not implemented)"));
            break;
         }
#if 0
         case TRC::SetProperty:  //0x1E
         {
            tBool bRet;
            ETG_TRACE_USR4(("Received SetProperty"));

            // Now find the owner of this PropertyID
            u8PropertyID = pcu8Data[2];
            pPropertyOwner = getPropertyOwner(u8PropertyID);
            if(pPropertyOwner != NULL) {
               bRet = pPropertyOwner->SetProperty(u8PropertyID, const_cast<tU8*>(pcu8Data));
               if(bRet == true)
               {
                  sendSetPropertyStatus(SOUNDTOOL_SUCCESS);
               }
               else
               {
                  sendSetPropertyStatus(SOUNDTOOL_FAILURE);
               }
            }
            break;
         }
         case TRC::GetProperty:  //0x1F
         {
             trPropertyData pProperty;
            ETG_TRACE_USR4(("Received GetProperty"));

            // Now find the owner of this PropertyID
            u8PropertyID = pcu8Data[2];

            pPropertyOwner = getPropertyOwner(u8PropertyID);
            if(pPropertyOwner != NULL) {
               // Fill the data structure
                  (void)pPropertyOwner->GetProperty(u8PropertyID, const_cast<tU8*>(pcu8Data), &pProperty);
               ETG_TRACE_USR4(("Property = min=%d,max=%d,val=%d",
                     pProperty.s16MinVal,pProperty.s16MaxVal,pProperty.s16Value));
               // Put it into a tU8 buffer and send it
                   sendGetProperty(u8PropertyID, const_cast<tU8*>(pcu8Data), &pProperty);
                }
            break;
         }
#endif
         case TRC::GetPropertyListLength: //0x1D
         {
            handleGetPropertyListLength(size, const_cast<tU8*>(pcu8Data));
            break;
         }
         default:
         {
            ETG_TRACE_ERR(("Error in vTraceRx default case for unknown message"));
            break;
         }
      }
   }

   return;
}

#if 0
/********************************************************************************
 * getPropertyOwner().
 *

 *******************************************************************************/
PropertyIF_Base* SoundToolClient::getPropertyOwner(tU8 u8PropertyID)
{
   // go through the vector and find the owner, if not found return NULL
   PropertyIF_Base* pPropertyIF_Base = NULL;

   tU32 u32size = m_PropertyIF_BaseList.size();

   if(u32size >= u8PropertyID)
   {
      pPropertyIF_Base = m_PropertyIF_BaseList.at(u8PropertyID);
      ETG_TRACE_USR4(("SoundToolClient found Property Owner %x",pPropertyIF_Base));
   }

   return pPropertyIF_Base;

}


/********************************************************************************
 * bRegisterPropertyOwner().
 *

 *******************************************************************************/
// this is static
tBool SoundToolClient::bRegisterPropertyOwner(PropertyIF_Base* pPropertyIF_Base)
{
   tBool bRes = false;
   if(poSoundToolClient != NULL)
   {
    ETG_TRACE_USR4(("vRegisterPropertyOwner ID %d, pOwner = %x",m_PropertyIF_BaseList.size(),
            pPropertyIF_Base));
      m_PropertyIF_BaseList.push_back(pPropertyIF_Base);
      bRes = true;
   }
   else
   {
      ETG_TRACE_USR4(("SoundToolClient::bRegisterPropertyOwner call before init"));
   }
   ETG_TRACE_USR4(("vRegisterPropertyOwner size=%x",m_PropertyIF_BaseList.size()));
   return bRes;

}
#endif

